from ......Internal.Core import Core
from ......Internal.CommandsGroup import CommandsGroup
from ......Internal import Conversions


# noinspection PyPep8Naming,PyAttributeOutsideInit,SpellCheckingInspection
class Delay:
	"""Delay commands group definition. 12 total commands, 1 Sub-groups, 6 group commands"""

	def __init__(self, core: Core, parent):
		self._core = core
		self._cmd_group = CommandsGroup("delay", core, parent)

	@property
	def tsp(self):
		"""tsp commands group. 1 Sub-classes, 5 commands."""
		if not hasattr(self, '_tsp'):
			from .Tsp import Tsp
			self._tsp = Tsp(self._core, self._cmd_group)
		return self._tsp

	def get_deviation(self) -> float:
		"""SCPI: [SOURce<HW>]:BB:T2DVb:DELay:DEViation \n
		Snippet: value: float = driver.source.bb.t2Dvb.delay.get_deviation() \n
		Sets the maximum permissible delay. \n
			:return: max_dev_time: float Range: 1E-6 to 500E-6
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:T2DVb:DELay:DEViation?')
		return Conversions.str_to_float(response)

	def set_deviation(self, max_dev_time: float) -> None:
		"""SCPI: [SOURce<HW>]:BB:T2DVb:DELay:DEViation \n
		Snippet: driver.source.bb.t2Dvb.delay.set_deviation(max_dev_time = 1.0) \n
		Sets the maximum permissible delay. \n
			:param max_dev_time: float Range: 1E-6 to 500E-6
		"""
		param = Conversions.decimal_value_to_str(max_dev_time)
		self._core.io.write(f'SOURce<HwInstance>:BB:T2DVb:DELay:DEViation {param}')

	def get_dynamic(self) -> float:
		"""SCPI: [SOURce<HW>]:BB:T2DVb:DELay:DYNamic \n
		Snippet: value: float = driver.source.bb.t2Dvb.delay.get_dynamic() \n
		Queries the transmission delay currently generated by the SFN delay . \n
			:return: dyn_delay: float Range: 0 to 8.0
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:T2DVb:DELay:DYNamic?')
		return Conversions.str_to_float(response)

	def get_mute_p_1(self) -> bool:
		"""SCPI: [SOURce<HW>]:BB:T2DVb:DELay:MUTep1 \n
		Snippet: value: bool = driver.source.bb.t2Dvb.delay.get_mute_p_1() \n
		Activates muting the P1 symbol of the first T2 frame in a super frame. To mute the P1 symbol, the symbol is set to zero. \n
			:return: mute_p_1: 0| 1| OFF| ON
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:T2DVb:DELay:MUTep1?')
		return Conversions.str_to_bool(response)

	def set_mute_p_1(self, mute_p_1: bool) -> None:
		"""SCPI: [SOURce<HW>]:BB:T2DVb:DELay:MUTep1 \n
		Snippet: driver.source.bb.t2Dvb.delay.set_mute_p_1(mute_p_1 = False) \n
		Activates muting the P1 symbol of the first T2 frame in a super frame. To mute the P1 symbol, the symbol is set to zero. \n
			:param mute_p_1: 0| 1| OFF| ON
		"""
		param = Conversions.bool_to_str(mute_p_1)
		self._core.io.write(f'SOURce<HwInstance>:BB:T2DVb:DELay:MUTep1 {param}')

	def get_process(self) -> float:
		"""SCPI: [SOURce<HW>]:BB:T2DVb:DELay:PROCess \n
		Snippet: value: float = driver.source.bb.t2Dvb.delay.get_process() \n
		Queries the delay from the modulator input up to the SFN delay . \n
			:return: proc_delay: float Range: 0 to 4.0
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:T2DVb:DELay:PROCess?')
		return Conversions.str_to_float(response)

	def get_static(self) -> float:
		"""SCPI: [SOURce<HW>]:BB:T2DVb:DELay:STATic \n
		Snippet: value: float = driver.source.bb.t2Dvb.delay.get_static() \n
		Sets the delay to shift the time of transmission positively or negatively. \n
			:return: static_delay: float Range: -4.0 to 4.0
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:T2DVb:DELay:STATic?')
		return Conversions.str_to_float(response)

	def set_static(self, static_delay: float) -> None:
		"""SCPI: [SOURce<HW>]:BB:T2DVb:DELay:STATic \n
		Snippet: driver.source.bb.t2Dvb.delay.set_static(static_delay = 1.0) \n
		Sets the delay to shift the time of transmission positively or negatively. \n
			:param static_delay: float Range: -4.0 to 4.0
		"""
		param = Conversions.decimal_value_to_str(static_delay)
		self._core.io.write(f'SOURce<HwInstance>:BB:T2DVb:DELay:STATic {param}')

	def get_total(self) -> float:
		"""SCPI: [SOURce<HW>]:BB:T2DVb:DELay:TOTal \n
		Snippet: value: float = driver.source.bb.t2Dvb.delay.get_total() \n
		Queries the sum of processing delay and dynamic delay. \n
			:return: total_delay: float Range: -4.0 to 16.0
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:T2DVb:DELay:TOTal?')
		return Conversions.str_to_float(response)

	def clone(self) -> 'Delay':
		"""Clones the group by creating new object from it and its whole existing sub-groups
		Also copies all the existing default Repeated Capabilities setting,
		which you can change independently without affecting the original group"""
		new_group = Delay(self._core, self._cmd_group.parent)
		self._cmd_group.synchronize_repcaps(new_group)
		return new_group
