from ......Internal.Core import Core
from ......Internal.CommandsGroup import CommandsGroup
from ......Internal import Conversions
from ...... import enums


# noinspection PyPep8Naming,PyAttributeOutsideInit,SpellCheckingInspection
class Delay:
	"""Delay commands group definition. 14 total commands, 2 Sub-groups, 9 group commands"""

	def __init__(self, core: Core, parent):
		self._core = core
		self._cmd_group = CommandsGroup("delay", core, parent)

	@property
	def mute(self):
		"""mute commands group. 0 Sub-classes, 1 commands."""
		if not hasattr(self, '_mute'):
			from .Mute import Mute
			self._mute = Mute(self._core, self._cmd_group)
		return self._mute

	@property
	def tsp(self):
		"""tsp commands group. 0 Sub-classes, 4 commands."""
		if not hasattr(self, '_tsp'):
			from .Tsp import Tsp
			self._tsp = Tsp(self._core, self._cmd_group)
		return self._tsp

	def get_deviation(self) -> float:
		"""SCPI: [SOURce<HW>]:BB:A3TSc:DELay:DEViation \n
		Snippet: value: float = driver.source.bb.a3Tsc.delay.get_deviation() \n
		Sets the maximum permissible delay. \n
			:return: deviation: float Range: 1.0 to 500.0, Unit: µs
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:A3TSc:DELay:DEViation?')
		return Conversions.str_to_float(response)

	def set_deviation(self, deviation: float) -> None:
		"""SCPI: [SOURce<HW>]:BB:A3TSc:DELay:DEViation \n
		Snippet: driver.source.bb.a3Tsc.delay.set_deviation(deviation = 1.0) \n
		Sets the maximum permissible delay. \n
			:param deviation: float Range: 1.0 to 500.0, Unit: µs
		"""
		param = Conversions.decimal_value_to_str(deviation)
		self._core.io.write(f'SOURce<HwInstance>:BB:A3TSc:DELay:DEViation {param}')

	def get_dispatch(self) -> float:
		"""SCPI: [SOURce<HW>]:BB:A3TSc:DELay:DISPatch \n
		Snippet: value: float = driver.source.bb.a3Tsc.delay.get_dispatch() \n
		Queries the time taken for the signal to travel from the playout center to the transmitter antenna for real transmission. \n
			:return: dispatch: float Range: -1000000.0 to 6000000.0, Unit: µs
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:A3TSc:DELay:DISPatch?')
		return Conversions.str_to_float(response)

	def get_dynamic(self) -> float:
		"""SCPI: [SOURce<HW>]:BB:A3TSc:DELay:DYNamic \n
		Snippet: value: float = driver.source.bb.a3Tsc.delay.get_dynamic() \n
		Queries the transmission delay currently generated by the SFN delay . \n
			:return: dynamic: float Range: -16000000.0 to 6000000.0, Unit: µs
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:A3TSc:DELay:DYNamic?')
		return Conversions.str_to_float(response)

	def get_maximum(self) -> float:
		"""SCPI: [SOURce<HW>]:BB:A3TSc:DELay:MAXImum \n
		Snippet: value: float = driver.source.bb.a3Tsc.delay.get_maximum() \n
		Queries the time taken for the signal to travel from the playout center to the transmitter antenna for regular
		transmission. \n
			:return: maximum_delay: float Range: 0.0 to 5000000.0, Unit: µs
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:A3TSc:DELay:MAXImum?')
		return Conversions.str_to_float(response)

	def get_network(self) -> float:
		"""SCPI: [SOURce<HW>]:BB:A3TSc:DELay:NETWork \n
		Snippet: value: float = driver.source.bb.a3Tsc.delay.get_network() \n
		Queries the time taken for the signal to travel from the playout center to the input of the signal processing. \n
			:return: network: float Range: 0.0 to 5000000.0, Unit: µs
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:A3TSc:DELay:NETWork?')
		return Conversions.str_to_float(response)

	def get_process(self) -> float:
		"""SCPI: [SOURce<HW>]:BB:A3TSc:DELay:PROCess \n
		Snippet: value: float = driver.source.bb.a3Tsc.delay.get_process() \n
		Queries the delay from the modulator input up to the SFN delay . \n
			:return: process: float Range: 0.0 to 10000000.0, Unit: µs
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:A3TSc:DELay:PROCess?')
		return Conversions.str_to_float(response)

	# noinspection PyTypeChecker
	def get_sfn_mode(self) -> enums.SfnMode:
		"""SCPI: [SOURce<HW>]:BB:A3TSc:DELay:SFNMode \n
		Snippet: value: enums.SfnMode = driver.source.bb.a3Tsc.delay.get_sfn_mode() \n
		Sets the SFN timestamp operation mode, that is fixed to relative timestamp mode. \n
			:return: sfn_mode: ABSolute| RELative
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:A3TSc:DELay:SFNMode?')
		return Conversions.str_to_scalar_enum(response, enums.SfnMode)

	def set_sfn_mode(self, sfn_mode: enums.SfnMode) -> None:
		"""SCPI: [SOURce<HW>]:BB:A3TSc:DELay:SFNMode \n
		Snippet: driver.source.bb.a3Tsc.delay.set_sfn_mode(sfn_mode = enums.SfnMode.ABSolute) \n
		Sets the SFN timestamp operation mode, that is fixed to relative timestamp mode. \n
			:param sfn_mode: ABSolute| RELative
		"""
		param = Conversions.enum_scalar_to_str(sfn_mode, enums.SfnMode)
		self._core.io.write(f'SOURce<HwInstance>:BB:A3TSc:DELay:SFNMode {param}')

	def get_static(self) -> float:
		"""SCPI: [SOURce<HW>]:BB:A3TSc:DELay:STATic \n
		Snippet: value: float = driver.source.bb.a3Tsc.delay.get_static() \n
		Sets the delay to shift the time of transmission positively or negatively. \n
			:return: static: float Range: -1000000.0 to 1000000.0, Unit: µs
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:A3TSc:DELay:STATic?')
		return Conversions.str_to_float(response)

	def set_static(self, static: float) -> None:
		"""SCPI: [SOURce<HW>]:BB:A3TSc:DELay:STATic \n
		Snippet: driver.source.bb.a3Tsc.delay.set_static(static = 1.0) \n
		Sets the delay to shift the time of transmission positively or negatively. \n
			:param static: float Range: -1000000.0 to 1000000.0, Unit: µs
		"""
		param = Conversions.decimal_value_to_str(static)
		self._core.io.write(f'SOURce<HwInstance>:BB:A3TSc:DELay:STATic {param}')

	def get_total(self) -> float:
		"""SCPI: [SOURce<HW>]:BB:A3TSc:DELay:TOTal \n
		Snippet: value: float = driver.source.bb.a3Tsc.delay.get_total() \n
		Queries the sum of processing delay and dynamic delay. \n
			:return: total_delay: float Range: -6000000.0 to 6000000.0, Unit: µs
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:A3TSc:DELay:TOTal?')
		return Conversions.str_to_float(response)

	def clone(self) -> 'Delay':
		"""Clones the group by creating new object from it and its whole existing sub-groups
		Also copies all the existing default Repeated Capabilities setting,
		which you can change independently without affecting the original group"""
		new_group = Delay(self._core, self._cmd_group.parent)
		self._cmd_group.synchronize_repcaps(new_group)
		return new_group
