"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const cdk = require("aws-cdk-lib");
const kinesis = require("aws-cdk-lib/aws-kinesis");
const iam = require("aws-cdk-lib/aws-iam");
const assertions_1 = require("aws-cdk-lib/assertions");
const iotTopicRuleProps = {
    topicRulePayload: {
        description: "sends data to kinesis streams",
        sql: "SELECT * FROM 'solutions/constructs'",
        actions: []
    }
};
test('check iot topic rule properties', () => {
    const stack = new cdk.Stack();
    const props = {
        iotTopicRuleProps
    };
    const construct = new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::IoT::TopicRule', {
        TopicRulePayload: {
            Actions: [
                {
                    Kinesis: {
                        StreamName: {
                            Ref: "testiotkinesisstreamsKinesisStreamA6FEF4AF"
                        },
                        RoleArn: {
                            "Fn::GetAtt": [
                                "testiotkinesisstreamsIotActionsRoleAE74F764",
                                "Arn"
                            ]
                        }
                    }
                }
            ],
            Description: "sends data to kinesis streams",
            RuleDisabled: false,
            Sql: "SELECT * FROM 'solutions/constructs'"
        }
    });
    template.hasResourceProperties('AWS::Kinesis::Stream', {
        ShardCount: 1,
        RetentionPeriodHours: 24,
        StreamEncryption: {
            EncryptionType: "KMS",
            KeyId: "alias/aws/kinesis"
        }
    });
    expect(construct.iotTopicRule).toBeDefined();
    expect(construct.iotActionsRole).toBeDefined();
    expect(construct.cloudwatchAlarms).toBeDefined();
    expect(construct.kinesisStream).toBeDefined();
});
test('check existing kinesis stream', () => {
    const stack = new cdk.Stack();
    const existingKinesisStream = new kinesis.Stream(stack, `existing-stream`, {
        shardCount: 2,
        streamName: 'testexistingstream',
        retentionPeriod: cdk.Duration.hours(25)
    });
    const props = {
        iotTopicRuleProps,
        existingStreamObj: existingKinesisStream,
        createCloudWatchAlarms: false
    };
    const construct = new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Kinesis::Stream', {
        ShardCount: 2,
        RetentionPeriodHours: 25,
        Name: 'testexistingstream'
    });
    template.resourceCountIs('AWS::CloudWatch::Alarm', 0);
    expect(construct.iotTopicRule).toBeDefined();
    expect(construct.iotActionsRole).toBeDefined();
    expect(construct.cloudwatchAlarms).toBeUndefined();
    expect(construct.kinesisStream).toBeDefined();
});
test('check new kinesis stream with override props', () => {
    const stack = new cdk.Stack();
    const props = {
        iotTopicRuleProps,
        createCloudWatchAlarms: false,
        kinesisStreamProps: {
            shardCount: 5,
            streamName: 'testnewstream',
            retentionPeriod: cdk.Duration.hours(30)
        }
    };
    const construct = new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Kinesis::Stream', {
        ShardCount: 5,
        RetentionPeriodHours: 30,
        Name: 'testnewstream'
    });
    const kinesisStream = construct.kinesisStream;
    expect(kinesisStream).toBeDefined();
});
test('check existing action in topic rule props', () => {
    const stack = new cdk.Stack();
    const existingKinesisStream = new kinesis.Stream(stack, `existing-stream`, {});
    const existingIamRole = new iam.Role(stack, 'existingRole', {
        assumedBy: new iam.ServicePrincipal('iot.amazonaws.com')
    });
    const existingKinesisActionProperty = {
        kinesis: {
            streamName: existingKinesisStream.streamName,
            roleArn: existingIamRole.roleArn
        }
    };
    const props = {
        iotTopicRuleProps: {
            topicRulePayload: {
                sql: "SELECT * FROM 'solutions/constructs'",
                actions: [existingKinesisActionProperty],
                ruleDisabled: true
            },
            ruleName: 'testiotrulename'
        },
        createCloudWatchAlarms: false,
        kinesisStreamProps: {
            shardCount: 5,
            streamName: 'testnewstream',
            retentionPeriod: cdk.Duration.hours(30)
        }
    };
    new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams', props);
    // Check multiple actions in the Topic Rule
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::IoT::TopicRule', {
        TopicRulePayload: {
            Actions: [
                {
                    Kinesis: {
                        StreamName: {
                            Ref: "testiotkinesisstreamsKinesisStreamA6FEF4AF"
                        },
                        RoleArn: {
                            "Fn::GetAtt": [
                                "testiotkinesisstreamsIotActionsRoleAE74F764",
                                "Arn"
                            ]
                        }
                    }
                },
                {
                    Kinesis: {
                        RoleArn: {
                            "Fn::GetAtt": [
                                "existingRole3E995BBA",
                                "Arn"
                            ]
                        },
                        StreamName: {
                            Ref: "existingstream0A902451"
                        }
                    }
                }
            ],
            RuleDisabled: true,
            Sql: "SELECT * FROM 'solutions/constructs'"
        },
        RuleName: "testiotrulename"
    });
    template.hasResourceProperties('AWS::Kinesis::Stream', {
        ShardCount: 5,
        RetentionPeriodHours: 30,
        Name: 'testnewstream'
    });
    template.resourceCountIs('AWS::Kinesis::Stream', 2);
});
test('check name confict', () => {
    const stack = new cdk.Stack();
    const props = {
        iotTopicRuleProps
    };
    new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams1', props);
    new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams2', props);
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::Kinesis::Stream', 2);
});
test('check construct chaining', () => {
    const stack = new cdk.Stack();
    const props1 = {
        iotTopicRuleProps
    };
    const construct = new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams1', props1);
    const props2 = {
        iotTopicRuleProps,
        existingStreamObj: construct.kinesisStream
    };
    new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams2', props2);
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::Kinesis::Stream', 1);
});
test('check error when stream props and existing stream is supplied', () => {
    const stack = new cdk.Stack();
    const existingKinesisStream = new kinesis.Stream(stack, `existing-stream`, {});
    const props = {
        iotTopicRuleProps,
        existingStreamObj: existingKinesisStream,
        kinesisStreamProps: {}
    };
    const app = () => {
        new lib_1.IotToKinesisStreams(stack, 'test-iot-kinesisstreams', props);
    };
    expect(app).toThrowError();
});
//# sourceMappingURL=data:application/json;base64,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