"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Expires = exports.StorageClass = exports.ServerSideEncryption = exports.CacheControl = exports.BucketDeployment = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const efs = require("@aws-cdk/aws-efs");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const lambda_layer_awscli_1 = require("@aws-cdk/lambda-layer-awscli");
const case_1 = require("case");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("@aws-cdk/core");
// tag key has a limit of 128 characters
const CUSTOM_RESOURCE_OWNER_TAG = 'aws-cdk:cr-owned';
/**
 * `BucketDeployment` populates an S3 bucket with the contents of .zip files from
 * other S3 buckets or from local disk
 */
class BucketDeployment extends core_1.Construct {
    constructor(scope, id, props) {
        var _d, _e;
        super(scope, id);
        this.requestDestinationArn = false;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_s3_deployment_BucketDeploymentProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        if (props.distributionPaths) {
            if (!props.distribution) {
                throw new Error('Distribution must be specified if distribution paths are specified');
            }
            if (!cdk.Token.isUnresolved(props.distributionPaths)) {
                if (!props.distributionPaths.every(distributionPath => cdk.Token.isUnresolved(distributionPath) || distributionPath.startsWith('/'))) {
                    throw new Error('Distribution paths must start with "/"');
                }
            }
        }
        if (props.useEfs && !props.vpc) {
            throw new Error('Vpc must be specified if useEfs is set');
        }
        const accessPointPath = '/lambda';
        let accessPoint;
        if (props.useEfs && props.vpc) {
            const accessMode = '0777';
            const fileSystem = this.getOrCreateEfsFileSystem(scope, {
                vpc: props.vpc,
                removalPolicy: cdk.RemovalPolicy.DESTROY,
            });
            accessPoint = fileSystem.addAccessPoint('AccessPoint', {
                path: accessPointPath,
                createAcl: {
                    ownerUid: '1001',
                    ownerGid: '1001',
                    permissions: accessMode,
                },
                posixUser: {
                    uid: '1001',
                    gid: '1001',
                },
            });
            accessPoint.node.addDependency(fileSystem.mountTargetsAvailable);
        }
        // Making VPC dependent on BucketDeployment so that CFN stack deletion is smooth.
        // Refer comments on https://github.com/aws/aws-cdk/pull/15220 for more details.
        if (props.vpc) {
            this.node.addDependency(props.vpc);
        }
        const mountPath = `/mnt${accessPointPath}`;
        const handler = new lambda.SingletonFunction(this, 'CustomResourceHandler', {
            uuid: this.renderSingletonUuid(props.memoryLimit, props.ephemeralStorageSize, props.vpc),
            code: lambda.Code.fromAsset(path.join(__dirname, 'lambda')),
            layers: [new lambda_layer_awscli_1.AwsCliLayer(this, 'AwsCliLayer')],
            runtime: lambda.Runtime.PYTHON_3_7,
            environment: props.useEfs ? {
                MOUNT_PATH: mountPath,
            } : undefined,
            handler: 'index.handler',
            lambdaPurpose: 'Custom::CDKBucketDeployment',
            timeout: cdk.Duration.minutes(15),
            role: props.role,
            memorySize: props.memoryLimit,
            ephemeralStorageSize: props.ephemeralStorageSize,
            vpc: props.vpc,
            vpcSubnets: props.vpcSubnets,
            filesystem: accessPoint ? lambda.FileSystem.fromEfsAccessPoint(accessPoint, mountPath) : undefined,
            logRetention: props.logRetention,
        });
        const handlerRole = handler.role;
        if (!handlerRole) {
            throw new Error('lambda.SingletonFunction should have created a Role');
        }
        const sources = props.sources.map((source) => source.bind(this, { handlerRole }));
        props.destinationBucket.grantReadWrite(handler);
        if (props.distribution) {
            handler.addToRolePolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['cloudfront:GetInvalidation', 'cloudfront:CreateInvalidation'],
                resources: ['*'],
            }));
        }
        // to avoid redundant stack updates, only include "SourceMarkers" if one of
        // the sources actually has markers.
        const hasMarkers = sources.some(source => source.markers);
        const crUniqueId = `CustomResource${this.renderUniqueId(props.memoryLimit, props.ephemeralStorageSize, props.vpc)}`;
        this.cr = new cdk.CustomResource(this, crUniqueId, {
            serviceToken: handler.functionArn,
            resourceType: 'Custom::CDKBucketDeployment',
            properties: {
                SourceBucketNames: sources.map(source => source.bucket.bucketName),
                SourceObjectKeys: sources.map(source => source.zipObjectKey),
                SourceMarkers: hasMarkers ? sources.map(source => { var _d; return (_d = source.markers) !== null && _d !== void 0 ? _d : {}; }) : undefined,
                DestinationBucketName: props.destinationBucket.bucketName,
                DestinationBucketKeyPrefix: props.destinationKeyPrefix,
                RetainOnDelete: props.retainOnDelete,
                Prune: (_d = props.prune) !== null && _d !== void 0 ? _d : true,
                Exclude: props.exclude,
                Include: props.include,
                UserMetadata: props.metadata ? mapUserMetadata(props.metadata) : undefined,
                SystemMetadata: mapSystemMetadata(props),
                DistributionId: (_e = props.distribution) === null || _e === void 0 ? void 0 : _e.distributionId,
                DistributionPaths: props.distributionPaths,
                // Passing through the ARN sequences dependencees on the deployment
                DestinationBucketArn: cdk.Lazy.string({ produce: () => this.requestDestinationArn ? props.destinationBucket.bucketArn : undefined }),
            },
        });
        let prefix = props.destinationKeyPrefix ?
            `:${props.destinationKeyPrefix}` :
            '';
        prefix += `:${this.cr.node.addr.slice(-8)}`;
        const tagKey = CUSTOM_RESOURCE_OWNER_TAG + prefix;
        // destinationKeyPrefix can be 104 characters before we hit
        // the tag key limit of 128
        // '/this/is/a/random/key/prefix/that/is/a/lot/of/characters/do/we/think/that/it/will/ever/be/this/long?????'
        // better to throw an error here than wait for CloudFormation to fail
        if (tagKey.length > 128) {
            throw new Error('The BucketDeployment construct requires that the "destinationKeyPrefix" be <=104 characters');
        }
        /*
         * This will add a tag to the deployment bucket in the format of
         * `aws-cdk:cr-owned:{keyPrefix}:{uniqueHash}`
         *
         * For example:
         * {
         *   Key: 'aws-cdk:cr-owned:deploy/here/:240D17B3',
         *   Value: 'true',
         * }
         *
         * This will allow for scenarios where there is a single S3 Bucket that has multiple
         * BucketDeployment resources deploying to it. Each bucket + keyPrefix can be "owned" by
         * 1 or more BucketDeployment resources. Since there are some scenarios where multiple BucketDeployment
         * resources can deploy to the same bucket and key prefix (e.g. using include/exclude) we
         * also append part of the id to make the key unique.
         *
         * As long as a bucket + keyPrefix is "owned" by a BucketDeployment resource, another CR
         * cannot delete data. There are a couple of scenarios where this comes into play.
         *
         * 1. If the LogicalResourceId of the CustomResource changes (e.g. the crUniqueId changes)
         * CloudFormation will first issue a 'Create' to create the new CustomResource and will
         * update the Tag on the bucket. CloudFormation will then issue a 'Delete' on the old CustomResource
         * and since the new CR "owns" the Bucket+keyPrefix it will not delete the contents of the bucket
         *
         * 2. If the BucketDeployment resource is deleted _and_ it is the only CR for that bucket+keyPrefix
         * then CloudFormation will first remove the tag from the bucket and then issue a "Delete" to the
         * CR. Since there are no tags indicating that this bucket+keyPrefix is "owned" then it will delete
         * the contents.
         *
         * 3. If the BucketDeployment resource is deleted _and_ it is *not* the only CR for that bucket:keyPrefix
         * then CloudFormation will first remove the tag from the bucket and then issue a "Delete" to the CR.
         * Since there are other CRs that also "own" that bucket+keyPrefix there will still be a tag on the bucket
         * and the contents will not be removed.
         *
         * 4. If the BucketDeployment resource _and_ the S3 Bucket are both removed, then CloudFormation will first
         * issue a "Delete" to the CR and since there is a tag on the bucket the contents will not be removed. If you
         * want the contents of the bucket to be removed on bucket deletion, then `autoDeleteObjects` property should
         * be set to true on the Bucket.
         */
        cdk.Tags.of(props.destinationBucket).add(tagKey, 'true');
    }
    /**
     * The bucket after the deployment
     *
     * If you want to reference the destination bucket in another construct and make sure the
     * bucket deployment has happened before the next operation is started, pass the other construct
     * a reference to `deployment.deployedBucket`.
     *
     * Doing this replaces calling `otherResource.node.addDependency(deployment)`.
     */
    get deployedBucket() {
        var _d;
        this.requestDestinationArn = true;
        this._deployedBucket = (_d = this._deployedBucket) !== null && _d !== void 0 ? _d : s3.Bucket.fromBucketArn(this, 'DestinationBucket', cdk.Token.asString(this.cr.getAtt('DestinationBucketArn')));
        return this._deployedBucket;
    }
    renderUniqueId(memoryLimit, ephemeralStorageSize, vpc) {
        let uuid = '';
        // if the user specifes a custom memory limit, we define another singleton handler
        // with this configuration. otherwise, it won't be possible to use multiple
        // configurations since we have a singleton.
        if (memoryLimit) {
            if (cdk.Token.isUnresolved(memoryLimit)) {
                throw new Error("Can't use tokens when specifying 'memoryLimit' since we use it to identify the singleton custom resource handler.");
            }
            uuid += `-${memoryLimit.toString()}MiB`;
        }
        // if the user specifies a custom ephemeral storage size, we define another singleton handler
        // with this configuration. otherwise, it won't be possible to use multiple
        // configurations since we have a singleton.
        if (ephemeralStorageSize) {
            if (ephemeralStorageSize.isUnresolved()) {
                throw new Error("Can't use tokens when specifying 'ephemeralStorageSize' since we use it to identify the singleton custom resource handler.");
            }
            uuid += `-${ephemeralStorageSize.toMebibytes().toString()}MiB`;
        }
        // if the user specifies a VPC, we define another singleton handler
        // with this configuration. otherwise, it won't be possible to use multiple
        // configurations since we have a singleton.
        // A VPC is a must if EFS storage is used and that's why we are only using VPC in uuid.
        if (vpc) {
            uuid += `-${vpc.node.addr}`;
        }
        return uuid;
    }
    renderSingletonUuid(memoryLimit, ephemeralStorageSize, vpc) {
        let uuid = '8693BB64-9689-44B6-9AAF-B0CC9EB8756C';
        uuid += this.renderUniqueId(memoryLimit, ephemeralStorageSize, vpc);
        return uuid;
    }
    /**
     * Function to get/create a stack singleton instance of EFS FileSystem per vpc.
     *
     * @param scope Construct
     * @param fileSystemProps EFS FileSystemProps
     */
    getOrCreateEfsFileSystem(scope, fileSystemProps) {
        var _d;
        const stack = cdk.Stack.of(scope);
        const uuid = `BucketDeploymentEFS-VPC-${fileSystemProps.vpc.node.addr}`;
        return (_d = stack.node.tryFindChild(uuid)) !== null && _d !== void 0 ? _d : new efs.FileSystem(scope, uuid, fileSystemProps);
    }
}
exports.BucketDeployment = BucketDeployment;
_a = JSII_RTTI_SYMBOL_1;
BucketDeployment[_a] = { fqn: "@aws-cdk/aws-s3-deployment.BucketDeployment", version: "1.156.1" };
/**
 * Metadata
 */
function mapUserMetadata(metadata) {
    const mapKey = (key) => key.toLowerCase();
    return Object.keys(metadata).reduce((o, key) => ({ ...o, [mapKey(key)]: metadata[key] }), {});
}
function mapSystemMetadata(metadata) {
    const res = {};
    if (metadata.cacheControl) {
        res['cache-control'] = metadata.cacheControl.map(c => c.value).join(', ');
    }
    if (metadata.expires) {
        res.expires = metadata.expires.date.toUTCString();
    }
    if (metadata.contentDisposition) {
        res['content-disposition'] = metadata.contentDisposition;
    }
    if (metadata.contentEncoding) {
        res['content-encoding'] = metadata.contentEncoding;
    }
    if (metadata.contentLanguage) {
        res['content-language'] = metadata.contentLanguage;
    }
    if (metadata.contentType) {
        res['content-type'] = metadata.contentType;
    }
    if (metadata.serverSideEncryption) {
        res.sse = metadata.serverSideEncryption;
    }
    if (metadata.storageClass) {
        res['storage-class'] = metadata.storageClass;
    }
    if (metadata.websiteRedirectLocation) {
        res['website-redirect'] = metadata.websiteRedirectLocation;
    }
    if (metadata.serverSideEncryptionAwsKmsKeyId) {
        res['sse-kms-key-id'] = metadata.serverSideEncryptionAwsKmsKeyId;
    }
    if (metadata.serverSideEncryptionCustomerAlgorithm) {
        res['sse-c-copy-source'] = metadata.serverSideEncryptionCustomerAlgorithm;
    }
    if (metadata.accessControl) {
        res.acl = case_1.kebab(metadata.accessControl.toString());
    }
    return Object.keys(res).length === 0 ? undefined : res;
}
/**
 * Used for HTTP cache-control header, which influences downstream caches.
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
class CacheControl {
    constructor(
    /**
     * The raw cache control setting.
     */
    value) {
        this.value = value;
    }
    /**
     * Sets 'must-revalidate'.
     */
    static mustRevalidate() { return new CacheControl('must-revalidate'); }
    /**
     * Sets 'no-cache'.
     */
    static noCache() { return new CacheControl('no-cache'); }
    /**
     * Sets 'no-transform'.
     */
    static noTransform() { return new CacheControl('no-transform'); }
    /**
     * Sets 'public'.
     */
    static setPublic() { return new CacheControl('public'); }
    /**
     * Sets 'private'.
     */
    static setPrivate() { return new CacheControl('private'); }
    /**
     * Sets 'proxy-revalidate'.
     */
    static proxyRevalidate() { return new CacheControl('proxy-revalidate'); }
    /**
     * Sets 'max-age=<duration-in-seconds>'.
     */
    static maxAge(t) { return new CacheControl(`max-age=${t.toSeconds()}`); }
    /**
     * Sets 's-maxage=<duration-in-seconds>'.
     */
    static sMaxAge(t) { return new CacheControl(`s-maxage=${t.toSeconds()}`); }
    /**
     * Constructs a custom cache control key from the literal value.
     */
    static fromString(s) { return new CacheControl(s); }
}
exports.CacheControl = CacheControl;
_b = JSII_RTTI_SYMBOL_1;
CacheControl[_b] = { fqn: "@aws-cdk/aws-s3-deployment.CacheControl", version: "1.156.1" };
/**
 * Indicates whether server-side encryption is enabled for the object, and whether that encryption is
 * from the AWS Key Management Service (AWS KMS) or from Amazon S3 managed encryption (SSE-S3).
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
var ServerSideEncryption;
(function (ServerSideEncryption) {
    /**
     * 'AES256'
     */
    ServerSideEncryption["AES_256"] = "AES256";
    /**
     * 'aws:kms'
     */
    ServerSideEncryption["AWS_KMS"] = "aws:kms";
})(ServerSideEncryption = exports.ServerSideEncryption || (exports.ServerSideEncryption = {}));
/**
 * Storage class used for storing the object.
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
var StorageClass;
(function (StorageClass) {
    /**
     * 'STANDARD'
     */
    StorageClass["STANDARD"] = "STANDARD";
    /**
     * 'REDUCED_REDUNDANCY'
     */
    StorageClass["REDUCED_REDUNDANCY"] = "REDUCED_REDUNDANCY";
    /**
     * 'STANDARD_IA'
     */
    StorageClass["STANDARD_IA"] = "STANDARD_IA";
    /**
     * 'ONEZONE_IA'
     */
    StorageClass["ONEZONE_IA"] = "ONEZONE_IA";
    /**
     * 'INTELLIGENT_TIERING'
     */
    StorageClass["INTELLIGENT_TIERING"] = "INTELLIGENT_TIERING";
    /**
     * 'GLACIER'
     */
    StorageClass["GLACIER"] = "GLACIER";
    /**
     * 'DEEP_ARCHIVE'
     */
    StorageClass["DEEP_ARCHIVE"] = "DEEP_ARCHIVE";
})(StorageClass = exports.StorageClass || (exports.StorageClass = {}));
/**
 * Used for HTTP expires header, which influences downstream caches. Does NOT influence deletion of the object.
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 *
 * @deprecated use core.Expiration
 */
class Expires {
    constructor(
    /**
     * The raw expiration date expression.
     */
    value) {
        this.value = value;
    }
    /**
     * Expire at the specified date
     * @param d date to expire at
     */
    static atDate(d) { try {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-s3-deployment.Expires#atDate", "use core.Expiration");
    }
    catch (error) {
        if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
            Error.captureStackTrace(error, this.atDate);
        }
        throw error;
    } return new Expires(d.toUTCString()); }
    /**
     * Expire at the specified timestamp
     * @param t timestamp in unix milliseconds
     */
    static atTimestamp(t) { try {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-s3-deployment.Expires#atTimestamp", "use core.Expiration");
    }
    catch (error) {
        if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
            Error.captureStackTrace(error, this.atTimestamp);
        }
        throw error;
    } return Expires.atDate(new Date(t)); }
    /**
     * Expire once the specified duration has passed since deployment time
     * @param t the duration to wait before expiring
     */
    static after(t) { try {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-s3-deployment.Expires#after", "use core.Expiration");
    }
    catch (error) {
        if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
            Error.captureStackTrace(error, this.after);
        }
        throw error;
    } return Expires.atDate(new Date(Date.now() + t.toMilliseconds())); }
    /**
     * Create an expiration date from a raw date string.
     */
    static fromString(s) { try {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-s3-deployment.Expires#fromString", "use core.Expiration");
    }
    catch (error) {
        if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
            Error.captureStackTrace(error, this.fromString);
        }
        throw error;
    } return new Expires(s); }
}
exports.Expires = Expires;
_c = JSII_RTTI_SYMBOL_1;
Expires[_c] = { fqn: "@aws-cdk/aws-s3-deployment.Expires", version: "1.156.1" };
//# sourceMappingURL=data:application/json;base64,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