"use strict";
const cluster_1 = require("../lib/cluster-resource-handler/cluster");
const mocks = require("./cluster-resource-handler-mocks");
module.exports = {
    setUp(callback) {
        mocks.reset();
        callback();
    },
    create: {
        async 'onCreate: minimal defaults (vpc + role)'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Create', mocks.MOCK_PROPS));
            await handler.onEvent();
            test.deepEqual(mocks.actualRequest.configureAssumeRoleRequest, {
                RoleArn: mocks.MOCK_ASSUME_ROLE_ARN,
                RoleSessionName: 'AWSCDK.EKSCluster.Create.fake-request-id',
            });
            test.deepEqual(mocks.actualRequest.createClusterRequest, {
                roleArn: 'arn:of:role',
                resourcesVpcConfig: {
                    subnetIds: ['subnet1', 'subnet2'],
                    securityGroupIds: ['sg1', 'sg2', 'sg3']
                },
                name: 'MyResourceId-fakerequestid'
            });
            test.done();
        },
        async 'generated cluster name does not exceed 100 characters'(test) {
            var _a, _b;
            // GIVEN
            const req = {
                StackId: 'fake-stack-id',
                RequestId: '602c078a-6181-4352-9676-4f00352445aa',
                ResourceType: 'Custom::EKSCluster',
                ServiceToken: 'boom',
                LogicalResourceId: 'hello'.repeat(30),
                ResponseURL: 'http://response-url',
                RequestType: 'Create',
                ResourceProperties: {
                    ServiceToken: 'boom',
                    Config: mocks.MOCK_PROPS,
                    AssumeRoleArn: mocks.MOCK_ASSUME_ROLE_ARN
                }
            };
            // WHEN
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, req);
            await handler.onEvent();
            // THEN
            test.equal((_a = mocks.actualRequest.createClusterRequest) === null || _a === void 0 ? void 0 : _a.name.length, 100);
            test.deepEqual((_b = mocks.actualRequest.createClusterRequest) === null || _b === void 0 ? void 0 : _b.name, 'hellohellohellohellohellohellohellohellohellohellohellohellohellohe-602c078a6181435296764f00352445aa');
            test.done();
        },
        async 'onCreate: explicit cluster name'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Create', {
                ...mocks.MOCK_PROPS,
                name: 'ExplicitCustomName'
            }));
            await handler.onEvent();
            test.deepEqual(mocks.actualRequest.createClusterRequest.name, 'ExplicitCustomName');
            test.done();
        },
        async 'with no specific version'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Create', {
                ...mocks.MOCK_PROPS,
                version: '12.34.56',
            }));
            await handler.onEvent();
            test.deepEqual(mocks.actualRequest.createClusterRequest.version, '12.34.56');
            test.done();
        },
        async 'isCreateComplete still not complete if cluster is not ACTIVE'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Create'));
            mocks.simulateResponse.describeClusterResponseMockStatus = 'CREATING';
            const resp = await handler.isComplete();
            test.deepEqual(mocks.actualRequest.describeClusterRequest.name, 'physical-resource-id');
            test.deepEqual(resp, { IsComplete: false });
            test.done();
        },
        async 'isCreateComplete is complete when cluster is ACTIVE'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Create'));
            mocks.simulateResponse.describeClusterResponseMockStatus = 'ACTIVE';
            const resp = await handler.isComplete();
            test.deepEqual(resp, {
                IsComplete: true,
                Data: {
                    Name: 'physical-resource-id',
                    Endpoint: 'http://endpoint',
                    Arn: 'arn:cluster-arn',
                    CertificateAuthorityData: 'certificateAuthority-data'
                }
            });
            test.done();
        },
    },
    delete: {
        async 'returns correct physical name'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Delete'));
            const resp = await handler.onEvent();
            test.deepEqual(mocks.actualRequest.deleteClusterRequest.name, 'physical-resource-id');
            test.deepEqual(resp, { PhysicalResourceId: 'physical-resource-id' });
            test.done();
        },
        async 'onDelete ignores ResourceNotFoundException'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Delete'));
            mocks.simulateResponse.deleteClusterErrorCode = 'ResourceNotFoundException';
            await handler.onEvent();
            test.done();
        },
        async 'isDeleteComplete returns false as long as describeCluster succeeds'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Delete'));
            const resp = await handler.isComplete();
            test.deepEqual(mocks.actualRequest.describeClusterRequest.name, 'physical-resource-id');
            test.ok(!resp.IsComplete);
            test.done();
        },
        async 'isDeleteComplete returns true when describeCluster throws a ResourceNotFound exception'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Delete'));
            mocks.simulateResponse.describeClusterExceptionCode = 'ResourceNotFoundException';
            const resp = await handler.isComplete();
            test.ok(resp.IsComplete);
            test.done();
        },
        async 'isDeleteComplete propagates other errors'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Delete'));
            mocks.simulateResponse.describeClusterExceptionCode = 'OtherException';
            let error;
            try {
                await handler.isComplete();
            }
            catch (e) {
                error = e;
            }
            test.equal(error.code, 'OtherException');
            test.done();
        }
    },
    update: {
        async 'no change'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', mocks.MOCK_PROPS, mocks.MOCK_PROPS));
            const resp = await handler.onEvent();
            test.equal(resp, undefined);
            test.equal(mocks.actualRequest.createClusterRequest, undefined);
            test.equal(mocks.actualRequest.updateClusterConfigRequest, undefined);
            test.equal(mocks.actualRequest.updateClusterVersionRequest, undefined);
            test.done();
        },
        async 'isUpdateComplete is not complete when status is not ACTIVE'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update'));
            mocks.simulateResponse.describeClusterResponseMockStatus = 'UPDATING';
            const resp = await handler.isComplete();
            test.deepEqual(resp.IsComplete, false);
            test.done();
        },
        async 'isUpdateComplete waits for ACTIVE'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update'));
            mocks.simulateResponse.describeClusterResponseMockStatus = 'ACTIVE';
            const resp = await handler.isComplete();
            test.deepEqual(resp.IsComplete, true);
            test.done();
        },
        'requires replacement': {
            'name change': {
                async 'explicit name change'(test) {
                    // GIVEN
                    const req = mocks.newRequest('Update', {
                        ...mocks.MOCK_PROPS,
                        name: 'new-cluster-name-1234'
                    }, {
                        name: 'old-cluster-name'
                    });
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, req);
                    // WHEN
                    const resp = await handler.onEvent();
                    // THEN
                    test.deepEqual(mocks.actualRequest.createClusterRequest, {
                        name: 'new-cluster-name-1234',
                        roleArn: 'arn:of:role',
                        resourcesVpcConfig: {
                            subnetIds: ['subnet1', 'subnet2'],
                            securityGroupIds: ['sg1', 'sg2', 'sg3']
                        }
                    });
                    test.deepEqual(resp, { PhysicalResourceId: 'new-cluster-name-1234' });
                    test.done();
                },
                async 'from auto-gen name to explicit name'(test) {
                    // GIVEN
                    const req = mocks.newRequest('Update', {
                        ...mocks.MOCK_PROPS,
                        name: undefined // auto-gen
                    }, {
                        name: 'explicit' // auto-gen
                    });
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, req);
                    // WHEN
                    const resp = await handler.onEvent();
                    // THEN
                    test.deepEqual(mocks.actualRequest.createClusterRequest, {
                        name: 'MyResourceId-fakerequestid',
                        roleArn: 'arn:of:role',
                        resourcesVpcConfig: {
                            subnetIds: ['subnet1', 'subnet2'],
                            securityGroupIds: ['sg1', 'sg2', 'sg3']
                        }
                    });
                    test.deepEqual(resp, { PhysicalResourceId: 'MyResourceId-fakerequestid' });
                    test.done();
                },
            },
            async 'subnets or security groups requires a replacement'(test) {
                const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                    ...mocks.MOCK_PROPS,
                    resourcesVpcConfig: {
                        subnetIds: ['subnet1', 'subnet2'],
                        securityGroupIds: ['sg1']
                    }
                }, {
                    ...mocks.MOCK_PROPS,
                    resourcesVpcConfig: {
                        subnetIds: ['subnet1'],
                        securityGroupIds: ['sg2']
                    }
                }));
                const resp = await handler.onEvent();
                test.deepEqual(resp, { PhysicalResourceId: 'MyResourceId-fakerequestid' });
                test.deepEqual(mocks.actualRequest.createClusterRequest, {
                    name: 'MyResourceId-fakerequestid',
                    roleArn: 'arn:of:role',
                    resourcesVpcConfig: {
                        subnetIds: ['subnet1', 'subnet2'],
                        securityGroupIds: ['sg1']
                    }
                });
                test.done();
            },
            async '"roleArn" requires a replcement'(test) {
                const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                    roleArn: 'new-arn'
                }, {
                    roleArn: 'old-arn'
                }));
                const resp = await handler.onEvent();
                test.deepEqual(resp, { PhysicalResourceId: 'MyResourceId-fakerequestid' });
                test.deepEqual(mocks.actualRequest.createClusterRequest, {
                    name: 'MyResourceId-fakerequestid',
                    roleArn: 'new-arn'
                });
                test.done();
            },
            async 'fails if cluster has an explicit "name" that is the same as the old "name"'(test) {
                // GIVEN
                const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                    roleArn: 'new-arn',
                    name: 'explicit-cluster-name'
                }, {
                    roleArn: 'old-arn',
                    name: 'explicit-cluster-name'
                }));
                // THEN
                let err;
                try {
                    await handler.onEvent();
                }
                catch (e) {
                    err = e;
                }
                test.equal(err === null || err === void 0 ? void 0 : err.message, 'Cannot replace cluster "explicit-cluster-name" since it has an explicit physical name. Either rename the cluster or remove the "name" configuration');
                test.done();
            },
            async 'succeeds if cluster had an explicit "name" and now it does not'(test) {
                // GIVEN
                const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                    roleArn: 'new-arn',
                    name: undefined // auto-gen
                }, {
                    roleArn: 'old-arn',
                    name: 'explicit-cluster-name'
                }));
                // WHEN
                const resp = await handler.onEvent();
                // THEN
                test.deepEqual(resp, { PhysicalResourceId: 'MyResourceId-fakerequestid' });
                test.deepEqual(mocks.actualRequest.createClusterRequest, {
                    name: 'MyResourceId-fakerequestid',
                    roleArn: 'new-arn'
                });
                test.done();
            },
            async 'succeeds if cluster had an explicit "name" and now it has a different name'(test) {
                // GIVEN
                const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                    roleArn: 'new-arn',
                    name: 'new-explicit-cluster-name'
                }, {
                    roleArn: 'old-arn',
                    name: 'old-explicit-cluster-name'
                }));
                // WHEN
                const resp = await handler.onEvent();
                // THEN
                test.deepEqual(resp, { PhysicalResourceId: 'new-explicit-cluster-name' });
                test.deepEqual(mocks.actualRequest.createClusterRequest, {
                    name: 'new-explicit-cluster-name',
                    roleArn: 'new-arn'
                });
                test.done();
            }
        },
        'in-place': {
            'version change': {
                async 'from undefined to a specific value'(test) {
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                        version: '12.34'
                    }, {
                        version: undefined
                    }));
                    const resp = await handler.onEvent();
                    test.equal(resp, undefined);
                    test.deepEqual(mocks.actualRequest.updateClusterVersionRequest, {
                        name: 'physical-resource-id',
                        version: '12.34'
                    });
                    test.equal(mocks.actualRequest.createClusterRequest, undefined);
                    test.done();
                },
                async 'from a specific value to another value'(test) {
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                        version: '2.0'
                    }, {
                        version: '1.1'
                    }));
                    const resp = await handler.onEvent();
                    test.equal(resp, undefined);
                    test.deepEqual(mocks.actualRequest.updateClusterVersionRequest, {
                        name: 'physical-resource-id',
                        version: '2.0'
                    });
                    test.equal(mocks.actualRequest.createClusterRequest, undefined);
                    test.done();
                },
                async 'to a new value that is already the current version'(test) {
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                        version: '1.0'
                    }));
                    const resp = await handler.onEvent();
                    test.equal(resp, undefined);
                    test.deepEqual(mocks.actualRequest.describeClusterRequest, { name: 'physical-resource-id' });
                    test.equal(mocks.actualRequest.updateClusterVersionRequest, undefined);
                    test.equal(mocks.actualRequest.createClusterRequest, undefined);
                    test.done();
                },
                async 'fails from specific value to undefined'(test) {
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                        version: undefined
                    }, {
                        version: '1.2'
                    }));
                    let error;
                    try {
                        await handler.onEvent();
                    }
                    catch (e) {
                        error = e;
                    }
                    test.equal(error.message, 'Cannot remove cluster version configuration. Current version is 1.2');
                    test.done();
                }
            }
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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