"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const eks = require("../lib");
const hello = require("./hello-k8s");
const util_1 = require("./util");
class EksClusterStack extends util_1.TestStack {
    constructor(scope, id) {
        super(scope, id);
        // allow all account users to assume this role in order to admin the cluster
        const mastersRole = new iam.Role(this, 'AdminRole', {
            assumedBy: new iam.AccountRootPrincipal()
        });
        // create the cluster with a default nodegroup capacity
        const cluster = new eks.Cluster(this, 'Cluster', {
            mastersRole,
            defaultCapacity: 2,
        });
        // // fargate profile for resources in the "default" namespace
        cluster.addFargateProfile('default', {
            selectors: [{ namespace: 'default' }]
        });
        // add some capacity to the cluster. The IAM instance role will
        // automatically be mapped via aws-auth to allow nodes to join the cluster.
        cluster.addCapacity('Nodes', {
            instanceType: new ec2.InstanceType('t2.medium'),
            minCapacity: 3,
        });
        // add bottlerocket nodes
        cluster.addCapacity('BottlerocketNodes', {
            instanceType: new ec2.InstanceType('t3.small'),
            minCapacity: 2,
            machineImageType: eks.MachineImageType.BOTTLEROCKET
        });
        // spot instances (up to 10)
        cluster.addCapacity('spot', {
            spotPrice: '0.1094',
            instanceType: new ec2.InstanceType('t3.large'),
            maxCapacity: 10,
            bootstrapOptions: {
                kubeletExtraArgs: '--node-labels foo=bar,goo=far',
                awsApiRetryAttempts: 5
            }
        });
        // add a extra nodegroup
        cluster.addNodegroup('extra-ng', {
            instanceType: new ec2.InstanceType('t3.small'),
            minSize: 1,
            // reusing the default capacity nodegroup instance role when available
            nodeRole: cluster.defaultCapacity ? cluster.defaultCapacity.role : undefined
        });
        // // apply a kubernetes manifest
        cluster.addResource('HelloApp', ...hello.resources);
        // // add two Helm charts to the cluster. This will be the Kubernetes dashboard and the Nginx Ingress Controller
        cluster.addChart('dashboard', { chart: 'kubernetes-dashboard', repository: 'https://kubernetes-charts.storage.googleapis.com' });
        cluster.addChart('nginx-ingress', { chart: 'nginx-ingress', repository: 'https://helm.nginx.com/stable', namespace: 'kube-system' });
        new core_1.CfnOutput(this, 'ClusterEndpoint', { value: cluster.clusterEndpoint });
        new core_1.CfnOutput(this, 'ClusterArn', { value: cluster.clusterArn });
        new core_1.CfnOutput(this, 'ClusterCertificateAuthorityData', { value: cluster.clusterCertificateAuthorityData });
        new core_1.CfnOutput(this, 'ClusterName', { value: cluster.clusterName });
    }
}
const app = new core_1.App();
// since the EKS optimized AMI is hard-coded here based on the region,
// we need to actually pass in a specific region.
new EksClusterStack(app, 'aws-cdk-eks-cluster-test');
app.synth();
//# sourceMappingURL=data:application/json;base64,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