"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cfn = require("@aws-cdk/aws-cloudformation");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const consts_1 = require("./cluster-resource-handler/consts");
const cluster_resource_provider_1 = require("./cluster-resource-provider");
/**
 * A low-level CFN resource Amazon EKS cluster implemented through a custom
 * resource.
 *
 * Implements EKS create/update/delete through a CloudFormation custom resource
 * in order to allow us to control the IAM role which creates the cluster. This
 * is required in order to be able to allow CloudFormation to interact with the
 * cluster via `kubectl` to enable Kubernetes management capabilities like apply
 * manifest and IAM role/user RBAC mapping.
 */
class ClusterResource extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.trustedPrincipals = [];
        const stack = core_1.Stack.of(this);
        const provider = cluster_resource_provider_1.ClusterResourceProvider.getOrCreate(this);
        if (!props.roleArn) {
            throw new Error('"roleArn" is required');
        }
        // the role used to create the cluster. this becomes the administrator role
        // of the cluster.
        this.creationRole = new iam.Role(this, 'CreationRole', {
            assumedBy: new iam.CompositePrincipal(...provider.roles.map(x => new iam.ArnPrincipal(x.roleArn)))
        });
        // the CreateCluster API will allow the cluster to assume this role, so we
        // need to allow the lambda execution role to pass it.
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: [props.roleArn]
        }));
        // if we know the cluster name, restrict the policy to only allow
        // interacting with this specific cluster otherwise, we will have to grant
        // this role to manage all clusters in the account. this must be lazy since
        // `props.name` may contain a lazy value that conditionally resolves to a
        // physical name.
        const resourceArns = core_1.Lazy.listValue({
            produce: () => {
                const arn = stack.formatArn(clusterArnComponents(stack.resolve(props.name)));
                return stack.resolve(props.name)
                    ? [arn, `${arn}/*`] // see https://github.com/aws/aws-cdk/issues/6060
                    : ['*'];
            }
        });
        const fargateProfileResourceArn = core_1.Lazy.stringValue({
            produce: () => stack.resolve(props.name)
                ? stack.formatArn({ service: 'eks', resource: 'fargateprofile', resourceName: stack.resolve(props.name) + '/*' })
                : '*'
        });
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['ec2:DescribeSubnets'],
            resources: ['*'],
        }));
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'eks:CreateCluster',
                'eks:DescribeCluster',
                'eks:DeleteCluster',
                'eks:UpdateClusterVersion',
                'eks:UpdateClusterConfig',
                'eks:CreateFargateProfile',
                'eks:TagResource',
                'eks:UntagResource'
            ],
            resources: resourceArns
        }));
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['eks:DescribeFargateProfile', 'eks:DeleteFargateProfile'],
            resources: [fargateProfileResourceArn]
        }));
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:GetRole'],
            resources: ['*'],
        }));
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:CreateServiceLinkedRole'],
            resources: ['*'],
        }));
        const resource = new cfn.CustomResource(this, 'Resource', {
            resourceType: consts_1.CLUSTER_RESOURCE_TYPE,
            provider: provider.provider,
            properties: {
                Config: props,
                AssumeRoleArn: this.creationRole.roleArn
            }
        });
        resource.node.addDependency(this.creationRole);
        this.ref = resource.ref;
        this.attrEndpoint = core_1.Token.asString(resource.getAtt('Endpoint'));
        this.attrArn = core_1.Token.asString(resource.getAtt('Arn'));
        this.attrCertificateAuthorityData = core_1.Token.asString(resource.getAtt('CertificateAuthorityData'));
    }
    /**
     * Returns the ARN of the cluster creation role and grants `trustedRole`
     * permissions to assume this role.
     */
    getCreationRoleArn(trustedRole) {
        if (!trustedRole) {
            return this.creationRole.roleArn;
        }
        if (!this.trustedPrincipals.includes(trustedRole.roleArn)) {
            if (!this.creationRole.assumeRolePolicy) {
                throw new Error('unexpected: cluster creation role must have trust policy');
            }
            this.creationRole.assumeRolePolicy.addStatements(new iam.PolicyStatement({
                actions: ['sts:AssumeRole'],
                principals: [new iam.ArnPrincipal(trustedRole.roleArn)]
            }));
            this.trustedPrincipals.push(trustedRole.roleArn);
        }
        return this.creationRole.roleArn;
    }
}
exports.ClusterResource = ClusterResource;
function clusterArnComponents(clusterName) {
    return {
        service: 'eks',
        resource: 'cluster',
        resourceName: clusterName,
    };
}
exports.clusterArnComponents = clusterArnComponents;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2x1c3Rlci1yZXNvdXJjZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImNsdXN0ZXItcmVzb3VyY2UudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSxtREFBbUQ7QUFDbkQsd0NBQXdDO0FBQ3hDLHdDQUE2RTtBQUM3RSw4REFBMEU7QUFDMUUsMkVBQXNFO0FBR3RFOzs7Ozs7Ozs7R0FTRztBQUNILE1BQWEsZUFBZ0IsU0FBUSxnQkFBUztJQWM1QyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXNCO1FBQzlELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFIRixzQkFBaUIsR0FBYSxFQUFFLENBQUM7UUFLaEQsTUFBTSxLQUFLLEdBQUcsWUFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUM3QixNQUFNLFFBQVEsR0FBRyxtREFBdUIsQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFM0QsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLEVBQUU7WUFDbEIsTUFBTSxJQUFJLEtBQUssQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO1NBQzFDO1FBRUQsMkVBQTJFO1FBQzNFLGtCQUFrQjtRQUNsQixJQUFJLENBQUMsWUFBWSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsY0FBYyxFQUFFO1lBQ3JELFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxHQUFHLFFBQVEsQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsSUFBSSxHQUFHLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO1NBQ25HLENBQUMsQ0FBQztRQUVILDBFQUEwRTtRQUMxRSxzREFBc0Q7UUFDdEQsSUFBSSxDQUFDLFlBQVksQ0FBQyxXQUFXLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQ3BELE9BQU8sRUFBRSxDQUFFLGNBQWMsQ0FBRTtZQUMzQixTQUFTLEVBQUUsQ0FBRSxLQUFLLENBQUMsT0FBTyxDQUFFO1NBQzdCLENBQUMsQ0FBQyxDQUFDO1FBRUosaUVBQWlFO1FBQ2pFLDBFQUEwRTtRQUMxRSwyRUFBMkU7UUFDM0UseUVBQXlFO1FBQ3pFLGlCQUFpQjtRQUNqQixNQUFNLFlBQVksR0FBRyxXQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2xDLE9BQU8sRUFBRSxHQUFHLEVBQUU7Z0JBQ1osTUFBTSxHQUFHLEdBQUcsS0FBSyxDQUFDLFNBQVMsQ0FBQyxvQkFBb0IsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQzdFLE9BQU8sS0FBSyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDO29CQUM5QixDQUFDLENBQUMsQ0FBRSxHQUFHLEVBQUUsR0FBRyxHQUFHLElBQUksQ0FBRSxDQUFDLGlEQUFpRDtvQkFDdkUsQ0FBQyxDQUFDLENBQUUsR0FBRyxDQUFFLENBQUM7WUFDZCxDQUFDO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsTUFBTSx5QkFBeUIsR0FBRyxXQUFJLENBQUMsV0FBVyxDQUFDO1lBQ2pELE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUM7Z0JBQ3RDLENBQUMsQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDLEVBQUUsT0FBTyxFQUFFLEtBQUssRUFBRSxRQUFRLEVBQUUsZ0JBQWdCLEVBQUUsWUFBWSxFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxHQUFHLElBQUksRUFBRSxDQUFDO2dCQUNqSCxDQUFDLENBQUMsR0FBRztTQUNSLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxZQUFZLENBQUMsV0FBVyxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUNwRCxPQUFPLEVBQUUsQ0FBRSxxQkFBcUIsQ0FBRTtZQUNsQyxTQUFTLEVBQUUsQ0FBRSxHQUFHLENBQUU7U0FDbkIsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsWUFBWSxDQUFDLFdBQVcsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDcEQsT0FBTyxFQUFFO2dCQUNQLG1CQUFtQjtnQkFDbkIscUJBQXFCO2dCQUNyQixtQkFBbUI7Z0JBQ25CLDBCQUEwQjtnQkFDMUIseUJBQXlCO2dCQUN6QiwwQkFBMEI7Z0JBQzFCLGlCQUFpQjtnQkFDakIsbUJBQW1CO2FBQ3BCO1lBQ0QsU0FBUyxFQUFFLFlBQVk7U0FDeEIsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsWUFBWSxDQUFDLFdBQVcsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDcEQsT0FBTyxFQUFFLENBQUUsNEJBQTRCLEVBQUUsMEJBQTBCLENBQUU7WUFDckUsU0FBUyxFQUFFLENBQUUseUJBQXlCLENBQUU7U0FDekMsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsWUFBWSxDQUFDLFdBQVcsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDcEQsT0FBTyxFQUFFLENBQUUsYUFBYSxDQUFFO1lBQzFCLFNBQVMsRUFBRSxDQUFFLEdBQUcsQ0FBRTtTQUNuQixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxZQUFZLENBQUMsV0FBVyxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUNwRCxPQUFPLEVBQUUsQ0FBRSw2QkFBNkIsQ0FBRTtZQUMxQyxTQUFTLEVBQUUsQ0FBRSxHQUFHLENBQUU7U0FDbkIsQ0FBQyxDQUFDLENBQUM7UUFFSixNQUFNLFFBQVEsR0FBRyxJQUFJLEdBQUcsQ0FBQyxjQUFjLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUN4RCxZQUFZLEVBQUUsOEJBQXFCO1lBQ25DLFFBQVEsRUFBRSxRQUFRLENBQUMsUUFBUTtZQUMzQixVQUFVLEVBQUU7Z0JBQ1YsTUFBTSxFQUFFLEtBQUs7Z0JBQ2IsYUFBYSxFQUFFLElBQUksQ0FBQyxZQUFZLENBQUMsT0FBTzthQUN6QztTQUNGLENBQUMsQ0FBQztRQUVILFFBQVEsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQztRQUUvQyxJQUFJLENBQUMsR0FBRyxHQUFHLFFBQVEsQ0FBQyxHQUFHLENBQUM7UUFDeEIsSUFBSSxDQUFDLFlBQVksR0FBRyxZQUFLLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQztRQUNoRSxJQUFJLENBQUMsT0FBTyxHQUFHLFlBQUssQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO1FBQ3RELElBQUksQ0FBQyw0QkFBNEIsR0FBRyxZQUFLLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsMEJBQTBCLENBQUMsQ0FBQyxDQUFDO0lBQ2xHLENBQUM7SUFFRDs7O09BR0c7SUFDSSxrQkFBa0IsQ0FBQyxXQUF1QjtRQUUvQyxJQUFJLENBQUMsV0FBVyxFQUFFO1lBQ2hCLE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQyxPQUFPLENBQUM7U0FDbEM7UUFFRCxJQUFJLENBQUMsSUFBSSxDQUFDLGlCQUFpQixDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUMsT0FBTyxDQUFDLEVBQUU7WUFDekQsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsZ0JBQWdCLEVBQUU7Z0JBQ3ZDLE1BQU0sSUFBSSxLQUFLLENBQUMsMERBQTBELENBQUMsQ0FBQzthQUM3RTtZQUVELElBQUksQ0FBQyxZQUFZLENBQUMsZ0JBQWdCLENBQUMsYUFBYSxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztnQkFDdkUsT0FBTyxFQUFFLENBQUUsZ0JBQWdCLENBQUU7Z0JBQzdCLFVBQVUsRUFBRSxDQUFFLElBQUksR0FBRyxDQUFDLFlBQVksQ0FBQyxXQUFXLENBQUMsT0FBTyxDQUFDLENBQUU7YUFDMUQsQ0FBQyxDQUFDLENBQUM7WUFFSixJQUFJLENBQUMsaUJBQWlCLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxPQUFPLENBQUMsQ0FBQztTQUNsRDtRQUNELE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQyxPQUFPLENBQUM7SUFDbkMsQ0FBQztDQUNGO0FBcElELDBDQW9JQztBQUVELFNBQWdCLG9CQUFvQixDQUFDLFdBQW1CO0lBQ3RELE9BQU87UUFDTCxPQUFPLEVBQUUsS0FBSztRQUNkLFFBQVEsRUFBRSxTQUFTO1FBQ25CLFlBQVksRUFBRSxXQUFXO0tBQzFCLENBQUM7QUFDSixDQUFDO0FBTkQsb0RBTUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjZm4gZnJvbSAnQGF3cy1jZGsvYXdzLWNsb3VkZm9ybWF0aW9uJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCB7IEFybkNvbXBvbmVudHMsIENvbnN0cnVjdCwgTGF6eSwgU3RhY2ssIFRva2VuIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBDTFVTVEVSX1JFU09VUkNFX1RZUEUgfSBmcm9tICcuL2NsdXN0ZXItcmVzb3VyY2UtaGFuZGxlci9jb25zdHMnO1xuaW1wb3J0IHsgQ2x1c3RlclJlc291cmNlUHJvdmlkZXIgfSBmcm9tICcuL2NsdXN0ZXItcmVzb3VyY2UtcHJvdmlkZXInO1xuaW1wb3J0IHsgQ2ZuQ2x1c3RlclByb3BzIH0gZnJvbSAnLi9la3MuZ2VuZXJhdGVkJztcblxuLyoqXG4gKiBBIGxvdy1sZXZlbCBDRk4gcmVzb3VyY2UgQW1hem9uIEVLUyBjbHVzdGVyIGltcGxlbWVudGVkIHRocm91Z2ggYSBjdXN0b21cbiAqIHJlc291cmNlLlxuICpcbiAqIEltcGxlbWVudHMgRUtTIGNyZWF0ZS91cGRhdGUvZGVsZXRlIHRocm91Z2ggYSBDbG91ZEZvcm1hdGlvbiBjdXN0b20gcmVzb3VyY2VcbiAqIGluIG9yZGVyIHRvIGFsbG93IHVzIHRvIGNvbnRyb2wgdGhlIElBTSByb2xlIHdoaWNoIGNyZWF0ZXMgdGhlIGNsdXN0ZXIuIFRoaXNcbiAqIGlzIHJlcXVpcmVkIGluIG9yZGVyIHRvIGJlIGFibGUgdG8gYWxsb3cgQ2xvdWRGb3JtYXRpb24gdG8gaW50ZXJhY3Qgd2l0aCB0aGVcbiAqIGNsdXN0ZXIgdmlhIGBrdWJlY3RsYCB0byBlbmFibGUgS3ViZXJuZXRlcyBtYW5hZ2VtZW50IGNhcGFiaWxpdGllcyBsaWtlIGFwcGx5XG4gKiBtYW5pZmVzdCBhbmQgSUFNIHJvbGUvdXNlciBSQkFDIG1hcHBpbmcuXG4gKi9cbmV4cG9ydCBjbGFzcyBDbHVzdGVyUmVzb3VyY2UgZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuICBwdWJsaWMgcmVhZG9ubHkgYXR0ckVuZHBvaW50OiBzdHJpbmc7XG4gIHB1YmxpYyByZWFkb25seSBhdHRyQXJuOiBzdHJpbmc7XG4gIHB1YmxpYyByZWFkb25seSBhdHRyQ2VydGlmaWNhdGVBdXRob3JpdHlEYXRhOiBzdHJpbmc7XG4gIHB1YmxpYyByZWFkb25seSByZWY6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIElBTSByb2xlIHdoaWNoIGNyZWF0ZWQgdGhlIGNsdXN0ZXIuIEluaXRpYWxseSB0aGlzIGlzIHRoZSBvbmx5IElBTSByb2xlXG4gICAqIHRoYXQgZ2V0cyBhZG1pbmlzdHJhdG9yIHByaXZpbGFnZXMgb24gdGhlIGNsdXN0ZXIgKGBzeXN0ZW06bWFzdGVyc2ApLCBhbmRcbiAgICogd2lsbCBiZSBhYmxlIHRvIGlzc3VlIGBrdWJlY3RsYCBjb21tYW5kcyBhZ2FpbnN0IGl0LlxuICAgKi9cbiAgcHJpdmF0ZSByZWFkb25seSBjcmVhdGlvblJvbGU6IGlhbS5Sb2xlO1xuICBwcml2YXRlIHJlYWRvbmx5IHRydXN0ZWRQcmluY2lwYWxzOiBzdHJpbmdbXSA9IFtdO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBDZm5DbHVzdGVyUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgY29uc3Qgc3RhY2sgPSBTdGFjay5vZih0aGlzKTtcbiAgICBjb25zdCBwcm92aWRlciA9IENsdXN0ZXJSZXNvdXJjZVByb3ZpZGVyLmdldE9yQ3JlYXRlKHRoaXMpO1xuXG4gICAgaWYgKCFwcm9wcy5yb2xlQXJuKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1wicm9sZUFyblwiIGlzIHJlcXVpcmVkJyk7XG4gICAgfVxuXG4gICAgLy8gdGhlIHJvbGUgdXNlZCB0byBjcmVhdGUgdGhlIGNsdXN0ZXIuIHRoaXMgYmVjb21lcyB0aGUgYWRtaW5pc3RyYXRvciByb2xlXG4gICAgLy8gb2YgdGhlIGNsdXN0ZXIuXG4gICAgdGhpcy5jcmVhdGlvblJvbGUgPSBuZXcgaWFtLlJvbGUodGhpcywgJ0NyZWF0aW9uUm9sZScsIHtcbiAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5Db21wb3NpdGVQcmluY2lwYWwoLi4ucHJvdmlkZXIucm9sZXMubWFwKHggPT4gbmV3IGlhbS5Bcm5QcmluY2lwYWwoeC5yb2xlQXJuKSkpXG4gICAgfSk7XG5cbiAgICAvLyB0aGUgQ3JlYXRlQ2x1c3RlciBBUEkgd2lsbCBhbGxvdyB0aGUgY2x1c3RlciB0byBhc3N1bWUgdGhpcyByb2xlLCBzbyB3ZVxuICAgIC8vIG5lZWQgdG8gYWxsb3cgdGhlIGxhbWJkYSBleGVjdXRpb24gcm9sZSB0byBwYXNzIGl0LlxuICAgIHRoaXMuY3JlYXRpb25Sb2xlLmFkZFRvUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGFjdGlvbnM6IFsgJ2lhbTpQYXNzUm9sZScgXSxcbiAgICAgIHJlc291cmNlczogWyBwcm9wcy5yb2xlQXJuIF1cbiAgICB9KSk7XG5cbiAgICAvLyBpZiB3ZSBrbm93IHRoZSBjbHVzdGVyIG5hbWUsIHJlc3RyaWN0IHRoZSBwb2xpY3kgdG8gb25seSBhbGxvd1xuICAgIC8vIGludGVyYWN0aW5nIHdpdGggdGhpcyBzcGVjaWZpYyBjbHVzdGVyIG90aGVyd2lzZSwgd2Ugd2lsbCBoYXZlIHRvIGdyYW50XG4gICAgLy8gdGhpcyByb2xlIHRvIG1hbmFnZSBhbGwgY2x1c3RlcnMgaW4gdGhlIGFjY291bnQuIHRoaXMgbXVzdCBiZSBsYXp5IHNpbmNlXG4gICAgLy8gYHByb3BzLm5hbWVgIG1heSBjb250YWluIGEgbGF6eSB2YWx1ZSB0aGF0IGNvbmRpdGlvbmFsbHkgcmVzb2x2ZXMgdG8gYVxuICAgIC8vIHBoeXNpY2FsIG5hbWUuXG4gICAgY29uc3QgcmVzb3VyY2VBcm5zID0gTGF6eS5saXN0VmFsdWUoe1xuICAgICAgcHJvZHVjZTogKCkgPT4ge1xuICAgICAgICBjb25zdCBhcm4gPSBzdGFjay5mb3JtYXRBcm4oY2x1c3RlckFybkNvbXBvbmVudHMoc3RhY2sucmVzb2x2ZShwcm9wcy5uYW1lKSkpO1xuICAgICAgICByZXR1cm4gc3RhY2sucmVzb2x2ZShwcm9wcy5uYW1lKVxuICAgICAgICAgID8gWyBhcm4sIGAke2Fybn0vKmAgXSAvLyBzZWUgaHR0cHM6Ly9naXRodWIuY29tL2F3cy9hd3MtY2RrL2lzc3Vlcy82MDYwXG4gICAgICAgICAgOiBbICcqJyBdO1xuICAgICAgfVxuICAgIH0pO1xuXG4gICAgY29uc3QgZmFyZ2F0ZVByb2ZpbGVSZXNvdXJjZUFybiA9IExhenkuc3RyaW5nVmFsdWUoe1xuICAgICAgcHJvZHVjZTogKCkgPT4gc3RhY2sucmVzb2x2ZShwcm9wcy5uYW1lKVxuICAgICAgICA/IHN0YWNrLmZvcm1hdEFybih7IHNlcnZpY2U6ICdla3MnLCByZXNvdXJjZTogJ2ZhcmdhdGVwcm9maWxlJywgcmVzb3VyY2VOYW1lOiBzdGFjay5yZXNvbHZlKHByb3BzLm5hbWUpICsgJy8qJyB9KVxuICAgICAgICA6ICcqJ1xuICAgIH0pO1xuXG4gICAgdGhpcy5jcmVhdGlvblJvbGUuYWRkVG9Qb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgYWN0aW9uczogWyAnZWMyOkRlc2NyaWJlU3VibmV0cycgXSxcbiAgICAgIHJlc291cmNlczogWyAnKicgXSxcbiAgICB9KSk7XG5cbiAgICB0aGlzLmNyZWF0aW9uUm9sZS5hZGRUb1BvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICBhY3Rpb25zOiBbXG4gICAgICAgICdla3M6Q3JlYXRlQ2x1c3RlcicsXG4gICAgICAgICdla3M6RGVzY3JpYmVDbHVzdGVyJyxcbiAgICAgICAgJ2VrczpEZWxldGVDbHVzdGVyJyxcbiAgICAgICAgJ2VrczpVcGRhdGVDbHVzdGVyVmVyc2lvbicsXG4gICAgICAgICdla3M6VXBkYXRlQ2x1c3RlckNvbmZpZycsXG4gICAgICAgICdla3M6Q3JlYXRlRmFyZ2F0ZVByb2ZpbGUnLFxuICAgICAgICAnZWtzOlRhZ1Jlc291cmNlJyxcbiAgICAgICAgJ2VrczpVbnRhZ1Jlc291cmNlJ1xuICAgICAgXSxcbiAgICAgIHJlc291cmNlczogcmVzb3VyY2VBcm5zXG4gICAgfSkpO1xuXG4gICAgdGhpcy5jcmVhdGlvblJvbGUuYWRkVG9Qb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgYWN0aW9uczogWyAnZWtzOkRlc2NyaWJlRmFyZ2F0ZVByb2ZpbGUnLCAnZWtzOkRlbGV0ZUZhcmdhdGVQcm9maWxlJyBdLFxuICAgICAgcmVzb3VyY2VzOiBbIGZhcmdhdGVQcm9maWxlUmVzb3VyY2VBcm4gXVxuICAgIH0pKTtcblxuICAgIHRoaXMuY3JlYXRpb25Sb2xlLmFkZFRvUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGFjdGlvbnM6IFsgJ2lhbTpHZXRSb2xlJyBdLFxuICAgICAgcmVzb3VyY2VzOiBbICcqJyBdLFxuICAgIH0pKTtcblxuICAgIHRoaXMuY3JlYXRpb25Sb2xlLmFkZFRvUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGFjdGlvbnM6IFsgJ2lhbTpDcmVhdGVTZXJ2aWNlTGlua2VkUm9sZScgXSxcbiAgICAgIHJlc291cmNlczogWyAnKicgXSxcbiAgICB9KSk7XG5cbiAgICBjb25zdCByZXNvdXJjZSA9IG5ldyBjZm4uQ3VzdG9tUmVzb3VyY2UodGhpcywgJ1Jlc291cmNlJywge1xuICAgICAgcmVzb3VyY2VUeXBlOiBDTFVTVEVSX1JFU09VUkNFX1RZUEUsXG4gICAgICBwcm92aWRlcjogcHJvdmlkZXIucHJvdmlkZXIsXG4gICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgIENvbmZpZzogcHJvcHMsXG4gICAgICAgIEFzc3VtZVJvbGVBcm46IHRoaXMuY3JlYXRpb25Sb2xlLnJvbGVBcm5cbiAgICAgIH1cbiAgICB9KTtcblxuICAgIHJlc291cmNlLm5vZGUuYWRkRGVwZW5kZW5jeSh0aGlzLmNyZWF0aW9uUm9sZSk7XG5cbiAgICB0aGlzLnJlZiA9IHJlc291cmNlLnJlZjtcbiAgICB0aGlzLmF0dHJFbmRwb2ludCA9IFRva2VuLmFzU3RyaW5nKHJlc291cmNlLmdldEF0dCgnRW5kcG9pbnQnKSk7XG4gICAgdGhpcy5hdHRyQXJuID0gVG9rZW4uYXNTdHJpbmcocmVzb3VyY2UuZ2V0QXR0KCdBcm4nKSk7XG4gICAgdGhpcy5hdHRyQ2VydGlmaWNhdGVBdXRob3JpdHlEYXRhID0gVG9rZW4uYXNTdHJpbmcocmVzb3VyY2UuZ2V0QXR0KCdDZXJ0aWZpY2F0ZUF1dGhvcml0eURhdGEnKSk7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyB0aGUgQVJOIG9mIHRoZSBjbHVzdGVyIGNyZWF0aW9uIHJvbGUgYW5kIGdyYW50cyBgdHJ1c3RlZFJvbGVgXG4gICAqIHBlcm1pc3Npb25zIHRvIGFzc3VtZSB0aGlzIHJvbGUuXG4gICAqL1xuICBwdWJsaWMgZ2V0Q3JlYXRpb25Sb2xlQXJuKHRydXN0ZWRSb2xlPzogaWFtLklSb2xlKTogc3RyaW5nIHtcblxuICAgIGlmICghdHJ1c3RlZFJvbGUpIHtcbiAgICAgIHJldHVybiB0aGlzLmNyZWF0aW9uUm9sZS5yb2xlQXJuO1xuICAgIH1cblxuICAgIGlmICghdGhpcy50cnVzdGVkUHJpbmNpcGFscy5pbmNsdWRlcyh0cnVzdGVkUm9sZS5yb2xlQXJuKSkge1xuICAgICAgaWYgKCF0aGlzLmNyZWF0aW9uUm9sZS5hc3N1bWVSb2xlUG9saWN5KSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcigndW5leHBlY3RlZDogY2x1c3RlciBjcmVhdGlvbiByb2xlIG11c3QgaGF2ZSB0cnVzdCBwb2xpY3knKTtcbiAgICAgIH1cblxuICAgICAgdGhpcy5jcmVhdGlvblJvbGUuYXNzdW1lUm9sZVBvbGljeS5hZGRTdGF0ZW1lbnRzKG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgYWN0aW9uczogWyAnc3RzOkFzc3VtZVJvbGUnIF0sXG4gICAgICAgIHByaW5jaXBhbHM6IFsgbmV3IGlhbS5Bcm5QcmluY2lwYWwodHJ1c3RlZFJvbGUucm9sZUFybikgXVxuICAgICAgfSkpO1xuXG4gICAgICB0aGlzLnRydXN0ZWRQcmluY2lwYWxzLnB1c2godHJ1c3RlZFJvbGUucm9sZUFybik7XG4gICAgfVxuICAgIHJldHVybiB0aGlzLmNyZWF0aW9uUm9sZS5yb2xlQXJuO1xuICB9XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBjbHVzdGVyQXJuQ29tcG9uZW50cyhjbHVzdGVyTmFtZTogc3RyaW5nKTogQXJuQ29tcG9uZW50cyB7XG4gIHJldHVybiB7XG4gICAgc2VydmljZTogJ2VrcycsXG4gICAgcmVzb3VyY2U6ICdjbHVzdGVyJyxcbiAgICByZXNvdXJjZU5hbWU6IGNsdXN0ZXJOYW1lLFxuICB9O1xufSJdfQ==