"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cloudformation_1 = require("@aws-cdk/aws-cloudformation");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const cr = require("@aws-cdk/custom-resources");
const path = require("path");
const HANDLER_DIR = path.join(__dirname, 'cluster-resource-handler');
const HANDLER_RUNTIME = lambda.Runtime.NODEJS_12_X;
/**
 * A custom resource provider that handles cluster operations. It serves
 * multiple custom resources such as the cluster resource and the fargate
 * resource.
 *
 * @internal
 */
class ClusterResourceProvider extends aws_cloudformation_1.NestedStack {
    constructor(scope, id) {
        super(scope, id);
        const onEvent = new lambda.Function(this, 'OnEventHandler', {
            code: lambda.Code.fromAsset(HANDLER_DIR),
            description: 'onEvent handler for EKS cluster resource provider',
            runtime: HANDLER_RUNTIME,
            handler: 'index.onEvent',
            timeout: core_1.Duration.minutes(1)
        });
        const isComplete = new lambda.Function(this, 'IsCompleteHandler', {
            code: lambda.Code.fromAsset(HANDLER_DIR),
            description: 'isComplete handler for EKS cluster resource provider',
            runtime: HANDLER_RUNTIME,
            handler: 'index.isComplete',
            timeout: core_1.Duration.minutes(1)
        });
        this.provider = new cr.Provider(this, 'Provider', {
            onEventHandler: onEvent,
            isCompleteHandler: isComplete,
            totalTimeout: core_1.Duration.hours(1),
            queryInterval: core_1.Duration.minutes(1)
        });
        this.roles = [onEvent.role, isComplete.role];
    }
    static getOrCreate(scope) {
        const stack = core_1.Stack.of(scope);
        const uid = '@aws-cdk/aws-eks.ClusterResourceProvider';
        return stack.node.tryFindChild(uid) || new ClusterResourceProvider(stack, uid);
    }
}
exports.ClusterResourceProvider = ClusterResourceProvider;
//# sourceMappingURL=data:application/json;base64,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