"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const k8s_resource_1 = require("./k8s-resource");
/**
 * Manages mapping between IAM users and roles to Kubernetes RBAC configuration.
 *
 * @see https://docs.aws.amazon.com/en_us/eks/latest/userguide/add-user-role.html
 */
class AwsAuth extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.roleMappings = new Array();
        this.userMappings = new Array();
        this.accounts = new Array();
        this.stack = core_1.Stack.of(this);
        new k8s_resource_1.KubernetesResource(this, 'manifest', {
            cluster: props.cluster,
            manifest: [
                {
                    apiVersion: 'v1',
                    kind: 'ConfigMap',
                    metadata: {
                        name: 'aws-auth',
                        namespace: 'kube-system'
                    },
                    data: {
                        mapRoles: this.synthesizeMapRoles(),
                        mapUsers: this.synthesizeMapUsers(),
                        mapAccounts: this.synthesizeMapAccounts(),
                    }
                }
            ]
        });
    }
    /**
     * Adds the specified IAM role to the `system:masters` RBAC group, which means
     * that anyone that can assume it will be able to administer this Kubernetes system.
     *
     * @param role The IAM role to add
     * @param username Optional user (defaults to the role ARN)
     */
    addMastersRole(role, username) {
        this.addRoleMapping(role, {
            username,
            groups: ['system:masters']
        });
    }
    /**
     * Adds a mapping between an IAM role to a Kubernetes user and groups.
     *
     * @param role The IAM role to map
     * @param mapping Mapping to k8s user name and groups
     */
    addRoleMapping(role, mapping) {
        this.roleMappings.push({ role, mapping });
    }
    /**
     * Adds a mapping between an IAM user to a Kubernetes user and groups.
     *
     * @param user The IAM user to map
     * @param mapping Mapping to k8s user name and groups
     */
    addUserMapping(user, mapping) {
        this.userMappings.push({ user, mapping });
    }
    /**
     * Additional AWS account to add to the aws-auth configmap.
     * @param accountId account number
     */
    addAccount(accountId) {
        this.accounts.push(accountId);
    }
    synthesizeMapRoles() {
        return core_1.Lazy.anyValue({
            produce: () => this.stack.toJsonString(this.roleMappings.map(m => {
                var _a;
                return ({
                    rolearn: m.role.roleArn,
                    username: (_a = m.mapping.username) !== null && _a !== void 0 ? _a : m.role.roleArn,
                    groups: m.mapping.groups
                });
            }))
        });
    }
    synthesizeMapUsers() {
        return core_1.Lazy.anyValue({
            produce: () => this.stack.toJsonString(this.userMappings.map(m => {
                var _a;
                return ({
                    userarn: m.user.userArn,
                    username: (_a = m.mapping.username) !== null && _a !== void 0 ? _a : m.user.userArn,
                    groups: m.mapping.groups
                });
            }))
        });
    }
    synthesizeMapAccounts() {
        return core_1.Lazy.anyValue({
            produce: () => this.stack.toJsonString(this.accounts)
        });
    }
}
exports.AwsAuth = AwsAuth;
//# sourceMappingURL=data:application/json;base64,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