# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['gqlalchemy',
 'gqlalchemy.graph_algorithms',
 'gqlalchemy.query_builders',
 'gqlalchemy.query_modules',
 'gqlalchemy.query_modules.push_streams',
 'gqlalchemy.vendors']

package_data = \
{'': ['*']}

install_requires = \
['adlfs>=2022.2.0,<2023.0.0',
 'dacite>=1.6.0,<2.0.0',
 'docker>=5.0.3,<6.0.0',
 'neo4j>=4.4.3,<5.0.0',
 'networkx>=2.5.1,<3.0.0',
 'psutil>=5.9.0,<6.0.0',
 'pyarrow>=7.0.0,<8.0.0',
 'pydantic>=1.8.2,<2.0.0',
 'pymgclient==1.2.0']

setup_kwargs = {
    'name': 'gqlalchemy',
    'version': '1.3.1',
    'description': 'GQLAlchemy is library developed with purpose of assisting writing and running queries on Memgraph.',
    'long_description': '# GQLAlchemy\n\n\n<p>\n    <a href="https://github.com/memgraph/gqlalchemy/actions"><img src="https://github.com/memgraph/gqlalchemy/workflows/Build%20and%20Test/badge.svg" /></a>\n    <a href="https://github.com/memgraph/gqlalchemy/blob/main/LICENSE"><img src="https://img.shields.io/github/license/memgraph/gqlalchemy" /></a>\n    <a href="https://pypi.org/project/gqlalchemy"><img src="https://img.shields.io/pypi/v/gqlalchemy" /></a>\n    <a href="https://github.com/psf/black"><img alt="Code style: black" src="https://img.shields.io/badge/code%20style-black-000000.svg"></a>\n    <a href="https://memgraph.com/docs/gqlalchemy" alt="Documentation"><img src="https://img.shields.io/badge/documentation-GQLAlchemy-orange" /></a>\n    <a href="https://github.com/memgraph/gqlalchemy/stargazers" alt="Stargazers"><img src="https://img.shields.io/github/stars/memgraph/gqlalchemy?style=social" /></a>\n</p>\n\n**GQLAlchemy** is a fully open-source Python library and **Object Graph Mapper** (OGM) - a link between graph database objects and Python objects.\n\nAn Object Graph Mapper or OGM provides a developer-friendly workflow that allows for writing object-oriented notation to communicate with graph databases. Instead of writing Cypher queries, you will be able to write object-oriented code, which the OGM will automatically translate into Cypher queries.\n\nGQLAlchemy is built on top of Memgraph\'s low-level Python client `pymgclient`\n([PyPI](https://pypi.org/project/pymgclient/) /\n[Documentation](https://memgraph.github.io/pymgclient/) /\n[GitHub](https://github.com/memgraph/pymgclient)).\n\n## Installation\n\nBefore you install `gqlalchemy`, make sure that you have `cmake` installed by running:\n```\ncmake --version\n```\nYou can install `cmake` by following the [official instructions](https://cgold.readthedocs.io/en/latest/first-step/installation.html#).\n\nTo install `gqlalchemy`, simply run the following command:\n```\npip install gqlalchemy\n```\n\n## Build & Test\n\nThe project uses [Poetry](https://python-poetry.org/) to build the GQLAlchemy Python library. To build and run tests, execute the following command:\n`poetry install`\n\nBefore starting the tests, make sure you have an active Memgraph instance running. Execute the following command:\n`poetry run pytest .`\n\n## GQLAlchemy capabilities\n\n<details>\n<summary>🗺️ Object graph mapper</summary>\n<br>\n\nBelow you can see an example of how to create `User` and `Language` node classes, and a relationship class of type `SPEAKS`. Along with that, you can see how to create a new node and relationship and how to save them in the database. After that, you can load those nodes and relationship from the database.\n<br>\n<br>\n\n```python\nfrom gqlalchemy import Memgraph, Node, Relationship, Field\nfrom typing import Optional\n\ndb = Memgraph()\n\nclass User(Node, index=True, db=db):\n    id: str = Field(index=True, exist=True, unique=True, db=db)\n\nclass Language(Node):\n    name: str = Field(unique=True, db=db)\n\nclass Speaks(Relationship, type="SPEAKS"):\n    pass\n\nuser = User(id="3", username="John").save(db)\nlanguage = Language(name="en").save(db)\nspeaks_rel = Speaks(\n    _start_node_id = user._id,\n    _end_node_id = language._id\n).save(db)\n\nloaded_user = User(id="3").load(db=db)\nprint(loaded_user)\nloaded_speaks = Speaks(\n        _start_node_id=user._id,\n        _end_node_id=language._id\n    ).load(db)\nprint(loaded_speaks)\n```\n</details>\n\n<details>\n<summary>🔨 Query builder</summary>\n<br>\nWhen building a Cypher query, you can use a set of methods that are wrappers around Cypher clauses. \n<br>\n<br>\n\n```python\nfrom gqlalchemy import create, match\nfrom gqlalchemy.query_builder import Operator\n\nquery_create = create()\n        .node(labels="Person", name="Leslie")\n        .to(relationship_type="FRIENDS_WITH")\n        .node(labels="Person", name="Ron")\n        .execute()\n\nquery_match = match()\n        .node(labels="Person", variable="p1")\n        .to()\n        .node(labels="Person", variable="p2")\n        .where(item="p1.name", operator=Operator.EQUAL, literal="Leslie")\n        .return_(results=["p1", ("p2", "second")])\n        .execute()\n```\n</details>\n\n<details>\n<summary>🚰 Manage streams</summary>\n<br>\n\nYou can create and start Kafka or Pulsar stream using GQLAlchemy. \n<br>\n\n**Kafka stream** \n```python\nfrom gqlalchemy import MemgraphPulsarStream\n\nstream = MemgraphPulsarStream(name="ratings_stream", topics=["ratings"], transform="movielens.rating", service_url="localhost:6650")\ndb.create_stream(stream)\ndb.start_stream(stream)\n```\n\n**Pulsar stream**\n```python\nfrom gqlalchemy import MemgraphKafkaStream\n\nstream = MemgraphKafkaStream(name="ratings_stream", topics=["ratings"], transform="movielens.rating", bootstrap_servers="localhost:9093")\ndb.create_stream(stream)\ndb.start_stream(stream)\n```\n</details>\n\n<details>\n<summary>🗄️ Import table data from different sources</summary>\n<br>\n\n**Import table data to a graph database**\n\nYou can translate table data from a file to graph data and import it to Memgraph. Currently, we support reading of CSV, Parquet, ORC and IPC/Feather/Arrow file formats via the PyArrow package.\n\nRead all about it in [table to graph importer how-to guide](https://memgraph.com/docs/gqlalchemy/how-to-guides/loaders/table-to-graph-importer).\n\n**Make a custom file system importer**\n\nIf you want to read from a file system not currently supported by GQLAlchemy, or use a file type currently not readable, you can implement your own by extending abstract classes `FileSystemHandler` and `DataLoader`, respectively.\n\nRead all about it in [custom file system importer how-to guide](https://memgraph.com/docs/gqlalchemy/how-to-guides/loaders/custom-file-system-importer).\n\n</details>\n\n<details>\n<summary>⚙️ Manage Memgraph instances</summary>\n<br>\n\nYou can start, stop, connect to and monitor Memgraph instances with GQLAlchemy.\n\n**Manage Memgraph Docker instance**\n\n```python\nfrom gqlalchemy.instance_runner import (\n    DockerImage,\n    MemgraphInstanceDocker\n)\n\nmemgraph_instance = MemgraphInstanceDocker(\n    docker_image=DockerImage.MEMGRAPH, docker_image_tag="latest", host="0.0.0.0", port=7687\n)\nmemgraph = memgraph_instance.start_and_connect(restart=False)\n\nmemgraph.execute_and_fetch("RETURN \'Memgraph is running\' AS result"))[0]["result"]\n```\n\n**Manage Memgraph binary instance**\n\n```python\nfrom gqlalchemy.instance_runner import MemgraphInstanceBinary\n\nmemgraph_instance = MemgraphInstanceBinary(\n    host="0.0.0.0", port=7698, binary_path="/usr/lib/memgraph/memgraph", user="memgraph"\n)\nmemgraph = memgraph_instance.start_and_connect(restart=False)\n\nmemgraph.execute_and_fetch("RETURN \'Memgraph is running\' AS result"))[0]["result"]\n```\n</details>\n\n<details>\n<summary>🔫 Manage database triggers</summary>\n<br>\n\nBecause Memgraph supports database triggers on `CREATE`, `UPDATE` and `DELETE` operations, GQLAlchemy also implements a simple interface for maintaining these triggers.\n\n```python\nfrom gqlalchemy import Memgraph, MemgraphTrigger\nfrom gqlalchemy.models import (\n    TriggerEventType,\n    TriggerEventObject,\n    TriggerExecutionPhase,\n)\n\ndb = Memgraph()\n\ntrigger = MemgraphTrigger(\n    name="ratings_trigger",\n    event_type=TriggerEventType.CREATE,\n    event_object=TriggerEventObject.NODE,\n    execution_phase=TriggerExecutionPhase.AFTER,\n    statement="UNWIND createdVertices AS node SET node.created_at = LocalDateTime()",\n)\n\ndb.create_trigger(trigger)\ntriggers = db.get_triggers()\nprint(triggers)\n```\n</details>\n\n<details>\n<summary>💽 On-disk storage</summary>\n<br>\n\nSince Memgraph is an in-memory graph database, the GQLAlchemy library provides an on-disk storage solution for large properties not used in graph algorithms. This is useful when nodes or relationships have metadata that doesn’t need to be used in any of the graph algorithms that need to be carried out in Memgraph, but can be fetched after. Learn all about it in the [on-disk storage how-to guide](https://memgraph.com/docs/gqlalchemy/how-to-guides/on-disk-storage).\n</details>\n\n<br>\n\nIf you want to learn more about OGM, query builder, managing streams, importing data from different source, managing Memgraph instances, managing database triggers and using on-disk storage, check out the GQLAlchemy [how-to guides](https://memgraph.com/docs/gqlalchemy/how-to-guides).\n\n## Development (how to build)\n```\npoetry run flake8 .\npoetry run black .\npoetry run pytest . -k "not slow"\n```\n\n## Documentation\n\nThe GQLAlchemy documentation is available on [memgraph.com/docs/gqlalchemy](https://memgraph.com/docs/gqlalchemy/).\n\nThe documentation can be generated by executing:\n```\npip3 install python-markdown\npython-markdown\n```\n\n## License\n\nCopyright (c) 2016-2022 [Memgraph Ltd.](https://memgraph.com)\n\nLicensed under the Apache License, Version 2.0 (the "License"); you may not use\nthis file except in compliance with the License. You may obtain a copy of the\nLicense at\n\n     http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software distributed\nunder the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR\nCONDITIONS OF ANY KIND, either express or implied. See the License for the\nspecific language governing permissions and limitations under the License.\n',
    'author': 'Boris Tasevski',
    'author_email': 'boris.tasevski@memgraph.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/memgraph/gqlalchemy',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
