"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
require("@aws-cdk/assert/jest");
const kinesis = require("@aws-cdk/aws-kinesis");
// --------------------------------------------------------------
// Test construct properties
// --------------------------------------------------------------
test('Test construct properties', () => {
    const stack = new core_1.Stack();
    const pattern = new lib_1.ApiGatewayToKinesisStreams(stack, 'api-gateway-kinesis', {});
    expect(pattern.apiGateway !== null);
    expect(pattern.apiGatewayRole !== null);
    expect(pattern.apiGatewayCloudWatchRole !== null);
    expect(pattern.apiGatewayLogGroup !== null);
    expect(pattern.kinesisStream !== null);
});
// --------------------------------------------------------------
// Test deployment w/ overwritten properties
// --------------------------------------------------------------
test('Test deployment w/ overwritten properties', () => {
    const stack = new core_1.Stack();
    new lib_1.ApiGatewayToKinesisStreams(stack, 'api-gateway-kinesis', {
        apiGatewayProps: {
            restApiName: 'my-api',
            deployOptions: {
                methodOptions: {
                    '/*/*': {
                        throttlingRateLimit: 100,
                        throttlingBurstLimit: 25
                    }
                }
            }
        },
        kinesisStreamProps: {
            shardCount: 1,
            streamName: 'my-stream'
        },
        putRecordRequestTemplate: `{ "Data": "$util.base64Encode($input.json('$.foo'))", "PartitionKey": "$input.path('$.bar')" }`,
        putRecordRequestModel: { schema: {} },
        putRecordsRequestTemplate: `{ "Records": [ #foreach($elem in $input.path('$.records')) { "Data": "$util.base64Encode($elem.foo)", "PartitionKey": "$elem.bar"}#if($foreach.hasNext),#end #end ] }`,
        putRecordsRequestModel: { schema: {} }
    });
    expect(stack).toHaveResourceLike('AWS::ApiGateway::Stage', {
        MethodSettings: [
            {
                DataTraceEnabled: false,
                HttpMethod: '*',
                LoggingLevel: 'INFO',
                ResourcePath: '/*'
            },
            {
                HttpMethod: '*',
                ResourcePath: '/*',
                ThrottlingRateLimit: 100,
                ThrottlingBurstLimit: 25
            }
        ]
    });
    expect(stack).toHaveResource('AWS::Kinesis::Stream', {
        ShardCount: 1,
        Name: 'my-stream'
    });
});
// --------------------------------------------------------------
// Test deployment w/ existing stream
// --------------------------------------------------------------
test('Test deployment w/ existing stream', () => {
    const stack = new core_1.Stack();
    new lib_1.ApiGatewayToKinesisStreams(stack, 'api-gateway-kinesis', {
        apiGatewayProps: {},
        existingStreamObj: new kinesis.Stream(stack, 'ExistingStream', {
            shardCount: 5,
            retentionPeriod: core_1.Duration.days(4)
        })
    });
    expect(stack).toHaveResource('AWS::Kinesis::Stream', {
        ShardCount: 5,
        RetentionPeriodHours: 96
    });
});
//# sourceMappingURL=data:application/json;base64,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