# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['telepay',
 'telepay.v1',
 'telepay.v1._async',
 'telepay.v1._sync',
 'telepay.v1.models']

package_data = \
{'': ['*']}

install_requires = \
['colorama>=0.4.4,<0.5.0',
 'fastapi>=0.75.1,<0.76.0',
 'httpx>=0.22.0,<0.23.0',
 'python-dotenv>=0.20.0,<0.21.0',
 'uvicorn>=0.17.6,<0.18.0']

setup_kwargs = {
    'name': 'telepay',
    'version': '0.0.6',
    'description': 'Python SDK for the TelePay API',
    'long_description': '# Python SDK for the TelePay API\n\n![TelePay Python](https://github.com/TelePay-cash/telepay-python/blob/main/docs/cover.jpg?raw=true)\n\nOfficial TelePay client library for the Python language, so you can easely process cryptocurrency payments using the REST API.\n\n[![License: MIT](https://img.shields.io/badge/License-MIT-green.svg)](https://opensource.org/licenses/MIT)\n[![Test](https://github.com/telepay-cash/telepay-python/workflows/CI/badge.svg)](https://github.com/telepay-cash/telepay-python/actions?query=workflow%3ACI)\n[![Version](https://img.shields.io/pypi/v/telepay?color=%2334D058&label=Version&style=flat-square)](https://pypi.org/project/telepay)\n[![Last commit](https://img.shields.io/github/last-commit/telepay-cash/telepay-python.svg?style=flat-square)](https://github.com/telepay-cash/telepay-python/commits)\n[![GitHub commit activity](https://img.shields.io/github/commit-activity/m/telepay-cash/telepay-python?style=flat-square)](https://github.com/telepay-cash/telepay-python/commits)\n[![Github Stars](https://img.shields.io/github/stars/telepay-cash/telepay-python?style=flat-square&logo=github&)](https://github.com/telepay-cash/telepay-python/stargazers)\n[![Github Forks](https://img.shields.io/github/forks/telepay-cash/telepay-python?style=flat-square&logo=github)](https://github.com/telepay-cash/telepay-python/network/members)\n[![Github Watchers](https://img.shields.io/github/watchers/telepay-cash/telepay-python?style=flat-square&logo=github)](https://github.com/telepay-cash/telepay-python)\n[![GitHub contributors](https://img.shields.io/github/contributors/telepay-cash/telepay-python?label=code%20contributors&style=flat-square)](https://github.com/telepay-cash/telepay-python/graphs/contributors)\n[![Downloads](https://pepy.tech/badge/telepay?style=flat-square)](https://pepy.tech/project/telepay)\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg?style=flat-square)](https://github.com/psf/black)\n[![Imports: isort](https://img.shields.io/badge/%20imports-isort-%231674b1?style=flat-square&labelColor=ef8336)](https://pycqa.github.io/isort/)\n[![Telegram](https://img.shields.io/badge/Telegram-2CA5E0?style=flat-squeare&logo=telegram&logoColor=white)](https://t.me/TelePayCash)\n[![Blog](https://img.shields.io/badge/RSS-FFA500?style=flat-square&logo=rss&logoColor=white)](https://blog.telepay.cash)\n\n## Installation\n\nInstall the package with pip:\n\n```bash\npip install telepay\n```\n\nOr using [Poetry](https://python-poetry.org/):\n\n```bash\npoetry add telepay\n```\n\n## Using the library\n\nRefer to the [TelePay Docs](https://telepay.readme.io) and follow the [first steps guide](https://telepay.readme.io/reference/first-steps), you\'ll get your TelePay account and API key.\n\nTo make requests to the TelePay API, you need to import a client. We have two clients:\n* `TelePaySyncClient`: make requests synchronously.\n* `TelePayAsyncClient` make requests asynchronously.\n\n**Import and use the client**\n\n```python\nfrom telepay.v1 import TelePaySyncClient, TelePayAsyncClient\n\nclient = TelePaySyncClient(secret_api_key)\nclient = TelePayAsyncClient(secret_api_key)\n```\n\n**Use the client as a context manager**\n\nWe recommend using the client as a context manager, like this:\n\n```python\nwith TelePaySyncClient(secret_api_key) as client:\n    # use the client\n    ...\n```\n\nor\n\n```python\nasync with TelePayAsyncClient(secret_api_key) as client:\n    # use the client\n    ...\n```\n\n## API endpoints\n\nThe API endpoints are documented in the [TelePay documentation](https://telepay.readme.io/reference/endpoints), refer to that pages to know more about them.\n\nTo manage the requests, if the client is async, you should use the `await` keyword, like this:\n\n```python\nresponse = await client.method(...)\n```\n\nWhere `method` is the endpoint method.\n\n**get_me**\n\nInfo about the current merchant. [Read docs](https://telepay.readme.io/reference/getme)\n\n```python\naccount = client.get_me()\n```\n\n**get_balance**\n\nGet your merchant wallet assets with corresponding balance. [Read docs](https://telepay.readme.io/reference/getbalance)\n\n```python\nwallets = client.get_balance()\n```\n\n**get_assets**\n\nGet assets suported by TelePay. [Read docs](https://telepay.readme.io/reference/getassets)\n\n```python\nassets = client.get_assets()\n```\n\n**get_invoices**\n\nGet your merchant invoices. [Read docs](https://telepay.readme.io/reference/getinvoices)\n\n```python\ninvoices = client.get_invoices()\n```\n\n**get_invoice**\n\nGet invoice details, by its number. [Read docs](https://telepay.readme.io/reference/getinvoice)\n\n```python\ninvoice = client.get_invoice(number)\n```\n\n**create_invoice**\n\nCreates an invoice, associated to your merchant. [Read docs](https://telepay.readme.io/reference/createinvoice)\n\n```python\ninvoice = client.create_invoice(\n    asset=\'TON\',\n    blockchain=\'TON\',\n    network=\'mainnet\',\n    amount=1,\n    description=\'Product\',\n    metadata={\n        \'color\': \'red\',\n        \'size\': \'large\',\n    },\n    success_url=\'https://example.com/success\',\n    cancel_url=\'https://example.com/cancel\',\n    expires_at=30\n)\n```\n\n**cancel_invoice**\n\nCancel invoice, by its number. [Read docs](https://telepay.readme.io/reference/cancelinvoice)\n\n```python\ninvoice = client.cancel_invoice(number)\n```\n\n**delete_invoice**\n\nDelete invoice, by its number. [Read docs](https://telepay.readme.io/reference/deleteinvoice)\n\n```python\nstatus = client.delete_invoice(number)\n```\n\n**transfer**\n\nTransfer funds between internal wallets. Off-chain operation. [Read docs](https://telepay.readme.io/reference/transfer)\n\n```python\nstatus = client.transfer(\n    asset=\'TON\',\n    blockchain=\'TON\',\n    network=\'mainnet\',\n    amount=1,\n    username=\'test\',\n    message=\'Thanks\'\n)\n```\n\n**get_withdraw_minimum**\n\nObtains minimum amount required to withdraw funds on a given asset. [Read docs](https://telepay.readme.io/reference/getwithdrawminimum)\n\n```python\nminimum = client.get_withdraw_minimum(\n    asset=\'TON\',\n    blockchain=\'TON\',\n    network=\'mainnet\',\n)\n```\n\n**get_withdraw_fee**\n\nGet estimated withdraw fee, composed of blockchain fee and processing fee. [Read docs](https://telepay.readme.io/reference/getwithdrawfee)\n\n```python\nfees = client.get_withdraw_fee(\n    to_address=\'EQCKYK7bYBt1t8UmdhImrbiSzC5ijfo_H3Zc_Hk8ksRpOkOk\',\n    asset=\'TON\',\n    blockchain=\'TON\',\n    network=\'mainnet\',\n    amount=1,\n    message=\'test\'\n)\n```\n\n**withdraw**\n\nWithdraw funds from merchant wallet to external wallet. On-chain operation. [Read docs](https://telepay.readme.io/reference/withdraw)\n\n```python\nstatus = client.withdraw(\n    to_address=\'EQCKYK7bYBt1t8UmdhImrbiSzC5ijfo_H3Zc_Hk8ksRpOkOk\',\n    asset=\'TON\',\n    blockchain=\'TON\',\n    network=\'mainnet\',\n    amount=1,\n    message=\'test\'\n)\n```\n\n## Webhooks\n\n> Webhooks allows you to get updates delivered to your app server whenever we have events on our side. We will send a POST request over HTTPS, with serialized JSON data of the event, to the endpoint defined by you in the Developers > Webhooks section, in your merchant dashboard. [Read more in the docs](https://telepay.readme.io/reference/webhooks).\n\n\nThe `telepay.v1.webhooks` module contains utilities to manage webhooks received on your side.\n\n* `get_signature(data, secret)`: Returns a webhook signature, used to verify data integrity of the webhook. This is optional, but highly recommended.\n* `TelePayWebhookListener`: A lightweight webhook listener, to receive webhooks from TelePay. You could build your own, like using django views, flask views, or any other web framework. Your choice.\n\n**Example using `TelePayWebhookListener`**\n\n```python\nfrom telepay.v1.webhooks import TelePayWebhookListener\n\n\ndef callback(headers, data):\n    print("Executing callback...")\n    # do your stuff here\n\n\nlistener = TelePayWebhookListener(\n    secret="SECRET",\n    callback=callback,\n    host="localhost",\n    port=5000,\n    url="/webhook",\n    log_level="error",\n)\n\nlistener.listen()\n```\n\nRunning the listener will output something like this:\n\n![Webhook listener](https://github.com/TelePay-cash/telepay-python/blob/main/docs/webhook-listener.png?raw=true)\n\nModify the listener parameters to your needs, knowing this:\n* `secret`: A secret token that your side knows and it\'s configured in the webhook definition, on the TelePay dashboard.\n* `callback`: The callback function that is called when new webhook arrives, receiving it\'s HTTP headers and data.\n* `host`: The host on which the listener will be running.\n* `port`: The port on which the listener will be exposed.\n* `url`: The webhook url, which is secret and should only be known by your app and TelePay. Otherwise, it could lead to security issues.\n* `log_level`: The listener logger level, like `"error"`, `"info"` or `"debug"`.\n\n## Contributors ✨\n\nThe library is made by ([emoji key](https://allcontributors.org/docs/en/emoji-key)):\n\n<!-- ALL-CONTRIBUTORS-LIST:START - Do not remove or modify this section -->\n<!-- prettier-ignore-start -->\n<!-- markdownlint-disable -->\n<table>\n  <tr>\n    <td align="center"><a href="https://lugodev.com"><img src="https://avatars.githubusercontent.com/u/18733370?v=4?s=100" width="100px;" alt=""/><br /><sub><b>Carlos Lugones</b></sub></a><br /><a href="https://github.com/telepay-cash/telepay-python/commits?author=lugodev" title="Code">💻</a></td>\n    <td align="center"><a href="http://showwcase.com/ravenclawldz"><img src="https://avatars.githubusercontent.com/u/68219934?v=4" width="100px;" alt=""/><br /><sub><b>Ravenclaw.ldz</b></sub></a><br /><a href="https://github.com/telepay-cash/telepay-python/commits?author=ravenclawldz" title="Code">💻</a></td>\n  </tr>\n</table>\n<!-- markdownlint-restore -->\n<!-- prettier-ignore-end -->\n\n<!-- ALL-CONTRIBUTORS-LIST:END -->\n\nThis project follows the [all-contributors](https://github.com/all-contributors/all-contributors) specification. Contributions of any kind welcome!\n',
    'author': 'Carlos Lugones',
    'author_email': 'contact@lugodev.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/TelePay-cash/telepay-python',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
