# (C) Copyright 1996- ECMWF.
#
# This software is licensed under the terms of the Apache Licence Version 2.0
# which can be obtained at http://www.apache.org/licenses/LICENSE-2.0.
# In applying this licence, ECMWF does not waive the privileges and immunities
# granted to it by virtue of its status as an intergovernmental organisation
# nor does it submit to any jurisdiction.

from logging import warning
from typing import Optional, Union

import attrs
import numpy as np
import scipy.interpolate
import scipy.special
import scipy.stats

from ..utils import (
    RunningWindowOverDaysOfYear,
    StatisticalModel,
    day_of_year,
    ecdf,
    get_mask_for_unique_subarray,
    get_years_and_yearly_means,
    iecdf,
    infer_and_create_time_arrays_if_not_given,
    interp_sorted_cdf_vals_on_given_length,
    month,
    quantile_map_non_parametically,
    quantile_map_x_on_y_non_parametically,
    sort_array_like_another_one,
    threshold_cdf_vals,
    year,
)
from ..variables import Variable
from ._debiaser import Debiaser
from ._isimip_options import isimip3_general_settings, isimip3_variable_settings


@attrs.define(slots=False)
class ISIMIP(Debiaser):
    """
    |br| Implements the ISIMIP3b and ISIMIP3BASD bias adjustment methodology based on Lange 2019 and Lange 2021.

    ISIMIP is a semi-parametric quantile mapping method that attempts to be trend-preserving in all quantiles by generating ‘pseudo future observations’ and executing the quantile mapping between the future climate model and the pseudo future observations. ISIMIP includes special cases for several the variables, and for a complete description of the methodology we refer to the ISIMIP documentation.

    `This notebook <https://nbviewer.org/github/esowc/ibicus/blob/main/notebooks/04%20ISIMIP%20Consistency.ipynb>`_ demonstrates that the results obtained with this implementation of ISIMIP are consistent with the reference implementation.

    ISIMIP covers the following variables::

        ["hurs", "pr", "prsnratio", "psl", "rsds", "rlds", "sfcwind", "tas", "tasrange", "taskew"]

    The variables tasmin, tasmax and prsn can be derived as: ``tasrange = tasmin - tasmax, tasskew = (tas-tasmin)/(tasmax-tasmin)`` and ``prsnratio = prsn/pr``.
    The ISIMIP methodology consists of 8 steps that cover the special cases and particularities for the different variables. We refer to Lange 2019 for a complete description. In the following the steps shall only be sketched.
    Let obs refer to the observation dataset and cm_hist and cm_fut to historical and future climate model values.

    **Steps:**

    1. :py:func:`step1`: (For rsds only). If ``scale_by_annual_cycle_of_upper_bounds = True`` then an annual cycle of upper bounds is calculated to scale the values in obs, cm_hist and cm_fut onto the interval [0,1]. This is to account physical upper bounds that vary over the year, while other variables have static upper and lower bounds. |brr|

    2. :py:func:`step2`: (For prsnratio only). If ``impute_missing_values = True`` missing values (passed in either as masked array or array with np.nan values) in obs, cm_hist and cm_fut are imputed via a somewhat trend preserving sampling from all other non-missing values within a given time window. |brr|

    3. :py:func:`step3`: (For tas, psl and rlds). If ``detrending = True`` the timeseries are detrended to account for trends within historical and future period (not for trends between historical and future one as this is done in step 5). Trends are estimated via linear regression on the annual means and then removed from the daily observations. If ``detrending_with_significance_test = True`` then a significance test is first used to determine whether the trend is to be removed. |brr|

    4. :py:func:`step4`: If the variable has either a lower or upper bound and threshold, values beyond the threshold are randomized within between the respective bound and threshold. This allows to quantile map these values in step 6. Randomization is done uniformly (different than from Lange 2019 to reflect a change in ISIMIP3b v2.2), however ranks are preserved. |brr|

    5. :py:func:`step5`: Here pseudo future observations are generated by transferring a climate change signal onto the observations. This is done differently depending on the trend_preservation_method: one of ``["additive", "multiplicative", "mixed", "bounded"]`` to reflect differences in the trends of different variables. For bounded variables if ``trend_transfer_only_for_values_within_threshold = True`` then trend transfer only happens for the values within thresholds, else for all values. |brr|

    6. :py:func:`step6`: Given the pseudo future observations from step 5 a quantile mapping is used mapping the values of cm_fut to the pseudo future observations to generate debiased future climate model values.

        - If ``event_likelihood_adjustment = True`` also an event likelihood adjustment is done, following Switanek et al. 2017 and as originally described in Lange 2019, but removed in the ISIMIP3b run.

        - By default the quantile mapping is done parametically using the distribution in self.distribution, however if ``nonparametric_qm = True`` or ``ks_test_for_goodness_of_cdf_fit = True`` and a Kolmogorov Smirnov statistic indicats a bigger misfit a nonparametric quantile mapping is used instead (default for hurs, psrnratio, rsds, tasskew).

        - If ``bias_correct_frequencies_of_values_beyond_thresholds = True`` (default) also the frequencies of values beyond thresholds are adjusted to. This adjusts eg. the number of precipitation dry days. |brr|

    7. :py:func:`step7`: If ``detrending = True`` then the trend removed in step 3 is restored onto the debiased cm_fut. |brr|

    8. :py:func:`step8`: (For rsds only). If ``scale_by_annual_cycle_of_upper_bounds = True`` then a debiased annual cycle of upper bounds is calculated to rescale the debiased climate model values from [0,1] back to the original scale. |brr|

    As for all other debiasers all steps are applied locationwise and if ``running_window_mode = True`` (default) in a running window with step size ``running_window_step_length`` and length ``running_window_length``. Otherwise steps are applied on a month by month basis to account for seasonality.

    .. warning:: Currently only uneven sizes are allowed for window length and window step length. This allows symmetrical windows of the form [window_center - window length//2, window_center + window length//2] given an arbitrary window center.

    All variables have different default settings and using the ``ISIMIP.from_variable(variable, **kwargs)`` method it is possible to initialize a debiaser with these settings eg.

    >>> debiaser = ISIMIP.from_variable("tas")

    However depending on the debiasing region and usecase at hand it might be important to modify some of the default settings, such as for example:

    >>> debiaser = ISIMIP.from_variable("tas", detrending = False)

    or

    >>> debiaser = ISIMIP.from_variable("tas")
    >>> debiaser.detrending = False

    If run with the default options, the results generated in this package are exactly identical to v3.0.1 (2022-06-27) of the `ISIMIP3BASD reference implementation <https://zenodo.org/record/6758997/>`_, as shown in the notebook `04 ISIMIP Consistency <https://nbviewer.org/github/esowc/ibicus/blob/main/notebooks/04%20ISIMIP%20Consistency.ipynb>`_. By setting parameters appropriately reproducing lower versions is possible and also consistency with future versions is considered.

    In contrast to the reference implementation implementation of ISIMIP this code includes further options for customization of the ISIMIP behavior, allows more flexible usage given that it operates on a numerical basis and handles a variety of cases that might lead to bugs in the reference implementation.

    **References**:

    - Lange, S. (2019). Trend-preserving bias adjustment and statistical downscaling with ISIMIP3BASD (v1.0). In Geoscientific Model Development (Vol. 12, Issue 7, pp. 3055–3070). Copernicus GmbH. https://doi.org/10.5194/gmd-12-3055-2019
    - Lange, S. (2022). ISIMIP3BASD (3.0.1) [Computer software]. Zenodo. https://doi.org/10.5281/ZENODO.6758997
    - Switanek, M. B., Troch, P. A., Castro, C. L., Leuprecht, A., Chang, H.-I., Mukherjee, R., & Demaria, E. M. C. (2017). Scaled distribution mapping: a bias correction method that preserves raw climate model projected changes. In Hydrology and Earth System Sciences (Vol. 21, Issue 6, pp. 2649–2666). Copernicus GmbH. https://doi.org/10.5194/hess-21-2649-2017.

    |br|
    **Usage information:**

    - Default settings exist for: ``["hurs", "pr", "prsnratio", "psl", "rlds", "rsds", "sfcWind", "tas", "tasrange", "tasskew"]``.

    - :py:func:`apply` requires: time arguments ``time_obs``, ``time_cm_hist``, and ``time_cm_future`` next to ``obs``, ``cm_hist`` and ``cm_future``. These are just 1d numpy-arrays of dates (multiple formats are possible as long as they as convertible to numpy or datetime dates) specifying the date for each value/timestep in ``obs``, ``cm_hist`` and ``cm_future``. If they are not specified they are inferred, assuming the first observation in all three observation/climate value arrays is on a 1st of January.

    - The debiaser assumes daily data.

    |br|
    **Examples:**

    Running without dates (they are inferred assuming the first value in ``obs``, ``cm_hist`` and ``cm_future`` always corresponds to a January 1st):

    >>> debiaser = ISIMIP.from_variable("tas")
    >>> debiaser.apply(obs, cm_hist, cm_future)

    Running with dates:

    >>> debiaser = ISIMIP.from_variable("tas")
    >>> debiaser.apply(obs, cm_hist, cm_future, time_obs = time_obs, time_cm_hist = time_cm_hist, time_cm_future = time_cm_future)

    |br|
    **Core Parameters:**

    :Parameters:
        **trend_preservation_method** : str
            One of ``["additive", "multiplicative", "mixed", "bounded"]``. Method of how the climate change trend is transferred to observations to generate the pseudo future observations. Depends on the kind of climate change trend expected in the variable.
        **distribution** : Union[scipy.stats.rv_continuous, scipy.stats.rv_discrete, scipy.stats.rv_histogram, StatisticalModel]
            Distribution or statistical model used for the quantile mapping in step 6 if ``nonparametric_qm = False``. |br|
            Usually a distribution in ``scipy.stats.rv_continuous``, but can also be an empirical distribution as given by ``scipy.stats.rv_histogram`` or a more complex statistical model as wrapped by the :py:class:`ibicus.utils.StatisticalModel` class.
        **nonparametric_qm** : bool
            Whether nonparametric quantile mapping is used for debiasing in step 6. If ``nonparametric_qm = False`` then quantile mapping is done using the distribution in self.distribution.
        **detrending** : bool
            Whether detrending is applied in step4 to account for within obs, cm_hist and cm_fut trends.

    |br|

    **Full list of attributes:**

    Attributes
    ----------
    lower_bound : float
        Variable specific: Lower bound of a variable. Default: ``-np.inf``.
    lower_threshold : float
        Variable specific: Lower threshold of a variable. Default: ``-np.inf``.
    upper_bound : float
        Variable specific: Upper bound of a variable. Default: ``np.inf``.
    upper_threshold : float
        Variable specific: Upper threshold of a variable. Default: ``np.inf``.

    scale_by_annual_cycle_of_upper_bounds** : bool
        Step 1: Whether in step 1 a scaling by an estimated annual cycle of upper bounds is done to scale obs, cm_hist and cm_fut values onto [0,1]. Default: ``False``.
    window_length_annual_cycle_of_upper_bounds** : int
        Step 1: Window length to estimate an annual cycle of upper bounds to rescale. Only relevant if ``scale_by_annual_cycle_of_upper_bounds = True``. Default: ``31``.

    impute_missing_values : bool
        Step 2: Whether in step 2 missing values (indicated either by a mask of ``np.nan`` values) are imputed using the other values within the window.
    detrending : bool
        Step 3: Whether detrending is applied in step4 to account for within obs, cm_hist and cm_fut trends.

    detrending_with_significance_test : bool
        Step 3: Whether detrending is applied conditional on a significance test for the trend. Only relevant if ``detrending = True``. Default: ``True``.

    trend_preservation_method : str
        Step 5: One of ``["additive", "multiplicative", "mixed", "bounded"]``. Method of how the climate change trend is transferred to observations to generate pseudo future ones.
    trend_transfer_only_for_values_within_threshold : bool
        Step 5: Whether the trend transfer is done only for the values within thresholds. Default: ``True``.

    distribution : Union[scipy.stats.rv_continuous, scipy.stats.rv_discrete, scipy.stats.rv_histogram, StatisticalModel]
        Step 6: Distribution or statistical model used for the quantile mapping in step 6 if ``nonparametric_qm = False``. |br|
        Usually a distribution in ``scipy.stats.rv_continuous``, but can also be an empirical distribution as given by ``scipy.stats.rv_histogram`` or a more complex statistical model as wrapped by the :py:class:`ibicus.utils.StatisticalModel` class.
    nonparametric_qm : bool
        Step 6: Whether nonparametric quantile mapping is used for debiasing in step 6. If ``nonparametric_qm = False`` then quantile mapping is done using the distribution in ``distribution``.
    bias_correct_frequencies_of_values_beyond_thresholds : bool
        Step 6: Whether frequencies of values beyond thresholds are adjusted/bias corrected. Default: ``True``.
    event_likelihood_adjustment : bool
        Step 6: Whether in conjuction with quantile mapping an event likelihood adjustment is used inspired by Switanek et al. 2017. Default: ``False``.
    ks_test_for_goodness_of_cdf_fit : bool
        Step 6: Whether a Kolmogorov Smirnov statistic is used to diagnose misfit of the parametric CDF prior to quantile mapping. If such misfit is diagnosed nonparametric quantile mapping is used. Only relevant if ``nonparametric_qm = False``. Default: ``True``.

    ecdf_method : str
        Computation: One of ``["kernel_density", "linear_interpolation", "step_function"]``. Method to calculate the empirical CDF. Default: ``"linear_interpolation"``.
    iecdf_method : str
        Computation: One of ``["inverted_cdf", "averaged_inverted_cdf", closest_observation", "interpolated_inverted_cdf", "hazen", "weibull", "linear", "median_unbiased", "normal_unbiased"]``. Method to calculate the inverse empirical CDF (empirical quantile function). Default: ``"linear"``.
    mode_non_parametric_qm : str
        Computation: One of ``["normal", "isimipv3.0"]``. Computation method used for nonparametric quantile mapping. If ``"normal"`` this is based on estimated empirical CDFs and inverse empirical CDFs. If ``"isimipv3.0"`` this is based on estimated ranks using ``scipy.stats.rankdata`` which can lead to inconsistencies. Default: ``"normal"``.

    running_window_mode : bool
        Iteration: Whether ISIMIP is used in running window mode to account for seasonalities. If ``running_window_mode = False`` then ISIMIP is applied by months. Default: ``True``.
    running_window_length : int
        Iteration: Length of the running window in days: how many values are used to the debiased climate model values. Only relevant if ``running_window_mode = True``. Default: ``31``.
    running_window_step_length : int
        Iteration: Step length of the running window in days: how many values are debiased inside the running window. Only relevant if ``running_window_mode = True``. Default: ``1``.
    """

    # Core arguments

    # Step 5
    trend_preservation_method: str = attrs.field(
        validator=attrs.validators.in_(["additive", "multiplicative", "mixed", "bounded"])
    )

    # Step 6
    distribution: Union[
        scipy.stats.rv_continuous, scipy.stats.rv_discrete, scipy.stats.rv_histogram, StatisticalModel, None
    ] = attrs.field(
        validator=attrs.validators.instance_of(
            (scipy.stats.rv_continuous, scipy.stats.rv_discrete, scipy.stats.rv_histogram, StatisticalModel, type(None))
        )
    )
    nonparametric_qm: bool = attrs.field(validator=attrs.validators.instance_of(bool))

    # Step 3
    detrending: bool = attrs.field(validator=attrs.validators.instance_of(bool))  # step 3

    # Variable bounds
    lower_bound: float = attrs.field(default=np.inf, validator=attrs.validators.instance_of(float), converter=float)
    lower_threshold: float = attrs.field(default=np.inf, validator=attrs.validators.instance_of(float), converter=float)
    upper_bound: float = attrs.field(default=-np.inf, validator=attrs.validators.instance_of(float), converter=float)
    upper_threshold: float = attrs.field(
        default=-np.inf, validator=attrs.validators.instance_of(float), converter=float
    )

    # ISIMIP steps
    # Step 1
    scale_by_annual_cycle_of_upper_bounds: bool = attrs.field(
        default=False, validator=attrs.validators.instance_of(bool)
    )
    window_length_annual_cycle_of_upper_bounds: int = attrs.field(
        default=31, validator=attrs.validators.instance_of(int), converter=int
    )

    # Step 2
    impute_missing_values: bool = attrs.field(default=False, validator=attrs.validators.instance_of(bool))

    # Step 3
    detrending_with_significance_test: bool = attrs.field(default=True, validator=attrs.validators.instance_of(bool))

    # Step 5
    trend_transfer_only_for_values_within_threshold: bool = attrs.field(
        default=True, validator=attrs.validators.instance_of(bool)
    )
    bias_correct_frequencies_of_values_beyond_thresholds: bool = attrs.field(
        default=True, validator=attrs.validators.instance_of(bool)
    )

    # Step 6
    event_likelihood_adjustment: bool = attrs.field(default=False, validator=attrs.validators.instance_of(bool))
    ks_test_for_goodness_of_cdf_fit: bool = attrs.field(default=True, validator=attrs.validators.instance_of(bool))

    # Computation arguments
    ecdf_method: str = attrs.field(
        default="linear_interpolation",
        validator=attrs.validators.in_(["kernel_density", "linear_interpolation", "step_function"]),
    )
    iecdf_method: str = attrs.field(
        default="linear",
        validator=attrs.validators.in_(
            [
                "inverted_cdf",
                "averaged_inverted_cdf",
                "closest_observation",
                "interpolated_inverted_cdf",
                "hazen",
                "weibull",
                "linear",
                "median_unbiased",
                "normal_unbiased",
            ]
        ),
    )
    mode_non_parametric_qm: str = attrs.field(
        default="normal", validator=attrs.validators.in_(["normal", "isimipv3.0"])
    )

    # Iteration
    running_window_mode: bool = attrs.field(default=True, validator=attrs.validators.instance_of(bool))
    running_window_length: int = attrs.field(
        default=31,
        validator=[attrs.validators.instance_of(int), attrs.validators.gt(0)],
    )
    running_window_step_length: int = attrs.field(
        default=1, validator=[attrs.validators.instance_of(int), attrs.validators.gt(0)]
    )

    def __attrs_post_init__(self):
        if self.running_window_mode:
            self.running_window = RunningWindowOverDaysOfYear(
                window_length_in_days=self.running_window_length,
                window_step_length_in_days=self.running_window_step_length,
            )
        if self.distribution is None and not self.nonparametric_qm:
            raise ValueError(
                "Either self.distribution needs to be specified or self.nonparametric_qm needs to be True."
            )

    @classmethod
    def from_variable(cls, variable: Union[str, Variable], **kwargs):
        return super()._from_variable(
            cls,
            variable=variable,
            default_settings_variable=isimip3_variable_settings,
            default_settings_general=isimip3_general_settings,
            **kwargs,
        )

    @property
    def has_lower_threshold(self):
        if self.lower_threshold is not None and self.lower_threshold > -np.inf:
            return True
        else:
            return False

    @property
    def has_lower_bound(self):
        if self.lower_bound is not None and self.lower_bound > -np.inf:
            return True
        else:
            return False

    @property
    def has_upper_threshold(self):
        if self.upper_threshold is not None and self.upper_threshold < np.inf:
            return True
        else:
            return False

    @property
    def has_upper_bound(self):
        if self.upper_bound is not None and self.upper_bound < np.inf:
            return True
        else:
            return False

    @property
    def has_bound(self):
        if self.has_upper_bound or self.has_lower_bound:
            return True
        else:
            return False

    @property
    def has_threshold(self):
        if self.has_upper_threshold or self.has_lower_threshold:
            return True
        else:
            return False

    # ----- Non public helpers: General ----- #

    def _get_mask_for_values_beyond_lower_threshold(self, x):
        return x <= self.lower_threshold

    def _get_mask_for_values_beyond_upper_threshold(self, x):
        return x >= self.upper_threshold

    def _get_mask_for_values_between_thresholds(self, x):
        return (x > self.lower_threshold) & (x < self.upper_threshold)

    def _get_values_between_thresholds(self, x):
        return x[self._get_mask_for_values_between_thresholds(x)]

    def get_proportion_of_days_beyond_lower_threshold(self, x):
        return self._get_mask_for_values_beyond_lower_threshold(x).mean()

    def get_proportion_of_days_beyond_upper_threshold(self, x):
        return self._get_mask_for_values_beyond_upper_threshold(x).mean()

    @staticmethod
    def _check_time_information_and_raise_error(obs, cm_hist, cm_future, time_obs, time_cm_hist, time_cm_future):
        if obs.size != time_obs.size or cm_hist.size != time_cm_hist.size or cm_future.size != time_cm_future.size:
            raise ValueError(
                """Dimensions of time information for one of time_obs, time_cm_hist, time_cm_future do not correspond to the dimensions of obs, cm_hist, cm_future. 
                Make sure that for each one of obs, cm_hist, cm_future time information is given for each value in the arrays."""
            )

    # ----- Non public helpers: ISIMIP-steps ----- #

    def _step1_get_annual_cycle_of_upper_bounds(self, vals, days_of_year_vals):

        # Sort vals and days_of_year_vals by days of year
        argsort_days_of_year_vals = days_of_year_vals.argsort()
        vals = vals[argsort_days_of_year_vals]
        days_of_year_vals = days_of_year_vals[argsort_days_of_year_vals]

        unique_days_of_year, idx_day_of_year = np.unique(days_of_year_vals, return_index=True)

        # Multiyear maximum values
        multiyear_daily_maximum_values = np.maximum.reduceat(vals, idx_day_of_year)

        # Running maximum:
        multiyear_daily_maximum_values = scipy.ndimage.maximum_filter1d(
            multiyear_daily_maximum_values, size=self.window_length_annual_cycle_of_upper_bounds, mode="wrap"
        )
        # Running mean:
        annual_cycle_of_upper_bounds = scipy.ndimage.uniform_filter1d(
            multiyear_daily_maximum_values, size=self.window_length_annual_cycle_of_upper_bounds, mode="wrap"
        )

        return annual_cycle_of_upper_bounds, unique_days_of_year

    @staticmethod
    def _step1_calculate_debiased_annual_cycle_of_upper_bounds(
        annual_cycle_obs_hist,
        unique_days_of_year_obs_hist,
        annual_cycle_cm_hist,
        unique_days_of_year_cm_hist,
        annual_cycle_cm_future,
        unique_days_of_year_cm_future,
    ):
        if np.array_equal(unique_days_of_year_cm_hist, unique_days_of_year_cm_future) and np.array_equal(
            unique_days_of_year_obs_hist, unique_days_of_year_cm_future
        ):
            factor = np.where(annual_cycle_cm_hist != 0, annual_cycle_cm_future / annual_cycle_cm_hist, 1)
            factor = np.maximum(0.1, np.minimum(10.0, factor))
            debiased_annual_cycle = annual_cycle_obs_hist * factor

        else:
            # One of obs, cm_hist, cm_future has one or multiple days of year more
            debiased_annual_cycle = annual_cycle_cm_future.copy()
            for index, day_of_year in enumerate(unique_days_of_year_cm_future):

                val_cm_future = annual_cycle_cm_future[index]
                val_cm_hist = annual_cycle_cm_hist[unique_days_of_year_cm_hist == day_of_year]
                val_obs_hist = annual_cycle_obs_hist[unique_days_of_year_obs_hist == day_of_year]

                # Day of year exists in unique_days_of_year_cm_hist
                if val_cm_hist.size > 0:
                    val_cm_hist = val_cm_hist[0]

                    # Day of year exists in unique_days_of_year_obs_hist
                    if val_obs_hist.size > 0:
                        val_obs_hist = val_obs_hist[0]

                        debiased_annual_cycle[index] = (
                            val_obs_hist * val_cm_future / val_cm_hist if val_cm_hist != 0 else val_obs_hist
                        )

        return debiased_annual_cycle

    @staticmethod
    def _step1_scale_by_annual_cycle_of_upper_bounds(
        vals, days_of_year_vals, annual_cycle_of_upper_bounds, days_of_year_annual_cycle_of_upper_bounds
    ):
        scaling = np.where(annual_cycle_of_upper_bounds == 0, 1.0, 1 / annual_cycle_of_upper_bounds)
        scaling = (
            scaling[days_of_year_vals - 1]
            if days_of_year_annual_cycle_of_upper_bounds.size == 366
            else np.array(
                [
                    scaling[days_of_year_annual_cycle_of_upper_bounds == day_of_year][0]
                    for day_of_year in days_of_year_vals
                ]
            )
        )
        return vals * scaling

    def _step2_get_mask_for_values_to_impute(self, x):
        return np.logical_or(np.isnan(x), np.isinf(x))

    def _step2_impute_values(self, x):

        mask_values_to_impute = self._step2_get_mask_for_values_to_impute(x)
        mask_valid_values = np.logical_not(mask_values_to_impute)
        valid_values = x[mask_valid_values]

        # If all values are invalid raise error
        if valid_values.size == 0:
            raise ValueError("Step2: Imputation not possible because all values are invalid in the given month/window.")

        # If only one valid value exist insert this one at locations of invalid values
        if valid_values.size == 1:
            x[mask_values_to_impute] = valid_values[0]
            return x

        # Sample invalid values from the valid ones
        sampled_values = iecdf(
            x=valid_values,
            p=np.random.random(size=mask_values_to_impute.sum()),
            method=self.iecdf_method,
        )

        # Compute a backsort of how sorted valid values would be reinserted
        indices_valid_values = np.where(mask_valid_values)[0]
        backsort_sorted_valid_values = np.argsort(np.argsort(valid_values))
        interpolated_backsort_valid_values = scipy.interpolate.interp1d(
            indices_valid_values, backsort_sorted_valid_values, fill_value="extrapolate"
        )

        # Use this backsort to compute where sorted sampled values for invalid values would be reinserted
        backsort_invalid_values = np.argsort(
            np.argsort(interpolated_backsort_valid_values(np.where(mask_values_to_impute)[0]))
        )

        # Reinserted sorted sampled values into the locations
        x[mask_values_to_impute] = np.sort(sampled_values)[backsort_invalid_values]

        return x

    def _step3_remove_trend(self, x, years):

        # Calculate annual trend
        unique_years, annual_means = get_years_and_yearly_means(x, years)
        regression = scipy.stats.linregress(unique_years, annual_means)
        if regression.pvalue < 0.05 and self.detrending_with_significance_test:
            annual_trend = regression.slope * (unique_years - np.mean(unique_years))
        else:
            annual_trend = np.zeros(years.size, dtype=x.dtype)

        # Map annual trend onto daily resolution
        trend = np.zeros_like(x)
        for index_unique_year, unique_year in enumerate(unique_years):
            trend[years == unique_year] = annual_trend[index_unique_year]
        x = x - trend
        return x, trend

    def _step4_randomize_values_between_lower_threshold_and_bound(self, vals):
        mask_vals_beyond_lower_threshold = self._get_mask_for_values_beyond_lower_threshold(vals)
        randomised_values_between_threshold_and_bound = np.sort(
            np.random.uniform(
                size=mask_vals_beyond_lower_threshold.sum(),
                low=self.lower_bound,
                high=self.lower_threshold,
            )
        )
        vals[mask_vals_beyond_lower_threshold] = sort_array_like_another_one(
            randomised_values_between_threshold_and_bound,
            vals[mask_vals_beyond_lower_threshold],
        )
        return vals

    def _step4_randomize_values_between_upper_threshold_and_bound(self, vals):
        mask_vals_beyond_upper_threshold = self._get_mask_for_values_beyond_upper_threshold(vals)
        randomised_values_between_threshold_and_bound = np.sort(
            np.random.uniform(
                size=mask_vals_beyond_upper_threshold.sum(),
                low=self.upper_threshold,
                high=self.upper_bound,
            )
        )
        vals[mask_vals_beyond_upper_threshold] = sort_array_like_another_one(
            randomised_values_between_threshold_and_bound,
            vals[mask_vals_beyond_upper_threshold],
        )
        return vals

    def _step5_transfer_trend(self, obs_hist, cm_hist, cm_future):
        # Compute p = F_obs_hist(x) with x in obs_hist
        p = ecdf(obs_hist, obs_hist, method=self.ecdf_method)

        # Compute q-vals: q = IECDF(p)
        q_obs_hist = obs_hist  # = iecdf(obs_hist, p, method=self.iecdf_method), appears in eq. 7
        q_cm_future = iecdf(cm_future, p, method=self.iecdf_method)
        q_cm_hist = iecdf(cm_hist, p, method=self.iecdf_method)

        if self.trend_preservation_method == "additive":
            delta_add = q_cm_future - q_cm_hist
            return obs_hist + delta_add
        elif self.trend_preservation_method == "multiplicative":
            delta_star_mult = np.where(q_cm_hist == 0, 1, q_cm_future / q_cm_hist)
            delta_mult = np.maximum(0.01, np.minimum(100, delta_star_mult))
            return obs_hist * delta_mult
        elif self.trend_preservation_method == "mixed":
            # Formula 7
            condition1 = q_cm_hist >= q_obs_hist
            condition2 = (q_cm_hist < q_obs_hist) & (q_obs_hist < 9 * q_cm_hist)

            gamma = np.zeros_like(obs_hist)
            gamma[condition1] = 1
            gamma[condition2] = 0.5 * (1 + np.cos((q_obs_hist[condition2] / q_cm_hist[condition2] - 1) * np.pi / 8))

            # Formula 6
            delta_add = q_cm_future - q_cm_hist
            delta_star_mult = np.where(q_cm_hist == 0, 1, q_cm_future / q_cm_hist)
            delta_mult = np.maximum(0.01, np.minimum(100, delta_star_mult))
            return gamma * obs_hist * delta_mult + (1 - gamma) * (obs_hist + delta_add)
        elif self.trend_preservation_method == "bounded":
            a = self.lower_bound
            b = self.upper_bound

            mask_negative_bias = q_cm_hist < q_obs_hist
            mask_zero_bias = np.isclose(q_cm_hist, q_obs_hist)
            mask_positive_bias = q_cm_hist > q_obs_hist
            mask_additive_correction = np.logical_or(
                np.logical_and(mask_negative_bias, q_cm_future < q_cm_hist),
                np.logical_and(mask_positive_bias, q_cm_future > q_cm_hist),
            )

            return_vals = np.empty_like(q_cm_future)

            # New and updated formula for climate change transfer to observations for bounded variables, >= isimipv2.3
            return_vals[mask_negative_bias] = b - (b - q_obs_hist[mask_negative_bias]) * (
                b - q_cm_future[mask_negative_bias]
            ) / (b - q_cm_hist[mask_negative_bias])

            return_vals[mask_zero_bias] = q_cm_future[mask_zero_bias]

            return_vals[mask_positive_bias] = a + (q_obs_hist[mask_positive_bias] - a) * (
                q_cm_future[mask_positive_bias] - a
            ) / (q_cm_hist[mask_positive_bias] - a)

            return_vals[mask_additive_correction] = (
                q_obs_hist[mask_additive_correction]
                + q_cm_future[mask_additive_correction]
                - q_cm_hist[mask_additive_correction]
            )

            # Enforce bounds:
            return_vals = np.maximum(a, np.minimum(return_vals, b))

            return return_vals
        else:
            raise ValueError(
                """ Wrong value for self.trend_preservation_method.
                    Needs to be one of ['additive', 'multiplicative', 'mixed', 'bounded'] """
            )

    def _step6_calculate_percent_values_beyond_threshold(
        mask_for_values_beyond_threshold,
    ):
        return mask_for_values_beyond_threshold.sum() / mask_for_values_beyond_threshold.size

    @staticmethod
    def _step6_get_P_obs_future(P_obs_hist, P_cm_hist, P_cm_future):
        if np.isclose(P_cm_hist, P_obs_hist):
            return P_cm_future
        elif P_cm_future <= P_cm_hist and P_cm_hist > P_obs_hist:
            return P_obs_hist * P_cm_future / P_cm_hist
        elif P_cm_future >= P_cm_hist and P_cm_hist < P_obs_hist:
            return 1 - (1 - P_obs_hist) * (1 - P_cm_future) / (1 - P_cm_hist)
        else:
            return P_obs_hist + P_cm_future - P_cm_hist

    def _step6_get_nr_of_entries_to_set_to_bound(
        self,
        mask_for_values_beyond_threshold_obs_hist_sorted,
        mask_for_values_beyond_threshold_cm_hist_sorted,
        mask_for_values_beyond_threshold_cm_future_sorted,
    ):

        if self.bias_correct_frequencies_of_values_beyond_thresholds:

            P_obs_hist = ISIMIP._step6_calculate_percent_values_beyond_threshold(
                mask_for_values_beyond_threshold_obs_hist_sorted
            )
            P_cm_hist = ISIMIP._step6_calculate_percent_values_beyond_threshold(
                mask_for_values_beyond_threshold_cm_hist_sorted
            )
            P_cm_future = ISIMIP._step6_calculate_percent_values_beyond_threshold(
                mask_for_values_beyond_threshold_cm_future_sorted
            )

            P_obs_future = ISIMIP._step6_get_P_obs_future(P_obs_hist, P_cm_hist, P_cm_future)

        else:

            P_obs_future = ISIMIP._step6_calculate_percent_values_beyond_threshold(
                mask_for_values_beyond_threshold_obs_hist_sorted
            )

        return round(mask_for_values_beyond_threshold_cm_future_sorted.size * P_obs_future)

    @staticmethod
    def _step6_scale_nr_of_entries_to_set_to_bounds(
        nr_of_entries_to_set_to_lower_bound, nr_of_entries_to_set_to_upper_bound, size_cm_future
    ):
        nr_of_entries_to_set_to_lower_bound = round(
            nr_of_entries_to_set_to_lower_bound
            * size_cm_future
            / (nr_of_entries_to_set_to_lower_bound + nr_of_entries_to_set_to_upper_bound)
        )
        nr_of_entries_to_set_to_upper_bound = round(
            nr_of_entries_to_set_to_upper_bound
            * size_cm_future
            / (nr_of_entries_to_set_to_lower_bound + nr_of_entries_to_set_to_upper_bound)
        )
        return nr_of_entries_to_set_to_lower_bound, nr_of_entries_to_set_to_upper_bound

    @staticmethod
    def _step6_get_mask_for_entries_to_set_to_lower_bound(nr, cm_future_sorted):
        mask = np.zeros_like(cm_future_sorted, dtype=bool)
        mask[0:nr] = True
        return mask

    @staticmethod
    def _step6_get_mask_for_entries_to_set_to_upper_bound(nr, cm_future_sorted):
        mask = np.zeros_like(cm_future_sorted, dtype=bool)
        mask[(cm_future_sorted.size - nr) :] = True
        return mask

    @staticmethod
    def _step6_fit_good_enough(data, distribution, fit):
        ks_statistic = scipy.stats.kstest(data, distribution.cdf, args=fit)[0]
        return ks_statistic <= 0.5

    def _step6_adjust_values_between_thresholds(
        self,
        obs_hist_sorted_entries_between_thresholds,
        obs_future_sorted_entries_between_thresholds,
        cm_hist_sorted_entries_between_thresholds,
        cm_future_sorted_entries_not_sent_to_bound,
        cm_future_sorted_entries_between_thresholds,
    ):

        # ISIMIP v2.5: entries between bounds are mapped non-parametically on entries between thresholds (previous to bound adjustment)
        if self.has_threshold and not self.nonparametric_qm and cm_future_sorted_entries_between_thresholds.size > 0:
            cm_future_sorted_entries_not_sent_to_bound = quantile_map_x_on_y_non_parametically(
                x=cm_future_sorted_entries_not_sent_to_bound,
                y=cm_future_sorted_entries_between_thresholds,
                mode=self.mode_non_parametric_qm,
                ecdf_method=self.ecdf_method,
                iecdf_method=self.iecdf_method,
            )

        if self.nonparametric_qm or cm_future_sorted_entries_between_thresholds.size == 0:
            if not self.nonparametric_qm and cm_future_sorted_entries_between_thresholds.size == 0:
                warning(
                    "Step 6: There are no values between thresholds in cm_future, but there should be some after bias adjustment. Using nonparametric quantile mapping (instead of parametric one) between the values in cm_future and the pseudo future observations to calculate the values between threshold."
                )
            return quantile_map_non_parametically(
                x=cm_future_sorted_entries_not_sent_to_bound,
                y=obs_future_sorted_entries_between_thresholds,
                vals=cm_future_sorted_entries_not_sent_to_bound,
                ecdf_method=self.ecdf_method,
                iecdf_method=self.iecdf_method,
            )

        # ISIMIP v2.5: fix location and scale as function of upper and lower threshold
        floc = self.lower_threshold if self.has_lower_threshold else None
        fscale = (
            self.upper_threshold - self.lower_threshold
            if self.has_lower_threshold and self.has_upper_threshold
            else None
        )

        if self.distribution in [scipy.stats.rice, scipy.stats.weibull_min]:
            fixed_args = {"floc": floc}
        else:
            fixed_args = {"floc": floc, "fscale": fscale}

        # Calculate cdf-fits
        fit_cm_future = self.distribution.fit(cm_future_sorted_entries_between_thresholds, **fixed_args)
        fit_obs_future = self.distribution.fit(obs_future_sorted_entries_between_thresholds, **fixed_args)

        # If ks_test_for_goodness_of_fit = True then test for goodness of fit and do nonparametric qm if too bad
        if self.ks_test_for_goodness_of_cdf_fit:

            if not (
                ISIMIP._step6_fit_good_enough(
                    cm_future_sorted_entries_between_thresholds, self.distribution, fit_cm_future
                )
                and ISIMIP._step6_fit_good_enough(
                    obs_future_sorted_entries_between_thresholds, self.distribution, fit_obs_future
                )
            ):
                warning(
                    "Step 6: Goodness of fit not good enough according to ks-test. Using nonparametric quantile mapping instead."
                )
                return quantile_map_non_parametically(
                    x=cm_future_sorted_entries_not_sent_to_bound,
                    y=obs_future_sorted_entries_between_thresholds,
                    vals=cm_future_sorted_entries_not_sent_to_bound,
                    ecdf_method=self.ecdf_method,
                    iecdf_method=self.iecdf_method,
                )

        # Get the cdf-vals of cm_future
        cdf_vals_cm_future = threshold_cdf_vals(
            self.distribution.cdf(cm_future_sorted_entries_not_sent_to_bound, *fit_cm_future)
        )

        # Event likelihood adjustment only happens in certain cases
        if not self.event_likelihood_adjustment:
            mapped_vals = self.distribution.ppf(cdf_vals_cm_future, *fit_obs_future)
        else:
            # Calculate additional needed cdf-fits
            fit_cm_hist = self.distribution.fit(cm_hist_sorted_entries_between_thresholds)
            fit_obs_hist = self.distribution.fit(obs_hist_sorted_entries_between_thresholds)

            # Get the cdf-vals and interpolate if there are unequal sample sizes (following Switanek 2017):
            cdf_vals_obs_hist = interp_sorted_cdf_vals_on_given_length(
                threshold_cdf_vals(self.distribution.cdf(obs_hist_sorted_entries_between_thresholds, *fit_obs_hist)),
                cdf_vals_cm_future.size,
            )
            cdf_vals_cm_hist = interp_sorted_cdf_vals_on_given_length(
                threshold_cdf_vals(self.distribution.cdf(cm_hist_sorted_entries_between_thresholds, *fit_cm_hist)),
                cdf_vals_cm_future.size,
            )

            # Calculate L-values and delta log-odds for mapping, following formula 11-14
            L_obs_hist = scipy.special.logit(cdf_vals_obs_hist)
            L_cm_hist = scipy.special.logit(cdf_vals_cm_hist)
            L_cm_future = scipy.special.logit(cdf_vals_cm_future)

            delta_log_odds = np.maximum(-np.log(10), np.minimum(np.log(10), L_cm_future - L_cm_hist))

            # Map values following formula 10
            mapped_vals = self.distribution.ppf(scipy.special.expit(L_obs_hist + delta_log_odds), *fit_obs_future)
        return mapped_vals

    @staticmethod
    def _step8_rescale_by_annual_cycle_of_upper_bounds(
        vals, days_of_year_vals, annual_cycle_of_upper_bounds, days_of_year_annual_cycle_of_upper_bounds
    ):
        scaling = (
            annual_cycle_of_upper_bounds[days_of_year_vals - 1]
            if days_of_year_annual_cycle_of_upper_bounds.size == 366
            else np.array(
                [
                    annual_cycle_of_upper_bounds[days_of_year_annual_cycle_of_upper_bounds == day_of_year][0]
                    for day_of_year in days_of_year_vals
                ]
            )
        )
        return vals * scaling

    # ----- ISIMIP calculation steps ----- #

    def step1(self, obs_hist, cm_hist, cm_future, time_obs_hist, time_cm_hist, time_cm_future):
        """
        Step 1: if ``scale_by_annual_cycle_of_upper_bounds = True`` (``rsds`` only).
        Scales obs, cm_hist and cm_future values to [0, 1] by computing an annual cycle of upper bounds as "running mean values of running maximum values of multiyear daily maximum values." (Lange 2019).
        Furthermore return a debiased annual cycle of upper bounds to rescale cm_future values in step8.
        """
        debiased_annual_cycle = None
        if self.scale_by_annual_cycle_of_upper_bounds:

            days_of_year_obs_hist = day_of_year(time_obs_hist)
            days_of_year_cm_hist = day_of_year(time_cm_hist)
            days_of_year_cm_future = day_of_year(time_cm_future)

            # Calculate the annual cycles
            annual_cycle_obs_hist, unique_days_of_year_obs_hist = self._step1_get_annual_cycle_of_upper_bounds(
                obs_hist, days_of_year_obs_hist
            )
            annual_cycle_cm_hist, unique_days_of_year_cm_hist = self._step1_get_annual_cycle_of_upper_bounds(
                cm_hist, days_of_year_cm_hist
            )
            annual_cycle_cm_future, unique_days_of_year_cm_future = self._step1_get_annual_cycle_of_upper_bounds(
                cm_future, days_of_year_cm_future
            )

            # Remove the annual cycle from observations and climate model values
            obs_hist = ISIMIP._step1_scale_by_annual_cycle_of_upper_bounds(
                obs_hist, days_of_year_obs_hist, annual_cycle_obs_hist, unique_days_of_year_obs_hist
            )
            cm_hist = ISIMIP._step1_scale_by_annual_cycle_of_upper_bounds(
                cm_hist, days_of_year_cm_hist, annual_cycle_cm_hist, unique_days_of_year_cm_hist
            )
            cm_future = ISIMIP._step1_scale_by_annual_cycle_of_upper_bounds(
                cm_future, days_of_year_cm_future, annual_cycle_cm_future, unique_days_of_year_cm_future
            )

            # Calculate the debiased annual cycle:
            debiased_annual_cycle = ISIMIP._step1_calculate_debiased_annual_cycle_of_upper_bounds(
                annual_cycle_obs_hist,
                unique_days_of_year_obs_hist,
                annual_cycle_cm_hist,
                unique_days_of_year_cm_hist,
                annual_cycle_cm_future,
                unique_days_of_year_cm_future,
            )

        return obs_hist, cm_hist, cm_future, debiased_annual_cycle

    def step2(self, obs_hist, cm_hist, cm_future):
        """
        Step 2: impute values for ``prsnratio`` which are missing on days where there is no precipitation. They are imputed by effectively sampling the iecdf (see Lange 2019 and ISIMIP3b factsheet for the method).
        """
        if self.impute_missing_values:
            obs_hist = self._step2_impute_values(obs_hist)
            cm_hist = self._step2_impute_values(cm_hist)
            cm_future = self._step2_impute_values(cm_future)

        return obs_hist, cm_hist, cm_future

    def step3(self, obs_hist, cm_hist, cm_future, years_obs_hist, years_cm_hist, years_cm_future):
        """
        Step 3: Linear trend removal if ``detrending = True``. This is because certain variables (eg. temp) can have substantial trends also within training and application period (not only between).
        These trends are removed to "prevent a confusion of these trends with interannual variability during quantile mapping (steps 5 and6)" (Lange 2019). The trend for cm_future is subsequently added again in step7.
        Trends are calculated by linearly regressing the yearly mean values y_i against years t_i. From each observation in year t_i then the slope * (t_i - mean(t_i)) is removed (normalising the trend such that the sum over all t_i is zero). See Lange 2019 for the exact method.

        If ``detrending_with_significance_test = True`` (default in ISIMIP v2.5) then linear trends are only removed subject it being significant (p-value < 0.05).
        """
        trend_cm_future = np.zeros_like(cm_future)
        if self.detrending:
            obs_hist, _ = self._step3_remove_trend(obs_hist, years_obs_hist)
            cm_hist, _ = self._step3_remove_trend(cm_hist, years_cm_hist)
            cm_future, trend_cm_future = self._step3_remove_trend(cm_future, years_cm_future)
        return obs_hist, cm_hist, cm_future, trend_cm_future

    def step4(self, obs_hist, cm_hist, cm_future):
        """
        Step4: If the variable is bounded then values between the threshold and corresponding bound (including values equal to the bound) are
        randomized uniformly between the threshold and bound and resorted according to the order in which they were before.
        """
        if self.has_lower_bound and self.has_lower_threshold:
            obs_hist = self._step4_randomize_values_between_lower_threshold_and_bound(obs_hist)
            cm_hist = self._step4_randomize_values_between_lower_threshold_and_bound(cm_hist)
            cm_future = self._step4_randomize_values_between_lower_threshold_and_bound(cm_future)

        if self.has_upper_bound and self.has_upper_threshold:
            obs_hist = self._step4_randomize_values_between_upper_threshold_and_bound(obs_hist)
            cm_hist = self._step4_randomize_values_between_upper_threshold_and_bound(cm_hist)
            cm_future = self._step4_randomize_values_between_upper_threshold_and_bound(cm_future)

        return obs_hist, cm_hist, cm_future

    # Generate pseudo future observations and transfer trends
    def step5(self, obs_hist, cm_hist, cm_future):
        """
        Step 5: generates pseudo-future observations by transfering simulated trends to historical recorded observations.
        This makes the ISIMIP-method trend-preserving.
        """
        if self.trend_transfer_only_for_values_within_threshold:
            mask_for_values_between_thresholds_obs_hist = self._get_mask_for_values_between_thresholds(obs_hist)
            obs_future = obs_hist.copy()
            if (
                any(mask_for_values_between_thresholds_obs_hist) > 0
                and (values_between_thresholds_cm_hist := self._get_values_between_thresholds(cm_hist)).size > 0
                and (values_between_thresholds_cm_future := self._get_values_between_thresholds(cm_future)).size > 0
            ):
                obs_future[mask_for_values_between_thresholds_obs_hist] = self._step5_transfer_trend(
                    obs_hist[mask_for_values_between_thresholds_obs_hist],
                    values_between_thresholds_cm_hist,
                    values_between_thresholds_cm_future,
                )
            return obs_future
        else:
            return self._step5_transfer_trend(obs_hist, cm_hist, cm_future)

    # Core of the isimip-method: parametric quantile mapping
    def step6(self, obs_hist, obs_future, cm_hist, cm_future):
        """
        Step 6: parametric quantile mapping between cm_future and obs_future (the pseudo-future observations) to debias the former. Core of the bias adjustment method.
        For (partly) bounded climate variables additionally the frequency of values is bias-adjusted before the other observations are quantile mapped.
        If ``event_likelihood_adjustment = True`` then additionally to "normal quantile mapping" the likelihood of individual events is adjusted (see Lange 2019 for the method which is based on Switanek 2017).
        """

        # Sort arrays to apply parametric quantile mapping (values of equal rank are mapped together).
        # Save sort-order of cm_future for backsorting
        cm_future_argsort = np.argsort(cm_future)
        cm_future_sorted = cm_future[cm_future_argsort]

        obs_hist_sorted = np.sort(obs_hist)
        obs_future_sorted = np.sort(obs_future)
        cm_hist_sorted = np.sort(cm_hist)

        # Vector to store quantile mapped values
        mapped_vals = cm_future_sorted.copy()

        # Calculate number of values that are set to lower bound for bounded/thresholded variables
        nr_of_entries_to_set_to_lower_bound = 0
        if self.has_lower_threshold:
            nr_of_entries_to_set_to_lower_bound = self._step6_get_nr_of_entries_to_set_to_bound(
                self._get_mask_for_values_beyond_lower_threshold(obs_hist_sorted),
                self._get_mask_for_values_beyond_lower_threshold(cm_hist_sorted),
                self._get_mask_for_values_beyond_lower_threshold(cm_future_sorted),
            )

        # Calculate number of values that are set to lower bound for bounded/thresholded variables
        nr_of_entries_to_set_to_upper_bound = 0
        if self.has_upper_threshold:
            nr_of_entries_to_set_to_upper_bound = self._step6_get_nr_of_entries_to_set_to_bound(
                self._get_mask_for_values_beyond_upper_threshold(obs_hist_sorted),
                self._get_mask_for_values_beyond_upper_threshold(cm_hist_sorted),
                self._get_mask_for_values_beyond_upper_threshold(cm_future_sorted),
            )

        # Adjust number of values by scaling if more values are supposed to be set to bound than cm_future has entries
        if nr_of_entries_to_set_to_lower_bound + nr_of_entries_to_set_to_upper_bound > cm_future_sorted.size:
            (
                nr_of_entries_to_set_to_lower_bound,
                nr_of_entries_to_set_to_upper_bound,
            ) = ISIMIP._step6_scale_nr_of_entries_to_set_to_bounds(
                nr_of_entries_to_set_to_lower_bound, nr_of_entries_to_set_to_upper_bound, cm_future_sorted.size
            )

        # Get mask for the number of entries to set to lower bound if this adjustment is to be done
        mask_for_entries_to_set_to_lower_bound = self._step6_get_mask_for_entries_to_set_to_lower_bound(
            nr_of_entries_to_set_to_lower_bound, cm_future_sorted
        )

        # Get mask for the number of entries to set to higher bound if this adjustment is to be done
        mask_for_entries_to_set_to_upper_bound = ISIMIP._step6_get_mask_for_entries_to_set_to_upper_bound(
            nr_of_entries_to_set_to_upper_bound, cm_future_sorted
        )

        # Set values to upper or lower bound for bounded/thresholded variables
        mapped_vals[mask_for_entries_to_set_to_lower_bound] = self.lower_bound
        mapped_vals[mask_for_entries_to_set_to_upper_bound] = self.upper_bound
        mask_for_entries_not_set_to_either_bound = np.logical_and(
            np.logical_not(mask_for_entries_to_set_to_lower_bound),
            np.logical_not(mask_for_entries_to_set_to_upper_bound),
        )

        # Calculate values between bounds (if any are to be calculated)
        if any(mask_for_entries_not_set_to_either_bound):
            mapped_vals[mask_for_entries_not_set_to_either_bound] = self._step6_adjust_values_between_thresholds(
                self._get_values_between_thresholds(obs_hist_sorted),
                self._get_values_between_thresholds(obs_future_sorted),
                self._get_values_between_thresholds(cm_hist_sorted),
                mapped_vals[mask_for_entries_not_set_to_either_bound],
                self._get_values_between_thresholds(cm_future_sorted),
            )

        # Return values inserted back at correct locations
        reverse_sorting_idx = np.argsort(cm_future_argsort)
        return mapped_vals[reverse_sorting_idx]

    def step7(self, cm_future, trend_cm_future):
        """
        Step 7: If ``detrending = True`` adds the trend removed in step 3 back to the debiased cm_future values.
        """
        if self.detrending:
            cm_future = cm_future + trend_cm_future
        return cm_future

    def step8(self, cm_future, debiased_annual_cycle, time_cm_future):
        """
        Step 8: if ``scale_by_annual_cycle_of_upper_bounds = True`` (rsds only).
        Scales debiased cm_future on [0, 1] back to the usual scale using the debiased annual cycle
        """
        if self.scale_by_annual_cycle_of_upper_bounds:

            days_of_year_cm_future = day_of_year(time_cm_future)
            cm_future = ISIMIP._step8_rescale_by_annual_cycle_of_upper_bounds(
                cm_future, days_of_year_cm_future, debiased_annual_cycle, np.unique(days_of_year_cm_future)
            )

        return cm_future

    # ----- Apply location function -----

    def _apply_on_window(
        self,
        obs_hist: np.ndarray,
        cm_hist: np.ndarray,
        cm_future: np.ndarray,
        years_obs_hist: np.ndarray = None,
        years_cm_hist: np.ndarray = None,
        years_cm_future: np.ndarray = None,
    ) -> np.ndarray:

        # Apply all ISIMIP steps
        # Step 1 & 8 are outside the running window iteration
        obs_hist, cm_hist, cm_future = self.step2(obs_hist, cm_hist, cm_future)
        obs_hist, cm_hist, cm_future, trend_cm_future = self.step3(
            obs_hist, cm_hist, cm_future, years_obs_hist, years_cm_hist, years_cm_future
        )
        obs_hist, cm_hist, cm_future = self.step4(obs_hist, cm_hist, cm_future)
        obs_future = self.step5(obs_hist, cm_hist, cm_future)
        cm_future = self.step6(obs_hist, obs_future, cm_hist, cm_future)
        cm_future = self.step7(cm_future, trend_cm_future)
        # Step 1 & 8 are outside the running window iteration
        return cm_future

    def apply_location(
        self,
        obs: np.ndarray,
        cm_hist: np.ndarray,
        cm_future: np.ndarray,
        time_obs: Optional[np.ndarray] = None,
        time_cm_hist: Optional[np.ndarray] = None,
        time_cm_future: Optional[np.ndarray] = None,
    ) -> np.ndarray:

        if time_obs is None or time_cm_hist is None or time_cm_future is None:
            warning(
                """
                    ISIMIP runs without time-information for at least one of obs, cm_hist or cm_future.
                    This information is inferred, assuming the first observation is on a January 1st. Observations are chunked according to the assumed time information. 
                    This might lead to slight numerical differences to the run with time information, however the debiasing is not fundamentally changed.
                    """
            )
            time_obs, time_cm_hist, time_cm_future = infer_and_create_time_arrays_if_not_given(
                obs, cm_hist, cm_future, time_obs, time_cm_hist, time_cm_future
            )

        ISIMIP._check_time_information_and_raise_error(obs, cm_hist, cm_future, time_obs, time_cm_hist, time_cm_future)

        years_obs = year(time_obs)
        years_cm_hist = year(time_cm_hist)
        years_cm_future = year(time_cm_future)

        # Step 1 & 8 are outside the running window iteration
        obs, cm_hist, cm_future, debiased_annual_cycle = self.step1(
            obs, cm_hist, cm_future, time_obs, time_cm_hist, time_cm_future
        )

        if self.running_window_mode:

            days_of_year_obs = day_of_year(time_obs)
            days_of_year_cm_hist = day_of_year(time_cm_hist)
            days_of_year_cm_future = day_of_year(time_cm_future)

            debiased_cm_future = np.zeros_like(cm_future)

            # Main iteration
            for window_center, indices_bias_corrected_values in self.running_window.use(
                days_of_year_cm_future, years_cm_future
            ):

                indices_window_obs = self.running_window.get_indices_vals_in_window(days_of_year_obs, window_center)
                indices_window_cm_hist = self.running_window.get_indices_vals_in_window(
                    days_of_year_cm_hist, window_center
                )
                indices_window_cm_future = self.running_window.get_indices_vals_in_window(
                    days_of_year_cm_future, window_center
                )

                debiased_cm_future[indices_bias_corrected_values] = self._apply_on_window(
                    obs_hist=obs[indices_window_obs],
                    cm_hist=cm_hist[indices_window_cm_hist],
                    cm_future=cm_future[indices_window_cm_future],
                    years_obs_hist=years_obs[indices_window_obs],
                    years_cm_hist=years_cm_hist[indices_window_cm_hist],
                    years_cm_future=years_cm_future[indices_window_cm_future],
                )[
                    np.logical_and(
                        np.in1d(indices_window_cm_future, indices_bias_corrected_values),
                        get_mask_for_unique_subarray(indices_window_cm_future),
                    )
                ]
        else:
            months_obs = month(time_obs)
            months_cm_hist = month(time_cm_hist)
            months_cm_future = month(time_cm_future)

            debiased_cm_future = np.zeros_like(cm_future)
            for i_month in range(1, 13):
                mask_i_month_in_obs = months_obs == i_month
                mask_i_month_in_cm_hist = months_cm_hist == i_month
                mask_i_month_in_cm_future = months_cm_future == i_month

                debiased_cm_future[mask_i_month_in_cm_future] = self._apply_on_window(
                    obs_hist=obs[mask_i_month_in_obs],
                    cm_hist=cm_hist[mask_i_month_in_cm_hist],
                    cm_future=cm_future[mask_i_month_in_cm_future],
                    years_obs_hist=years_obs[mask_i_month_in_obs],
                    years_cm_hist=years_cm_hist[mask_i_month_in_cm_hist],
                    years_cm_future=years_cm_future[mask_i_month_in_cm_future],
                )

        # Step 1 & 8 are outside the running window iteration
        debiased_cm_future = self.step8(debiased_cm_future, debiased_annual_cycle, time_cm_future)

        return debiased_cm_future
