# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['prometheus_alert_model']

package_data = \
{'': ['*']}

install_requires = \
['pydantic>=1.7.2,<2.0.0', 'typing-extensions>=3.7.4,<4.0.0']

setup_kwargs = {
    'name': 'prometheus-alert-model',
    'version': '0.1.0',
    'description': 'Pydantic model of the Prometheus Alertmanager alert payload',
    'long_description': '<!-- omit in toc -->\n# Prometheus Alert Model *for Python*\n\n[![Current Package Version](https://badge.fury.io/py/prometheus-alert-model.svg)](https://pypi.python.org/pypi/prometheus-alert-model)\n[![Maintenance](https://img.shields.io/badge/maintained%3F-yes-green.svg)](https://GitHub.com/Naereen/StrapDown.js/graphs/commit-activity)\n[![Supported Python Versions](https://img.shields.io/pypi/pyversions/prometheus-alert-model.svg)](https://pypi.python.org/pypi/prometheus-alert-model)\n[![Downloads](https://pepy.tech/badge/prometheus-alert-model/month)](https://pepy.tech/project/prometheus-alert-model/month)\n[![docs](https://img.shields.io/badge/docs-here-blue)](https://trallnag.github.io/prometheus-alert-model-for-python/)\n\n![release](https://github.com/trallnag/prometheus-alert-model-for-python/workflows/release/badge.svg)\n![commit](https://github.com/trallnag/prometheus-alert-model-for-python/workflows/commit/badge.svg)\n[![codecov](https://codecov.io/gh/trallnag/prometheus-alert-model-for-python/branch/master/graph/badge.svg)](https://codecov.io/gh/trallnag/prometheus-alert-model-for-python)\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n\nThis project provides the Pydantic models `AlertGroup` and `Alert` that\nrepresent a payload from Prometheus Alertmanager. In addition, it also includes\na number of useful utility methods that perform actions on alert data.  It can\nbe used as a drop-in wherever you want to work with Prometheus Alertmanager\nalert payload data.\n\nA prominent example for using it is a FastAPI route that receives alert payloads.\nSimply add `AlertGroup` as a parameter to the handler.\n\n<!-- omit in toc -->\n## Features\n\n- Pydantic models that matches the official Alertmanager payload schema.\n- Fields `specific_annotations` and `specific_labels` in every alert that\n    contain elements that are specific to the respective alert.\n- Methods to update common annotations and labels.\n- Methods to remove, add, update, override and prefix annotations and labels.\n- Every single method is well covered by tests.\n\n<!-- omit in toc -->\n## Table of Contents\n\n- [Usage](#usage)\n- [Motivation](#motivation)\n- [Development](#development)\n\n## Usage\n\nUsing the model is pretty straight-forward. Take a look at the automatically\ngenerated docs or the source code itself. \n\nHere is a very short example how you could use the model. It removes all\nannotations and labels starting with two underscores and adds a prefix that\ncontains namespace info to the summary label. Finally it prints specific\nelements for all alerts in the group.\n\n```python\n\nfrom prometheus_alert_model import AlertGroup\nfrom fastapi import FastAPI\nfrom re import compile\n\napp = FastAPI()\n\n@app.post("/alert")\ndef post_alert(alert_group: AlertGroup):\n    alert_group.remove_re(\n        annotations=r"^(__.*)$",\n        labels=compile(r"^(__.*)$")\n    )\n\n    alert_group.add_prefix(labels={"summary": "Prototyping system: "})\n    \n    for alert in alert_group.alerts:\n        print(alert.specific_annotations\n        print(alert.specific_labels)\n```\n\n<!-- omit in toc -->\n### `Alert` Model\n\nIn the following all attributes you can find within a `Alert`. Notice\nthe custom attributes `specific_annotations` and `specific_labels` that\ninclude elements that are specific to the respective alert in context of the\ncomplete `AlertGroup` / payload.\n\n```python\nfingerprint: str\nstatus: str\nstarts_at: datetime\nends_at: datetime\ngenerator_url: str\nannotations: Dict[str, str]\nlabels: Dict[str, str]\nspecific_annotations: Dict[str, str]\nspecific_labels: Dict[str, str]\n```\n\n<!-- omit in toc -->\n### `AlertGroup` Model\n\nIn the following all attributes you can find within a `AlertGroup`. It\nrepresents a single payload from Alertmanager.\n\n```python\nreceiver: str\nstatus: str\nexternal_url: str = Field(alias="externalURL")\nversion: str\ngroup_key: str = Field(alias="groupKey")\ntruncated_alerts: int = Field(alias="truncatedAlerts", default=0)\ngroup_labels: Dict[str, str] = Field(alias="groupLabels")\ncommon_annotations: Dict[str, str] = Field(alias="commonAnnotations")\ncommon_labels: Dict[str, str] = Field(alias="commonLabels")\nalerts: List[Alert]\n```\n\nHere is a short summary over the included utility methods (for full\ndocumentation please refer to type hints or the automatically generated docs):\n\n- `update_specific_elements`: Updates specific labels and annotations.\n- `update_specific_annotations`: Updates specific annotations.\n- `update_specific_labels`: Updates specific labels.\n- `update_common_elements`: Updates common annotations and labels.\n- `update_common_annotations`: Updates common annotations.\n- `update_common_labels`: Updates common labels.\n- `remove`: Removes annotations and labels by name.\n- `remove_re`: Removes annotations and labels by matching names with regex.\n- `add`: Adds annotations and labels but skips existing elements.\n- `override`: Adds annotations and labels and overrides existing elements.\n- `add_prefix`: Adds prefix to annotations and labels.\n\n## Motivation\n\nI have a bunch of Python scripts that work in some shape or form with Prometheus\nAlertmanager data. Instead of duplicating the model across all of them I prefer\nto have a single small package that is well tested and reuse it again and again.\nThis way I don\'t have to reimplement utility functions / methods.\n\n## Development\n\nPlease refer to ["DEVELOPMENT.md"](DEVELOPMENT.md).\n',
    'author': 'Tim Schwenke',
    'author_email': 'tim.and.trallnag+code@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/trallnag/prometheus-alert-model-for-python',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
