import os, sys, multiprocessing, hashlib, ast
from fractions import Fraction
from typing import Union, Callable
import numpy as np
from .kshell_exceptions import KshellDataStructureError
from .general_utilities import level_plot, level_density, gamma_strength_function_average
from .parameters import atomic_numbers

def parity_string_to_integer(parity: str):
    if parity == "+":
        res = 1
    elif parity == "-":
        res = -1
    else:
        msg = f"Invalid parity read from file. Got: '{parity}'."
        raise KshellDataStructureError(msg)

    return res

def generate_states(
    start: int = 0,
    stop: int = 14,
    n_states: int = 100,
    parity: Union[str, int] = "both"
    ):
    """
    Generate correct string for input to `kshell_ui.py` when asked for
    which states to calculate. Copy the string generated by this
    function and paste it into `kshell_ui.py` when it prompts for
    states.

    Parameters
    ----------
    start : int
        The lowest spin value.

    stop : int
        The largest spin value.

    n_states : int
        The number of states per spin value.

    parity : Union[str, int]
        The parity of the states. Allowed values are: 1, -1, 'both',
        'positive', 'negative', 'pos', 'neg', '+', '-'.

    Examples
    --------
    ``` python
    >>> import kshell_utilities as ksutil
    >>> ksutil.generate_states(start=0, stop=3, n_states=100, parity="both")
    0+100, 0.5+100, 1+100, 1.5+100, 2+100, 2.5+100, 3+100, 0-100, 0.5-100, 1-100, 1.5-100, 2-100, 2.5-100, 3-100,
    ```
    """
    allowed_positive_parity_inputs = ["positive", "pos", "+", "1", "+1", 1, "both"]
    allowed_negative_parity_inputs = ["negative", "neg", "-", "-1", -1, "both"]
    
    def correct_syntax(lst):
        for elem in lst:
            print(elem, end=", ")
    
    if parity in allowed_positive_parity_inputs:
        positive = [f"{i:g}{'+'}{n_states}" for i in np.arange(start, stop+0.5, 0.5)]
        correct_syntax(positive)
    
    if parity in allowed_negative_parity_inputs:
        negative = [f"{i:g}{'-'}{n_states}" for i in np.arange(start, stop+0.5, 0.5)]
        correct_syntax(negative)

def _generate_unique_identifier(path: str) -> str:
    """
    Generate a unique identifier based on the shell script and the
    save_input file from KSHELL.

    Parameters
    ----------
    path : str
        The path to a summary file or a directory with a summary file.
    """
    shell_file_content = ""
    save_input_content = ""
    msg = "Not able to generate unique identifier!"
    if os.path.isfile(path):
        """
        If a file is specified, extract the directory from the path.
        """
        directory = path.rsplit("/", 1)[0]
        if directory == path:
            """
            Example: path is 'summary.txt'
            """
            directory = "."

        for elem in os.listdir(directory):
            """
            Loop over all elements in the directory and find the shell
            script and save_input file.
            """
            if elem.endswith(".sh"):
                with open(f"{directory}/{elem}", "r") as infile:
                    shell_file_content = infile.read()
            elif elem.endswith(".input"):
                with open(f"{directory}/{elem}", "r") as infile:
                    save_input_content = infile.read()
    else:
        print(msg)

    if (shell_file_content == "") and (save_input_content == ""):
        print(msg)

    
    return hashlib.sha1((shell_file_content + save_input_content).encode()).hexdigest()

def _load_energy_levels(infile):
    """
    Example
    -------
    Energy levels

    N    J prty N_Jp    T     E(MeV)  Ex(MeV)  log-file

    1   5/2 +     1   3/2    -16.565    0.000  log_O19_sdpf-mu_m1p.txt 
    2   3/2 +     1   3/2    -15.977    0.588  log_O19_sdpf-mu_m1p.txt 
    3   1/2 +     1   3/2    -15.192    1.374  log_O19_sdpf-mu_m1p.txt 
    4   9/2 +     1   3/2    -13.650    2.915  log_O19_sdpf-mu_m1p.txt 
    5   7/2 +     1   3/2    -13.267    3.298  log_O19_sdpf-mu_m1p.txt 
    6   5/2 +     2   3/2    -13.074    3.491  log_O19_sdpf-mu_m1p.txt
    """
    levels = []
    negative_spin_counts = 0
    for _ in range(3): infile.readline()
    for line in infile:
        try:
            tmp = line.split()
            
            if tmp[1] == "-1":
                """
                -1 spin states in the KSHELL data file indicates
                bad states which should not be included.
                """
                negative_spin_counts += 1  # Debug.
                continue
            
            parity = 1 if tmp[2] == "+" else -1
            levels.append([float(tmp[5]), 2*float(Fraction(tmp[1])), parity])
        except IndexError:
            """
            End of energies.
            """
            break

    return levels, negative_spin_counts

def _load_transition_probabilities_old(infile):
        """
        For summary files with old syntax (pre 2021-11-24).
        Parameters
        ----------
        infile:
            The KSHELL summary file.
        """
        reduced_transition_prob_decay_list = []
        negative_spin_counts = 0
        for _ in range(2): infile.readline()
        for line in infile:
            try:
                """
                Example of possible lines in file:
                J_i    Ex_i     J_f    Ex_f   dE        B(M1)->         B(M1)<- 
                2+(11) 18.393 2+(10) 17.791 0.602 0.1(  0.0) 0.1( 0.0)
                3/2+( 1) 0.072 5/2+( 1) 0.000 0.071 0.127( 0.07) 0.084( 0.05)
                2+(10) 17.791 2+( 1) 5.172 12.619 0.006( 0.00) 0.006( 0.00)
                3+( 8) 19.503 2+(11) 18.393 1.111 0.000( 0.00) 0.000( 0.00)
                1+( 7) 19.408 2+( 9) 16.111 3.297 0.005( 0.00) 0.003( 0.00)
                5.0+(60) 32.170  4.0+(100) 31.734  0.436    0.198( 0.11)    0.242( 0.14)
                4.0-( 3)  3.191  3.0+(10)  3.137  0.054      0.0(  0.0)      0.0(  0.0)
                0.0+(46)', '47.248', '1.0+(97)', '45.384', '1.864', '23.973(13.39)', '7.991(', '4.46)
                """
                tmp = line.split()
                len_tmp = len(tmp)
                case_ = None # Used for identifying which if-else case reads wrong.
                
                # Location of initial parity is common for all cases.
                parity_idx = tmp[0].index("(") - 1 # Find index of initial parity.
                parity_initial = 1 if tmp[0][parity_idx] == "+" else -1
                parity_initial_symbol = tmp[0][parity_idx]
                
                # Location of initial spin is common for all cases.
                spin_initial = float(Fraction(tmp[0][:parity_idx]))
                
                if (tmp[1][-1] != ")") and (tmp[3][-1] != ")") and (len_tmp == 9):
                    """
                    Example:
                    J_i    Ex_i     J_f    Ex_f   dE        B(M1)->         B(M1)<- 
                    2+(11)   18.393  2+(10)    17.791  0.602    0.1(    0.0)    0.1(    0.0)
                    5.0+(60) 32.170  4.0+(100) 31.734  0.436    0.198( 0.11)    0.242( 0.14)
                    """
                    case_ = 0
                    E_gamma = float(tmp[4])
                    Ex_initial = float(tmp[1])
                    reduced_transition_prob_decay = float(tmp[5][:-1])
                    reduced_transition_prob_excite = float(tmp[7][:-1])
                    parity_final_symbol = tmp[2].split("(")[0][-1]
                    spin_final = float(Fraction(tmp[2].split(parity_final_symbol)[0]))
                    Ex_final = float(tmp[3])

                elif (tmp[1][-1] != ")") and (tmp[3][-1] == ")") and (len_tmp == 10):
                    """
                    Example:
                    J_i    Ex_i     J_f    Ex_f   dE        B(M1)->         B(M1)<- 
                    2+(10) 17.791 2+( 1) 5.172 12.619 0.006( 0.00) 0.006( 0.00)
                    """
                    case_ = 1
                    E_gamma = float(tmp[5])
                    Ex_initial = float(tmp[1])
                    reduced_transition_prob_decay = float(tmp[6][:-1])
                    reduced_transition_prob_excite = float(tmp[8][:-1])
                    parity_final_symbol = tmp[2].split("(")[0][-1]
                    # spin_final = float(Fraction(tmp[2][:-2]))
                    spin_final = float(Fraction(tmp[2].split(parity_final_symbol)[0]))
                    Ex_final = float(tmp[4])
                
                elif (tmp[1][-1] == ")") and (tmp[4][-1] != ")") and (len_tmp == 10):
                    """
                    Example:
                    J_i    Ex_i     J_f    Ex_f   dE        B(M1)->         B(M1)<- 
                    3+( 8)   19.503 2+(11)    18.393 1.111 0.000( 0.00) 0.000( 0.00)
                    1.0+( 1) 5.357  0.0+(103) 0.000  5.357 0.002( 0.00) 0.007( 0.00)
                    4.0-( 3)  3.191  3.0+(10)  3.137  0.054      0.0(  0.0)      0.0(  0.0)
                    """
                    case_ = 2
                    E_gamma = float(tmp[5])
                    Ex_initial = float(tmp[2])
                    reduced_transition_prob_decay = float(tmp[6][:-1])
                    reduced_transition_prob_excite = float(tmp[8][:-1])
                    parity_final_symbol = tmp[3].split("(")[0][-1]
                    spin_final = float(Fraction(tmp[3].split(parity_final_symbol)[0]))
                    Ex_final = float(tmp[4])

                elif (tmp[1][-1] == ")") and (tmp[4][-1] == ")") and (len_tmp == 11):
                    """
                    Example:
                    J_i    Ex_i     J_f    Ex_f   dE        B(M1)->         B(M1)<- 
                    1+( 7) 19.408 2+( 9) 16.111 3.297 0.005( 0.00) 0.003( 0.00)
                    """
                    case_ = 3
                    E_gamma = float(tmp[6])
                    Ex_initial = float(tmp[2])
                    reduced_transition_prob_decay = float(tmp[7][:-1])
                    reduced_transition_prob_excite = float(tmp[9][:-1])
                    parity_final_symbol = tmp[3].split("(")[0][-1]
                    # spin_final = float(Fraction(tmp[3][:-2]))
                    spin_final = float(Fraction(tmp[3].split(parity_final_symbol)[0]))
                    Ex_final = float(tmp[5])

                elif (tmp[5][-1] == ")") and (tmp[2][-1] == ")") and (len_tmp == 8):
                    """
                    Example:
                    J_i    Ex_i     J_f    Ex_f   dE        B(M1)->         B(M1)<- 
                    0.0+(46) 47.248  1.0+(97) 45.384  1.864   23.973(13.39)    7.991( 4.46)
                    """
                    case_ = 4
                    E_gamma = float(tmp[4])
                    Ex_initial = float(tmp[1])
                    reduced_transition_prob_decay = float(tmp[5].split("(")[0])
                    reduced_transition_prob_excite = float(tmp[6][:-1])
                    parity_final_symbol = tmp[2].split("(")[0][-1]
                    spin_final = float(Fraction(tmp[2].split(parity_final_symbol)[0]))
                    Ex_final = float(tmp[3])

                else:
                    msg = "ERROR: Structure not accounted for!"
                    msg += f"\n{line=}"
                    raise KshellDataStructureError(msg)

                if parity_final_symbol == "+":
                    parity_final = 1
                elif parity_final_symbol == "-":
                    parity_final = -1
                else:
                    msg = f"Could not properly read the final parity! {case_=}"
                    raise KshellDataStructureError(msg)

                if (spin_final == -1) or (spin_initial == -1):
                    """
                    -1 spin states in the KSHELL data file indicates
                    bad states which should not be included.
                    """
                    negative_spin_counts += 1  # Debug.
                    continue

                reduced_transition_prob_decay_list.append([
                    2*spin_initial, parity_initial, Ex_initial, 2*spin_final,
                    parity_final, Ex_final, E_gamma, reduced_transition_prob_decay,
                    reduced_transition_prob_excite
                ])

            except ValueError as err:
                """
                One of the float conversions failed indicating that
                the structure of the line is not accounted for.
                """
                msg = "\n" + err.__str__() + f"\n{case_=}" + f"\n{line=}"
                raise KshellDataStructureError(msg)

            except IndexError:
                """
                End of probabilities.
                """
                break
    
        return reduced_transition_prob_decay_list, negative_spin_counts
    
def _load_transition_probabilities(infile):
        """
        Example structure:
        B(E2)  ( > -0.0 W.u.)  mass = 50    1 W.u. = 10.9 e^2 fm^4
        e^2 fm^4 (W.u.)
        J_i  pi_i idx_i Ex_i    J_f  pi_f idx_f Ex_f      dE         B(E2)->         B(E2)->[wu]     B(E2)<-         B(E2)<-[wu]
        5    +    1     0.036   6    +    1     0.000     0.036     70.43477980      6.43689168     59.59865983      5.44660066
        4    +    1     0.074   6    +    1     0.000     0.074     47.20641983      4.31409897     32.68136758      2.98668391
        Parameters
        ----------
        infile:
            The KSHELL summary file.
        """
        reduced_transition_prob_decay_list = []
        negative_spin_counts = 0
        for _ in range(2): infile.readline()
        for line in infile:
            line_split = line.split()
            if not line_split: break
            
            spin_initial = float(Fraction(line_split[0]))
            parity_initial = parity_string_to_integer(line_split[1])
            Ex_initial = float(line_split[3])

            spin_final = float(Fraction(line_split[4]))
            parity_final = parity_string_to_integer(line_split[5])
            Ex_final = float(line_split[7])

            E_gamma = float(line_split[8])
            reduced_transition_prob_decay = float(line_split[9])
            reduced_transition_prob_excite = float(line_split[11])

            if (spin_final < 0) or (spin_initial < 0):
                """
                -1 spin states in the KSHELL data file indicates
                bad states which should not be included.
                """
                negative_spin_counts += 1  # Debug.
                continue

            reduced_transition_prob_decay_list.append([
                2*spin_initial, parity_initial, Ex_initial, 2*spin_final,
                parity_final, Ex_final, E_gamma, reduced_transition_prob_decay,
                reduced_transition_prob_excite
            ])
    
        return reduced_transition_prob_decay_list, negative_spin_counts

def _load_parallel(arg_list):
    """
    For parallel data loads.
    [self.fname_summary, "Energy", self._load_energy_levels, None]
    """
    fname, condition, loader = arg_list
    with open(fname, "r") as infile:
        for line in infile:
            if condition in line:
                return loader(infile)

class ReadKshellOutput:
    """
    Read `KSHELL` data files and store the values as instance
    attributes.

    Attributes
    ----------
    Ex : np.ndarray
        Array of excitation energies zeroed at the ground state energy.

    BM1 : np.ndarray
        Array of [[E, B_decay_prob, E_gamma], ...].
        Reduced transition probabilities for M1.

    BE2 : np.ndarray
        Array of [[E, B_decay_prob, E_gamma], ...].
        Reduced transition probabilities for E2.

    levels : np.ndarray
        Array containing energy, spin, and parity for each excited
        state. [[E, 2*spin, parity], ...].

    transitions : np.ndarray
        OLD:
        Mx8 array containing [2*spin_final, parity_initial, Ex_final,
        2*spin_initial, parity_initial, Ex_initial, E_gamma, B(.., i->f)].
        NEW:
        [2*spin_initial, parity_initial, Ex_initial, 2*spin_final,
        parity_final, Ex_final, E_gamma, B(.., i->f), B(.., f<-i)]
    """
    def __init__(self, path: str, load_and_save_to_file: bool, old_or_new: str):
        """
        Parameters
        ----------
        path : string
            Path of `KSHELL` output file directory, or path to a
            specific `KSHELL` data file.

        load_and_save_to_file : bool
            Toggle saving data as `.npy` files on / off. If `overwrite`,
            saved `.npy` files are overwritten.

        old_or_new : str
            Choose between old and new summary file syntax. All summary
            files generated pre 2021-11-24 use old style.
            New:
            J_i  pi_i idx_i Ex_i    J_f  pi_f idx_f Ex_f      dE         B(E2)->         B(E2)->[wu]     B(E2)<-         B(E2)<-[wu]
            5    +    1     0.036   6    +    1     0.000     0.036     70.43477980      6.43689168     59.59865983      5.44660066
            Old:
            J_i    Ex_i     J_f    Ex_f   dE        B(M1)->         B(M1)<- 
            2+(11) 18.393 2+(10) 17.791 0.602 0.1(  0.0) 0.1( 0.0)
        """

        self.path = path
        self.load_and_save_to_file = load_and_save_to_file
        self.old_or_new = old_or_new
        # Some attributes might not be altered, depending on the input file.
        self.fname_summary = None
        self.fname_ptn = None
        self.nucleus = None
        self.model_space = None
        self.proton_partition = None
        self.neutron_partition = None
        self.levels = None
        self.transitions_BM1 = [None]
        self.transitions_BE2 = [None]
        self.transitions_BE1 = [None]
        self.truncation = None
        # Debug.
        self.negative_spin_counts = np.array([0, 0, 0, 0])  # The number of skipped -1 spin states for [levels, BM1, BE2, BE1].

        if isinstance(self.load_and_save_to_file, str) and (self.load_and_save_to_file != "overwrite"):
            msg = "Allowed values for 'load_and_save_to_file' are: 'True', 'False', 'overwrite'."
            msg += f" Got '{self.load_and_save_to_file}'."
            raise ValueError(msg)

        if os.path.isdir(path):
            """
            If input 'path' is a directory containing KSHELL files,
            extract info from both summary and .ptn file.
            """
            for elem in os.listdir(path):
                if elem.startswith("summary"):
                    self.fname_summary = f"{path}/{elem}"
                    self._extract_info_from_summary_fname()
                    self._read_summary()

                elif elem.endswith(".ptn"):
                    self.fname_ptn = f"{path}/{elem}"
                    self._extract_info_from_ptn_fname()
                    self.read_ptn()

        else:
            """
            'path' is a single file, not a directory.
            """
            fname = path.split("/")[-1]

            if fname.startswith("summary"):
                self.fname_summary = path
                self._extract_info_from_summary_fname()
                self._read_summary()

            elif fname.endswith(".ptn"):
                self.fname_ptn = path
                self._extract_info_from_ptn_fname()
                self._read_ptn()

            else:
                msg = f"Handling for file {fname} is not implemented."
                raise KshellDataStructureError(msg)

    def _extract_info_from_ptn_fname(self):
        """
        Extract nucleus and model space name.
        """
        fname_split = self.fname_ptn.split("/")[-1]
        fname_split = fname_split.split("_")
        self.nucleus = fname_split[0]
        self.model_space = fname_split[1]

    def _read_ptn(self):
        """
        Read `KSHELL` partition file (.ptn) and extract proton
        partition, neutron partition, and particle-hole truncation data.
        Save as instance attributes.
        """

        line_number = 0
        line_number_inner = 0
        self.truncation = []

        with open(self.fname_ptn, "r") as infile:
            for line in infile:
                line_number += 1
                
                if line.startswith("# proton partition"):
                    for line_inner in infile:
                        """
                        Read until next '#'.
                        """
                        line_number_inner += 1
                        if line_inner.startswith("#"):
                            line = line_inner
                            break
                    
                    self.proton_partition = np.loadtxt(
                        fname = self.fname_ptn,
                        skiprows = line_number,
                        max_rows = line_number_inner
                    )
                    line_number += line_number_inner
                    line_number_inner = 0
                
                if line.startswith("# neutron partition"):
                    for line_inner in infile:
                        """
                        Read until next '#'.
                        """
                        line_number_inner += 1
                        if line_inner.startswith("#"):
                            line = line_inner
                            break
                    
                    self.neutron_partition = np.loadtxt(
                        fname = self.fname_ptn,
                        skiprows = line_number,
                        max_rows = line_number_inner
                    )
                    line_number += line_number_inner
                    line_number_inner = 0

                if line.startswith("# particle-hole truncation"):
                    for line_inner in infile:
                        """
                        Loop over all particle-hole truncation lines.
                        """
                        line_number += 1
                        line_inner_split = line_inner.split()

                        if (len(line_inner_split) < 2):
                            """
                            Condition will probably not get fulfilled.
                            Safety precaution due to indexing in this
                            loop.
                            """
                            break

                        if (line_inner_split[1]).startswith("["):
                            """
                            '[' indicates that 'line_inner' is still
                            containing truncation information.
                            """
                            for colon_index, elem in enumerate(line_inner_split):
                                """
                                Find the index of the colon ':' to
                                decide the orbit numbers and occupation
                                numbers.
                                """
                                if (elem == ":"): break

                            occupation = [int(occ) for occ in line_inner_split[colon_index + 1:]]   # [min, max].
                            orbit_numbers = "".join(line_inner_split[1:colon_index])
                            orbit_numbers = orbit_numbers.replace("[", "")
                            orbit_numbers = orbit_numbers.replace("]", "")
                            orbit_numbers = orbit_numbers.replace(" ", "")  # This can prob. be removed because of the earlier split.
                            orbit_numbers = orbit_numbers.split(",")
                            orbit_numbers = [int(orbit) for orbit in orbit_numbers]
                            
                            for orbit in orbit_numbers:
                                self.truncation.append((orbit, occupation))
                        
                        else:
                            """
                            Line does not contain '[' and thus does not
                            contain truncation information.
                            """
                            break

    def _extract_info_from_summary_fname(self):
        """
        Extract nucleus and model space name.
        """
        fname_split = self.fname_summary.split("/")[-1]  # Remove path.
        fname_split = fname_split.split("_")
        self.nucleus = fname_split[1]
        self.model_space = fname_split[2][:-4]  # Remove .txt and keep model space name.

    def _read_summary(self):
        """
        Read energy level data, transition probabilities and transition
        strengths from `KSHELL` output files.

        Raises
        ------
        KshellDataStructureError
            If the `KSHELL` file has unexpected structure / syntax.
        """
        npy_path = "tmp"
        base_fname = self.path.split("/")[-1][:-4]

        try:
            os.mkdir(npy_path)
        except FileExistsError:
            pass
        
        unique_id = _generate_unique_identifier(self.path)
        levels_fname = f"{npy_path}/{base_fname}_levels_{unique_id}.npy"
        transitions_BM1_fname = f"{npy_path}/{base_fname}_transitions_BM1_{unique_id}.npy"
        transitions_BE2_fname = f"{npy_path}/{base_fname}_transitions_BE2_{unique_id}.npy"
        transitions_BE1_fname = f"{npy_path}/{base_fname}_transitions_BE1_{unique_id}.npy"
        debug_fname = f"{npy_path}/{base_fname}_debug_{unique_id}.npy"

        fnames = [
            levels_fname, transitions_BE2_fname, transitions_BM1_fname,
            transitions_BE1_fname, debug_fname
        ]

        if self.load_and_save_to_file != "overwrite":
            """
            Do not load files if overwrite parameter has been passed.
            """
            if all([os.path.isfile(fname) for fname in fnames]) and self.load_and_save_to_file:
                """
                If all files exist, load them. If any of the files do
                not exist, all will be generated.
                """
                self.levels = np.load(file=levels_fname, allow_pickle=True)
                self.transitions_BM1 = np.load(file=transitions_BM1_fname, allow_pickle=True)
                self.transitions_BE2 = np.load(file=transitions_BE2_fname, allow_pickle=True)
                self.transitions_BE1 = np.load(file=transitions_BE1_fname, allow_pickle=True)
                self.debug = np.load(file=debug_fname, allow_pickle=True)
                msg = "Summary data loaded from .npy!"
                msg += " Use loadtxt parameter load_and_save_to_file = 'overwrite'"
                msg += " to re-read data from the summary file."
                print(msg)
                return

        if self.old_or_new == "new":
            parallel_args = [
                [self.fname_summary, "Energy", _load_energy_levels],
                [self.fname_summary, "B(M1)", _load_transition_probabilities],
                [self.fname_summary, "B(E2)", _load_transition_probabilities],
                [self.fname_summary, "B(E1)", _load_transition_probabilities],
            ]
        elif self.old_or_new == "old":
            parallel_args = [
                [self.fname_summary, "Energy", _load_energy_levels],
                [self.fname_summary, "B(M1)", _load_transition_probabilities_old],
                [self.fname_summary, "B(E2)", _load_transition_probabilities_old],
                [self.fname_summary, "B(E1)", _load_transition_probabilities_old],
            ]

        pool = multiprocessing.Pool()
        pool_res = pool.map(_load_parallel, parallel_args)

        try:
            self.levels, self.negative_spin_counts[0] = pool_res[0]
        except TypeError:
            """
            If no energy values are found in the file:
            TypeError: cannot unpack non-iterable NoneType object
            """
            pass

        try:
            self.transitions_BM1, self.negative_spin_counts[1] = pool_res[1]
        except (TypeError, IndexError):
            """
            If no BM1 values are found in the file:
            TypeError: cannot unpack non-iterable NoneType object
            """
            pass
            
        try:
            self.transitions_BE2, self.negative_spin_counts[2] = pool_res[2]
        except (TypeError, IndexError):
            """
            If no BE2 values are found in the file:
            TypeError: cannot unpack non-iterable NoneType object
            """
            pass

        try:
            self.transitions_BE1, self.negative_spin_counts[3] = pool_res[3]
        except (TypeError, IndexError):
            """
            If no BE1 values are found in the file:
            TypeError: cannot unpack non-iterable NoneType object
            """
            pass

        self.levels = np.array(self.levels)
        self.transitions_BM1 = np.array(self.transitions_BM1)
        self.transitions_BE2 = np.array(self.transitions_BE2)
        self.transitions_BE1 = np.array(self.transitions_BE1)
        self.debug = "DEBUG\n"
        self.debug += f"skipped -1 states in levels: {self.negative_spin_counts[0]}\n"
        self.debug += f"skipped -1 states in BM1: {self.negative_spin_counts[1]}\n"
        self.debug += f"skipped -1 states in BE2: {self.negative_spin_counts[2]}\n"
        self.debug += f"skipped -1 states in BE1: {self.negative_spin_counts[3]}\n"
        self.debug = np.array(self.debug)

        if self.load_and_save_to_file:
            np.save(file=levels_fname, arr=self.levels, allow_pickle=True)
            np.save(file=transitions_BM1_fname, arr=self.transitions_BM1, allow_pickle=True)
            np.save(file=transitions_BE2_fname, arr=self.transitions_BE2, allow_pickle=True)
            np.save(file=transitions_BE1_fname, arr=self.transitions_BE1, allow_pickle=True)
            np.save(file=debug_fname, arr=self.debug, allow_pickle=True)

    def level_plot(self,
        max_spin_states: int = 1000,
        filter_spins: Union[None, list] = None
        ):
        """
        Wrapper method to include level plot as an attribute to this
        class. Generate a level plot for a single isotope. Spin on the x
        axis, energy on the y axis.

        Parameters
        ----------
        max_spin_states : int
            The maximum amount of states to plot for each spin. Default
            set to a large number to indicate ≈ no limit.

        filter_spins : Union[None, list]
            Which spins to include in the plot. If `None`, all spins are
            plotted. Defaults to `None`
        """
        level_plot(
            levels = self.levels,
            max_spin_states = max_spin_states,
            filter_spins = filter_spins
        )

    def level_density_plot(self,
            bin_size: Union[int, float],
            plot: bool = True
        ):
        """
        Wrapper method to include level density plotting as
        an attribute to this class. Generate the level density with the
        input bin size.

        Parameters
        ----------
        bin_size : Union[int, float]
            Energy interval of which to calculate the density.

        plot : bool
            Toogle plotting on / off.

        Returns
        -------
        bins : np.ndarray
            The corresponding bins (x value for plotting).

        density : np.ndarray
            The level density.
        """
        bins, density = level_density(
            energy_levels = self.levels[:, 0],
            bin_size = bin_size,
            plot = plot
        )

        return bins, density

    def gamma_strength_function_average_plot(self,
        bin_width: Union[float, int],
        Ex_min: Union[float, int],
        Ex_max: Union[float, int],
        multipole_type: str = "M1",
        plot: bool = True
        ):
        """
        Wrapper method to include gamma ray strength function
        calculations as an attribute to this class.

        Parameters
        ----------
        bin_width : Union[float, int]
            The width of the energy bins. A bin width of 0.2 contains 20
            states of uniform spacing of 0.01.

        Ex_min : Union[float, int]
            Lower limit for initial level excitation energy, usually in MeV.

        Ex_max : Union[float, int]
            Upper limit for initial level excitation energy, usually in MeV.

        multipole_type : str
            Choose whether to calculate for 'E1', 'M1' or 'E2'. NOTE:
            Currently only M1 is implemented.

        plot : bool
            Toogle plotting on / off.
        """
        transitions_dict = {
            "M1": self.transitions_BM1,
            "E2": self.transitions_BE2
        }
        bins, gsf = gamma_strength_function_average(
            levels = self.levels,
            transitions = transitions_dict[multipole_type],
            bin_width = bin_width,
            Ex_min = Ex_min,
            Ex_max = Ex_max,
            multipole_type = multipole_type,
            plot = plot
        )

        return bins, gsf

    def gsf(self,
        bin_width: Union[float, int],
        Ex_min: Union[float, int],
        Ex_max: Union[float, int],
        multipole_type: str = "M1",
        plot: bool = True
        ):
        """
        Alias for gamma_strength_function_average_plot.
        """
        return self.gamma_strength_function_average_plot(bin_width, Ex_min, Ex_max, multipole_type, plot)

    @property
    def help(self):
        """
        Generate a list of instance attributes without magic and private
        methods.

        Returns
        -------
        help_list : list
            A list of non-magic instance attributes.
        """
        help_list = []
        for elem in dir(self):
            if not elem.startswith("_"):   # Omit magic and private methods.
                help_list.append(elem)
        
        return help_list

    @property
    def parameters(self) -> dict:
        """
        Get the KSHELL parameters from the shell file.

        Returns
        -------
        : dict
            A dictionary of KSHELL parameters.
        """
        path = self.path
        if os.path.isfile(path):
            path = path.rsplit("/", 1)[0]
        return get_parameters(path)

def _process_kshell_output_in_parallel(args):
    """
    Simple wrapper for parallelizing loading of KSHELL files.
    """
    filepath, load_and_save_to_file, old_or_new = args
    print(filepath)
    return ReadKshellOutput(filepath, load_and_save_to_file, old_or_new)

def loadtxt(
    path: str,
    is_directory: bool = False,
    filter_: Union[None, str] = None,
    load_and_save_to_file: Union[bool, str] = True,
    old_or_new = "new"
    ) -> list:
    """
    Wrapper for using ReadKshellOutput class as a function.
    TODO: Consider changing 'path' to 'fname' to be the same as
    np.loadtxt.

    Parameters
    ----------
    path : str
        Filename (and path) of `KSHELL` output data file, or path to
        directory containing sub-directories with `KSHELL` output data.
    
    is_directory : bool
        If True, and 'path' is a directory containing sub-directories
        with `KSHELL` data files, the contents of 'path' will be scanned
        for `KSHELL` data files. Currently supports only summary files.

    filter_ : Union[None, str]
        NOTE: Shouldnt the type be list, not str?

    load_and_save_to_file : Union[bool, str]
        Toggle saving data as `.npy` files on / off. If 'overwrite',
        saved `.npy` files are overwritten.

    old_or_new : str
        Choose between old and new summary file syntax. All summary
        files generated pre 2021-11-24 use old style.
        New:
        J_i  pi_i idx_i Ex_i    J_f  pi_f idx_f Ex_f      dE         B(E2)->         B(E2)->[wu]     B(E2)<-         B(E2)<-[wu]
        5    +    1     0.036   6    +    1     0.000     0.036     70.43477980      6.43689168     59.59865983      5.44660066
        Old:
        J_i    Ex_i     J_f    Ex_f   dE        B(M1)->         B(M1)<- 
        2+(11) 18.393 2+(10) 17.791 0.602 0.1(  0.0) 0.1( 0.0)

    Returns
    -------
    data : list
        List of instances with data from `KSHELL` data file as
        attributes.
    """
    all_fnames = None
    data = []
    if old_or_new not in (old_or_new_allowed := ["old", "new"]):
        msg = f"'old_or_new' argument must be in {old_or_new_allowed}!"
        msg += f" Got '{old_or_new}'."
        raise ValueError(msg)

    if (is_directory) and (not os.path.isdir(path)):
        msg = f"{path} is not a directory"
        raise NotADirectoryError(msg)

    elif (not is_directory) and (not os.path.isfile(path)):
        msg = f"{path} is not a file"
        raise FileNotFoundError(msg)

    elif (is_directory) and (os.path.isdir(path)):
        all_fnames = {}

        for element in sorted(os.listdir(path)):
            """
            List all content in path.
            """
            if os.path.isdir(path + element):
                """
                If element is a directory, enter it to find data files.
                """
                all_fnames[element] = []    # Create blank list entry in dict for current element.
                for isotope in os.listdir(path + element):
                    """
                    List all content in the element directory.
                    """
                    if isotope.startswith("summary") and isotope.endswith(".txt"):
                        """
                        Extract summary data files.
                        """
                        try:
                            """
                            Example: O16.
                            """
                            n_neutrons = int(isotope[9:11])
                        except ValueError:
                            """
                            Example: Ne20.
                            """
                            n_neutrons = int(isotope[10:12])

                        n_neutrons -= atomic_numbers[element.split("_")[1]]
                        all_fnames[element].append([element + "/" + isotope, n_neutrons])
        
        pool = multiprocessing.Pool()
        for key in all_fnames:
            """
            Sort each list in the dict by the number of neutrons. Loop
            over all directories in 'all_fnames' and extract KSHELL data
            and append to a list.
            """
            if filter_ is not None:
                if key.split("_")[1] not in filter_:
                    """
                    Skip elements not in filter_.
                    """
                    continue

            all_fnames[key].sort(key=lambda tup: tup[1])   # Why not do this when directory is listed?
            sub_fnames = all_fnames[key]
            arg_list = [(path + i[0], load_and_save_to_file, old_or_new) for i in sub_fnames]
            data += pool.map(_process_kshell_output_in_parallel, arg_list)

    else:
        """
        Only a single KSHELL data file.
        """
        data.append(ReadKshellOutput(path, load_and_save_to_file, old_or_new))

    if not data:
        msg = "No KSHELL data loaded. Most likely error is that the given"
        msg += f" directory has no KSHELL data files. {path=}"
        raise RuntimeError(msg)

    return data

def _get_timing_data(path: str):
    """
    Get timing data from KSHELL log files.

    Parameters
    ----------
    path : str
        Path to log file.

    Examples
    --------
    Last 10 lines of log_Ar30_usda_m0p.txt:
    ```
          total      20.899         2    10.44928   1.0000
    pre-process       0.029         1     0.02866   0.0014
        operate       3.202      1007     0.00318   0.1532
     re-orthog.      11.354       707     0.01606   0.5433
  thick-restart       0.214        12     0.01781   0.0102
   diag tri-mat       3.880       707     0.00549   0.1857
           misc       2.220                         0.1062

           tmp        0.002       101     0.00002   0.0001
    ```
    """

    if "log" not in path:
        msg = f"Unknown log file name! Got '{path}'"
        raise KshellDataStructureError(msg)

    if not os.path.isfile(path):
        raise FileNotFoundError(path)

    res = os.popen(f'tail -n 20 {path}').read()    # Get the final 10 lines.
    res = res.split("\n")
    total = None
    
    if "tr" not in path:
        """
        KSHELL log.
        """
        for elem in res:
            tmp = elem.split()
            try:
                if tmp[0] == "total":
                    total = float(tmp[1])
                    break
            except IndexError:
                continue
        
    elif "tr" in path:
        """
        Transit log.
        """
        for elem in res:
            tmp = elem.split()
            try:
                if tmp[0] == "total":
                    total = float(tmp[3])
                    break
            except IndexError:
                continue

    if total is None:
        msg = f"Not able to extract timing data from '{path}'!"
        raise KshellDataStructureError(msg)
    
    return total

def _get_memory_usage(path: str) -> Union[float, None]:
    """
    Get memory usage from KSHELL log files.

    Parameters
    ----------
    path : str
        Path to a single log file.

    Returns
    -------
    total : float, None
        Memory usage in GB or None if memory usage could not be read.
    """
    total = None
    
    if "tr" not in path:
        """
        KSHELL log.
        """
        with open(path, "r") as infile:
            for line in infile:
                if line.startswith("Total Memory for Lanczos vectors:"):
                    try:
                        total = float(line.split()[-2])
                    except ValueError:
                        msg = f"Error reading memory usage from '{path}'."
                        msg += f" Got '{line.split()[-2]}'."
                        raise KshellDataStructureError(msg)
                    break
        
    elif "tr" in path:
        """
        Transit log. NOTE: Not yet implemented.
        """
        return 0

    if total is None:
        msg = f"Not able to extract memory data from '{path.split('/')[-1]}'!"
        raise KshellDataStructureError(msg)
    
    return total

def _get_data_general(path: str, func: Callable):
    """
    General input handling for timing data and memory data.

    Parameters
    ----------
    path : str
        Path to a single log file or path to a directory of log files.

    func : Callable
        _get_timing_data or _get_memory_usage.
    """
    if os.path.isfile(path):
        return func(path)
    
    elif os.path.isdir(path):
        total = 0
        for elem in os.listdir(path):
            if elem.startswith("log_") and elem.endswith(".txt"):
                total += func(f"{path}/{elem}")
        
        return total

    else:
        msg = f"'{path}' is neither a file nor a directory!"
        raise NotADirectoryError(msg)

def get_timing_data(path: str) -> float:
    """
    Wrapper for _get_timing_data. Input a single log filename and get
    the timing data. Input a path to a directory several log files and
    get the summed timing data. In units of seconds.

    Parameters
    ----------
    path : str
        Path to a single log file or path to a directory of log files.

    Returns
    -------
    : float
        The summed times for all input log files.
    """
    return _get_data_general(path, _get_timing_data)

def get_memory_usage(path: str) -> float:
    """
    Wrapper for _get_memory_usage. Input a single log filename and get
    the memory data. Input a path to a directory several log files and
    get the summed memory data. In units of GB.

    Parameters
    ----------
    path : str
        Path to a single log file or path to a directory of log files.

    Returns
    -------
    : float
        The summed memory usage for all input log files.
    """
    return _get_data_general(path, _get_memory_usage)

def get_parameters(path: str, verbose: bool = True) -> dict:
    """
    Extract the parameters which are fed to KSHELL throught the shell
    script.

    Parameters
    ----------
    path : str
        Path to a KSHELL work directory.

    Returns
    -------
    res : dict
        A dictionary where the keys are the parameter names and the
        values are the corresponding values.
    """
    res = {}
    shell_filename = None
    if os.path.isdir(path):
        for elem in os.listdir(path):
            if elem.endswith(".sh"):
                shell_filename = f"{path}/{elem}"
                break
    else:
        print("Directly specifying path to .sh file not yet implemented!")

    if shell_filename is None:
        if verbose:
            msg = f"No .sh file found in path '{path}'!"
            print(msg)

        return res
    
    with open(shell_filename, "r") as infile:
        for line in infile:
            if line.startswith(r"&input"):
                break
        
        for line in infile:
            if line.startswith(r"&end"):
                """
                End of parameters.
                """
                break
            
            tmp = line.split("=")
            key = tmp[0].strip()
            value = tmp[1].strip()

            try:
                value = ast.literal_eval(value)
            except ValueError:
                """
                Cant convert strings. Keep them as strings.
                """
                pass
            except SyntaxError:
                """
                Cant convert Fortran booleans (.true., .false.). Keep
                them as strings.
                """
                pass
            
            res[key] = value

    return res
