from __future__ import annotations

import dataclasses
from dataclasses import dataclass
from datetime import datetime, timezone
from pathlib import Path
from typing import Any

import torch
from torch import Tensor
from torch.nn import Module
from torch.serialization import MAP_LOCATION

import lightly_train
from lightly_train._models.embedding_model import EmbeddingModel

CHECKPOINT_LIGHTLY_TRAIN_KEY = "lightly_train"


@dataclass(frozen=True)
class CheckpointLightlyTrain:
    version: str
    date: datetime
    models: CheckpointLightlyTrainModels

    def to_dict(self) -> dict[str, Any]:
        d = dataclasses.asdict(self)
        d["date"] = d["date"].isoformat()
        d["models"] = self.models.to_dict()
        return d

    @staticmethod
    def from_dict(checkpoint_info: dict[str, Any]) -> CheckpointLightlyTrain:
        return CheckpointLightlyTrain(
            version=checkpoint_info["version"],
            date=datetime.fromisoformat(checkpoint_info["date"]),
            models=CheckpointLightlyTrainModels.from_dict(checkpoint_info["models"]),
        )

    @staticmethod
    def from_now(models: CheckpointLightlyTrainModels) -> CheckpointLightlyTrain:
        return CheckpointLightlyTrain(
            version=lightly_train.__version__,
            date=datetime.now(timezone.utc).astimezone(),
            models=models,
        )

    @staticmethod
    def from_checkpoint_dict(checkpoint: dict[str, Any]) -> CheckpointLightlyTrain:
        return CheckpointLightlyTrain.from_dict(
            checkpoint[CHECKPOINT_LIGHTLY_TRAIN_KEY]
        )


@dataclass(frozen=True)
class CheckpointLightlyTrainModels:
    model: Module
    embedding_model: EmbeddingModel

    def to_dict(self) -> dict[str, Any]:
        return {
            "model": self.model,
            "embedding_model": self.embedding_model,
        }

    @staticmethod
    def from_dict(models: dict[str, Any]) -> CheckpointLightlyTrainModels:
        return CheckpointLightlyTrainModels(
            model=models["model"],
            embedding_model=models["embedding_model"],
        )


@dataclass(frozen=True)
class Checkpoint:
    """Checkpoint as generated by PyTorch Lightning Trainer with the ModelCheckpoint
    callback from lightly_train.callbacks.checkpoint.
    """

    state_dict: dict[str, Tensor]
    lightly_train: CheckpointLightlyTrain

    def to_dict(self) -> dict[str, Any]:
        return {
            "state_dict": self.state_dict,
            CHECKPOINT_LIGHTLY_TRAIN_KEY: self.lightly_train.to_dict(),
        }

    @staticmethod
    def from_dict(checkpoint: dict[str, Any]) -> Checkpoint:
        return Checkpoint(
            state_dict=checkpoint["state_dict"],
            lightly_train=CheckpointLightlyTrain.from_checkpoint_dict(
                checkpoint=checkpoint
            ),
        )

    @staticmethod
    def from_path(
        checkpoint: Path, map_location: MAP_LOCATION | None = "cpu"
    ) -> Checkpoint:
        checkpoint_dict = torch.load(checkpoint, map_location=map_location)
        return Checkpoint.from_dict(checkpoint=checkpoint_dict)

    def save(self, path: Path) -> None:
        path.parent.mkdir(parents=True, exist_ok=True)
        torch.save(self.to_dict(), path)
