from collections.abc import Mapping
from typing import Any, TypeVar, cast

from attrs import (
    define as _attrs_define,
    field as _attrs_field,
)

T = TypeVar("T", bound="MaterialConfig")


@_attrs_define
class MaterialConfig:
    """Configuration option for a material that defines variant attributes

    Example:
        {'id': 101, 'name': 'Grade', 'values': ['Premium', 'Standard', 'Economy'], 'material_id': 1}

    Attributes:
        id (int): Unique identifier
        name (str): Configuration attribute name (e.g., Grade, Thickness, Color)
        values (list[str]): Array of possible values for this configuration attribute
        material_id (int): ID of the material this configuration belongs to
    """

    id: int
    name: str
    values: list[str]
    material_id: int
    additional_properties: dict[str, Any] = _attrs_field(init=False, factory=dict)

    def to_dict(self) -> dict[str, Any]:
        id = self.id

        name = self.name

        values = self.values

        material_id = self.material_id

        field_dict: dict[str, Any] = {}
        field_dict.update(self.additional_properties)
        field_dict.update(
            {
                "id": id,
                "name": name,
                "values": values,
                "material_id": material_id,
            }
        )

        return field_dict

    @classmethod
    def from_dict(cls: type[T], src_dict: Mapping[str, Any]) -> T:
        d = dict(src_dict)
        id = d.pop("id")

        name = d.pop("name")

        values = cast(list[str], d.pop("values"))

        material_id = d.pop("material_id")

        material_config = cls(
            id=id,
            name=name,
            values=values,
            material_id=material_id,
        )

        material_config.additional_properties = d
        return material_config

    @property
    def additional_keys(self) -> list[str]:
        return list(self.additional_properties.keys())

    def __getitem__(self, key: str) -> Any:
        return self.additional_properties[key]

    def __setitem__(self, key: str, value: Any) -> None:
        self.additional_properties[key] = value

    def __delitem__(self, key: str) -> None:
        del self.additional_properties[key]

    def __contains__(self, key: str) -> bool:
        return key in self.additional_properties
