import datetime
from collections.abc import Mapping
from typing import TYPE_CHECKING, Any, TypeVar

from attrs import define as _attrs_define
from dateutil.parser import isoparse

from ..client_types import UNSET, Unset

if TYPE_CHECKING:
    from ..models.create_sales_return_row_request import CreateSalesReturnRowRequest


T = TypeVar("T", bound="CreateSalesReturnRequest")


@_attrs_define
class CreateSalesReturnRequest:
    """Request payload for creating a new sales return to process customer product returns and refunds

    Example:
        {'customer_id': 1001, 'sales_order_id': 2001, 'order_no': 'SR-2023-001', 'return_location_id': 1, 'currency':
            'USD', 'order_created_date': '2023-10-10T10:00:00Z', 'additional_info': 'Customer reported damaged items during
            shipping', 'sales_return_rows': [{'variant_id': 2002, 'quantity': 2, 'return_reason_id': 1, 'notes': 'Packaging
            was damaged', 'unit_price': 25.0, 'total_price': 50.0}]}

    Attributes:
        customer_id (int): ID of the customer initiating the return
        order_no (str): Return order reference number
        return_location_id (int): ID of the location where items are being returned to
        sales_return_rows (list['CreateSalesReturnRowRequest']): Array of items being returned
        sales_order_id (Union[Unset, int]): ID of the original sales order being returned
        currency (Union[Unset, str]): Currency code (e.g., USD, EUR)
        order_created_date (Union[Unset, datetime.datetime]): Date when the original order was created
        additional_info (Union[Unset, str]): Optional notes or comments about the return
    """

    customer_id: int
    order_no: str
    return_location_id: int
    sales_return_rows: list["CreateSalesReturnRowRequest"]
    sales_order_id: Unset | int = UNSET
    currency: Unset | str = UNSET
    order_created_date: Unset | datetime.datetime = UNSET
    additional_info: Unset | str = UNSET

    def to_dict(self) -> dict[str, Any]:
        customer_id = self.customer_id

        order_no = self.order_no

        return_location_id = self.return_location_id

        sales_return_rows = []
        for sales_return_rows_item_data in self.sales_return_rows:
            sales_return_rows_item = sales_return_rows_item_data.to_dict()
            sales_return_rows.append(sales_return_rows_item)

        sales_order_id = self.sales_order_id

        currency = self.currency

        order_created_date: Unset | str = UNSET
        if not isinstance(self.order_created_date, Unset):
            order_created_date = self.order_created_date.isoformat()

        additional_info = self.additional_info

        field_dict: dict[str, Any] = {}

        field_dict.update(
            {
                "customer_id": customer_id,
                "order_no": order_no,
                "return_location_id": return_location_id,
                "sales_return_rows": sales_return_rows,
            }
        )
        if sales_order_id is not UNSET:
            field_dict["sales_order_id"] = sales_order_id
        if currency is not UNSET:
            field_dict["currency"] = currency
        if order_created_date is not UNSET:
            field_dict["order_created_date"] = order_created_date
        if additional_info is not UNSET:
            field_dict["additional_info"] = additional_info

        return field_dict

    @classmethod
    def from_dict(cls: type[T], src_dict: Mapping[str, Any]) -> T:
        from ..models.create_sales_return_row_request import CreateSalesReturnRowRequest

        d = dict(src_dict)
        customer_id = d.pop("customer_id")

        order_no = d.pop("order_no")

        return_location_id = d.pop("return_location_id")

        sales_return_rows = []
        _sales_return_rows = d.pop("sales_return_rows")
        for sales_return_rows_item_data in _sales_return_rows:
            sales_return_rows_item = CreateSalesReturnRowRequest.from_dict(
                sales_return_rows_item_data
            )

            sales_return_rows.append(sales_return_rows_item)

        sales_order_id = d.pop("sales_order_id", UNSET)

        currency = d.pop("currency", UNSET)

        _order_created_date = d.pop("order_created_date", UNSET)
        order_created_date: Unset | datetime.datetime
        if isinstance(_order_created_date, Unset):
            order_created_date = UNSET
        else:
            order_created_date = isoparse(_order_created_date)

        additional_info = d.pop("additional_info", UNSET)

        create_sales_return_request = cls(
            customer_id=customer_id,
            order_no=order_no,
            return_location_id=return_location_id,
            sales_return_rows=sales_return_rows,
            sales_order_id=sales_order_id,
            currency=currency,
            order_created_date=order_created_date,
            additional_info=additional_info,
        )

        return create_sales_return_request
