"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const client_secrets_manager_1 = require("@aws-sdk/client-secrets-manager");
const core_1 = require("@octokit/core");
const github_secret_encryptor_1 = require("../github-secret-encryptor");
const github_secret_name_validator_1 = require("../github-secret-name-validator");
const onEvent = async (event) => {
    console.log(`Event: ${JSON.stringify(event)}`);
    github_secret_name_validator_1.validateSecretName(event.ResourceProperties.repositorySecretName);
    const smClient = new client_secrets_manager_1.SecretsManagerClient({ region: event.ResourceProperties.awsRegion });
    const githubTokenSecret = await smClient.send(new client_secrets_manager_1.GetSecretValueCommand({ SecretId: event.ResourceProperties.githubTokenSecret }));
    const octokit = new core_1.Octokit({ auth: githubTokenSecret.SecretString });
    const requestType = event.RequestType;
    switch (requestType) {
        case 'Create':
            return onCreate(event, octokit, smClient);
        case 'Update':
            return onUpdate(event, octokit, smClient);
        case 'Delete':
            return onDelete(event, octokit);
        default:
            throw new Error(`Unexpected request type: '${requestType}'`);
    }
};
const onCreate = async (event, octokit, smClient) => {
    const props = event.ResourceProperties;
    const secretName = props.repositorySecretName;
    console.log('Create new ActionEnvironmentSecret with props ' + JSON.stringify(props));
    await createOrUpdateEnvironmentSecret(event, octokit, smClient);
    return { PhysicalResourceId: secretName };
};
const onUpdate = async (event, octokit, smClient) => {
    const props = event.ResourceProperties;
    const secretName = props.repositorySecretName;
    console.log(`Update ActionEnvironmentSecret ${secretName} with props ${JSON.stringify(props)}`);
    await createOrUpdateEnvironmentSecret(event, octokit, smClient);
    return { PhysicalResourceId: secretName };
};
const onDelete = async (event, octokit) => {
    const secretName = event.ResourceProperties.repositorySecretName;
    console.log('Delete ActionEnvironmentSecret ' + secretName);
    await deleteEnvironmentSecret(event, octokit);
    return { PhysicalResourceId: secretName };
};
const createOrUpdateEnvironmentSecret = async (event, octokit, smClient) => {
    const { repositoryOwner: owner, repositoryName: repo, repositorySecretName: secret_name, environment: environment_name, sourceSecretArn: secretId, } = event.ResourceProperties;
    const secretToEncrypt = await smClient.send(new client_secrets_manager_1.GetSecretValueCommand({ SecretId: secretId }));
    console.log(`Encrypt value of secret with id: ${secretId}`);
    const secretString = secretToEncrypt.SecretString;
    if (!secretString) {
        throw new Error('SecretString is empty from secret with id: ' + secretId);
    }
    const { data } = await octokit.request('GET /repos/{owner}/{repo}/actions/secrets/public-key', { owner, repo });
    const encryptedSecret = await github_secret_encryptor_1.encryptValue(secretString, data.key);
    console.log('Encrypted secret, attempting to create/update github secret');
    const repository_id = await getRepositoryId(event, octokit);
    const secretResponse = await octokit.request('PUT /repositories/{repository_id}/environments/{environment_name}/secrets/{secret_name}', {
        repository_id,
        environment_name,
        secret_name,
        encrypted_value: encryptedSecret,
        key_id: data.key_id,
    });
    console.log(JSON.stringify(secretResponse));
    return secretResponse;
};
const deleteEnvironmentSecret = async (event, octokit) => {
    const { environment: environment_name, repositorySecretName: secret_name } = event.ResourceProperties;
    const repository_id = await getRepositoryId(event, octokit);
    const deleteSecretResponse = await octokit.request('DELETE /repositories/{repository_id}/environments/{environment_name}/secrets/{secret_name}', {
        repository_id,
        environment_name,
        secret_name,
    });
    console.log(`Delete: ${JSON.stringify(deleteSecretResponse)}`);
    return deleteSecretResponse;
};
const getRepositoryId = async (event, octokit) => {
    const { repositoryOwner: owner, repositoryName: repo } = event.ResourceProperties;
    const { data } = await octokit.request('GET /repos/{owner}/{repo}', {
        owner,
        repo,
    });
    return data.id;
};
exports.handler = onEvent;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYWN0aW9uLWVudmlyb25tZW50LXNlY3JldC1oYW5kbGVyLmxhbWJkYS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uLy4uL3NyYy9oYW5kbGVyL3NlY3JldHMvYWN0aW9uLWVudmlyb25tZW50LXNlY3JldHMvYWN0aW9uLWVudmlyb25tZW50LXNlY3JldC1oYW5kbGVyLmxhbWJkYS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSw0RUFBOEY7QUFDOUYsd0NBQXdDO0FBR3hDLHdFQUEwRDtBQUMxRCxrRkFBcUU7QUFFckUsTUFBTSxPQUFPLEdBQUcsS0FBSyxFQUFFLEtBQXdELEVBQUUsRUFBRTtJQUNqRixPQUFPLENBQUMsR0FBRyxDQUFDLFVBQVUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLENBQUM7SUFDL0MsaURBQWtCLENBQUMsS0FBSyxDQUFDLGtCQUFrQixDQUFDLG9CQUFvQixDQUFDLENBQUM7SUFDbEUsTUFBTSxRQUFRLEdBQUcsSUFBSSw2Q0FBb0IsQ0FBQyxFQUFFLE1BQU0sRUFBRSxLQUFLLENBQUMsa0JBQWtCLENBQUMsU0FBUyxFQUFFLENBQUMsQ0FBQztJQUMxRixNQUFNLGlCQUFpQixHQUFHLE1BQU0sUUFBUSxDQUFDLElBQUksQ0FBQyxJQUFJLDhDQUFxQixDQUFDLEVBQUUsUUFBUSxFQUFFLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxpQkFBaUIsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUNuSSxNQUFNLE9BQU8sR0FBRyxJQUFJLGNBQU8sQ0FBQyxFQUFFLElBQUksRUFBRSxpQkFBaUIsQ0FBQyxZQUFZLEVBQUUsQ0FBQyxDQUFDO0lBRXRFLE1BQU0sV0FBVyxHQUFHLEtBQUssQ0FBQyxXQUFXLENBQUM7SUFDdEMsUUFBUSxXQUFXLEVBQUU7UUFDbkIsS0FBSyxRQUFRO1lBQ1gsT0FBTyxRQUFRLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRSxRQUFRLENBQUMsQ0FBQztRQUM1QyxLQUFLLFFBQVE7WUFDWCxPQUFPLFFBQVEsQ0FBQyxLQUFLLEVBQUUsT0FBTyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQzVDLEtBQUssUUFBUTtZQUNYLE9BQU8sUUFBUSxDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztRQUNsQztZQUNFLE1BQU0sSUFBSSxLQUFLLENBQUMsNkJBQTZCLFdBQVcsR0FBRyxDQUFDLENBQUM7S0FDaEU7QUFDSCxDQUFDLENBQUM7QUFFRixNQUFNLFFBQVEsR0FBRyxLQUFLLEVBQ3BCLEtBQXdELEVBQ3hELE9BQWdCLEVBQ2hCLFFBQThCLEVBQzlCLEVBQUU7SUFDRixNQUFNLEtBQUssR0FBRyxLQUFLLENBQUMsa0JBQWtCLENBQUM7SUFDdkMsTUFBTSxVQUFVLEdBQUcsS0FBSyxDQUFDLG9CQUFvQixDQUFDO0lBQzlDLE9BQU8sQ0FBQyxHQUFHLENBQUMsZ0RBQWdELEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO0lBRXRGLE1BQU0sK0JBQStCLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRSxRQUFRLENBQUMsQ0FBQztJQUNoRSxPQUFPLEVBQUUsa0JBQWtCLEVBQUUsVUFBVSxFQUFFLENBQUM7QUFDNUMsQ0FBQyxDQUFDO0FBRUYsTUFBTSxRQUFRLEdBQUcsS0FBSyxFQUNwQixLQUF3RCxFQUN4RCxPQUFnQixFQUNoQixRQUE4QixFQUM5QixFQUFFO0lBQ0YsTUFBTSxLQUFLLEdBQUcsS0FBSyxDQUFDLGtCQUFrQixDQUFDO0lBQ3ZDLE1BQU0sVUFBVSxHQUFHLEtBQUssQ0FBQyxvQkFBb0IsQ0FBQztJQUM5QyxPQUFPLENBQUMsR0FBRyxDQUFDLGtDQUFrQyxVQUFVLGVBQWUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLENBQUM7SUFFaEcsTUFBTSwrQkFBK0IsQ0FBQyxLQUFLLEVBQUUsT0FBTyxFQUFFLFFBQVEsQ0FBQyxDQUFDO0lBQ2hFLE9BQU8sRUFBRSxrQkFBa0IsRUFBRSxVQUFVLEVBQUUsQ0FBQztBQUM1QyxDQUFDLENBQUM7QUFFRixNQUFNLFFBQVEsR0FBRyxLQUFLLEVBQ3BCLEtBQXdELEVBQ3hELE9BQWdCLEVBQ2hCLEVBQUU7SUFDRixNQUFNLFVBQVUsR0FBRyxLQUFLLENBQUMsa0JBQWtCLENBQUMsb0JBQW9CLENBQUM7SUFDakUsT0FBTyxDQUFDLEdBQUcsQ0FBQyxpQ0FBaUMsR0FBRyxVQUFVLENBQUMsQ0FBQztJQUM1RCxNQUFNLHVCQUF1QixDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztJQUM5QyxPQUFPLEVBQUUsa0JBQWtCLEVBQUUsVUFBVSxFQUFFLENBQUM7QUFDNUMsQ0FBQyxDQUFDO0FBRUYsTUFBTSwrQkFBK0IsR0FBRyxLQUFLLEVBQzNDLEtBQXdELEVBQ3hELE9BQWdCLEVBQ2hCLFFBQThCLEVBQzlCLEVBQUU7SUFDRixNQUFNLEVBQ0osZUFBZSxFQUFFLEtBQUssRUFDdEIsY0FBYyxFQUFFLElBQUksRUFDcEIsb0JBQW9CLEVBQUUsV0FBVyxFQUNqQyxXQUFXLEVBQUUsZ0JBQWdCLEVBQzdCLGVBQWUsRUFBRSxRQUFRLEdBQzFCLEdBQUcsS0FBSyxDQUFDLGtCQUFrQixDQUFDO0lBRTdCLE1BQU0sZUFBZSxHQUFHLE1BQU0sUUFBUSxDQUFDLElBQUksQ0FBQyxJQUFJLDhDQUFxQixDQUFDLEVBQUUsUUFBUSxFQUFFLFFBQVEsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUMvRixPQUFPLENBQUMsR0FBRyxDQUFDLG9DQUFvQyxRQUFRLEVBQUUsQ0FBQyxDQUFDO0lBRTVELE1BQU0sWUFBWSxHQUFHLGVBQWUsQ0FBQyxZQUFZLENBQUM7SUFDbEQsSUFBSSxDQUFDLFlBQVksRUFBRTtRQUNqQixNQUFNLElBQUksS0FBSyxDQUFDLDZDQUE2QyxHQUFHLFFBQVEsQ0FBQyxDQUFDO0tBQzNFO0lBRUQsTUFBTSxFQUFFLElBQUksRUFBRSxHQUFHLE1BQU0sT0FBTyxDQUFDLE9BQU8sQ0FBQyxzREFBc0QsRUFBRSxFQUFFLEtBQUssRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO0lBRWhILE1BQU0sZUFBZSxHQUFHLE1BQU0sc0NBQVksQ0FBQyxZQUFZLEVBQUUsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ25FLE9BQU8sQ0FBQyxHQUFHLENBQUMsNkRBQTZELENBQUMsQ0FBQztJQUUzRSxNQUFNLGFBQWEsR0FBRyxNQUFNLGVBQWUsQ0FBQyxLQUFLLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFDNUQsTUFBTSxjQUFjLEdBQUcsTUFBTSxPQUFPLENBQUMsT0FBTyxDQUFDLHlGQUF5RixFQUFFO1FBQ3RJLGFBQWE7UUFDYixnQkFBZ0I7UUFDaEIsV0FBVztRQUNYLGVBQWUsRUFBRSxlQUFlO1FBQ2hDLE1BQU0sRUFBRSxJQUFJLENBQUMsTUFBTTtLQUNwQixDQUFDLENBQUM7SUFFSCxPQUFPLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQztJQUM1QyxPQUFPLGNBQWMsQ0FBQztBQUN4QixDQUFDLENBQUM7QUFFRixNQUFNLHVCQUF1QixHQUFHLEtBQUssRUFDbkMsS0FBd0QsRUFDeEQsT0FBZ0IsRUFDaEIsRUFBRTtJQUNGLE1BQU0sRUFBRSxXQUFXLEVBQUUsZ0JBQWdCLEVBQUUsb0JBQW9CLEVBQUUsV0FBVyxFQUFFLEdBQUcsS0FBSyxDQUFDLGtCQUFrQixDQUFDO0lBQ3RHLE1BQU0sYUFBYSxHQUFHLE1BQU0sZUFBZSxDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztJQUM1RCxNQUFNLG9CQUFvQixHQUFHLE1BQU0sT0FBTyxDQUFDLE9BQU8sQ0FBQyw0RkFBNEYsRUFBRTtRQUMvSSxhQUFhO1FBQ2IsZ0JBQWdCO1FBQ2hCLFdBQVc7S0FDWixDQUFDLENBQUM7SUFDSCxPQUFPLENBQUMsR0FBRyxDQUFDLFdBQVcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxvQkFBb0IsQ0FBQyxFQUFFLENBQUMsQ0FBQztJQUMvRCxPQUFPLG9CQUFvQixDQUFDO0FBQzlCLENBQUMsQ0FBQztBQUVGLE1BQU0sZUFBZSxHQUFHLEtBQUssRUFDM0IsS0FBd0QsRUFDeEQsT0FBZ0IsRUFDaEIsRUFBRTtJQUNGLE1BQU0sRUFBRSxlQUFlLEVBQUUsS0FBSyxFQUFFLGNBQWMsRUFBRSxJQUFJLEVBQUUsR0FBRyxLQUFLLENBQUMsa0JBQWtCLENBQUM7SUFDbEYsTUFBTSxFQUFFLElBQUksRUFBRSxHQUFHLE1BQU0sT0FBTyxDQUFDLE9BQU8sQ0FBQywyQkFBMkIsRUFBRTtRQUNsRSxLQUFLO1FBQ0wsSUFBSTtLQUNMLENBQUMsQ0FBQztJQUNILE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQztBQUNqQixDQUFDLENBQUM7QUFHVyxRQUFBLE9BQU8sR0FBRyxPQUFPLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBTZWNyZXRzTWFuYWdlckNsaWVudCwgR2V0U2VjcmV0VmFsdWVDb21tYW5kIH0gZnJvbSAnQGF3cy1zZGsvY2xpZW50LXNlY3JldHMtbWFuYWdlcic7XG5pbXBvcnQgeyBPY3Rva2l0IH0gZnJvbSAnQG9jdG9raXQvY29yZSc7XG5pbXBvcnQgdHlwZSB7IE9uRXZlbnRSZXF1ZXN0LCBBY3Rpb25FbnZpcm9ubWVudFNlY3JldEV2ZW50UHJvcHMgfSBmcm9tICcuLi8uLi8uLi90eXBlcyc7XG5cbmltcG9ydCB7IGVuY3J5cHRWYWx1ZSB9IGZyb20gJy4uL2dpdGh1Yi1zZWNyZXQtZW5jcnlwdG9yJztcbmltcG9ydCB7IHZhbGlkYXRlU2VjcmV0TmFtZSB9IGZyb20gJy4uL2dpdGh1Yi1zZWNyZXQtbmFtZS12YWxpZGF0b3InO1xuXG5jb25zdCBvbkV2ZW50ID0gYXN5bmMgKGV2ZW50OiBPbkV2ZW50UmVxdWVzdDxBY3Rpb25FbnZpcm9ubWVudFNlY3JldEV2ZW50UHJvcHM+KSA9PiB7XG4gIGNvbnNvbGUubG9nKGBFdmVudDogJHtKU09OLnN0cmluZ2lmeShldmVudCl9YCk7XG4gIHZhbGlkYXRlU2VjcmV0TmFtZShldmVudC5SZXNvdXJjZVByb3BlcnRpZXMucmVwb3NpdG9yeVNlY3JldE5hbWUpO1xuICBjb25zdCBzbUNsaWVudCA9IG5ldyBTZWNyZXRzTWFuYWdlckNsaWVudCh7IHJlZ2lvbjogZXZlbnQuUmVzb3VyY2VQcm9wZXJ0aWVzLmF3c1JlZ2lvbiB9KTtcbiAgY29uc3QgZ2l0aHViVG9rZW5TZWNyZXQgPSBhd2FpdCBzbUNsaWVudC5zZW5kKG5ldyBHZXRTZWNyZXRWYWx1ZUNvbW1hbmQoeyBTZWNyZXRJZDogZXZlbnQuUmVzb3VyY2VQcm9wZXJ0aWVzLmdpdGh1YlRva2VuU2VjcmV0IH0pKTtcbiAgY29uc3Qgb2N0b2tpdCA9IG5ldyBPY3Rva2l0KHsgYXV0aDogZ2l0aHViVG9rZW5TZWNyZXQuU2VjcmV0U3RyaW5nIH0pO1xuXG4gIGNvbnN0IHJlcXVlc3RUeXBlID0gZXZlbnQuUmVxdWVzdFR5cGU7XG4gIHN3aXRjaCAocmVxdWVzdFR5cGUpIHtcbiAgICBjYXNlICdDcmVhdGUnOlxuICAgICAgcmV0dXJuIG9uQ3JlYXRlKGV2ZW50LCBvY3Rva2l0LCBzbUNsaWVudCk7XG4gICAgY2FzZSAnVXBkYXRlJzpcbiAgICAgIHJldHVybiBvblVwZGF0ZShldmVudCwgb2N0b2tpdCwgc21DbGllbnQpO1xuICAgIGNhc2UgJ0RlbGV0ZSc6XG4gICAgICByZXR1cm4gb25EZWxldGUoZXZlbnQsIG9jdG9raXQpO1xuICAgIGRlZmF1bHQ6XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYFVuZXhwZWN0ZWQgcmVxdWVzdCB0eXBlOiAnJHtyZXF1ZXN0VHlwZX0nYCk7XG4gIH1cbn07XG5cbmNvbnN0IG9uQ3JlYXRlID0gYXN5bmMgKFxuICBldmVudDogT25FdmVudFJlcXVlc3Q8QWN0aW9uRW52aXJvbm1lbnRTZWNyZXRFdmVudFByb3BzPixcbiAgb2N0b2tpdDogT2N0b2tpdCxcbiAgc21DbGllbnQ6IFNlY3JldHNNYW5hZ2VyQ2xpZW50LFxuKSA9PiB7XG4gIGNvbnN0IHByb3BzID0gZXZlbnQuUmVzb3VyY2VQcm9wZXJ0aWVzO1xuICBjb25zdCBzZWNyZXROYW1lID0gcHJvcHMucmVwb3NpdG9yeVNlY3JldE5hbWU7XG4gIGNvbnNvbGUubG9nKCdDcmVhdGUgbmV3IEFjdGlvbkVudmlyb25tZW50U2VjcmV0IHdpdGggcHJvcHMgJyArIEpTT04uc3RyaW5naWZ5KHByb3BzKSk7XG5cbiAgYXdhaXQgY3JlYXRlT3JVcGRhdGVFbnZpcm9ubWVudFNlY3JldChldmVudCwgb2N0b2tpdCwgc21DbGllbnQpO1xuICByZXR1cm4geyBQaHlzaWNhbFJlc291cmNlSWQ6IHNlY3JldE5hbWUgfTtcbn07XG5cbmNvbnN0IG9uVXBkYXRlID0gYXN5bmMgKFxuICBldmVudDogT25FdmVudFJlcXVlc3Q8QWN0aW9uRW52aXJvbm1lbnRTZWNyZXRFdmVudFByb3BzPixcbiAgb2N0b2tpdDogT2N0b2tpdCxcbiAgc21DbGllbnQ6IFNlY3JldHNNYW5hZ2VyQ2xpZW50LFxuKSA9PiB7XG4gIGNvbnN0IHByb3BzID0gZXZlbnQuUmVzb3VyY2VQcm9wZXJ0aWVzO1xuICBjb25zdCBzZWNyZXROYW1lID0gcHJvcHMucmVwb3NpdG9yeVNlY3JldE5hbWU7XG4gIGNvbnNvbGUubG9nKGBVcGRhdGUgQWN0aW9uRW52aXJvbm1lbnRTZWNyZXQgJHtzZWNyZXROYW1lfSB3aXRoIHByb3BzICR7SlNPTi5zdHJpbmdpZnkocHJvcHMpfWApO1xuXG4gIGF3YWl0IGNyZWF0ZU9yVXBkYXRlRW52aXJvbm1lbnRTZWNyZXQoZXZlbnQsIG9jdG9raXQsIHNtQ2xpZW50KTtcbiAgcmV0dXJuIHsgUGh5c2ljYWxSZXNvdXJjZUlkOiBzZWNyZXROYW1lIH07XG59O1xuXG5jb25zdCBvbkRlbGV0ZSA9IGFzeW5jIChcbiAgZXZlbnQ6IE9uRXZlbnRSZXF1ZXN0PEFjdGlvbkVudmlyb25tZW50U2VjcmV0RXZlbnRQcm9wcz4sXG4gIG9jdG9raXQ6IE9jdG9raXQsXG4pID0+IHtcbiAgY29uc3Qgc2VjcmV0TmFtZSA9IGV2ZW50LlJlc291cmNlUHJvcGVydGllcy5yZXBvc2l0b3J5U2VjcmV0TmFtZTtcbiAgY29uc29sZS5sb2coJ0RlbGV0ZSBBY3Rpb25FbnZpcm9ubWVudFNlY3JldCAnICsgc2VjcmV0TmFtZSk7XG4gIGF3YWl0IGRlbGV0ZUVudmlyb25tZW50U2VjcmV0KGV2ZW50LCBvY3Rva2l0KTtcbiAgcmV0dXJuIHsgUGh5c2ljYWxSZXNvdXJjZUlkOiBzZWNyZXROYW1lIH07XG59O1xuXG5jb25zdCBjcmVhdGVPclVwZGF0ZUVudmlyb25tZW50U2VjcmV0ID0gYXN5bmMgKFxuICBldmVudDogT25FdmVudFJlcXVlc3Q8QWN0aW9uRW52aXJvbm1lbnRTZWNyZXRFdmVudFByb3BzPixcbiAgb2N0b2tpdDogT2N0b2tpdCxcbiAgc21DbGllbnQ6IFNlY3JldHNNYW5hZ2VyQ2xpZW50LFxuKSA9PiB7XG4gIGNvbnN0IHtcbiAgICByZXBvc2l0b3J5T3duZXI6IG93bmVyLFxuICAgIHJlcG9zaXRvcnlOYW1lOiByZXBvLFxuICAgIHJlcG9zaXRvcnlTZWNyZXROYW1lOiBzZWNyZXRfbmFtZSxcbiAgICBlbnZpcm9ubWVudDogZW52aXJvbm1lbnRfbmFtZSxcbiAgICBzb3VyY2VTZWNyZXRBcm46IHNlY3JldElkLFxuICB9ID0gZXZlbnQuUmVzb3VyY2VQcm9wZXJ0aWVzO1xuXG4gIGNvbnN0IHNlY3JldFRvRW5jcnlwdCA9IGF3YWl0IHNtQ2xpZW50LnNlbmQobmV3IEdldFNlY3JldFZhbHVlQ29tbWFuZCh7IFNlY3JldElkOiBzZWNyZXRJZCB9KSk7XG4gIGNvbnNvbGUubG9nKGBFbmNyeXB0IHZhbHVlIG9mIHNlY3JldCB3aXRoIGlkOiAke3NlY3JldElkfWApO1xuXG4gIGNvbnN0IHNlY3JldFN0cmluZyA9IHNlY3JldFRvRW5jcnlwdC5TZWNyZXRTdHJpbmc7XG4gIGlmICghc2VjcmV0U3RyaW5nKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKCdTZWNyZXRTdHJpbmcgaXMgZW1wdHkgZnJvbSBzZWNyZXQgd2l0aCBpZDogJyArIHNlY3JldElkKTtcbiAgfVxuXG4gIGNvbnN0IHsgZGF0YSB9ID0gYXdhaXQgb2N0b2tpdC5yZXF1ZXN0KCdHRVQgL3JlcG9zL3tvd25lcn0ve3JlcG99L2FjdGlvbnMvc2VjcmV0cy9wdWJsaWMta2V5JywgeyBvd25lciwgcmVwbyB9KTtcblxuICBjb25zdCBlbmNyeXB0ZWRTZWNyZXQgPSBhd2FpdCBlbmNyeXB0VmFsdWUoc2VjcmV0U3RyaW5nLCBkYXRhLmtleSk7XG4gIGNvbnNvbGUubG9nKCdFbmNyeXB0ZWQgc2VjcmV0LCBhdHRlbXB0aW5nIHRvIGNyZWF0ZS91cGRhdGUgZ2l0aHViIHNlY3JldCcpO1xuXG4gIGNvbnN0IHJlcG9zaXRvcnlfaWQgPSBhd2FpdCBnZXRSZXBvc2l0b3J5SWQoZXZlbnQsIG9jdG9raXQpO1xuICBjb25zdCBzZWNyZXRSZXNwb25zZSA9IGF3YWl0IG9jdG9raXQucmVxdWVzdCgnUFVUIC9yZXBvc2l0b3JpZXMve3JlcG9zaXRvcnlfaWR9L2Vudmlyb25tZW50cy97ZW52aXJvbm1lbnRfbmFtZX0vc2VjcmV0cy97c2VjcmV0X25hbWV9Jywge1xuICAgIHJlcG9zaXRvcnlfaWQsXG4gICAgZW52aXJvbm1lbnRfbmFtZSxcbiAgICBzZWNyZXRfbmFtZSxcbiAgICBlbmNyeXB0ZWRfdmFsdWU6IGVuY3J5cHRlZFNlY3JldCxcbiAgICBrZXlfaWQ6IGRhdGEua2V5X2lkLFxuICB9KTtcblxuICBjb25zb2xlLmxvZyhKU09OLnN0cmluZ2lmeShzZWNyZXRSZXNwb25zZSkpO1xuICByZXR1cm4gc2VjcmV0UmVzcG9uc2U7XG59O1xuXG5jb25zdCBkZWxldGVFbnZpcm9ubWVudFNlY3JldCA9IGFzeW5jIChcbiAgZXZlbnQ6IE9uRXZlbnRSZXF1ZXN0PEFjdGlvbkVudmlyb25tZW50U2VjcmV0RXZlbnRQcm9wcz4sXG4gIG9jdG9raXQ6IE9jdG9raXQsXG4pID0+IHtcbiAgY29uc3QgeyBlbnZpcm9ubWVudDogZW52aXJvbm1lbnRfbmFtZSwgcmVwb3NpdG9yeVNlY3JldE5hbWU6IHNlY3JldF9uYW1lIH0gPSBldmVudC5SZXNvdXJjZVByb3BlcnRpZXM7XG4gIGNvbnN0IHJlcG9zaXRvcnlfaWQgPSBhd2FpdCBnZXRSZXBvc2l0b3J5SWQoZXZlbnQsIG9jdG9raXQpO1xuICBjb25zdCBkZWxldGVTZWNyZXRSZXNwb25zZSA9IGF3YWl0IG9jdG9raXQucmVxdWVzdCgnREVMRVRFIC9yZXBvc2l0b3JpZXMve3JlcG9zaXRvcnlfaWR9L2Vudmlyb25tZW50cy97ZW52aXJvbm1lbnRfbmFtZX0vc2VjcmV0cy97c2VjcmV0X25hbWV9Jywge1xuICAgIHJlcG9zaXRvcnlfaWQsXG4gICAgZW52aXJvbm1lbnRfbmFtZSxcbiAgICBzZWNyZXRfbmFtZSxcbiAgfSk7XG4gIGNvbnNvbGUubG9nKGBEZWxldGU6ICR7SlNPTi5zdHJpbmdpZnkoZGVsZXRlU2VjcmV0UmVzcG9uc2UpfWApO1xuICByZXR1cm4gZGVsZXRlU2VjcmV0UmVzcG9uc2U7XG59O1xuXG5jb25zdCBnZXRSZXBvc2l0b3J5SWQgPSBhc3luYyAoXG4gIGV2ZW50OiBPbkV2ZW50UmVxdWVzdDxBY3Rpb25FbnZpcm9ubWVudFNlY3JldEV2ZW50UHJvcHM+LFxuICBvY3Rva2l0OiBPY3Rva2l0LFxuKSA9PiB7XG4gIGNvbnN0IHsgcmVwb3NpdG9yeU93bmVyOiBvd25lciwgcmVwb3NpdG9yeU5hbWU6IHJlcG8gfSA9IGV2ZW50LlJlc291cmNlUHJvcGVydGllcztcbiAgY29uc3QgeyBkYXRhIH0gPSBhd2FpdCBvY3Rva2l0LnJlcXVlc3QoJ0dFVCAvcmVwb3Mve293bmVyfS97cmVwb30nLCB7XG4gICAgb3duZXIsXG4gICAgcmVwbyxcbiAgfSk7XG4gIHJldHVybiBkYXRhLmlkO1xufTtcblxuXG5leHBvcnQgY29uc3QgaGFuZGxlciA9IG9uRXZlbnQ7XG4iXX0=