# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['gns3fy']

package_data = \
{'': ['*']}

install_requires = \
['pydantic>=1.0,<2.0', 'requests>=2.22,<3.0']

setup_kwargs = {
    'name': 'gns3fy',
    'version': '0.7.1',
    'description': 'Python wrapper around GNS3 Server API',
    'long_description': '# gns3fy\n\n[![Circle CI](https://circleci.com/gh/davidban77/gns3fy/tree/develop.svg?style=shield&circle-token=:circle-token)](https://circleci.com/gh/davidban77/gns3fy/tree/develop)\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/ambv/black)\n[![codecov](https://codecov.io/gh/davidban77/gns3fy/branch/develop/graph/badge.svg)](https://codecov.io/gh/davidban77/gns3fy)\n[![Total alerts](https://img.shields.io/lgtm/alerts/g/davidban77/gns3fy.svg?logo=lgtm&logoWidth=18)](https://lgtm.com/projects/g/davidban77/gns3fy/alerts/)\n[![Language grade: Python](https://img.shields.io/lgtm/grade/python/g/davidban77/gns3fy.svg?logo=lgtm&logoWidth=18)](https://lgtm.com/projects/g/davidban77/gns3fy/context:python)\n[![pypi](https://img.shields.io/pypi/v/gns3fy.svg)](https://pypi.python.org/pypi/gns3fy)\n[![versions](https://img.shields.io/pypi/pyversions/gns3fy.svg)](https://github.com/davidban77/gns3fy)\n\nPython wrapper around [GNS3 Server API](http://api.gns3.net/en/2.2/index.html). Minimal GNS3 version is 2.2.\n\nIts main objective is to interact with the GNS3 server in a programatic way, so it can be integrated with the likes of Ansible, docker and scripts. Ideal for network CI/CD pipeline tooling.\n\n## Documentation\n\nCheck out the [Documentation](https://davidban77.github.io/gns3fy/) to explore use cases and the API Reference\n\n## Use cases\n\nHere are some examples where gns3fy is used in a programmatic way:\n\n- [Ansible-collection-gns3](https://galaxy.ansible.com/davidban77/gns3): Useful for CI/CD pipelines to interact with GNS3 server using Ansible. It can create/delete projects, nodes and links in an ansible playbook.\n- Terraform: Coming soon... (although it might be a Go version of it)\n- [Migrate templates between GNS3 servers](https://davidban77.github.io/gns3fy/user-guide/#migrate-templates-between-gns3-servers)\n- [Check server usage](https://davidban77.github.io/gns3fy/user-guide/#check-server-cpu-and-memory-usage) before turning up resource-hungry nodes\n- [Manipulate project snapshots](https://davidban77.github.io/gns3fy/user-guide/#create-and-list-project-snapshots) like create, delete or list the snapshots configured for the project.\n\n## Install\n\n```shell\npip install gns3fy\n```\n\n### Development version\n\nUse [poetry](https://github.com/sdispater/poetry) to install the package when cloning it.\n\n## How it works\n\nYou can start the library and use the `Gns3Connector` object and the `Project` object.\n\nFor example:\n\n```python\n>>> import gns3fy\n>>> from tabulate import tabulate\n\n# Define the server object to establish the connection\n>>> gns3_server = gns3fy.Gns3Connector("http://<server address>:3080")\n\n# Show the available projects on the server\n>>> print(\n        tabulate(\n            gns3_server.projects_summary(is_print=False),\n            headers=["Project Name", "Project ID", "Total Nodes", "Total Links", "Status"],\n        )\n    )\n"""\nProject Name    Project ID                              Total Nodes    Total Links  Status\n--------------  ------------------------------------  -------------  -------------  --------\ntest2           c9dc56bf-37b9-453b-8f95-2845ce8908e3             10              9  opened\nAPI_TEST        4b21dfb3-675a-4efa-8613-2f7fb32e76fe              6              4  opened\nmpls-bgpv2      f5de5917-0ac5-4850-82b1-1d7e3c777fa1             30             40  closed\n"""\n\n# Define the lab you want to load and assign the server connector\n>>> lab = gns3fy.Project(name="API_TEST", connector=gns3_server)\n\n# Retrieve its information and display\n>>> lab.get()\n>>> print(lab)\n"Project(project_id=\'4b21dfb3-675a-4efa-8613-2f7fb32e76fe\', name=\'API_TEST\', status=\'opened\', ...)"\n\n# Access the project attributes\n>>> print(f"Name: {lab.name} -- Status: {lab.status} -- Is auto_closed?: {lab.auto_close}")\n"Name: API_TEST -- Status: closed -- Is auto_closed?: False"\n\n# Open the project\n>>> lab.open()\n>>> lab.status\nopened\n\n# Verify the stats\n>>> lab.stats\n{\'drawings\': 0, \'links\': 4, \'nodes\': 6, \'snapshots\': 0}\n\n# List the names and status of all the nodes in the project\n>>> for node in lab.nodes:\n...    print(f"Node: {node.name} -- Node Type: {node.node_type} -- Status: {node.status}")\n\n"Node: Ethernetswitch-1 -- Node Type: ethernet_switch -- Status: started"\n...\n```\n\nTake a look at the API documentation for complete information about the attributes retrieved.\n\n### Usage of Node and Link objects\n\nYou have access to the `Node` and `Link` objects as well, this gives you the ability to start, stop, suspend the individual element in a GNS3 project.\n\n```python\n>>> from gns3fy import Node, Link, Gns3Connector\n\n>>> PROJECT_ID = "<some project id>"\n>>> server = Gns3Connector("http://<server address>:3080")\n\n>>> alpine1 = Node(project_id=PROJECT_ID, name="alpine-1", connector=server)\n\n>>> alpine1.get()\n>>> print(alpine1)\n"Node(name=\'alpine-1\', node_type=\'docker\', node_directory= ...)"\n\n# And you can access the attributes the same way as the project\n>>> print(f"Name: {alpine1.name} -- Status: {alpine1.status} -- Console: {alpine1.console}")\n"Name: alpine-1 -- Status: started -- Console: 5005"\n\n# Stop the node and start (you can just restart it as well)\n>>> alpine1.stop()\n>>> alpine1.status\nstopped\n\n>>> alpine1.start()\n>>> alpine1.status\nstarted\n\n# You can also see the Link objects assigned to this node\n>>> alpine1.links\n[Link(link_id=\'4d9f1235-7fd1-466b-ad26-0b4b08beb778\', link_type=\'ethernet\', ...)]\n\n# And in the same way you can interact with a Link object\n>>> link1 = alpine1.links[0]\n>>> print(f"Link Type: {link1.link_type} -- Capturing?: {link1.capturing} -- Endpoints: {link1.nodes}")\n"Link Type: ethernet -- Capturing?: False -- Endpoints: [{\'adapter_number\': 2, ...}]"\n```\n\n### Useful functions\n\nYou also have some commodity methods like the `nodes_summary` and `links_summary`, that if used with a library like `tabulate` you can see the following:\n\n```python\n\n>>> from tabulate import tabulate\n\n>>> nodes_summary = lab.nodes_summary(is_print=False)\n\n>>> print(\n...     tabulate(nodes_summary, headers=["Node", "Status", "Console Port", "ID"])\n... )\n"""\nNode              Status      Console Port  ID\n----------------  --------  --------------  ------------------------------------\nEthernetswitch-1  started             5000  da28e1c0-9465-4f7c-b42c-49b2f4e1c64d\nIOU1              started             5001  de23a89a-aa1f-446a-a950-31d4bf98653c\nIOU2              started             5002  0d10d697-ef8d-40af-a4f3-fafe71f5458b\nvEOS-4.21.5F-1    started             5003  8283b923-df0e-4bc1-8199-be6fea40f500\nalpine-1          started             5005  ef503c45-e998-499d-88fc-2765614b313e\nCloud-1           started                   cde85a31-c97f-4551-9596-a3ed12c08498\n"""\n>>> links_summary = lab.links_summary(is_print=False)\n>>> print(\n...     tabulate(links_summary, headers=["Node A", "Port A", "Node B", "Port B"])\n... )\n"""\nNode A          Port A       Node B            Port B\n--------------  -----------  ----------------  -----------\nIOU1            Ethernet1/0  IOU2              Ethernet1/0\nvEOS-4.21.5F-1  Management1  Ethernetswitch-1  Ethernet0\nvEOS-4.21.5F-1  Ethernet1    alpine-1          eth0\nCloud-1         eth1         Ethernetswitch-1  Ethernet7\n"""\n```\n',
    'author': 'David Flores',
    'author_email': 'davidflores7_8@hotmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/davidban77/gns3fy',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
