# -*- coding: utf-8 -*-
#
# Copyright (C) 2014-2019 Bitergia
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
#
# Authors:
#     Santiago Dueñas <sduenas@bitergia.com>
#

import argparse
import collections
import logging
import re

from .. import api
from ..command import Command, CMD_SUCCESS, HELP_LIST
from ..exceptions import NotFoundError, InvalidValueError


AUTOPROFILE_COMMAND_USAGE_MSG = """%(prog)s autoprofile <source> ... <source>"""
EMAIL_ADDRESS_REGEX = r"^(?P<email>[^\s@]+@[^\s@.]+\.[^\s@]+)$"

logger = logging.getLogger(__name__)


class AutoProfile(Command):
    """Auto complete profile information.

    This command autocompletes the profiles information related to
    a set of unique identities. To update the profile, this command
    uses a list of sources ordered by priority. Only those unique
    identities which have one or more identities from any of these
    sources will be updated.
    """
    def __init__(self, **kwargs):
        super(AutoProfile, self).__init__(**kwargs)

        self.parser = argparse.ArgumentParser(description=self.description,
                                              usage=self.usage)

        # Positional arguments
        self.parser.add_argument('source', nargs='+',
                                 help="list of sources used to autocomplete ordered by priority")

        # Exit early if help is requested
        if 'cmd_args' in kwargs and [i for i in kwargs['cmd_args'] if i in HELP_LIST]:
            return

        self._set_database(**kwargs)

    @property
    def description(self):
        return """Autocomplete profiles information."""

    @property
    def usage(self):
        return AUTOPROFILE_COMMAND_USAGE_MSG

    def run(self, *args):
        """Autocomplete profile information."""

        params = self.parser.parse_args(args)
        sources = params.source
        code = self.autocomplete(sources)

        return code

    def autocomplete(self, sources):
        """Autocomplete unique identities profiles.

        Autocomplete unique identities profiles using the information
        of their identities. The selection of the data used to fill
        the profile is prioritized using a list of sources.
        """
        email_pattern = re.compile(EMAIL_ADDRESS_REGEX)

        identities = self.__select_autocomplete_identities(sources)

        for uuid, ids in identities.items():
            # Among the identities (with the same priority) selected
            # to complete the profile, it will choose the longest 'name'.
            # If no name is available, it will use the field 'username'.
            name = None
            email = None

            for identity in ids:
                oldname = name

                if not name:
                    name = identity.name or identity.username
                elif identity.name and len(identity.name) > len(name):
                    name = identity.name

                # Do not set email addresses on the name field
                if name and email_pattern.match(name):
                    name = oldname

                if not email and identity.email:
                    email = identity.email

            kw = {
                'name': name,
                'email': email
            }

            try:
                api.edit_profile(self.db, uuid, **kw)
                self.display('autoprofile.tmpl', identity=identity)
            except (NotFoundError, InvalidValueError) as e:
                self.error(str(e))
                return e.code

        return CMD_SUCCESS

    def __select_autocomplete_identities(self, sources):
        """Select the identities used for autocompleting"""

        MIN_PRIORITY = 99999999

        checked = {}

        for source in sources:
            uids = api.unique_identities(self.db, source=source)

            for uid in uids:
                if uid.uuid in checked:
                    continue

                max_priority = MIN_PRIORITY
                selected = []

                for identity in sorted(uid.identities, key=lambda x: x.id):
                    try:
                        priority = sources.index(identity.source)

                        if priority < max_priority:
                            selected = [identity]
                            max_priority = priority
                        elif priority == max_priority:
                            selected.append(identity)
                    except ValueError:
                        continue

                checked[uid.uuid] = selected

        identities = collections.OrderedDict(sorted(checked.items(),
                                             key=lambda t: t[0]))

        return identities
