"use strict";
/**
 * @upsetjs/jupyter_widget
 * https://github.com/upsetjs/upsetjs_jupyter_widget
 *
 * Copyright (c) 2020 Samuel Gratzl <sam@sgratzl.com>
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.UpSetView = exports.UpSetModel = void 0;
const base_1 = require("@jupyter-widgets/base");
const version_1 = require("./version");
const bundle_1 = require("@upsetjs/bundle");
const venn_js_1 = require("@upsetjs/venn.js");
const utils_1 = require("./utils");
const adapter = bundle_1.createVennJSAdapter(venn_js_1.layout);
class UpSetModel extends base_1.DOMWidgetModel {
    defaults() {
        return Object.assign(Object.assign({}, super.defaults()), { _model_name: UpSetModel.model_name, _model_module: UpSetModel.model_module, _model_module_version: UpSetModel.model_module_version, _view_name: UpSetModel.view_name, _view_module: UpSetModel.view_module, _view_module_version: UpSetModel.view_module_version });
    }
}
exports.UpSetModel = UpSetModel;
UpSetModel.serializers = Object.assign({}, base_1.DOMWidgetModel.serializers);
UpSetModel.model_name = 'UpSetModel';
UpSetModel.model_module = version_1.MODULE_NAME;
UpSetModel.model_module_version = version_1.MODULE_VERSION;
UpSetModel.view_name = 'UpSetView'; // Set to null if no view
UpSetModel.view_module = version_1.MODULE_NAME; // Set to null if no view
UpSetModel.view_module_version = version_1.MODULE_VERSION;
class UpSetView extends base_1.DOMWidgetView {
    constructor() {
        super(...arguments);
        this.props = {
            sets: [],
            width: 300,
            height: 300,
        };
        this.elems = [];
        this.elemToIndex = new Map();
        this.attrs = [];
        this.changeSelection = (s) => {
            this.model.off('change', this.changed_prop, null);
            if (!s) {
                this.model.set('value', null);
            }
            else {
                const setish = {
                    name: s.name,
                    type: s.type,
                    cardinality: s.cardinality,
                    elems: s.elems.map((e) => this.elemToIndex.get(e)),
                };
                if (s.type !== 'set') {
                    setish.degree = s.degree;
                    setish.set_names = Array.from(s.sets).map((s) => s.name);
                }
                this.model.set('value', setish);
            }
            this.props.selection = s;
            this.renderImpl();
            this.touch();
            this.model.on('change', this.changed_prop, this);
        };
    }
    render() {
        this.model.on('change', this.changed_prop, this);
        this.updateProps(this.stateToProps());
    }
    stateToProps() {
        const props = {};
        const state = this.model.get_state(true);
        const toCamelCase = (v) => v.replace(/([-_]\w)/g, (g) => g[1].toUpperCase());
        const toPropName = (key) => {
            if (key === 'value') {
                return 'selection';
            }
            return toCamelCase(key);
        };
        Object.keys(state).forEach((key) => {
            let v = state[key];
            if (v == null || (Array.isArray(v) && v.length === 0) || key.startsWith('_') || key === 'layout') {
                return;
            }
            const propName = toPropName(key);
            if (propName === 'fontSizes') {
                const converted = {};
                Object.keys(v).forEach((key) => {
                    const vi = v[key];
                    if (vi != null) {
                        converted[toCamelCase(key)] = typeof vi === 'number' ? `${vi}px` : String(vi);
                    }
                });
                v = converted;
            }
            props[propName] = v;
        });
        return props;
    }
    updateProps(delta) {
        if (delta) {
            Object.assign(this.props, delta);
        }
        this.fixProps(delta);
        this.renderImpl();
    }
    syncAddons() {
        if (this.attrs.length === 0) {
            delete this.props.setAddons;
            delete this.props.combinationAddons;
            return;
        }
        const toAddon = (attr, vertical = false) => {
            if (attr.type === 'number') {
                return bundle_1.boxplotAddon((v) => v.attrs[attr.name], { min: attr.domain[0], max: attr.domain[1] }, {
                    name: attr.name,
                    quantiles: 'hinges',
                    orient: vertical ? 'vertical' : 'horizontal',
                });
            }
            return bundle_1.categoricalAddon((v) => v.attrs[attr.name], {
                categories: attr.categories,
            }, {
                name: attr.name,
                orient: vertical ? 'vertical' : 'horizontal',
            });
        };
        this.props.setAddons = this.attrs.map((attr) => toAddon(attr, false));
        this.props.combinationAddons = this.attrs.map((attr) => toAddon(attr, true));
    }
    fixProps(delta) {
        var _a, _b, _c;
        const props = this.props;
        if (delta.elems)
            if (delta.elems != null) {
                this.attrs = (_a = delta.attrs) !== null && _a !== void 0 ? _a : this.attrs;
                const lookups = this.attrs.map((attr) => (attr.elems ? new Map(attr.elems.map((e, i) => [e, i])) : null));
                this.elems = delta.elems.map((name, i) => {
                    const attrs = {};
                    this.attrs.forEach((attr, j) => { var _a; return (attrs[attr.name] = attr.values[lookups[j] ? (_a = lookups[j].get(name)) !== null && _a !== void 0 ? _a : i : i]); });
                    return { name, attrs };
                });
                this.elemToIndex.clear();
                this.elems.forEach((e, i) => this.elemToIndex.set(e, i));
                this.syncAddons();
            }
            else if (delta.attrs != null) {
                // only attrs same elems
                this.attrs = delta.attrs;
                const lookups = this.attrs.map((attr) => (attr.elems ? new Map(attr.elems.map((e, i) => [e, i])) : null));
                this.elems.forEach((elem, i) => {
                    const attrs = {};
                    this.attrs.forEach((attr, j) => { var _a; return (attrs[attr.name] = attr.values[lookups[j] ? (_a = lookups[j].get(elem.name)) !== null && _a !== void 0 ? _a : i : i]); });
                    elem.attrs = attrs;
                });
                this.syncAddons();
            }
        delete this.props.elems;
        delete this.props.attrs;
        if (delta.sets != null) {
            props.sets = utils_1.fixSets(delta.sets, this.elems);
        }
        if (delta.combinations != null) {
            const c = utils_1.fixCombinations(delta.combinations, props.sets, this.elems);
            if (c == null) {
                delete props.combinations;
            }
            else {
                props.combinations = c;
            }
        }
        if (delta.selection) {
            props.selection = utils_1.resolveSet(delta.selection, (_b = props.sets) !== null && _b !== void 0 ? _b : [], ((_c = props.combinations) !== null && _c !== void 0 ? _c : []), this.elems);
        }
        if (delta.queries) {
            props.queries = delta.queries.map((query) => {
                var _a, _b;
                if (bundle_1.isSetQuery(query)) {
                    return Object.assign({}, query, {
                        set: utils_1.resolveSet(query.set, (_a = props.sets) !== null && _a !== void 0 ? _a : [], ((_b = props.combinations) !== null && _b !== void 0 ? _b : []), this.elems),
                    });
                }
                else if (bundle_1.isElemQuery(query)) {
                    return Object.assign({}, query, {
                        elems: bundle_1.fromIndicesArray(query.elems, this.elems),
                    });
                }
                return query;
            });
        }
        delete props.onHover;
        delete props.onClick;
        delete props.onContextMenu;
        if (this.model.get('mode') === 'click') {
            props.onClick = this.changeSelection;
        }
        else if (this.model.get('mode') === 'hover') {
            props.onHover = this.changeSelection;
        }
        else if (this.model.get('mode') === 'contextMenu') {
            props.onContextMenu = this.changeSelection;
        }
    }
    changed_prop(model) {
        this.updateProps(model.changed);
    }
    renderImpl() {
        const bb = this.el.getBoundingClientRect();
        const p = Object.assign({}, this.props);
        const renderMode = this.model.get('_render_mode');
        if (!bb.width || !bb.height) {
            requestAnimationFrame(() => {
                const bb2 = this.el.getBoundingClientRect();
                p.width = bb2.width || 600;
                p.height = bb2.height || 400;
                if (renderMode === 'venn') {
                    delete p.layout;
                    bundle_1.renderVennDiagram(this.el, p);
                }
                else if (renderMode === 'euler') {
                    p.layout = adapter;
                    bundle_1.renderVennDiagram(this.el, p);
                }
                else {
                    bundle_1.render(this.el, p);
                }
            });
            return;
        }
        p.width = bb.width;
        p.height = bb.height;
        if (renderMode === 'venn') {
            delete p.layout;
            bundle_1.renderVennDiagram(this.el, p);
        }
        else if (renderMode === 'euler') {
            p.layout = adapter;
            bundle_1.renderVennDiagram(this.el, p);
        }
        else {
            bundle_1.render(this.el, p);
        }
    }
}
exports.UpSetView = UpSetView;
//# sourceMappingURL=widget.js.map