# -*- coding: utf-8 -*-

"""Tests for knowledge graph factories."""

import unittest
from typing import Iterable

from geometric_graphs.generators import (
    BarbellGenerator,
    ChainGenerator,
    LollipopGenerator,
    StarGenerator,
    TadpoleGenerator,
    WheelGenerator,
    _hex_grid_helper,
)
from geometric_graphs.util import Generator


class TestGenerators(unittest.TestCase):
    """A test case for knowledge graph generators."""

    def assert_generator_sizes(self, generator: Generator) -> None:
        """Test the size estimation of the generator."""
        triples = set(generator.iterate_triples())
        nodes = generator.number_of_nodes()
        if nodes is not None:
            heads = {h for h, _, _ in triples}
            tails = {t for _, _, t in triples}
            self.assertEqual(nodes, len(heads | tails), msg="wrong number of nodes")
        edges = generator.number_of_edges()
        if edges is not None:
            self.assertEqual(edges, len(triples), msg="wrong number of edges")

    def assert_generator(self, expected: Iterable[tuple[int, int, int]], generator: Generator):
        """Assert the triples are generated by the generator (order insensitive)."""
        self.assertIsInstance(generator, Generator)
        self.assert_generator_sizes(generator)
        self.assertEqual(set(expected), set(generator.iterate_triples()))

    def test_star(self):
        """Test creating star graphs."""
        triples = [
            (0, 0, 1),
            (0, 0, 2),
            (0, 0, 3),
        ]
        generator = StarGenerator(3, sink=False)
        self.assert_generator(triples, generator)

        triples = [
            (1, 0, 0),
            (2, 0, 0),
            (3, 0, 0),
        ]
        generator = StarGenerator(3, sink=True)
        self.assert_generator(triples, generator)

    def test_wheel(self):
        """Test creating wheel graphs."""
        triples = [
            (0, 0, 1),
            (0, 0, 2),
            (0, 0, 3),
            (1, 1, 2),
            (2, 1, 3),
            (3, 1, 1),
        ]
        generator = WheelGenerator(3, sink=False)
        self.assert_generator(triples, generator)

        triples = [
            (1, 0, 0),
            (2, 0, 0),
            (3, 0, 0),
            (1, 1, 2),
            (2, 1, 3),
            (3, 1, 1),
        ]
        generator = WheelGenerator(3, sink=True)
        self.assert_generator(triples, generator)

    def test_hex_rows(self):
        """Test generating a two dimensional hexagonal grid graph."""
        for rows, columns, expected in [
            # one row
            (
                1,
                1,
                [[0], [1, 2], [3, 4], [5]],
            ),
            (1, 2, [[0, 1], [2, 3, 4], [5, 6, 7], [8, 9]]),
            (
                1,
                3,
                [[0, 1, 2], [3, 4, 5, 6], [7, 8, 9, 10], [11, 12, 13]],
            ),
            # two rows
            (
                2,
                1,
                [
                    [0],
                    [1, 2],
                    [3, 4],
                    [5, 6, 7],
                    [8, 9, 10],
                    [11, 12],
                ],
            ),
            (
                2,
                2,
                [
                    [0, 1],
                    [2, 3, 4],
                    [5, 6, 7],
                    [8, 9, 10, 11],
                    [12, 13, 14, 15],
                    [16, 17, 18],
                ],
            ),
            (
                2,
                3,
                [
                    [0, 1, 2],
                    [3, 4, 5, 6],
                    [7, 8, 9, 10],
                    [11, 12, 13, 14, 15],
                    [16, 17, 18, 19, 20],
                    [21, 22, 23, 24],
                ],
            ),
            # three rows
            (
                3,
                1,
                [
                    [0],
                    [1, 2],
                    [3, 4],
                    [5, 6, 7],
                    [8, 9, 10],
                    [11, 12],
                    [13, 14],
                    [15],
                ],
            ),
            (
                3,
                2,
                [
                    [0, 1],
                    [2, 3, 4],
                    [5, 6, 7],
                    [8, 9, 10, 11],
                    [12, 13, 14, 15],
                    [16, 17, 18],
                    [19, 20, 21],
                    [22, 23],
                ],
            ),
            (
                3,
                3,
                [
                    [0, 1, 2],
                    [3, 4, 5, 6],
                    [7, 8, 9, 10],
                    [11, 12, 13, 14, 15],
                    [16, 17, 18, 19, 20],
                    [21, 22, 23, 24],
                    [25, 26, 27, 28],
                    [29, 30, 31],
                ],
            ),
            # four rows
            (
                4,
                1,
                [
                    [0],
                    [1, 2],
                    [3, 4],
                    [5, 6, 7],
                    [8, 9, 10],
                    [11, 12],
                    [13, 14],
                    [15, 16, 17],
                    [18, 19, 20],
                    [21, 22],
                ],
            ),
        ]:
            with self.subTest(rows=rows, columns=columns):
                grid = _hex_grid_helper(rows=rows, columns=columns)
                self.assertEqual(columns, len(grid[0]), msg="first row have length of columns")
                self.assertEqual(
                    columns + (rows + 1) % 2,
                    len(grid[-1]),
                    msg="# rows even -> columns + 1 in last row; odd -> columns in last row",
                )
                self.assertEqual(
                    2 * (1 + rows),
                    len(expected),
                    msg="test data is wrong: grid should be 2 * (1 + rows) long",
                )
                self.assertEqual(len(expected), len(grid), msg="grid should be 2 * (1 + rows) long")
                self.assertEqual(expected, grid)

    def test_chain(self):
        """Test the chain generator."""
        generator = ChainGenerator(length=3, width=1, heterogeneous=False)
        triples = [
            (0, 0, 1),
            (0, 0, 2),
            (1, 0, 3),
            (2, 0, 3),
            (3, 0, 4),
            (3, 0, 5),
            (4, 0, 6),
            (5, 0, 6),
        ]
        self.assert_generator(triples, generator)

        generator = ChainGenerator(length=3, width=2, heterogeneous=False)
        triples = [
            (0, 0, 1),
            (0, 0, 2),
            (1, 0, 3),
            (2, 0, 4),
            (3, 0, 5),
            (4, 0, 5),
            (5, 0, 6),
            (5, 0, 7),
            (6, 0, 8),
            (7, 0, 9),
            (8, 0, 10),
            (9, 0, 10),
        ]
        self.assert_generator(triples, generator)

    def test_barbell(self):
        """Test the barbell generator."""
        triples = [(0, 0, 1), (0, 0, 2), (1, 0, 2), (3, 0, 4), (3, 0, 5), (4, 0, 5), (0, 1, 3)]
        self.assert_generator(triples, BarbellGenerator(3))

    def test_tadpole(self):
        """Test the tadpole generator."""
        triples = [
            (0, 0, 1),
            (1, 0, 2),
            (2, 0, 3),
            (3, 0, 0),
            (3, 1, 4),
            (4, 1, 5),
        ]
        self.assert_generator(triples, TadpoleGenerator(4, 2, sink=False))

        triples = [
            (0, 0, 1),
            (1, 0, 2),
            (2, 0, 3),
            (3, 0, 0),
            (4, 1, 3),
            (5, 1, 4),
        ]
        self.assert_generator(triples, TadpoleGenerator(4, 2, sink=True))

    def test_lollipop(self):
        """Test the tadpole generator."""
        triples = [
            (0, 0, 1),
            (0, 0, 2),
            (0, 0, 3),
            (1, 0, 2),
            (1, 0, 3),
            (2, 0, 3),
            (3, 1, 4),
            (4, 1, 5),
        ]
        self.assert_generator(triples, LollipopGenerator(4, 2, sink=False))
