# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pygradethis']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'pygradethis',
    'version': '0.2.1',
    'description': '"Python autograder to facilitate code output and static code checking."',
    'long_description': '# pygradethis\n\n[![PyPI version](https://badge.fury.io/py/pygradethis.svg)](https://badge.fury.io/py/pygradethis)\n[![PyPI - License](https://img.shields.io/pypi/l/pygradethis)](LICENSE)\n[![Downloads](https://pepy.tech/badge/pygradethis)](https://pepy.tech/project/pygradethis)\n\nA Python package to facilitate checking code output or static code checking\nusing AST analysis. It can either be used with R using the [`learnr`](https://rstudio.github.io/learnr/) package, as \na mirror of [`gradethis`](https://rstudio-education.github.io/gradethis/index.html) package, or as a standalone package for general Python \nuse in educational settings.\n\n**Note**: This package is in early development and will undergo rapid changes.\n\n## Install\n\n```\npip install pygradethis\n```\n\n## Dependencies\n\n- Python 3.9+\n\n## Dev Dependencies\n\n- pytest\n\n## Features\n\n- Simple output checking based on pass / fail conditions with feedback\n- Simple static code checking (AST), with feedback on how student\'s code differs from solution\n\n## Output checks\n\n`pygradethis` mimics the cadence to `gradethis::grade_result`. For e.g., we can\ncheck that the student supplies the `mpg` dataset like so:\n\n```python\npython_grade_result(\n  python_pass_if(mpg, "You also got the mpg dataframe!"),\n  python_fail_if(None, "")\n)\n```\n\nInternally, these `python_pass_if(output, message)` or `python_fail_if(output, message)` will be checked sequentially in\nthe order of arguments and return on first condition we match. The `None` here can be used\nif you simply want to execute a condition if none of the other conditions matched.\n\nIf we match a `python_pass_if` or `python_fail_if`, we will present a feedback message wrapped in a convenient `dict`:\n\n```python\ndict(\n    message = str,\n    correct = True|False,\n    type = "auto|success|info|warning|error|custom",\n    location = "append|prepend|replace"\n)\n```\n\nThe `message` is the feedback, the `correct` is whether or not the student\'s solution is correct, `type` is the type of feedback. When \nused with `learnr` the `location` field here is useful for where the message is situated in the tutorial. However, for those using \nthis package as a standalone the `location` is not an important field and it can be ignored. More on the flags [here](https://rstudio.github.io/learnr/exercises.html#Exercise_Checking).\n\nInternally, a random praise/encouragement message will be appended before any custom message supplied. \n\n```python\npython_pass_if(x = mpg, message = "You also got the mpg dataframe!")\n```\nFeedback:\n> Bravo! You also got the mpg dataframe!\n\n```python\npython_fail_if(x = None, message = "")\n```\nFeedback:\n> Try it again. You get better each time.\n\n## Code checks\n\nFor static code checking, we follow a similar cadence for `gradethis::grade_code`. \n\nWhen there is a solution code being supplied, `grade_code(user_code, solution_code)` can be used to check the AST of\nthe user and solution code, making sure to standardize function calls and producing a helpful message for the student\nto diagnose their issue.\n\nExample:\n\n```python\ngrade_code(\n  student_code="2 + sqrt(log(2))", \n  solution_code="2 + sqrt(log(1))"\n)\n\n```\nFeedback:\n> I expected `log(1)`, but what you wrote was interpreted as `log(2)` in `sqrt(log(2))` at line 1.\n\nNote how the feedback narrows in on the expression in which the problem occurs (`sqrt(log(2))`)\nso that the student can focus on the most relevant outer expression of the problem. In this case, the \n`log(2)` is the problem and the `2` on the left operand of \nthe addition is not as relevant.\n\nSimilarly, here the feedback points out that the 2 within the `log` function is incorrect, similar to the \n`gradethis` [example](https://rstudio-education.github.io/gradethis/reference/grade_code.html).\n\n### Call Standardization\n`pygradethis` also knows how to take user\'s function call code and map positional arguments \nto proper parameter names and set defaults if not supplied. This is so that you don\'t penalize\na student\'s code just because they did not explicitly spell out positional argument names, or\nwrite the default arguments out.\n\nFor e.g. suppose a student is calling the following silly function `foo`:\n\n```python\ndef foo(a, b=1): \n  pass\n```\n\nGrading the code with\n\n```python\ngrade_code(\n  student_code="foo(1)", \n  solution_code="foo(1)"\n)\n```\n\nIn the example above, the `grade_code` doesn\'t give us a feedback message since they are equivalent expressions.\n\nHowever, if the student supplies `foo(2)`\n\n```python\ngrade_code(\n  student_code="foo(2)", \n  solution_code="foo(1)"\n)\n```\n\nwe get back this feedback:\n> I expected `1`, but what you wrote was interpreted as `2` in `foo(2)` at line 1.\n\n**Note:** Although underneath the hood we do standardize the arguments of both the student and the solution code\nbefore checking, we don\'t surface this standardized form to the feedback message. This is certainly possible to\nachieve but in certain cases can hinder learning by revealing too much information. For example, the builtin functions\nlike `sum` is normally called without specifying its actual formal parameters (e.g. `sum(1)` versus `sum(iterable=[1], start=0)`. In the future, a `verbose` mode could be made available such that the formal parameters are pointed out.\n\nFor call standardizing to work, the function definitions corresponding to function \ncalls must be defined  and \'live\' in the environment, whether that is the `globals()`/`locals()`,\n`builtins`, or custom module imports `pandas`. This works if the student/solution source code also \nincludes the definition (like `foo` above) in their own source code or it\'s included by instructor. \n\nCurrently, common modules like `math` is imported for grading within `check_functions.py`, but more modules \nwill be included to serve data science grading as well, such as `pandas` or `numpy` in the future. \nWe plan to make the code more extensible for the instructor to add them as dependencies.\n',
    'author': 'nischalshrestha',
    'author_email': 'bitsorific@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/nischalshrestha/pygradethis',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.9.9,<4.0.0',
}


setup(**setup_kwargs)
