"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SmtpCredentials = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const ssm = require("@aws-cdk/aws-ssm");
const cdk = require("@aws-cdk/core");
const smtp_credentials_provider_1 = require("./smtp-credentials-provider");
/**
 * This construct creates an IAM user, with a policy permitting emails to be sent via SES from a specified email address, creates an access key associated with this user, and converts the access key to SMTP credentials.
 *
 * The generated SMTP credentials are stored as a parameter in Parameter Store, and the name of
 * this parameter is output as a CloudFormation output. The parameter may be safely deleted, once
 * the credentials have been accessed.
 *
 * @stability stable
 * @example
 *
 * new SmtpCredentials(this, 'SmtpCredentials', {
 *     emailAddress: 'me@charles.fish',
 * });
 */
class SmtpCredentials extends cdk.Construct {
    /**
     * @param scope A reference to the stack which this construct will be created in.
     * @param id A unique identifier, within the context that this construct is created.
     * @param props Configuration defining how this construct should be created.
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        const { emailAddress } = props;
        const domainName = this.extractDomainName(emailAddress);
        const { userArn, userName } = this.createIamUser(domainName, emailAddress);
        const { accessKey, smtpPassword } = this.createSmtpCredentials(userArn, userName);
        new cdk.CfnOutput(this, 'SmtpCredentialsParameterName', {
            value: new ssm.StringParameter(this, 'SmtpCredentials', {
                stringValue: JSON.stringify({
                    AccessKey: accessKey,
                    SmtpPassword: smtpPassword,
                }),
            }).parameterName,
        });
    }
    extractDomainName(emailAddress) {
        if (emailAddress.indexOf('@') === -1) {
            throw Error('Invalid email address supplied.');
        }
        return emailAddress.substring(emailAddress.lastIndexOf('@') + 1);
    }
    createIamUser(domainName, emailAddress) {
        const user = new iam.User(this, 'SmtpUser');
        user.attachInlinePolicy(new iam.Policy(this, 'SmtpPolicy', {
            statements: [
                new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    actions: ['ses:SendRawEmail'],
                    resources: [
                        `arn:aws:ses:${process.env.CDK_DEFAULT_REGION}:${process.env.CDK_DEFAULT_ACCOUNT}:identity/${domainName}`,
                    ],
                    conditions: {
                        StringEquals: {
                            'ses:FromAddress': emailAddress,
                        },
                    },
                }),
            ],
        }));
        return user;
    }
    createSmtpCredentials(userArn, userName) {
        const { serviceToken } = new smtp_credentials_provider_1.SmtpCredentialsProvider(this, 'SmtpCredentialsProvider', { userArn });
        const credentials = new cdk.CustomResource(this, 'SmtpCredentialsLambda', {
            serviceToken,
            properties: {
                UserName: userName,
            },
        });
        return {
            accessKey: credentials.getAttString('AccessKey'),
            smtpPassword: credentials.getAttString('SmtpPassword'),
        };
    }
}
exports.SmtpCredentials = SmtpCredentials;
_a = JSII_RTTI_SYMBOL_1;
SmtpCredentials[_a] = { fqn: "@charlesdotfish/smtp-credentials-cdk-construct.SmtpCredentials", version: "1.0.3" };
//# sourceMappingURL=data:application/json;base64,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