"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsAuth = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const k8s_manifest_1 = require("./k8s-manifest");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("@aws-cdk/core");
/**
 * Manages mapping between IAM users and roles to Kubernetes RBAC configuration.
 *
 * @see https://docs.aws.amazon.com/en_us/eks/latest/userguide/add-user-role.html
 * @stability stable
 */
class AwsAuth extends core_2.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.roleMappings = new Array();
        this.userMappings = new Array();
        this.accounts = new Array();
        this.stack = core_1.Stack.of(this);
        new k8s_manifest_1.KubernetesManifest(this, 'manifest', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'v1',
                    kind: 'ConfigMap',
                    metadata: {
                        name: 'aws-auth',
                        namespace: 'kube-system',
                    },
                    data: {
                        mapRoles: this.synthesizeMapRoles(),
                        mapUsers: this.synthesizeMapUsers(),
                        mapAccounts: this.synthesizeMapAccounts(),
                    },
                },
            ],
        });
    }
    /**
     * Adds the specified IAM role to the `system:masters` RBAC group, which means that anyone that can assume it will be able to administer this Kubernetes system.
     *
     * @param role The IAM role to add.
     * @param username Optional user (defaults to the role ARN).
     * @stability stable
     */
    addMastersRole(role, username) {
        this.addRoleMapping(role, {
            username,
            groups: ['system:masters'],
        });
    }
    /**
     * Adds a mapping between an IAM role to a Kubernetes user and groups.
     *
     * @param role The IAM role to map.
     * @param mapping Mapping to k8s user name and groups.
     * @stability stable
     */
    addRoleMapping(role, mapping) {
        this.assertSameStack(role);
        this.roleMappings.push({ role, mapping });
    }
    /**
     * Adds a mapping between an IAM user to a Kubernetes user and groups.
     *
     * @param user The IAM user to map.
     * @param mapping Mapping to k8s user name and groups.
     * @stability stable
     */
    addUserMapping(user, mapping) {
        this.assertSameStack(user);
        this.userMappings.push({ user, mapping });
    }
    /**
     * Additional AWS account to add to the aws-auth configmap.
     *
     * @param accountId account number.
     * @stability stable
     */
    addAccount(accountId) {
        this.accounts.push(accountId);
    }
    assertSameStack(construct) {
        const thisStack = core_1.Stack.of(this);
        if (core_1.Stack.of(construct) !== thisStack) {
            // aws-auth is always part of the cluster stack, and since resources commonly take
            // a dependency on the cluster, allowing those resources to be in a different stack,
            // will create a circular dependency. granted, it won't always be the case,
            // but we opted for the more causious and restrictive approach for now.
            throw new Error(`${construct.node.path} should be defined in the scope of the ${thisStack.stackName} stack to prevent circular dependencies`);
        }
    }
    synthesizeMapRoles() {
        return core_1.Lazy.any({
            produce: () => this.stack.toJsonString(this.roleMappings.map(m => {
                var _b;
                return ({
                    rolearn: m.role.roleArn,
                    username: (_b = m.mapping.username) !== null && _b !== void 0 ? _b : m.role.roleArn,
                    groups: m.mapping.groups,
                });
            })),
        });
    }
    synthesizeMapUsers() {
        return core_1.Lazy.any({
            produce: () => this.stack.toJsonString(this.userMappings.map(m => {
                var _b;
                return ({
                    userarn: m.user.userArn,
                    username: (_b = m.mapping.username) !== null && _b !== void 0 ? _b : m.user.userArn,
                    groups: m.mapping.groups,
                });
            })),
        });
    }
    synthesizeMapAccounts() {
        return core_1.Lazy.any({
            produce: () => this.stack.toJsonString(this.accounts),
        });
    }
}
exports.AwsAuth = AwsAuth;
_a = JSII_RTTI_SYMBOL_1;
AwsAuth[_a] = { fqn: "@aws-cdk/aws-eks.AwsAuth", version: "1.121.0" };
//# sourceMappingURL=data:application/json;base64,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