"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Job = exports.MetricType = exports.JobState = exports.WorkerType = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const _1 = require(".");
const glue_generated_1 = require("./glue.generated");
/**
 * (experimental) The type of predefined worker that is allocated when a job runs.
 *
 * If you need to use a WorkerType that doesn't exist as a static member, you
 * can instantiate a `WorkerType` object, e.g: `WorkerType.of('other type')`.
 *
 * @experimental
 */
class WorkerType {
    constructor(name) {
        this.name = name;
    }
    /**
     * (experimental) Custom worker type.
     *
     * @param workerType custom worker type.
     * @experimental
     */
    static of(workerType) {
        return new WorkerType(workerType);
    }
}
exports.WorkerType = WorkerType;
_a = JSII_RTTI_SYMBOL_1;
WorkerType[_a] = { fqn: "@aws-cdk/aws-glue.WorkerType", version: "1.142.0" };
/**
 * (experimental) Each worker provides 4 vCPU, 16 GB of memory and a 50GB disk, and 2 executors per worker.
 *
 * @experimental
 */
WorkerType.STANDARD = new WorkerType('Standard');
/**
 * (experimental) Each worker maps to 1 DPU (4 vCPU, 16 GB of memory, 64 GB disk), and provides 1 executor per worker.
 *
 * Suitable for memory-intensive jobs.
 *
 * @experimental
 */
WorkerType.G_1X = new WorkerType('G.1X');
/**
 * (experimental) Each worker maps to 2 DPU (8 vCPU, 32 GB of memory, 128 GB disk), and provides 1 executor per worker.
 *
 * Suitable for memory-intensive jobs.
 *
 * @experimental
 */
WorkerType.G_2X = new WorkerType('G.2X');
/**
 * (experimental) Job states emitted by Glue to CloudWatch Events.
 *
 * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/EventTypes.html#glue-event-types for more information.
 * @experimental
 */
var JobState;
(function (JobState) {
    JobState["SUCCEEDED"] = "SUCCEEDED";
    JobState["FAILED"] = "FAILED";
    JobState["TIMEOUT"] = "TIMEOUT";
    JobState["STARTING"] = "STARTING";
    JobState["RUNNING"] = "RUNNING";
    JobState["STOPPING"] = "STOPPING";
    JobState["STOPPED"] = "STOPPED";
})(JobState = exports.JobState || (exports.JobState = {}));
/**
 * (experimental) The Glue CloudWatch metric type.
 *
 * @see https://docs.aws.amazon.com/glue/latest/dg/monitoring-awsglue-with-cloudwatch-metrics.html
 * @experimental
 */
var MetricType;
(function (MetricType) {
    MetricType["GAUGE"] = "gauge";
    MetricType["COUNT"] = "count";
})(MetricType = exports.MetricType || (exports.MetricType = {}));
class JobBase extends cdk.Resource {
    /**
     * (experimental) Create a CloudWatch Event Rule for this Glue Job when it's in a given state.
     *
     * @param id construct id.
     * @param options event options.
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/EventTypes.html#glue-event-types
     * @experimental
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.glue'],
            detailType: ['Glue Job State Change', 'Glue Job Run Status'],
            detail: {
                jobName: [this.jobName],
            },
        });
        return rule;
    }
    /**
     * (experimental) Create a CloudWatch Event Rule for the transition into the input jobState.
     *
     * @param id construct id.
     * @param jobState the job state.
     * @param options optional event options.
     * @experimental
     */
    onStateChange(id, jobState, options = {}) {
        const rule = this.onEvent(id, {
            description: `Rule triggered when Glue job ${this.jobName} is in ${jobState} state`,
            ...options,
        });
        rule.addEventPattern({
            detail: {
                state: [jobState],
            },
        });
        return rule;
    }
    /**
     * (experimental) Create a CloudWatch Event Rule matching JobState.SUCCEEDED.
     *
     * @param id construct id.
     * @param options optional event options.
     * @experimental
     */
    onSuccess(id, options = {}) {
        return this.onStateChange(id, JobState.SUCCEEDED, options);
    }
    /**
     * (experimental) Return a CloudWatch Event Rule matching FAILED state.
     *
     * @param id construct id.
     * @param options optional event options.
     * @experimental
     */
    onFailure(id, options = {}) {
        return this.onStateChange(id, JobState.FAILED, options);
    }
    /**
     * (experimental) Return a CloudWatch Event Rule matching TIMEOUT state.
     *
     * @param id construct id.
     * @param options optional event options.
     * @experimental
     */
    onTimeout(id, options = {}) {
        return this.onStateChange(id, JobState.TIMEOUT, options);
    }
    /**
     * (experimental) Create a CloudWatch metric.
     *
     * @param metricName name of the metric typically prefixed with `glue.driver.`, `glue.<executorId>.` or `glue.ALL.`.
     * @param type the metric type.
     * @param props metric options.
     * @see https://docs.aws.amazon.com/glue/latest/dg/monitoring-awsglue-with-cloudwatch-metrics.html
     * @experimental
     */
    metric(metricName, type, props) {
        return new cloudwatch.Metric({
            metricName,
            namespace: 'Glue',
            dimensionsMap: {
                JobName: this.jobName,
                JobRunId: 'ALL',
                Type: type,
            },
            ...props,
        }).attachTo(this);
    }
    /**
     * (experimental) Return a CloudWatch Metric indicating job success.
     *
     * This metric is based on the Rule returned by no-args onSuccess() call.
     *
     * @experimental
     */
    metricSuccess(props) {
        return metricRule(this.metricJobStateRule('SuccessMetricRule', JobState.SUCCEEDED), props);
    }
    /**
     * (experimental) Return a CloudWatch Metric indicating job failure.
     *
     * This metric is based on the Rule returned by no-args onFailure() call.
     *
     * @experimental
     */
    metricFailure(props) {
        return metricRule(this.metricJobStateRule('FailureMetricRule', JobState.FAILED), props);
    }
    /**
     * (experimental) Return a CloudWatch Metric indicating job timeout.
     *
     * This metric is based on the Rule returned by no-args onTimeout() call.
     *
     * @experimental
     */
    metricTimeout(props) {
        return metricRule(this.metricJobStateRule('TimeoutMetricRule', JobState.TIMEOUT), props);
    }
    /**
     * Creates or retrieves a singleton event rule for the input job state for use with the metric JobState methods.
     *
     * @param id construct id.
     * @param jobState the job state.
     * @private
     */
    metricJobStateRule(id, jobState) {
        var _c;
        return (_c = this.node.tryFindChild(id)) !== null && _c !== void 0 ? _c : this.onStateChange(id, jobState);
    }
}
/**
 * (experimental) A Glue Job.
 *
 * @experimental
 */
class Job extends JobBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _c, _d, _e, _f, _g, _h;
        super(scope, id, {
            physicalName: props.jobName,
        });
        jsiiDeprecationWarnings._aws_cdk_aws_glue_JobProps(props);
        const executable = props.executable.bind();
        this.role = (_c = props.role) !== null && _c !== void 0 ? _c : new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('glue.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSGlueServiceRole')],
        });
        this.grantPrincipal = this.role;
        const sparkUI = ((_d = props.sparkUI) === null || _d === void 0 ? void 0 : _d.enabled) ? this.setupSparkUI(executable, this.role, props.sparkUI) : undefined;
        this.sparkUILoggingLocation = sparkUI === null || sparkUI === void 0 ? void 0 : sparkUI.location;
        const continuousLoggingArgs = ((_e = props.continuousLogging) === null || _e === void 0 ? void 0 : _e.enabled) ? this.setupContinuousLogging(this.role, props.continuousLogging) : {};
        const profilingMetricsArgs = props.enableProfilingMetrics ? { '--enable-metrics': '' } : {};
        const defaultArguments = {
            ...this.executableArguments(executable),
            ...continuousLoggingArgs,
            ...profilingMetricsArgs,
            ...sparkUI === null || sparkUI === void 0 ? void 0 : sparkUI.args,
            ...this.checkNoReservedArgs(props.defaultArguments),
        };
        const jobResource = new glue_generated_1.CfnJob(this, 'Resource', {
            name: props.jobName,
            description: props.description,
            role: this.role.roleArn,
            command: {
                name: executable.type.name,
                scriptLocation: this.codeS3ObjectUrl(executable.script),
                pythonVersion: executable.pythonVersion,
            },
            glueVersion: executable.glueVersion.name,
            workerType: (_f = props.workerType) === null || _f === void 0 ? void 0 : _f.name,
            numberOfWorkers: props.workerCount,
            maxCapacity: props.maxCapacity,
            maxRetries: props.maxRetries,
            executionProperty: props.maxConcurrentRuns ? { maxConcurrentRuns: props.maxConcurrentRuns } : undefined,
            notificationProperty: props.notifyDelayAfter ? { notifyDelayAfter: props.notifyDelayAfter.toMinutes() } : undefined,
            timeout: (_g = props.timeout) === null || _g === void 0 ? void 0 : _g.toMinutes(),
            connections: props.connections ? { connections: props.connections.map((connection) => connection.connectionName) } : undefined,
            securityConfiguration: (_h = props.securityConfiguration) === null || _h === void 0 ? void 0 : _h.securityConfigurationName,
            tags: props.tags,
            defaultArguments,
        });
        const resourceName = this.getResourceNameAttribute(jobResource.ref);
        this.jobArn = jobArn(this, resourceName);
        this.jobName = resourceName;
    }
    /**
     * (experimental) Creates a Glue Job.
     *
     * @param scope The scope creating construct (usually `this`).
     * @param id The construct's id.
     * @param attrs Import attributes.
     * @experimental
     */
    static fromJobAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings._aws_cdk_aws_glue_JobAttributes(attrs);
        class Import extends JobBase {
            constructor() {
                var _c;
                super(...arguments);
                this.jobName = attrs.jobName;
                this.jobArn = jobArn(scope, attrs.jobName);
                this.grantPrincipal = (_c = attrs.role) !== null && _c !== void 0 ? _c : new iam.UnknownPrincipal({ resource: this });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Check no usage of reserved arguments.
     *
     * @see https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html
     */
    checkNoReservedArgs(defaultArguments) {
        if (defaultArguments) {
            const reservedArgs = new Set(['--conf', '--debug', '--mode', '--JOB_NAME']);
            Object.keys(defaultArguments).forEach((arg) => {
                if (reservedArgs.has(arg)) {
                    throw new Error(`The ${arg} argument is reserved by Glue. Don't set it`);
                }
            });
        }
        return defaultArguments;
    }
    executableArguments(config) {
        var _c;
        const args = {};
        args['--job-language'] = config.language;
        if (config.className) {
            args['--class'] = config.className;
        }
        if (config.extraJars && ((_c = config.extraJars) === null || _c === void 0 ? void 0 : _c.length) > 0) {
            args['--extra-jars'] = config.extraJars.map(code => this.codeS3ObjectUrl(code)).join(',');
        }
        if (config.extraPythonFiles && config.extraPythonFiles.length > 0) {
            args['--extra-py-files'] = config.extraPythonFiles.map(code => this.codeS3ObjectUrl(code)).join(',');
        }
        if (config.extraFiles && config.extraFiles.length > 0) {
            args['--extra-files'] = config.extraFiles.map(code => this.codeS3ObjectUrl(code)).join(',');
        }
        if (config.extraJarsFirst) {
            args['--user-jars-first'] = 'true';
        }
        return args;
    }
    setupSparkUI(executable, role, props) {
        var _c;
        if (_1.JobType.PYTHON_SHELL === executable.type) {
            throw new Error('Spark UI is not available for JobType.PYTHON_SHELL jobs');
        }
        const bucket = (_c = props.bucket) !== null && _c !== void 0 ? _c : new s3.Bucket(this, 'SparkUIBucket');
        bucket.grantReadWrite(role);
        const args = {
            '--enable-spark-ui': 'true',
            '--spark-event-logs-path': bucket.s3UrlForObject(props.prefix),
        };
        return {
            location: {
                prefix: props.prefix,
                bucket,
            },
            args,
        };
    }
    setupContinuousLogging(role, props) {
        var _c;
        const args = {
            '--enable-continuous-cloudwatch-log': 'true',
            '--enable-continuous-log-filter': ((_c = props.quiet) !== null && _c !== void 0 ? _c : true).toString(),
        };
        if (props.logGroup) {
            args['--continuous-log-logGroup'] = props.logGroup.logGroupName;
            props.logGroup.grantWrite(role);
        }
        if (props.logStreamPrefix) {
            args['--continuous-log-logStreamPrefix'] = props.logStreamPrefix;
        }
        if (props.conversionPattern) {
            args['--continuous-log-conversionPattern'] = props.conversionPattern;
        }
        return args;
    }
    codeS3ObjectUrl(code) {
        const s3Location = code.bind(this, this.role).s3Location;
        return `s3://${s3Location.bucketName}/${s3Location.objectKey}`;
    }
}
exports.Job = Job;
_b = JSII_RTTI_SYMBOL_1;
Job[_b] = { fqn: "@aws-cdk/aws-glue.Job", version: "1.142.0" };
/**
 * Create a CloudWatch Metric that's based on Glue Job events
 * {@see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/EventTypes.html#glue-event-types}
 * The metric has namespace = 'AWS/Events', metricName = 'TriggeredRules' and RuleName = rule.ruleName dimension.
 *
 * @param rule for use in setting RuleName dimension value
 * @param props metric properties
 */
function metricRule(rule, props) {
    return new cloudwatch.Metric({
        namespace: 'AWS/Events',
        metricName: 'TriggeredRules',
        dimensionsMap: { RuleName: rule.ruleName },
        statistic: cloudwatch.Statistic.SUM,
        ...props,
    }).attachTo(rule);
}
/**
 * Returns the job arn
 * @param scope
 * @param jobName
 */
function jobArn(scope, jobName) {
    return cdk.Stack.of(scope).formatArn({
        service: 'glue',
        resource: 'job',
        resourceName: jobName,
    });
}
//# sourceMappingURL=data:application/json;base64,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