"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitlabRunnerAutoscaling = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const asg = require("@aws-cdk/aws-autoscaling");
const aws_autoscaling_hooktargets_1 = require("@aws-cdk/aws-autoscaling-hooktargets");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const logs = require("@aws-cdk/aws-logs");
const assets = require("@aws-cdk/aws-s3-assets");
const sns = require("@aws-cdk/aws-sns");
const subscriptions = require("@aws-cdk/aws-sns-subscriptions");
const cdk = require("@aws-cdk/core");
const cr = require("@aws-cdk/custom-resources");
/**
 * GitlabRunnerAutoscaling Construct for create Autoscaling Gitlab Runner.
 */
class GitlabRunnerAutoscaling extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const defaultProps = {
            instanceType: 't3.micro',
            tags: ['gitlab', 'awscdk', 'runner'],
            gitlabUrl: 'https://gitlab.com/',
            gitlabRunnerImage: 'public.ecr.aws/gitlab/gitlab-runner:latest',
            alarms: [
                {
                    AlarmName: 'GitlabRunnerDiskUsage',
                    MetricName: 'disk_used_percent',
                },
            ],
        };
        const runnerProps = { ...defaultProps, ...props };
        const asset = new assets.Asset(this, 'GitlabRunnerUserDataAsset', {
            path: path.join(__dirname, '../assets/userdata/amazon-cloudwatch-agent.json'),
        });
        const userData = ec2.UserData.forLinux();
        userData.addS3DownloadCommand({
            bucket: asset.bucket,
            bucketKey: asset.s3ObjectKey,
            localFile: '/opt/aws/amazon-cloudwatch-agent/etc/amazon-cloudwatch-agent.json',
        });
        userData.addCommands(...this.createUserData(runnerProps));
        this.instanceRole =
            runnerProps.instanceRole ??
                new iam.Role(this, 'GitlabRunnerInstanceRole', {
                    assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
                    description: 'For EC2 Instance (Gitlab Runner) Role',
                    managedPolicies: [
                        iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
                        iam.ManagedPolicy.fromAwsManagedPolicyName('CloudWatchAgentServerPolicy'),
                        iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonS3ReadOnlyAccess'),
                    ],
                });
        this.vpc = runnerProps.vpc ?? new ec2.Vpc(this, 'VPC');
        this.securityGroup = new ec2.SecurityGroup(this, 'GitlabRunnerSecurityGroup', {
            vpc: this.vpc,
        });
        const instanceProfile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [this.instanceRole.roleName],
        });
        const lt = new ec2.CfnLaunchTemplate(this, 'GitlabRunnerLaunchTemplate', {
            launchTemplateData: {
                imageId: ec2.MachineImage.latestAmazonLinux({
                    generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
                }).getImage(this).imageId,
                instanceType: runnerProps.instanceType,
                instanceMarketOptions: {
                    marketType: runnerProps.spotInstance ? 'spot' : undefined,
                    spotOptions: runnerProps.spotInstance ? {
                        spotInstanceType: 'one-time',
                    } : undefined,
                },
                userData: cdk.Fn.base64(userData.render()),
                blockDeviceMappings: [
                    {
                        deviceName: '/dev/xvda',
                        ebs: {
                            volumeSize: runnerProps.ebsSize ?? 60,
                        },
                    },
                ],
                iamInstanceProfile: {
                    arn: instanceProfile.attrArn,
                },
                securityGroupIds: this.securityGroup.connections.securityGroups.map((m) => m.securityGroupId),
            },
        });
        this.autoscalingGroup = new asg.AutoScalingGroup(this, 'GitlabRunnerAutoscalingGroup', {
            instanceType: new ec2.InstanceType(runnerProps.instanceType),
            autoScalingGroupName: `Gitlab Runners (${runnerProps.instanceType})`,
            vpc: this.vpc,
            vpcSubnets: runnerProps.vpcSubnet,
            machineImage: ec2.MachineImage.latestAmazonLinux({
                generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
            }),
            minCapacity: runnerProps.minCapacity,
            maxCapacity: runnerProps.maxCapacity,
            desiredCapacity: runnerProps.desiredCapacity,
        });
        const cfnAsg = this.autoscalingGroup.node.tryFindChild('ASG');
        cfnAsg.addPropertyDeletionOverride('LaunchConfigurationName');
        cfnAsg.addPropertyOverride('LaunchTemplate', {
            LaunchTemplateId: lt.ref,
            Version: lt.attrLatestVersionNumber,
        });
        this.autoscalingGroup.node.tryRemoveChild('LaunchConfig');
        this.topicAlarm = new sns.Topic(this, 'GitlabRunnerAlarm');
        const alarms = JSON.stringify(runnerProps.alarms);
        // Put alarms at launch
        const registerFunction = new lambda.Function(this, 'GitlabRunnerRegisterFunction', {
            code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
            handler: 'autoscaling_events.register',
            runtime: lambda.Runtime.PYTHON_3_8,
            timeout: cdk.Duration.seconds(60),
            logRetention: logs.RetentionDays.ONE_DAY,
            environment: {
                ALARMS: alarms,
                SNS_TOPIC_ARN: this.topicAlarm.topicArn,
            },
        });
        registerFunction.role?.addToPrincipalPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            resources: ['*'],
            actions: [
                'cloudwatch:PutMetricAlarm',
            ],
        }));
        this.autoscalingGroup.addLifecycleHook('GitlabRunnerLifeCycleHookLaunching', {
            lifecycleTransition: asg.LifecycleTransition.INSTANCE_LAUNCHING,
            notificationTarget: new aws_autoscaling_hooktargets_1.FunctionHook(registerFunction),
            defaultResult: asg.DefaultResult.CONTINUE,
            heartbeatTimeout: cdk.Duration.seconds(60),
        });
        // Add an alarm action to terminate invalid instances
        const alarmAction = new lambda.Function(this, 'GitlabRunnerAlarmAction', {
            code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
            handler: 'autoscaling_events.on_alarm',
            runtime: lambda.Runtime.PYTHON_3_8,
            timeout: cdk.Duration.seconds(60),
            logRetention: logs.RetentionDays.ONE_DAY,
        });
        alarmAction.role?.addToPrincipalPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            resources: ['*'],
            actions: [
                'autoscaling:SetInstanceHealth',
            ],
        }));
        const alarmSubscription = new subscriptions.LambdaSubscription(alarmAction);
        this.topicAlarm.addSubscription(alarmSubscription);
        // Unregister gitlab runners and remove alarms on instance termination or CFn stack deletion
        const unregisterRole = new iam.Role(this, 'GitlabRunnerUnregisterRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            description: 'For Gitlab Runner Unregistering Function Role',
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ],
        });
        unregisterRole.addToPrincipalPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            resources: ['*'],
            actions: [
                'ssm:SendCommand',
                'autoscaling:DescribeAutoScalingGroups',
                'cloudwatch:DeleteAlarms',
            ],
        }));
        const unregisterFunction = new lambda.Function(this, 'GitlabRunnerUnregisterFunction', {
            code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
            handler: 'autoscaling_events.unregister',
            runtime: lambda.Runtime.PYTHON_3_8,
            timeout: cdk.Duration.seconds(60),
            role: unregisterRole,
            logRetention: logs.RetentionDays.ONE_DAY,
            environment: {
                ALARMS: alarms,
            },
        });
        this.autoscalingGroup.addLifecycleHook('GitlabRunnerLifeCycleHookTerminating', {
            lifecycleTransition: asg.LifecycleTransition.INSTANCE_TERMINATING,
            notificationTarget: new aws_autoscaling_hooktargets_1.FunctionHook(unregisterFunction),
            defaultResult: asg.DefaultResult.CONTINUE,
            heartbeatTimeout: cdk.Duration.seconds(60),
        });
        const unregisterCustomResource = new lambda.Function(this, 'GitlabRunnerUnregisterCustomResource', {
            code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
            handler: 'autoscaling_events.on_event',
            runtime: lambda.Runtime.PYTHON_3_8,
            role: unregisterRole,
            logRetention: logs.RetentionDays.ONE_DAY,
            environment: {
                ALARMS: alarms,
            },
        });
        const unregisterProvider = new cr.Provider(this, 'GitlabRunnerUnregisterProvider', {
            onEventHandler: unregisterCustomResource,
        });
        const customResource = new cdk.CustomResource(this, 'GitlabRunnerCustomResource', {
            serviceToken: unregisterProvider.serviceToken,
            properties: {
                AutoScalingGroupNames: [this.autoscalingGroup.autoScalingGroupName],
            },
        });
        customResource.node.addDependency(unregisterProvider);
        new cdk.CfnOutput(this, 'GitlabRunnerAutoScalingGroupArn', {
            value: this.autoscalingGroup.autoScalingGroupArn,
        });
    }
    dockerVolumesList(dockerVolume) {
        let tempString = '--docker-volumes "/var/run/docker.sock:/var/run/docker.sock"';
        if (dockerVolume) {
            let tempList = [];
            dockerVolume.forEach(e => {
                tempList.push(`"${e.hostPath}:${e.containerPath}"`);
            });
            tempList.forEach(e => {
                tempString = `${tempString} --docker-volumes ${e}`;
            });
        }
        return tempString;
    }
    /**
     * @param props
     * @returns Array.
     */
    createUserData(props) {
        return [
            'yum update -y',
            'sleep 15 && amazon-linux-extras install docker && yum install -y amazon-cloudwatch-agent && systemctl start docker && usermod -aG docker ec2-user && chmod 777 /var/run/docker.sock',
            'systemctl restart docker && systemctl enable docker && systemctl start amazon-cloudwatch-agent && systemctl enable amazon-cloudwatch-agent',
            `docker run -d -v /home/ec2-user/.gitlab-runner:/etc/gitlab-runner -v /var/run/docker.sock:/var/run/docker.sock \
      --name gitlab-runner-register ${props.gitlabRunnerImage} register --non-interactive --url ${props.gitlabUrl} --registration-token ${props.gitlabToken} \
      --docker-pull-policy if-not-present ${this.dockerVolumesList(props?.dockerVolumes)} \
      --executor docker --docker-image "alpine:latest" --description "A Runner on EC2 Instance (${props.instanceType})" \
      --tag-list "${props.tags?.join(',')}" --docker-privileged`,
            `sleep 2 && docker run --restart always -d -v /home/ec2-user/.gitlab-runner:/etc/gitlab-runner -v /var/run/docker.sock:/var/run/docker.sock --name gitlab-runner ${props.gitlabRunnerImage}`,
        ];
    }
}
exports.GitlabRunnerAutoscaling = GitlabRunnerAutoscaling;
_a = JSII_RTTI_SYMBOL_1;
GitlabRunnerAutoscaling[_a] = { fqn: "cdk-gitlab-runner.GitlabRunnerAutoscaling", version: "1.115.1326" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZ2l0bGFiLXJ1bm5lci1hdXRvc2NhbGluZy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9naXRsYWItcnVubmVyLWF1dG9zY2FsaW5nLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsNkJBQTZCO0FBQzdCLGdEQUFnRDtBQUNoRCxzRkFBb0U7QUFDcEUsd0NBQXdDO0FBQ3hDLHdDQUF3QztBQUN4Qyw4Q0FBOEM7QUFDOUMsMENBQTBDO0FBQzFDLGlEQUFpRDtBQUNqRCx3Q0FBd0M7QUFDeEMsZ0VBQWdFO0FBQ2hFLHFDQUFxQztBQUNyQyxnREFBZ0Q7QUF1TWhEOztHQUVHO0FBQ0gsTUFBYSx1QkFBd0IsU0FBUSxHQUFHLENBQUMsU0FBUztJQTJCeEQsWUFBWSxLQUFvQixFQUFFLEVBQVUsRUFBRSxLQUFtQztRQUMvRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQ2pCLE1BQU0sWUFBWSxHQUFHO1lBQ25CLFlBQVksRUFBRSxVQUFVO1lBQ3hCLElBQUksRUFBRSxDQUFDLFFBQVEsRUFBRSxRQUFRLEVBQUUsUUFBUSxDQUFDO1lBQ3BDLFNBQVMsRUFBRSxxQkFBcUI7WUFDaEMsaUJBQWlCLEVBQUUsNENBQTRDO1lBQy9ELE1BQU0sRUFBRTtnQkFDTjtvQkFDRSxTQUFTLEVBQUUsdUJBQXVCO29CQUNsQyxVQUFVLEVBQUUsbUJBQW1CO2lCQUNoQzthQUNGO1NBQ0YsQ0FBQztRQUNGLE1BQU0sV0FBVyxHQUFHLEVBQUUsR0FBRyxZQUFZLEVBQUUsR0FBRyxLQUFLLEVBQUUsQ0FBQztRQUVsRCxNQUFNLEtBQUssR0FBRyxJQUFJLE1BQU0sQ0FBQyxLQUFLLENBQUMsSUFBSSxFQUFFLDJCQUEyQixFQUFFO1lBQ2hFLElBQUksRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxpREFBaUQsQ0FBQztTQUM5RSxDQUFDLENBQUM7UUFFSCxNQUFNLFFBQVEsR0FBRyxHQUFHLENBQUMsUUFBUSxDQUFDLFFBQVEsRUFBRSxDQUFDO1FBQ3pDLFFBQVEsQ0FBQyxvQkFBb0IsQ0FBQztZQUM1QixNQUFNLEVBQUUsS0FBSyxDQUFDLE1BQU07WUFDcEIsU0FBUyxFQUFFLEtBQUssQ0FBQyxXQUFXO1lBQzVCLFNBQVMsRUFBRSxtRUFBbUU7U0FDL0UsQ0FBQyxDQUFDO1FBQ0gsUUFBUSxDQUFDLFdBQVcsQ0FBQyxHQUFHLElBQUksQ0FBQyxjQUFjLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQztRQUUxRCxJQUFJLENBQUMsWUFBWTtZQUNmLFdBQVcsQ0FBQyxZQUFZO2dCQUN4QixJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLDBCQUEwQixFQUFFO29CQUM3QyxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsbUJBQW1CLENBQUM7b0JBQ3hELFdBQVcsRUFBRSx1Q0FBdUM7b0JBQ3BELGVBQWUsRUFBRTt3QkFDZixHQUFHLENBQUMsYUFBYSxDQUFDLHdCQUF3QixDQUFDLDhCQUE4QixDQUFDO3dCQUMxRSxHQUFHLENBQUMsYUFBYSxDQUFDLHdCQUF3QixDQUFDLDZCQUE2QixDQUFDO3dCQUN6RSxHQUFHLENBQUMsYUFBYSxDQUFDLHdCQUF3QixDQUFDLHdCQUF3QixDQUFDO3FCQUNyRTtpQkFDRixDQUFDLENBQUM7UUFFTCxJQUFJLENBQUMsR0FBRyxHQUFHLFdBQVcsQ0FBQyxHQUFHLElBQUksSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLElBQUksRUFBRSxLQUFLLENBQUMsQ0FBQztRQUV2RCxJQUFJLENBQUMsYUFBYSxHQUFHLElBQUksR0FBRyxDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQUUsMkJBQTJCLEVBQUU7WUFDNUUsR0FBRyxFQUFFLElBQUksQ0FBQyxHQUFHO1NBQ2QsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxlQUFlLEdBQUcsSUFBSSxHQUFHLENBQUMsa0JBQWtCLENBQUMsSUFBSSxFQUFFLGlCQUFpQixFQUFFO1lBQzFFLEtBQUssRUFBRSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsUUFBUSxDQUFDO1NBQ3BDLENBQUMsQ0FBQztRQUNILE1BQU0sRUFBRSxHQUFHLElBQUksR0FBRyxDQUFDLGlCQUFpQixDQUFDLElBQUksRUFBRSw0QkFBNEIsRUFBRTtZQUN2RSxrQkFBa0IsRUFBRTtnQkFDbEIsT0FBTyxFQUFFLEdBQUcsQ0FBQyxZQUFZLENBQUMsaUJBQWlCLENBQUM7b0JBQzFDLFVBQVUsRUFBRSxHQUFHLENBQUMscUJBQXFCLENBQUMsY0FBYztpQkFDckQsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPO2dCQUN6QixZQUFZLEVBQUUsV0FBVyxDQUFDLFlBQVk7Z0JBQ3RDLHFCQUFxQixFQUFFO29CQUNyQixVQUFVLEVBQUUsV0FBVyxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxTQUFTO29CQUN6RCxXQUFXLEVBQUUsV0FBVyxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUM7d0JBQ3RDLGdCQUFnQixFQUFFLFVBQVU7cUJBQzdCLENBQUMsQ0FBQyxDQUFDLFNBQVM7aUJBQ2Q7Z0JBQ0QsUUFBUSxFQUFFLEdBQUcsQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLFFBQVEsQ0FBQyxNQUFNLEVBQUUsQ0FBQztnQkFDMUMsbUJBQW1CLEVBQUU7b0JBQ25CO3dCQUNFLFVBQVUsRUFBRSxXQUFXO3dCQUN2QixHQUFHLEVBQUU7NEJBQ0gsVUFBVSxFQUFFLFdBQVcsQ0FBQyxPQUFPLElBQUksRUFBRTt5QkFDdEM7cUJBQ0Y7aUJBQ0Y7Z0JBQ0Qsa0JBQWtCLEVBQUU7b0JBQ2xCLEdBQUcsRUFBRSxlQUFlLENBQUMsT0FBTztpQkFDN0I7Z0JBQ0QsZ0JBQWdCLEVBQUUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxXQUFXLENBQUMsY0FBYyxDQUFDLEdBQUcsQ0FDakUsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxlQUFlLENBQ3pCO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsZ0JBQWdCLEdBQUcsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxFQUFFLDhCQUE4QixFQUFFO1lBQ3JGLFlBQVksRUFBRSxJQUFJLEdBQUcsQ0FBQyxZQUFZLENBQUMsV0FBVyxDQUFDLFlBQVksQ0FBQztZQUM1RCxvQkFBb0IsRUFBRSxtQkFBbUIsV0FBVyxDQUFDLFlBQVksR0FBRztZQUNwRSxHQUFHLEVBQUUsSUFBSSxDQUFDLEdBQUc7WUFDYixVQUFVLEVBQUUsV0FBVyxDQUFDLFNBQVM7WUFDakMsWUFBWSxFQUFFLEdBQUcsQ0FBQyxZQUFZLENBQUMsaUJBQWlCLENBQUM7Z0JBQy9DLFVBQVUsRUFBRSxHQUFHLENBQUMscUJBQXFCLENBQUMsY0FBYzthQUNyRCxDQUFDO1lBQ0YsV0FBVyxFQUFFLFdBQVcsQ0FBQyxXQUFXO1lBQ3BDLFdBQVcsRUFBRSxXQUFXLENBQUMsV0FBVztZQUNwQyxlQUFlLEVBQUUsV0FBVyxDQUFDLGVBQWU7U0FDN0MsQ0FBQyxDQUFDO1FBRUgsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUE0QixDQUFDO1FBQ3pGLE1BQU0sQ0FBQywyQkFBMkIsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1FBQzlELE1BQU0sQ0FBQyxtQkFBbUIsQ0FBQyxnQkFBZ0IsRUFBRTtZQUMzQyxnQkFBZ0IsRUFBRSxFQUFFLENBQUMsR0FBRztZQUN4QixPQUFPLEVBQUUsRUFBRSxDQUFDLHVCQUF1QjtTQUNwQyxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxjQUFjLENBQUMsQ0FBQztRQUUxRCxJQUFJLENBQUMsVUFBVSxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxJQUFJLEVBQUUsbUJBQW1CLENBQUMsQ0FBQztRQUMzRCxNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLFdBQVcsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUVsRCx1QkFBdUI7UUFDdkIsTUFBTSxnQkFBZ0IsR0FBRyxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLDhCQUE4QixFQUFFO1lBQ2pGLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxxQkFBcUIsQ0FBQyxDQUFDO1lBQ3hFLE9BQU8sRUFBRSw2QkFBNkI7WUFDdEMsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsVUFBVTtZQUNsQyxPQUFPLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO1lBQ2pDLFlBQVksRUFBRSxJQUFJLENBQUMsYUFBYSxDQUFDLE9BQU87WUFDeEMsV0FBVyxFQUFFO2dCQUNYLE1BQU0sRUFBRSxNQUFNO2dCQUNkLGFBQWEsRUFBRSxJQUFJLENBQUMsVUFBVSxDQUFDLFFBQVE7YUFDeEM7U0FDRixDQUFDLENBQUM7UUFDSCxnQkFBZ0IsQ0FBQyxJQUFJLEVBQUUsb0JBQW9CLENBQ3pDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUN0QixNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO1lBQ3hCLFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQztZQUNoQixPQUFPLEVBQUU7Z0JBQ1AsMkJBQTJCO2FBQzVCO1NBQ0YsQ0FBQyxDQUNILENBQUM7UUFFRixJQUFJLENBQUMsZ0JBQWdCLENBQUMsZ0JBQWdCLENBQUMsb0NBQW9DLEVBQUU7WUFDM0UsbUJBQW1CLEVBQUUsR0FBRyxDQUFDLG1CQUFtQixDQUFDLGtCQUFrQjtZQUMvRCxrQkFBa0IsRUFBRSxJQUFJLDBDQUFZLENBQUMsZ0JBQWdCLENBQUM7WUFDdEQsYUFBYSxFQUFFLEdBQUcsQ0FBQyxhQUFhLENBQUMsUUFBUTtZQUN6QyxnQkFBZ0IsRUFBRSxHQUFHLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7U0FDM0MsQ0FBQyxDQUFDO1FBRUgscURBQXFEO1FBQ3JELE1BQU0sV0FBVyxHQUFHLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUseUJBQXlCLEVBQUU7WUFDdkUsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLHFCQUFxQixDQUFDLENBQUM7WUFDeEUsT0FBTyxFQUFFLDZCQUE2QjtZQUN0QyxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxVQUFVO1lBQ2xDLE9BQU8sRUFBRSxHQUFHLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7WUFDakMsWUFBWSxFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsT0FBTztTQUN6QyxDQUFDLENBQUM7UUFDSCxXQUFXLENBQUMsSUFBSSxFQUFFLG9CQUFvQixDQUNwQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDdEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSztZQUN4QixTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7WUFDaEIsT0FBTyxFQUFFO2dCQUNQLCtCQUErQjthQUNoQztTQUNGLENBQUMsQ0FDSCxDQUFDO1FBQ0YsTUFBTSxpQkFBaUIsR0FBRyxJQUFJLGFBQWEsQ0FBQyxrQkFBa0IsQ0FBQyxXQUFXLENBQUMsQ0FBQztRQUM1RSxJQUFJLENBQUMsVUFBVSxDQUFDLGVBQWUsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO1FBRW5ELDRGQUE0RjtRQUM1RixNQUFNLGNBQWMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLDRCQUE0QixFQUFFO1lBQ3RFLFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxzQkFBc0IsQ0FBQztZQUMzRCxXQUFXLEVBQUUsK0NBQStDO1lBQzVELGVBQWUsRUFBRTtnQkFDZixHQUFHLENBQUMsYUFBYSxDQUFDLHdCQUF3QixDQUFDLDBDQUEwQyxDQUFDO2FBQ3ZGO1NBQ0YsQ0FBQyxDQUFDO1FBQ0gsY0FBYyxDQUFDLG9CQUFvQixDQUNqQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDdEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSztZQUN4QixTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7WUFDaEIsT0FBTyxFQUFFO2dCQUNQLGlCQUFpQjtnQkFDakIsdUNBQXVDO2dCQUN2Qyx5QkFBeUI7YUFDMUI7U0FDRixDQUFDLENBQ0gsQ0FBQztRQUVGLE1BQU0sa0JBQWtCLEdBQUcsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxnQ0FBZ0MsRUFBRTtZQUNyRixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUscUJBQXFCLENBQUMsQ0FBQztZQUN4RSxPQUFPLEVBQUUsK0JBQStCO1lBQ3hDLE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFVBQVU7WUFDbEMsT0FBTyxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztZQUNqQyxJQUFJLEVBQUUsY0FBYztZQUNwQixZQUFZLEVBQUUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxPQUFPO1lBQ3hDLFdBQVcsRUFBRTtnQkFDWCxNQUFNLEVBQUUsTUFBTTthQUNmO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLGdCQUFnQixDQUFDLGdCQUFnQixDQUFDLHNDQUFzQyxFQUFFO1lBQzdFLG1CQUFtQixFQUFFLEdBQUcsQ0FBQyxtQkFBbUIsQ0FBQyxvQkFBb0I7WUFDakUsa0JBQWtCLEVBQUUsSUFBSSwwQ0FBWSxDQUFDLGtCQUFrQixDQUFDO1lBQ3hELGFBQWEsRUFBRSxHQUFHLENBQUMsYUFBYSxDQUFDLFFBQVE7WUFDekMsZ0JBQWdCLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO1NBQzNDLENBQUMsQ0FBQztRQUVILE1BQU0sd0JBQXdCLEdBQUcsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxzQ0FBc0MsRUFBRTtZQUNqRyxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUscUJBQXFCLENBQUMsQ0FBQztZQUN4RSxPQUFPLEVBQUUsNkJBQTZCO1lBQ3RDLE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFVBQVU7WUFDbEMsSUFBSSxFQUFFLGNBQWM7WUFDcEIsWUFBWSxFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsT0FBTztZQUN4QyxXQUFXLEVBQUU7Z0JBQ1gsTUFBTSxFQUFFLE1BQU07YUFDZjtTQUNGLENBQUMsQ0FBQztRQUVILE1BQU0sa0JBQWtCLEdBQUcsSUFBSSxFQUFFLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxnQ0FBZ0MsRUFBRTtZQUNqRixjQUFjLEVBQUUsd0JBQXdCO1NBQ3pDLENBQUMsQ0FBQztRQUVILE1BQU0sY0FBYyxHQUFHLElBQUksR0FBRyxDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUsNEJBQTRCLEVBQUU7WUFDaEYsWUFBWSxFQUFFLGtCQUFrQixDQUFDLFlBQVk7WUFDN0MsVUFBVSxFQUFFO2dCQUNWLHFCQUFxQixFQUFFLENBQUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLG9CQUFvQixDQUFDO2FBQ3BFO1NBQ0YsQ0FBQyxDQUFDO1FBQ0gsY0FBYyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsa0JBQWtCLENBQUMsQ0FBQztRQUV0RCxJQUFJLEdBQUcsQ0FBQyxTQUFTLENBQUMsSUFBSSxFQUFFLGlDQUFpQyxFQUFFO1lBQ3pELEtBQUssRUFBRSxJQUFJLENBQUMsZ0JBQWdCLENBQUMsbUJBQW1CO1NBQ2pELENBQUMsQ0FBQztJQUNMLENBQUM7SUFFTyxpQkFBaUIsQ0FBQyxZQUF5QztRQUNqRSxJQUFJLFVBQVUsR0FBVyw4REFBOEQsQ0FBQztRQUN4RixJQUFJLFlBQVksRUFBRSxDQUFDO1lBQ2pCLElBQUksUUFBUSxHQUFhLEVBQUUsQ0FBQztZQUM1QixZQUFZLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFO2dCQUN2QixRQUFRLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsSUFBSSxDQUFDLENBQUMsYUFBYSxHQUFHLENBQUMsQ0FBQztZQUN0RCxDQUFDLENBQUMsQ0FBQztZQUNILFFBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUU7Z0JBQ25CLFVBQVUsR0FBRyxHQUFHLFVBQVUscUJBQXFCLENBQUMsRUFBRSxDQUFDO1lBQ3JELENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQztRQUNELE9BQU8sVUFBVSxDQUFDO0lBQ3BCLENBQUM7SUFDRDs7O09BR0c7SUFDSSxjQUFjLENBQUMsS0FBbUM7UUFDdkQsT0FBTztZQUNMLGVBQWU7WUFDZixxTEFBcUw7WUFDckwsNElBQTRJO1lBQzVJO3NDQUNnQyxLQUFLLENBQUMsaUJBQWlCLHFDQUFxQyxLQUFLLENBQUMsU0FBUyx5QkFBeUIsS0FBSyxDQUFDLFdBQVc7NENBQy9HLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsYUFBYSxDQUFDO2tHQUNVLEtBQUssQ0FBQyxZQUFZO29CQUNoRyxLQUFLLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxHQUFHLENBQUMsdUJBQXVCO1lBQzFELG1LQUFtSyxLQUFLLENBQUMsaUJBQWlCLEVBQUU7U0FDN0wsQ0FBQztJQUNKLENBQUM7O0FBbFJILDBEQW1SQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgKiBhcyBhc2cgZnJvbSAnQGF3cy1jZGsvYXdzLWF1dG9zY2FsaW5nJztcbmltcG9ydCB7IEZ1bmN0aW9uSG9vayB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1hdXRvc2NhbGluZy1ob29rdGFyZ2V0cyc7XG5pbXBvcnQgKiBhcyBlYzIgZnJvbSAnQGF3cy1jZGsvYXdzLWVjMic7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBsYW1iZGEgZnJvbSAnQGF3cy1jZGsvYXdzLWxhbWJkYSc7XG5pbXBvcnQgKiBhcyBsb2dzIGZyb20gJ0Bhd3MtY2RrL2F3cy1sb2dzJztcbmltcG9ydCAqIGFzIGFzc2V0cyBmcm9tICdAYXdzLWNkay9hd3MtczMtYXNzZXRzJztcbmltcG9ydCAqIGFzIHNucyBmcm9tICdAYXdzLWNkay9hd3Mtc25zJztcbmltcG9ydCAqIGFzIHN1YnNjcmlwdGlvbnMgZnJvbSAnQGF3cy1jZGsvYXdzLXNucy1zdWJzY3JpcHRpb25zJztcbmltcG9ydCAqIGFzIGNkayBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCAqIGFzIGNyIGZyb20gJ0Bhd3MtY2RrL2N1c3RvbS1yZXNvdXJjZXMnO1xuaW1wb3J0IHsgRG9ja2VyVm9sdW1lcyB9IGZyb20gJy4vZ2l0bGFiLXJ1bm5lci1pbnRlcmZhY2VzJztcblxuLyoqXG4gKiBHaXRsYWJSdW5uZXJBdXRvc2NhbGluZyBQcm9wcy5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBHaXRsYWJSdW5uZXJBdXRvc2NhbGluZ1Byb3BzIHtcbiAgLyoqXG4gICAqIEdpdGxhYiB0b2tlbi5cbiAgICpcbiAgICogQGV4YW1wbGVcbiAgICogbmV3IEdpdGxhYlJ1bm5lckF1dG9zY2FsaW5nKHN0YWNrLCAncnVubmVyJywgeyBnaXRsYWJUb2tlbjogJ0dJVExBQl9UT0tFTicgfSk7XG4gICAqL1xuICByZWFkb25seSBnaXRsYWJUb2tlbjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBJbWFnZSBVUkwgb2YgR2l0bGFiIFJ1bm5lci5cbiAgICpcbiAgICogQGV4YW1wbGVcbiAgICogbmV3IEdpdGxhYlJ1bm5lckF1dG9zY2FsaW5nKHN0YWNrLCAncnVubmVyJywgeyBnaXRsYWJUb2tlbjogJ0dJVExBQl9UT0tFTicsIGdpdGxhYlJ1bm5lckltYWdlOiAnZ2l0bGFiL2dpdGxhYi1ydW5uZXI6YWxwaW5lJyB9KTtcbiAgICpcbiAgICogQGRlZmF1bHQgcHVibGljLmVjci5hd3MvZ2l0bGFiL2dpdGxhYi1ydW5uZXI6bGF0ZXN0XG4gICAqXG4gICAqL1xuICByZWFkb25seSBnaXRsYWJSdW5uZXJJbWFnZT86IHN0cmluZztcblxuICAvKipcbiAgICogUnVubmVyIGRlZmF1bHQgRUMyIGluc3RhbmNlIHR5cGUuXG4gICAqXG4gICAqIEBleGFtcGxlXG4gICAqIG5ldyBHaXRsYWJSdW5uZXJBdXRvc2NhbGluZyhzdGFjaywgJ3J1bm5lcicsIHsgZ2l0bGFiVG9rZW46ICdHSVRMQUJfVE9LRU4nLCBpbnN0YW5jZVR5cGU6ICd0My5zbWFsbCcgfSk7XG4gICAqXG4gICAqIEBkZWZhdWx0IC0gdDMubWljcm9cbiAgICpcbiAgICovXG4gIHJlYWRvbmx5IGluc3RhbmNlVHlwZT86IHN0cmluZztcblxuICAvKipcbiAgICogVlBDIGZvciB0aGUgR2l0bGFiIFJ1bm5lciAuXG4gICAqXG4gICAqIEBleGFtcGxlXG4gICAqIGNvbnN0IG5ld1ZwYyA9IG5ldyBWcGMoc3RhY2ssICdOZXdWUEMnLCB7XG4gICAqICAgY2lkcjogJzEwLjEuMC4wLzE2JyxcbiAgICogICBtYXhBenM6IDIsXG4gICAqICAgc3VibmV0Q29uZmlndXJhdGlvbjogW3tcbiAgICogICAgIGNpZHJNYXNrOiAyNixcbiAgICogICAgIG5hbWU6ICdSdW5uZXJWUEMnLFxuICAgKiAgICAgc3VibmV0VHlwZTogU3VibmV0VHlwZS5QVUJMSUMsXG4gICAqICAgfV0sXG4gICAqICAgbmF0R2F0ZXdheXM6IDAsXG4gICAqIH0pO1xuICAgKlxuICAgKiBuZXcgR2l0bGFiUnVubmVyQXV0b3NjYWxpbmcoc3RhY2ssICdydW5uZXInLCB7IGdpdGxhYlRva2VuOiAnR0lUTEFCX1RPS0VOJywgdnBjOiBuZXdWcGMgfSk7XG4gICAqXG4gICAqIEBkZWZhdWx0IC0gQSBuZXcgVlBDIHdpbGwgYmUgY3JlYXRlZC5cbiAgICpcbiAgICovXG4gIHJlYWRvbmx5IHZwYz86IGVjMi5JVnBjO1xuXG4gIC8qKlxuICAgKiBJQU0gcm9sZSBmb3IgdGhlIEdpdGxhYiBSdW5uZXIgSW5zdGFuY2UgLlxuICAgKlxuICAgKiBAZXhhbXBsZVxuICAgKiBjb25zdCByb2xlID0gbmV3IFJvbGUoc3RhY2ssICdydW5uZXItcm9sZScsIHtcbiAgICogICBhc3N1bWVkQnk6IG5ldyBTZXJ2aWNlUHJpbmNpcGFsKCdlYzIuYW1hem9uYXdzLmNvbScpLFxuICAgKiAgIGRlc2NyaXB0aW9uOiAnRm9yIEdpdGxhYiBSdW5uZXIgVGVzdCBSb2xlJyxcbiAgICogICByb2xlTmFtZTogJ1J1bm5lci1Sb2xlJyxcbiAgICogfSk7XG4gICAqXG4gICAqIG5ldyBHaXRsYWJSdW5uZXJBdXRvc2NhbGluZyhzdGFjaywgJ3J1bm5lcicsIHsgZ2l0bGFiVG9rZW46ICdHSVRMQUJfVE9LRU4nLCBpbnN0YW5jZVJvbGU6IHJvbGUgfSk7XG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbmV3IFJvbGUgZm9yIEdpdGxhYiBSdW5uZXIgSW5zdGFuY2UgLCBhdHRhY2ggQW1hem9uU1NNTWFuYWdlZEluc3RhbmNlQ29yZSBQb2xpY3kgLlxuICAgKlxuICAgKi9cbiAgcmVhZG9ubHkgaW5zdGFuY2VSb2xlPzogaWFtLklSb2xlO1xuXG4gIC8qKlxuICAgKiBSdW4gd29ya2VyIG5vZGVzIGFzIEVDMiBTcG90XG4gICAqXG4gICAqIEBkZWZhdWx0IC0gZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IHNwb3RJbnN0YW5jZT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIE1pbmltdW0gY2FwYWNpdHkgbGltaXQgZm9yIGF1dG9zY2FsaW5nIGdyb3VwLlxuICAgKlxuICAgKiBAZXhhbXBsZVxuICAgKiBuZXcgR2l0bGFiUnVubmVyQXV0b3NjYWxpbmcoc3RhY2ssICdydW5uZXInLCB7IGdpdGxhYlRva2VuOiAnR0lUTEFCX1RPS0VOJywgbWluQ2FwYWNpdHk6IDIgfSk7XG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbWluQ2FwYWNpdHk6IDFcbiAgICpcbiAgICovXG4gIHJlYWRvbmx5IG1pbkNhcGFjaXR5PzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBNYXhpbXVtIGNhcGFjaXR5IGxpbWl0IGZvciBhdXRvc2NhbGluZyBncm91cC5cbiAgICpcbiAgICogQGV4YW1wbGVcbiAgICogbmV3IEdpdGxhYlJ1bm5lckF1dG9zY2FsaW5nKHN0YWNrLCAncnVubmVyJywgeyBnaXRsYWJUb2tlbjogJ0dJVExBQl9UT0tFTicsIG1heENhcGFjaXR5OiA0IH0pO1xuICAgKlxuICAgKiBAZGVmYXVsdCAtIGRlc2lyZWRDYXBhY2l0eVxuICAgKlxuICAgKi9cbiAgcmVhZG9ubHkgbWF4Q2FwYWNpdHk/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIERlc2lyZWQgY2FwYWNpdHkgbGltaXQgZm9yIGF1dG9zY2FsaW5nIGdyb3VwLlxuICAgKlxuICAgKiBAZXhhbXBsZVxuICAgKiBuZXcgR2l0bGFiUnVubmVyQXV0b3NjYWxpbmcoc3RhY2ssICdydW5uZXInLCB7IGdpdGxhYlRva2VuOiAnR0lUTEFCX1RPS0VOJywgZGVzaXJlZENhcGFjaXR5OiAyIH0pO1xuICAgKlxuICAgKiBAZGVmYXVsdCAtIG1pbkNhcGFjaXR5LCBhbmQgbGVhdmUgdW5jaGFuZ2VkIGR1cmluZyBkZXBsb3ltZW50XG4gICAqXG4gICAqL1xuICByZWFkb25seSBkZXNpcmVkQ2FwYWNpdHk/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIHRhZ3MgZm9yIHRoZSBydW5uZXJcbiAgICpcbiAgICogQGRlZmF1bHQgLSBbJ3J1bm5lcicsICdnaXRsYWInLCAnYXdzY2RrJ11cbiAgICovXG4gIHJlYWRvbmx5IHRhZ3M/OiBzdHJpbmdbXTtcblxuICAvKipcbiAgICogR2l0bGFiIFJ1bm5lciByZWdpc3RlciB1cmwgLlxuICAgKlxuICAgKiBAZXhhbXBsZVxuICAgKiBjb25zdCBydW5uZXIgPSBuZXcgR2l0bGFiUnVubmVyQXV0b3NjYWxpbmcoc3RhY2ssICdydW5uZXInLCB7IGdpdGxhYlRva2VuOiAnR0lUTEFCX1RPS0VOJyxnaXRsYWJVcmw6ICdodHRwczovL2dpdGxhYi5jb20vJ30pO1xuICAgKlxuICAgKiBAZGVmYXVsdCAtIGh0dHBzOi8vZ2l0bGFiLmNvbS8gLCBUaGUgdHJhaWxpbmcgc2xhc2ggaXMgbWFuZGF0b3J5LlxuICAgKlxuICAgKi9cbiAgcmVhZG9ubHkgZ2l0bGFiVXJsPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBHaXRsYWIgUnVubmVyIGluc3RhbmNlIEVCUyBzaXplIC5cbiAgICpcbiAgICogQGV4YW1wbGVcbiAgICogY29uc3QgcnVubmVyID0gbmV3IEdpdGxhYlJ1bm5lckF1dG9zY2FsaW5nKHN0YWNrLCAncnVubmVyJywgeyBnaXRsYWJUb2tlbjogJ0dJVExBQl9UT0tFTicsIGVic1NpemU6IDEwMH0pO1xuICAgKlxuICAgKiBAZGVmYXVsdCAtIGVic1NpemU9NjBcbiAgICpcbiAgICovXG4gIHJlYWRvbmx5IGVic1NpemU/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFZQQyBzdWJuZXRcbiAgICpcbiAgICogQGV4YW1wbGVcbiAgICogY29uc3QgdnBjID0gbmV3IFZwYyhzdGFjaywgJ25hdCcsIHtcbiAgICogbmF0R2F0ZXdheXM6IDEsXG4gICAqIG1heEF6czogMixcbiAgICogfSk7XG4gICAqIGNvbnN0IHJ1bm5lciA9IG5ldyBHaXRsYWJSdW5uZXJBdXRvc2NhbGluZyhzdGFjaywgJ3Rlc3RpbmcnLCB7XG4gICAqICAgZ2l0bGFiVG9rZW46ICdHSVRMQUJfVE9LRU4nLFxuICAgKiAgIGluc3RhbmNlVHlwZTogJ3QzLmxhcmdlJyxcbiAgICogICBpbnN0YW5jZVJvbGU6IHJvbGUsXG4gICAqICAgZWJzU2l6ZTogMTAwLFxuICAgKiAgIHZwYzogdnBjLFxuICAgKiAgIHZwY1N1Ym5ldDoge1xuICAgKiAgICAgc3VibmV0VHlwZTogU3VibmV0VHlwZS5QVUJMSUMsXG4gICAqICAgfSxcbiAgICogfSk7XG4gICAqXG4gICAqIEBkZWZhdWx0IC0gcHJpdmF0ZSBzdWJuZXRcbiAgICovXG4gIHJlYWRvbmx5IHZwY1N1Ym5ldD86IGVjMi5TdWJuZXRTZWxlY3Rpb247XG5cbiAgLyoqXG4gICAqIGFkZCBhbm90aGVyIEdpdGxhYiBDb250YWluZXIgUnVubmVyIERvY2tlciBWb2x1bWVzIFBhdGggYXQgam9iIHJ1bm5lciBydW50aW1lLlxuICAgKlxuICAgKiBtb3JlIGRldGFpbCBzZWUgaHR0cHM6Ly9kb2NzLmdpdGxhYi5jb20vcnVubmVyL2NvbmZpZ3VyYXRpb24vYWR2YW5jZWQtY29uZmlndXJhdGlvbi5odG1sI3RoZS1ydW5uZXJzZG9ja2VyLXNlY3Rpb25cbiAgICpcbiAgICogQGRlZmF1bHQgLSBhbHJlYWR5IG1vdW50IFwiL3Zhci9ydW4vZG9ja2VyLnNvY2s6L3Zhci9ydW4vZG9ja2VyLnNvY2tcIlxuICAgKlxuICAgKiBAZXhhbXBsZVxuICAgKiBkb2NrZXJWb2x1bWVzOiBbXG4gICAqICAge1xuICAgKiAgICAgaG9zdFBhdGg6ICcvdG1wL2NhY2hlJyxcbiAgICogICAgIGNvbnRhaW5lclBhdGg6ICcvdG1wL2NhY2hlJyxcbiAgICogICB9LFxuICAgKiBdLFxuICAgKi9cbiAgcmVhZG9ubHkgZG9ja2VyVm9sdW1lcz86IERvY2tlclZvbHVtZXNbXTtcblxuICAvKipcbiAgICogUGFyYW1ldGVycyBvZiBwdXRfbWV0cmljX2FsYXJtIGZ1bmN0aW9uXG4gICAqXG4gICAqIGh0dHBzOi8vYm90bzMuYW1hem9uYXdzLmNvbS92MS9kb2N1bWVudGF0aW9uL2FwaS9sYXRlc3QvcmVmZXJlbmNlL3NlcnZpY2VzL2Nsb3Vkd2F0Y2guaHRtbCNDbG91ZFdhdGNoLkNsaWVudC5wdXRfbWV0cmljX2FsYXJtXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gW3tcbiAgICogICAgIEFsYXJtTmFtZTogJ0dpdGxhYlJ1bm5lckRpc2tVc2FnZScsXG4gICAqICAgICBNZXRyaWNOYW1lOiAnZGlza191c2VkX3BlcmNlbnQnLFxuICAgKiB9XVxuICAgKlxuICAgKi9cbiAgcmVhZG9ubHkgYWxhcm1zPzogb2JqZWN0W107XG59XG5cbi8qKlxuICogR2l0bGFiUnVubmVyQXV0b3NjYWxpbmcgQ29uc3RydWN0IGZvciBjcmVhdGUgQXV0b3NjYWxpbmcgR2l0bGFiIFJ1bm5lci5cbiAqL1xuZXhwb3J0IGNsYXNzIEdpdGxhYlJ1bm5lckF1dG9zY2FsaW5nIGV4dGVuZHMgY2RrLkNvbnN0cnVjdCB7XG4gIC8qKlxuICAgKiBUaGUgSUFNIHJvbGUgYXNzdW1lZCBieSB0aGUgUnVubmVyIGluc3RhbmNlLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGluc3RhbmNlUm9sZTogaWFtLklSb2xlO1xuXG4gIC8qKlxuICAgKiBUaGlzIHJlcHJlc2VudHMgYSBSdW5uZXIgQXV0byBTY2FsaW5nIEdyb3VwXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgYXV0b3NjYWxpbmdHcm91cDogYXNnLkF1dG9TY2FsaW5nR3JvdXA7XG5cbiAgLyoqXG4gICAqIFRoZSBFQzIgcnVubmVyJ3MgVlBDLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHZwYzogZWMyLklWcGM7XG5cbiAgLyoqXG4gICAqIFRoZSBFQzIgcnVubmVyJ3MgZGVmYXVsdCBTZWN1cml0eUdyb3VwLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHNlY3VyaXR5R3JvdXA6IGVjMi5JU2VjdXJpdHlHcm91cDtcblxuICAvKipcbiAgICogVGhlIFNOUyB0b3BpYyB0byBzdXNjcmliZSBhbGFybXMgZm9yIEVDMiBydW5uZXIncyBtZXRyaWNzLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHRvcGljQWxhcm06IHNucy5JVG9waWM7XG5cblxuICBjb25zdHJ1Y3RvcihzY29wZTogY2RrLkNvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEdpdGxhYlJ1bm5lckF1dG9zY2FsaW5nUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuICAgIGNvbnN0IGRlZmF1bHRQcm9wcyA9IHtcbiAgICAgIGluc3RhbmNlVHlwZTogJ3QzLm1pY3JvJyxcbiAgICAgIHRhZ3M6IFsnZ2l0bGFiJywgJ2F3c2NkaycsICdydW5uZXInXSxcbiAgICAgIGdpdGxhYlVybDogJ2h0dHBzOi8vZ2l0bGFiLmNvbS8nLFxuICAgICAgZ2l0bGFiUnVubmVySW1hZ2U6ICdwdWJsaWMuZWNyLmF3cy9naXRsYWIvZ2l0bGFiLXJ1bm5lcjpsYXRlc3QnLFxuICAgICAgYWxhcm1zOiBbXG4gICAgICAgIHtcbiAgICAgICAgICBBbGFybU5hbWU6ICdHaXRsYWJSdW5uZXJEaXNrVXNhZ2UnLFxuICAgICAgICAgIE1ldHJpY05hbWU6ICdkaXNrX3VzZWRfcGVyY2VudCcsXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgIH07XG4gICAgY29uc3QgcnVubmVyUHJvcHMgPSB7IC4uLmRlZmF1bHRQcm9wcywgLi4ucHJvcHMgfTtcblxuICAgIGNvbnN0IGFzc2V0ID0gbmV3IGFzc2V0cy5Bc3NldCh0aGlzLCAnR2l0bGFiUnVubmVyVXNlckRhdGFBc3NldCcsIHtcbiAgICAgIHBhdGg6IHBhdGguam9pbihfX2Rpcm5hbWUsICcuLi9hc3NldHMvdXNlcmRhdGEvYW1hem9uLWNsb3Vkd2F0Y2gtYWdlbnQuanNvbicpLFxuICAgIH0pO1xuXG4gICAgY29uc3QgdXNlckRhdGEgPSBlYzIuVXNlckRhdGEuZm9yTGludXgoKTtcbiAgICB1c2VyRGF0YS5hZGRTM0Rvd25sb2FkQ29tbWFuZCh7XG4gICAgICBidWNrZXQ6IGFzc2V0LmJ1Y2tldCxcbiAgICAgIGJ1Y2tldEtleTogYXNzZXQuczNPYmplY3RLZXksXG4gICAgICBsb2NhbEZpbGU6ICcvb3B0L2F3cy9hbWF6b24tY2xvdWR3YXRjaC1hZ2VudC9ldGMvYW1hem9uLWNsb3Vkd2F0Y2gtYWdlbnQuanNvbicsXG4gICAgfSk7XG4gICAgdXNlckRhdGEuYWRkQ29tbWFuZHMoLi4udGhpcy5jcmVhdGVVc2VyRGF0YShydW5uZXJQcm9wcykpO1xuXG4gICAgdGhpcy5pbnN0YW5jZVJvbGUgPVxuICAgICAgcnVubmVyUHJvcHMuaW5zdGFuY2VSb2xlID8/XG4gICAgICBuZXcgaWFtLlJvbGUodGhpcywgJ0dpdGxhYlJ1bm5lckluc3RhbmNlUm9sZScsIHtcbiAgICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2VjMi5hbWF6b25hd3MuY29tJyksXG4gICAgICAgIGRlc2NyaXB0aW9uOiAnRm9yIEVDMiBJbnN0YW5jZSAoR2l0bGFiIFJ1bm5lcikgUm9sZScsXG4gICAgICAgIG1hbmFnZWRQb2xpY2llczogW1xuICAgICAgICAgIGlhbS5NYW5hZ2VkUG9saWN5LmZyb21Bd3NNYW5hZ2VkUG9saWN5TmFtZSgnQW1hem9uU1NNTWFuYWdlZEluc3RhbmNlQ29yZScpLFxuICAgICAgICAgIGlhbS5NYW5hZ2VkUG9saWN5LmZyb21Bd3NNYW5hZ2VkUG9saWN5TmFtZSgnQ2xvdWRXYXRjaEFnZW50U2VydmVyUG9saWN5JyksXG4gICAgICAgICAgaWFtLk1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKCdBbWF6b25TM1JlYWRPbmx5QWNjZXNzJyksXG4gICAgICAgIF0sXG4gICAgICB9KTtcblxuICAgIHRoaXMudnBjID0gcnVubmVyUHJvcHMudnBjID8/IG5ldyBlYzIuVnBjKHRoaXMsICdWUEMnKTtcblxuICAgIHRoaXMuc2VjdXJpdHlHcm91cCA9IG5ldyBlYzIuU2VjdXJpdHlHcm91cCh0aGlzLCAnR2l0bGFiUnVubmVyU2VjdXJpdHlHcm91cCcsIHtcbiAgICAgIHZwYzogdGhpcy52cGMsXG4gICAgfSk7XG4gICAgY29uc3QgaW5zdGFuY2VQcm9maWxlID0gbmV3IGlhbS5DZm5JbnN0YW5jZVByb2ZpbGUodGhpcywgJ0luc3RhbmNlUHJvZmlsZScsIHtcbiAgICAgIHJvbGVzOiBbdGhpcy5pbnN0YW5jZVJvbGUucm9sZU5hbWVdLFxuICAgIH0pO1xuICAgIGNvbnN0IGx0ID0gbmV3IGVjMi5DZm5MYXVuY2hUZW1wbGF0ZSh0aGlzLCAnR2l0bGFiUnVubmVyTGF1bmNoVGVtcGxhdGUnLCB7XG4gICAgICBsYXVuY2hUZW1wbGF0ZURhdGE6IHtcbiAgICAgICAgaW1hZ2VJZDogZWMyLk1hY2hpbmVJbWFnZS5sYXRlc3RBbWF6b25MaW51eCh7XG4gICAgICAgICAgZ2VuZXJhdGlvbjogZWMyLkFtYXpvbkxpbnV4R2VuZXJhdGlvbi5BTUFaT05fTElOVVhfMixcbiAgICAgICAgfSkuZ2V0SW1hZ2UodGhpcykuaW1hZ2VJZCxcbiAgICAgICAgaW5zdGFuY2VUeXBlOiBydW5uZXJQcm9wcy5pbnN0YW5jZVR5cGUsXG4gICAgICAgIGluc3RhbmNlTWFya2V0T3B0aW9uczoge1xuICAgICAgICAgIG1hcmtldFR5cGU6IHJ1bm5lclByb3BzLnNwb3RJbnN0YW5jZSA/ICdzcG90JyA6IHVuZGVmaW5lZCxcbiAgICAgICAgICBzcG90T3B0aW9uczogcnVubmVyUHJvcHMuc3BvdEluc3RhbmNlID8ge1xuICAgICAgICAgICAgc3BvdEluc3RhbmNlVHlwZTogJ29uZS10aW1lJyxcbiAgICAgICAgICB9IDogdW5kZWZpbmVkLFxuICAgICAgICB9LFxuICAgICAgICB1c2VyRGF0YTogY2RrLkZuLmJhc2U2NCh1c2VyRGF0YS5yZW5kZXIoKSksXG4gICAgICAgIGJsb2NrRGV2aWNlTWFwcGluZ3M6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBkZXZpY2VOYW1lOiAnL2Rldi94dmRhJyxcbiAgICAgICAgICAgIGViczoge1xuICAgICAgICAgICAgICB2b2x1bWVTaXplOiBydW5uZXJQcm9wcy5lYnNTaXplID8/IDYwLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgICBpYW1JbnN0YW5jZVByb2ZpbGU6IHtcbiAgICAgICAgICBhcm46IGluc3RhbmNlUHJvZmlsZS5hdHRyQXJuLFxuICAgICAgICB9LFxuICAgICAgICBzZWN1cml0eUdyb3VwSWRzOiB0aGlzLnNlY3VyaXR5R3JvdXAuY29ubmVjdGlvbnMuc2VjdXJpdHlHcm91cHMubWFwKFxuICAgICAgICAgIChtKSA9PiBtLnNlY3VyaXR5R3JvdXBJZCxcbiAgICAgICAgKSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICB0aGlzLmF1dG9zY2FsaW5nR3JvdXAgPSBuZXcgYXNnLkF1dG9TY2FsaW5nR3JvdXAodGhpcywgJ0dpdGxhYlJ1bm5lckF1dG9zY2FsaW5nR3JvdXAnLCB7XG4gICAgICBpbnN0YW5jZVR5cGU6IG5ldyBlYzIuSW5zdGFuY2VUeXBlKHJ1bm5lclByb3BzLmluc3RhbmNlVHlwZSksXG4gICAgICBhdXRvU2NhbGluZ0dyb3VwTmFtZTogYEdpdGxhYiBSdW5uZXJzICgke3J1bm5lclByb3BzLmluc3RhbmNlVHlwZX0pYCxcbiAgICAgIHZwYzogdGhpcy52cGMsXG4gICAgICB2cGNTdWJuZXRzOiBydW5uZXJQcm9wcy52cGNTdWJuZXQsXG4gICAgICBtYWNoaW5lSW1hZ2U6IGVjMi5NYWNoaW5lSW1hZ2UubGF0ZXN0QW1hem9uTGludXgoe1xuICAgICAgICBnZW5lcmF0aW9uOiBlYzIuQW1hem9uTGludXhHZW5lcmF0aW9uLkFNQVpPTl9MSU5VWF8yLFxuICAgICAgfSksXG4gICAgICBtaW5DYXBhY2l0eTogcnVubmVyUHJvcHMubWluQ2FwYWNpdHksXG4gICAgICBtYXhDYXBhY2l0eTogcnVubmVyUHJvcHMubWF4Q2FwYWNpdHksXG4gICAgICBkZXNpcmVkQ2FwYWNpdHk6IHJ1bm5lclByb3BzLmRlc2lyZWRDYXBhY2l0eSxcbiAgICB9KTtcblxuICAgIGNvbnN0IGNmbkFzZyA9IHRoaXMuYXV0b3NjYWxpbmdHcm91cC5ub2RlLnRyeUZpbmRDaGlsZCgnQVNHJykgYXMgYXNnLkNmbkF1dG9TY2FsaW5nR3JvdXA7XG4gICAgY2ZuQXNnLmFkZFByb3BlcnR5RGVsZXRpb25PdmVycmlkZSgnTGF1bmNoQ29uZmlndXJhdGlvbk5hbWUnKTtcbiAgICBjZm5Bc2cuYWRkUHJvcGVydHlPdmVycmlkZSgnTGF1bmNoVGVtcGxhdGUnLCB7XG4gICAgICBMYXVuY2hUZW1wbGF0ZUlkOiBsdC5yZWYsXG4gICAgICBWZXJzaW9uOiBsdC5hdHRyTGF0ZXN0VmVyc2lvbk51bWJlcixcbiAgICB9KTtcbiAgICB0aGlzLmF1dG9zY2FsaW5nR3JvdXAubm9kZS50cnlSZW1vdmVDaGlsZCgnTGF1bmNoQ29uZmlnJyk7XG5cbiAgICB0aGlzLnRvcGljQWxhcm0gPSBuZXcgc25zLlRvcGljKHRoaXMsICdHaXRsYWJSdW5uZXJBbGFybScpO1xuICAgIGNvbnN0IGFsYXJtcyA9IEpTT04uc3RyaW5naWZ5KHJ1bm5lclByb3BzLmFsYXJtcyk7XG5cbiAgICAvLyBQdXQgYWxhcm1zIGF0IGxhdW5jaFxuICAgIGNvbnN0IHJlZ2lzdGVyRnVuY3Rpb24gPSBuZXcgbGFtYmRhLkZ1bmN0aW9uKHRoaXMsICdHaXRsYWJSdW5uZXJSZWdpc3RlckZ1bmN0aW9uJywge1xuICAgICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICcuLi9hc3NldHMvZnVuY3Rpb25zJykpLFxuICAgICAgaGFuZGxlcjogJ2F1dG9zY2FsaW5nX2V2ZW50cy5yZWdpc3RlcicsXG4gICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5QWVRIT05fM184LFxuICAgICAgdGltZW91dDogY2RrLkR1cmF0aW9uLnNlY29uZHMoNjApLFxuICAgICAgbG9nUmV0ZW50aW9uOiBsb2dzLlJldGVudGlvbkRheXMuT05FX0RBWSxcbiAgICAgIGVudmlyb25tZW50OiB7XG4gICAgICAgIEFMQVJNUzogYWxhcm1zLFxuICAgICAgICBTTlNfVE9QSUNfQVJOOiB0aGlzLnRvcGljQWxhcm0udG9waWNBcm4sXG4gICAgICB9LFxuICAgIH0pO1xuICAgIHJlZ2lzdGVyRnVuY3Rpb24ucm9sZT8uYWRkVG9QcmluY2lwYWxQb2xpY3koXG4gICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICdjbG91ZHdhdGNoOlB1dE1ldHJpY0FsYXJtJyxcbiAgICAgICAgXSxcbiAgICAgIH0pLFxuICAgICk7XG5cbiAgICB0aGlzLmF1dG9zY2FsaW5nR3JvdXAuYWRkTGlmZWN5Y2xlSG9vaygnR2l0bGFiUnVubmVyTGlmZUN5Y2xlSG9va0xhdW5jaGluZycsIHtcbiAgICAgIGxpZmVjeWNsZVRyYW5zaXRpb246IGFzZy5MaWZlY3ljbGVUcmFuc2l0aW9uLklOU1RBTkNFX0xBVU5DSElORyxcbiAgICAgIG5vdGlmaWNhdGlvblRhcmdldDogbmV3IEZ1bmN0aW9uSG9vayhyZWdpc3RlckZ1bmN0aW9uKSxcbiAgICAgIGRlZmF1bHRSZXN1bHQ6IGFzZy5EZWZhdWx0UmVzdWx0LkNPTlRJTlVFLFxuICAgICAgaGVhcnRiZWF0VGltZW91dDogY2RrLkR1cmF0aW9uLnNlY29uZHMoNjApLFxuICAgIH0pO1xuXG4gICAgLy8gQWRkIGFuIGFsYXJtIGFjdGlvbiB0byB0ZXJtaW5hdGUgaW52YWxpZCBpbnN0YW5jZXNcbiAgICBjb25zdCBhbGFybUFjdGlvbiA9IG5ldyBsYW1iZGEuRnVuY3Rpb24odGhpcywgJ0dpdGxhYlJ1bm5lckFsYXJtQWN0aW9uJywge1xuICAgICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICcuLi9hc3NldHMvZnVuY3Rpb25zJykpLFxuICAgICAgaGFuZGxlcjogJ2F1dG9zY2FsaW5nX2V2ZW50cy5vbl9hbGFybScsXG4gICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5QWVRIT05fM184LFxuICAgICAgdGltZW91dDogY2RrLkR1cmF0aW9uLnNlY29uZHMoNjApLFxuICAgICAgbG9nUmV0ZW50aW9uOiBsb2dzLlJldGVudGlvbkRheXMuT05FX0RBWSxcbiAgICB9KTtcbiAgICBhbGFybUFjdGlvbi5yb2xlPy5hZGRUb1ByaW5jaXBhbFBvbGljeShcbiAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgICByZXNvdXJjZXM6IFsnKiddLFxuICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgJ2F1dG9zY2FsaW5nOlNldEluc3RhbmNlSGVhbHRoJyxcbiAgICAgICAgXSxcbiAgICAgIH0pLFxuICAgICk7XG4gICAgY29uc3QgYWxhcm1TdWJzY3JpcHRpb24gPSBuZXcgc3Vic2NyaXB0aW9ucy5MYW1iZGFTdWJzY3JpcHRpb24oYWxhcm1BY3Rpb24pO1xuICAgIHRoaXMudG9waWNBbGFybS5hZGRTdWJzY3JpcHRpb24oYWxhcm1TdWJzY3JpcHRpb24pO1xuXG4gICAgLy8gVW5yZWdpc3RlciBnaXRsYWIgcnVubmVycyBhbmQgcmVtb3ZlIGFsYXJtcyBvbiBpbnN0YW5jZSB0ZXJtaW5hdGlvbiBvciBDRm4gc3RhY2sgZGVsZXRpb25cbiAgICBjb25zdCB1bnJlZ2lzdGVyUm9sZSA9IG5ldyBpYW0uUm9sZSh0aGlzLCAnR2l0bGFiUnVubmVyVW5yZWdpc3RlclJvbGUnLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnbGFtYmRhLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgIGRlc2NyaXB0aW9uOiAnRm9yIEdpdGxhYiBSdW5uZXIgVW5yZWdpc3RlcmluZyBGdW5jdGlvbiBSb2xlJyxcbiAgICAgIG1hbmFnZWRQb2xpY2llczogW1xuICAgICAgICBpYW0uTWFuYWdlZFBvbGljeS5mcm9tQXdzTWFuYWdlZFBvbGljeU5hbWUoJ3NlcnZpY2Utcm9sZS9BV1NMYW1iZGFCYXNpY0V4ZWN1dGlvblJvbGUnKSxcbiAgICAgIF0sXG4gICAgfSk7XG4gICAgdW5yZWdpc3RlclJvbGUuYWRkVG9QcmluY2lwYWxQb2xpY3koXG4gICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICdzc206U2VuZENvbW1hbmQnLFxuICAgICAgICAgICdhdXRvc2NhbGluZzpEZXNjcmliZUF1dG9TY2FsaW5nR3JvdXBzJyxcbiAgICAgICAgICAnY2xvdWR3YXRjaDpEZWxldGVBbGFybXMnLFxuICAgICAgICBdLFxuICAgICAgfSksXG4gICAgKTtcblxuICAgIGNvbnN0IHVucmVnaXN0ZXJGdW5jdGlvbiA9IG5ldyBsYW1iZGEuRnVuY3Rpb24odGhpcywgJ0dpdGxhYlJ1bm5lclVucmVnaXN0ZXJGdW5jdGlvbicsIHtcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChwYXRoLmpvaW4oX19kaXJuYW1lLCAnLi4vYXNzZXRzL2Z1bmN0aW9ucycpKSxcbiAgICAgIGhhbmRsZXI6ICdhdXRvc2NhbGluZ19ldmVudHMudW5yZWdpc3RlcicsXG4gICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5QWVRIT05fM184LFxuICAgICAgdGltZW91dDogY2RrLkR1cmF0aW9uLnNlY29uZHMoNjApLFxuICAgICAgcm9sZTogdW5yZWdpc3RlclJvbGUsXG4gICAgICBsb2dSZXRlbnRpb246IGxvZ3MuUmV0ZW50aW9uRGF5cy5PTkVfREFZLFxuICAgICAgZW52aXJvbm1lbnQ6IHtcbiAgICAgICAgQUxBUk1TOiBhbGFybXMsXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgdGhpcy5hdXRvc2NhbGluZ0dyb3VwLmFkZExpZmVjeWNsZUhvb2soJ0dpdGxhYlJ1bm5lckxpZmVDeWNsZUhvb2tUZXJtaW5hdGluZycsIHtcbiAgICAgIGxpZmVjeWNsZVRyYW5zaXRpb246IGFzZy5MaWZlY3ljbGVUcmFuc2l0aW9uLklOU1RBTkNFX1RFUk1JTkFUSU5HLFxuICAgICAgbm90aWZpY2F0aW9uVGFyZ2V0OiBuZXcgRnVuY3Rpb25Ib29rKHVucmVnaXN0ZXJGdW5jdGlvbiksXG4gICAgICBkZWZhdWx0UmVzdWx0OiBhc2cuRGVmYXVsdFJlc3VsdC5DT05USU5VRSxcbiAgICAgIGhlYXJ0YmVhdFRpbWVvdXQ6IGNkay5EdXJhdGlvbi5zZWNvbmRzKDYwKSxcbiAgICB9KTtcblxuICAgIGNvbnN0IHVucmVnaXN0ZXJDdXN0b21SZXNvdXJjZSA9IG5ldyBsYW1iZGEuRnVuY3Rpb24odGhpcywgJ0dpdGxhYlJ1bm5lclVucmVnaXN0ZXJDdXN0b21SZXNvdXJjZScsIHtcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChwYXRoLmpvaW4oX19kaXJuYW1lLCAnLi4vYXNzZXRzL2Z1bmN0aW9ucycpKSxcbiAgICAgIGhhbmRsZXI6ICdhdXRvc2NhbGluZ19ldmVudHMub25fZXZlbnQnLFxuICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuUFlUSE9OXzNfOCxcbiAgICAgIHJvbGU6IHVucmVnaXN0ZXJSb2xlLFxuICAgICAgbG9nUmV0ZW50aW9uOiBsb2dzLlJldGVudGlvbkRheXMuT05FX0RBWSxcbiAgICAgIGVudmlyb25tZW50OiB7XG4gICAgICAgIEFMQVJNUzogYWxhcm1zLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIGNvbnN0IHVucmVnaXN0ZXJQcm92aWRlciA9IG5ldyBjci5Qcm92aWRlcih0aGlzLCAnR2l0bGFiUnVubmVyVW5yZWdpc3RlclByb3ZpZGVyJywge1xuICAgICAgb25FdmVudEhhbmRsZXI6IHVucmVnaXN0ZXJDdXN0b21SZXNvdXJjZSxcbiAgICB9KTtcblxuICAgIGNvbnN0IGN1c3RvbVJlc291cmNlID0gbmV3IGNkay5DdXN0b21SZXNvdXJjZSh0aGlzLCAnR2l0bGFiUnVubmVyQ3VzdG9tUmVzb3VyY2UnLCB7XG4gICAgICBzZXJ2aWNlVG9rZW46IHVucmVnaXN0ZXJQcm92aWRlci5zZXJ2aWNlVG9rZW4sXG4gICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgIEF1dG9TY2FsaW5nR3JvdXBOYW1lczogW3RoaXMuYXV0b3NjYWxpbmdHcm91cC5hdXRvU2NhbGluZ0dyb3VwTmFtZV0sXG4gICAgICB9LFxuICAgIH0pO1xuICAgIGN1c3RvbVJlc291cmNlLm5vZGUuYWRkRGVwZW5kZW5jeSh1bnJlZ2lzdGVyUHJvdmlkZXIpO1xuXG4gICAgbmV3IGNkay5DZm5PdXRwdXQodGhpcywgJ0dpdGxhYlJ1bm5lckF1dG9TY2FsaW5nR3JvdXBBcm4nLCB7XG4gICAgICB2YWx1ZTogdGhpcy5hdXRvc2NhbGluZ0dyb3VwLmF1dG9TY2FsaW5nR3JvdXBBcm4sXG4gICAgfSk7XG4gIH1cblxuICBwcml2YXRlIGRvY2tlclZvbHVtZXNMaXN0KGRvY2tlclZvbHVtZTogRG9ja2VyVm9sdW1lc1tdIHwgdW5kZWZpbmVkKTogc3RyaW5nIHtcbiAgICBsZXQgdGVtcFN0cmluZzogc3RyaW5nID0gJy0tZG9ja2VyLXZvbHVtZXMgXCIvdmFyL3J1bi9kb2NrZXIuc29jazovdmFyL3J1bi9kb2NrZXIuc29ja1wiJztcbiAgICBpZiAoZG9ja2VyVm9sdW1lKSB7XG4gICAgICBsZXQgdGVtcExpc3Q6IHN0cmluZ1tdID0gW107XG4gICAgICBkb2NrZXJWb2x1bWUuZm9yRWFjaChlID0+IHtcbiAgICAgICAgdGVtcExpc3QucHVzaChgXCIke2UuaG9zdFBhdGh9OiR7ZS5jb250YWluZXJQYXRofVwiYCk7XG4gICAgICB9KTtcbiAgICAgIHRlbXBMaXN0LmZvckVhY2goZSA9PiB7XG4gICAgICAgIHRlbXBTdHJpbmcgPSBgJHt0ZW1wU3RyaW5nfSAtLWRvY2tlci12b2x1bWVzICR7ZX1gO1xuICAgICAgfSk7XG4gICAgfVxuICAgIHJldHVybiB0ZW1wU3RyaW5nO1xuICB9XG4gIC8qKlxuICAgKiBAcGFyYW0gcHJvcHNcbiAgICogQHJldHVybnMgQXJyYXkuXG4gICAqL1xuICBwdWJsaWMgY3JlYXRlVXNlckRhdGEocHJvcHM6IEdpdGxhYlJ1bm5lckF1dG9zY2FsaW5nUHJvcHMpOiBzdHJpbmdbXSB7XG4gICAgcmV0dXJuIFtcbiAgICAgICd5dW0gdXBkYXRlIC15JyxcbiAgICAgICdzbGVlcCAxNSAmJiBhbWF6b24tbGludXgtZXh0cmFzIGluc3RhbGwgZG9ja2VyICYmIHl1bSBpbnN0YWxsIC15IGFtYXpvbi1jbG91ZHdhdGNoLWFnZW50ICYmIHN5c3RlbWN0bCBzdGFydCBkb2NrZXIgJiYgdXNlcm1vZCAtYUcgZG9ja2VyIGVjMi11c2VyICYmIGNobW9kIDc3NyAvdmFyL3J1bi9kb2NrZXIuc29jaycsXG4gICAgICAnc3lzdGVtY3RsIHJlc3RhcnQgZG9ja2VyICYmIHN5c3RlbWN0bCBlbmFibGUgZG9ja2VyICYmIHN5c3RlbWN0bCBzdGFydCBhbWF6b24tY2xvdWR3YXRjaC1hZ2VudCAmJiBzeXN0ZW1jdGwgZW5hYmxlIGFtYXpvbi1jbG91ZHdhdGNoLWFnZW50JyxcbiAgICAgIGBkb2NrZXIgcnVuIC1kIC12IC9ob21lL2VjMi11c2VyLy5naXRsYWItcnVubmVyOi9ldGMvZ2l0bGFiLXJ1bm5lciAtdiAvdmFyL3J1bi9kb2NrZXIuc29jazovdmFyL3J1bi9kb2NrZXIuc29jayBcXFxuICAgICAgLS1uYW1lIGdpdGxhYi1ydW5uZXItcmVnaXN0ZXIgJHtwcm9wcy5naXRsYWJSdW5uZXJJbWFnZX0gcmVnaXN0ZXIgLS1ub24taW50ZXJhY3RpdmUgLS11cmwgJHtwcm9wcy5naXRsYWJVcmx9IC0tcmVnaXN0cmF0aW9uLXRva2VuICR7cHJvcHMuZ2l0bGFiVG9rZW59IFxcXG4gICAgICAtLWRvY2tlci1wdWxsLXBvbGljeSBpZi1ub3QtcHJlc2VudCAke3RoaXMuZG9ja2VyVm9sdW1lc0xpc3QocHJvcHM/LmRvY2tlclZvbHVtZXMpfSBcXFxuICAgICAgLS1leGVjdXRvciBkb2NrZXIgLS1kb2NrZXItaW1hZ2UgXCJhbHBpbmU6bGF0ZXN0XCIgLS1kZXNjcmlwdGlvbiBcIkEgUnVubmVyIG9uIEVDMiBJbnN0YW5jZSAoJHtwcm9wcy5pbnN0YW5jZVR5cGV9KVwiIFxcXG4gICAgICAtLXRhZy1saXN0IFwiJHtwcm9wcy50YWdzPy5qb2luKCcsJyl9XCIgLS1kb2NrZXItcHJpdmlsZWdlZGAsXG4gICAgICBgc2xlZXAgMiAmJiBkb2NrZXIgcnVuIC0tcmVzdGFydCBhbHdheXMgLWQgLXYgL2hvbWUvZWMyLXVzZXIvLmdpdGxhYi1ydW5uZXI6L2V0Yy9naXRsYWItcnVubmVyIC12IC92YXIvcnVuL2RvY2tlci5zb2NrOi92YXIvcnVuL2RvY2tlci5zb2NrIC0tbmFtZSBnaXRsYWItcnVubmVyICR7cHJvcHMuZ2l0bGFiUnVubmVySW1hZ2V9YCxcbiAgICBdO1xuICB9XG59XG4iXX0=