"""Utility functions for parsing Haskell."""
from logging import getLogger

logger = getLogger(__name__)


def lmap(f, xs):
    """Like map but returns a list instead of a generator."""
    return [f(x) for x in xs]


def apply(fs, t):
    """Weirdly similar to Haskell's ap for subscriptable types.

    Parameters
    ----------
    fs : iterable
        List of tuples which have a `callable` as second member.
    t : iterable
        Arguments for the `callable`.

    Returns
    -------
    applied : tuple
        Results of the application of the callables
        to the arguments in `t` with the same index.
    """
    return tuple(f[1](x) for f, x in zip(fs, t))


def strip_io(tp):
    """IO is somewhat disregarded in the FFI exports. IO CDouble
    looks the same as CDouble from Python's side. So we remove
    the monadic part from our type to process the rest.

    Parameters
    ----------
    tp : str
        Haskell type statement.

    Returns
    -------
    stripped : (str, str)
        Tuple of `'IO '` if there was an IO statement or
        the empty string if there was none and the rest of the type statement.
    """
    io = tp.find("IO ")
    if io < 0:
        return "", tp
    else:
        return "IO ", tp[io + 3 :]


def tuple_types(hs_type):
    """Extracts the types declarations inside a Haskell tuple type statement.

    Parameters
    ----------
    hs_type : str
        Haskell type statement for a tuple.

    Returns
    -------
    types : list(str)
        Haskell type statements inside the tuple.
    """
    match = lambda x: match_parens(hs_type, x)

    openp = hs_type.find("(")
    parens = list()

    while openp > -1:
        closep = match(openp)
        parens.append((openp, closep))

        openp = hs_type.find("(", parens[-1][-1])

    if parens:  # Pythas' autogenerated Tuples dissect like this
        types = [hs_type[start:end] for start, end in parens]
    else:  # User generated Tuples
        types = hs_type.split(" ")

    logger.debug("Tuple {} was found to contain types: {}".format(hs_type, types))
    return types


def match_parens(s, i):
    """Given a string and the index of the opening
    parenthesis returns the index of the closing one.

    Parameters
    ----------
    s : str
        The string to match parentheses on.
    i : int
        The initial index to start from.

    Returns
    -------
    closing : int
        Index of the next matching closing paranthesis.
    """
    x = 0
    for it in range(i, len(s)):
        c = s[it]
        if c == "(":
            x += 1

        elif c == ")":
            x -= 1

        if x == 0:
            getLogger(__name__).debug("Parens at {} close at {} in {}".format(i, it, s))
            return it
    else:
        getLogger(__name__).debug(
            "Cannot find match for parenthesis opening at {} in {}".format(i, s)
        )
        return len(s)


def parse_generator(f_llist, f_carray, f_tuple, f_string, f_default):
    """Parser generator for parsing Haskell type statements.

    Parameters
    ----------
    f_llist : callable
        Function to call in case of linked lists.
    f_carray : callable
        Function to call in case of arrays.
    f_tuple : callable
        Function to call in case of tuples.
    f_string : callable
        Function to call in case of string.
    f_default : callable
        Function to call in case of string.

    Returns
    -------
    parser : callable
        Function taking a string object with a Haskell type statement
        and parsing it using the appropriate function.
    """

    def parser(hs_type):
        """Parser for Haskell type statements.

        Parameters
        ----------
        hs_type : str
            Haskell type statement.

        See also
        --------
        parse_generator
        """
        ll = hs_type.find("CList ")
        arr = hs_type.find("CArray ")
        tup = hs_type.find("CTuple")
        st = hs_type.find("CWString")
        ## Linked List first
        if ll + 1 and (ll < arr or arr < 0) and (ll < tup or tup < 0):
            logger.debug("Found linked list at {} in {}".format(ll, hs_type))
            return f_llist(hs_type[ll + len("CList ") :])
        ## Array first
        elif arr + 1 and (arr < tup or tup < 0):
            logger.debug("Found array at {} in {}".format(arr, hs_type))
            return f_carray(hs_type[arr + len("CArray ") :])
        ## Tuple first
        elif tup + 1:
            logger.debug("Found tuple at {} in {}".format(tup, hs_type))
            return f_tuple(hs_type[tup + len("CTupleX ") :])
        ## String first
        elif st + 1:
            logger.debug("Found string at {} in {}".format(st, hs_type))
            return f_string(hs_type[st + len("CWString ") :])
        else:
            return f_default(hs_type)

    return parser


class TypeWarning(Warning):
    """Warning reminding the user that the types used by
    them are currently only under experimental support.

    See also
    --------
    Warning
    """

    pass
