# -*- coding: utf-8 -*-
"""
Lacework Vulnerability API wrapper.
"""

import json
import logging

logger = logging.getLogger(__name__)


class VulnerabilityAPI(object):
    """
    Lacework Vulnerability API.
    """

    def __init__(self, session):
        """
        Initializes the VulnerabilityAPI object.

        :param session: An instance of the HttpSession class

        :return VulnerabilityAPI object.
        """

        super(VulnerabilityAPI, self).__init__()

        self._session = session

    def get_container_evaluations_by_date(self,
                                          start_time=None,
                                          end_time=None):
        """
        A method to get a list of container vulnerability assessments for the specified date range.

        :param start_time: A "%Y-%m-%dT%H:%M:%SZ" structured timestamp to begin from.
        :param end_time: A "%Y-%m-%dT%H:%M:%S%Z" structured timestamp to end at.

        :return: response json
        """

        logger.info("Getting container evaluations from Lacework...")

        # Build the Host Vulnerabilities request URI
        api_uri = f"/api/v1/external/vulnerabilities/container/GetEvaluationsForDateRange?"

        if start_time and end_time:
            api_uri += f"&START_TIME={start_time}&END_TIME={end_time}"

        response = self._session.get(api_uri)

        return response.json()

    def get_container_vulnerabilities(self,
                                      image_digest=None,
                                      image_id=None,
                                      severity=None,
                                      fixable=None,
                                      start_time=None,
                                      end_time=None):
        """
        A method to get the last scan data of the specified container.

        :param image_digest: A string representing the container image digest for which to fetch vulnerabilities.
        :param image_id: A string representing the container image ID for which to fetch vulnerabilities.
        :param severity: A string representing the severity of vulnerabilities to fetch.
        :param fixable: A boolean which filters for fixable vulnerabilities.
        :param start_time: A "%Y-%m-%dT%H:%M:%S%z" structured timestamp to begin from.
        :param end_time: A "%Y-%m-%dT%H:%M:%S%z" structured timestamp to end at.

        :return: response json
        """

        logger.info("Getting container vulnerabilities from Lacework...")

        if image_digest:
            # Build the Container Vulnerability request URI
            api_uri = f"/api/v1/external/vulnerabilities/container/imageDigest/{image_digest}?"
        elif image_id:
            # Build the Container Vulnerability request URI
            api_uri = f"/api/v1/external/vulnerabilities/container/imageId/{image_id}?"
        else:
            logger.error("An Image Digest or Image ID must be specified.")
            exit()

        if fixable is not None:
            api_uri += f"&fixable={fixable}"

        if severity:
            api_uri += f"&severity={severity}"

        if start_time and end_time:
            api_uri += f"&StartTime={start_time}&EndTime={end_time}"

        response = self._session.get(api_uri)

        return response.json()

    def initiate_container_scan(self, registry, repository, tag):
        """
        A method to initiate a container vulnerability scan.

        :param registry: A string representing the container registry.
        :param repository: A string representing the container repository.
        :param tag: A string representing the container tag.

        :return: response json
        """

        logger.info("Initiating container vulnerability scan in Lacework...")

        # Build the Container Image Scan request URI
        api_uri = f"/api/v1/external/vulnerabilities/container/repository/images/scan"

        data = {
            "registry": registry,
            "repository": repository,
            "tag": tag
        }

        response = self._session.post(api_uri, data=data)

        return response.json()

    def get_container_scan_status(self, request_id, severity=None, fixable=None):
        """
        A method to get the status/results of a container vulnerability scan from Lacework.

        :param request_id: A string representing the request ID to be queried.
        :param severity: A string representing the severity of vulnerabilities to fetch.
        :param fixable: A boolean which filters for fixable vulnerabilities.

        :return: response json
        """

        logger.info("Getting container vulnerability scan status from Lacework...")

        # Build the Container Image Scan request URI
        api_uri = f"/api/v1/external/vulnerabilities/container/reqId/{request_id}?"

        if fixable is not None:
            api_uri += f"&fixable={fixable}"

        if severity:
            api_uri += f"&severity={severity}"

        response = self._session.get(api_uri)

        return response.json()

    def get_host_vulnerabilities(self,
                                 fixable=None,
                                 severity=None,
                                 start_time=None,
                                 end_time=None,
                                 cve=None):
        """
        A method to get the Host Vulnerabilities found by Lacework.

        :param fixable: A boolean which filters for fixable vulnerabilities.
        :param severity: A string representing the severity for which to filter returned results.
        :param start_time: A "%Y-%m-%dT%H:%M:%S%z" structured timestamp to begin from.
        :param end_time: A "%Y-%m-%dT%H:%M:%S%z" structured timestamp to end at.
        :param cve: A string representing the CVE ID for which to filter returned results.

        :return: response json
        """

        logger.info("Getting host vulnerabilities from Lacework...")

        # Build the Host Vulnerabilities request URI
        api_uri = f"/api/v1/external/vulnerabilities/host?"

        if fixable is not None:
            api_uri += f"&fixable={fixable}"

        if severity:
            api_uri += f"&severity={severity}"

        if start_time and end_time:
            api_uri += f"&StartTime={start_time}&EndTime={end_time}"

        if cve:
            api_uri += f"&vuln_id={cve}"

        response = self._session.get(api_uri)

        return response.json()

    def get_host_vulnerabilities_by_cve(self,
                                        cve,
                                        hostname=None,
                                        status=None):
        """
        A method to get the Host Vulnerabilities by CVE.

        :param cve: A string representing the CVE ID for which to filter returned results.
        :param hostname: A string representing a hostname for which to filter returned results.
        :param status: A string representing a status for which to filter results.

        :return: response json
        """

        logger.info("Getting host vulnerabilities by CVE from Lacework...")

        # Build the Host Vulnerabilities by CVE URI
        api_uri = f"/api/v1/external/vulnerabilities/host/cveId/{cve}?"

        if hostname:
            api_uri += f"&hostname={hostname}"

        if status:
            if status.capitalize() in ["New", "Active", "Fixed"]:
                api_uri += f"&status={status}"
            else:
                print(f"Invalid status parameter '{status}' provided.")
                return None

        response = self._session.get(api_uri)

        return response.json()

    def get_host_vulnerabilities_by_machine_id(self,
                                               machine,
                                               fixable=None,
                                               severity=None,
                                               start_time=None,
                                               end_time=None,
                                               cve=None):
        """
        A method to get the Host Vulnerabilities by Machine ID.

        :param machine: A string representing the Lacework Machine ID.
        :param fixable: A boolean which filters for fixable vulnerabilities.
        :param severity: A string representing a severity for which to filter returned results.
        :param start_time: A "%Y-%m-%dT%H:%M:%S%z" structured timestamp to begin from.
        :param end_time: A "%Y-%m-%dT%H:%M:%S%z" structured timestamp to end at.
        :param cve: A string representing the CVE ID for which to filter returned results.

        :return: response json
        """

        logger.info("Getting host vulnerabilities by machine ID from Lacework...")

        # Build the Host Vulnerabilities by Machine ID URI
        api_uri = f"/api/v1/external/vulnerabilities/host/machineId/{machine}?"

        if fixable is not None:
            api_uri += f"&fixable={fixable}"

        if severity:
            api_uri += f"&severity={severity}"

        if start_time and end_time:
            api_uri += f"&StartTime={start_time}&EndTime={end_time}"

        if cve:
            api_uri += f"&vuln_id={cve}"

        response = self._session.get(api_uri)

        return response.json()

    def initiate_package_scan(self, os=None, os_version=None, package=None, package_version=None):
        """
        A method to initiate a package vulnerability scan in Lacework.

        :param os: A comma-separated string or list representing the operating system(s) for which to initiate a scan.
        :param os_version: A comma-separated string or list representing the operating system version(s) for which to initiate a scan.
        :param package: A comma-separated string or list representing the software package(s) for which to initiate a scan.
        :param package_version: A comma-separated string or list representing the software package version(s) for which to initiate a scan.

        :return: response json
        """

        logger.info("Initiating package vulnerability scan in Lacework...")

        # Build the Host Vulnerabilities request URI
        api_uri = f"/api/v1/external/vulnerabilities/scan"

        # If strings were provided, make them lists
        if isinstance(os, str):
            os = os.split(",")
        if isinstance(os_version, str):
            os_version = os_version.split(",")
        if isinstance(package, str):
            package = package.split(",")
        if isinstance(package_version, str):
            package_version = package_version.split(",")

        packages = []

        # Expand all package queries
        for current_os in os:
            for current_os_version in os_version:
                for current_package in package:
                    for current_package_version in package_version:
                        packages.append({
                            "os": current_os,
                            "os_ver": current_os_version,
                            "pkg": current_package,
                            "pkg_ver": current_package_version
                        })

        data = {
            "os_pkg_info_list": packages
        }

        response = self._session.post(api_uri, data=data)

        return response.json()
