# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/00_learner.ipynb (unless otherwise specified).

__all__ = ['Config', 'energy_max', 'save_tmp', 'EnsembleLearner']

# Cell
import shutil, gc, joblib, json, numpy as np, pandas as pd
import torch, torch.nn as nn, torch.nn.functional as F
from dataclasses import dataclass, field, asdict
from pathlib import Path

from sklearn import svm
from sklearn.model_selection import KFold
from sklearn.pipeline import Pipeline
from sklearn.preprocessing import StandardScaler

from fastcore.basics import patch, GetAttr
from fastcore.foundation import add_docs
from fastprogress import progress_bar
from fastai import optimizer
from fastai.torch_core import TensorImage
from fastai.learner import Learner
from fastai.callback.tracker import SaveModelCallback
from fastai.data.core import DataLoaders
from fastai.data.transforms import get_image_files, get_files, Normalize
from fastai.vision.augment import Brightness

from .metrics import Dice_f1, Iou
from .losses import WeightedSoftmaxCrossEntropy
from .callbacks import ElasticDeformCallback
from .models import get_default_shapes
from .data import TileDataset, RandomTileDataset, _read_img, _read_msk, calculate_weights
from .utils import iou, plot_results, get_label_fn, calc_iterations, save_mask, save_unc
import deepflash2.tta as tta

# Cell
@dataclass
class Config:
    "Config class for settings."

    #Project
    proj_dir:str = 'deepflash2'

    #GT Estimation Settings
    staple_thres:float = 0.5
    staple_fval:int= 1
    mv_undec:int = 0

    #Train General Settings
    n:int = 4
    max_splits:int=5
    repo:str = 'matjesg/deepflash2'
    arch:str = 'unext50_deepflash2'
    pretrained:str = None
    random_state:int = 42

    #Train Data Settings
    c:int = 2
    il:bool = False

    #Train Settings
    lr:float = 0.005
    bs:int = 4
    wd:float = 0.001
    mpt:bool = False
    optim:str = 'ranger'
    n_iter:int = 1000
    stats:tuple = None

    #Train Validation Settings
    tta:bool = True

    #Train Data Augmentation
    light:float = 0.1
    flip:bool = True
    rot:int = 360
    def_grid:int = 150
    def_mag:int = 10
    bwf:int = 25
    bws:int = 10
    fds:int = 10
    fbr:float = 0.5

    #Pred Settings
    pred_tta:bool = True

    #OOD Settings
    kernel:str = 'rbf'
    nu:float = 0.01
    gamma:float = 0.01

    #Folder Structure
    gt_dir:str = 'GT_Estimation'
    train_dir:str = 'Training'
    pred_dir:str = 'Prediction'
    ens_dir:str = 'ensemble'
    val_dir:str = 'valid'

    @property
    def mw_kwargs(self):
        mw_vars = ['bwf', 'bws', 'fds', 'fbr']
        return dict(filter(lambda x: x[0] in mw_vars, self.__dict__.items()))

    @property
    def svm_kwargs(self):
        svm_vars = ['kernel', 'nu', 'gamma']
        return dict(filter(lambda x: x[0] in svm_vars, self.__dict__.items()))

    def save(self, path):
        'Save configuration to path'
        path = Path(path)
        with open(path.with_suffix('.json'), 'w') as config_file:
            json.dump(asdict(self), config_file)
        print(f'Saved current configuration to {path}.json')

    def load(self, path):
        'Load configuration from path'
        path = Path(path)
        try:
            with open(path) as config_file: c = json.load(config_file)
            if not Path(c['proj_dir']).is_dir(): c['proj_dir']='deepflash2'
            for k,v in c.items(): setattr(self, k, v)
            print(f'Successsfully loaded configuration from {path}')
        except:
            print('Error! Select valid config file (.json)')

# Cell
_optim_dict = {
    'ranger' : optimizer.ranger,
    'Adam' : optimizer.Adam,
    'RAdam' : optimizer.RAdam,
    'QHAdam' :optimizer.QHAdam,
    'Larc' : optimizer.Larc,
    'Lamb' : optimizer.Lamb,
    'SGD' : optimizer.SGD,
    'RMSProp' : optimizer.RMSProp,
}

# Cell
_archs = ["unext50_deepflash2", "unet_deepflash2",  "unet_falk2019", "unet_ronnberger2015"]

# Cell
_pretrained = ["new", "wue_cFOS", "wue_Parv", "wue_GFP", "wue_OPN3"]

# Cell
@patch
def apply_dropout(self:Learner):
    "If a module contains 'dropout', it will be switched to .train() mode."
    for m in self.model.modules():
        if isinstance(m, nn.Dropout):
            m.train()

# Cell
def energy_max(e, ks=20, dim=None):
    e = torch.as_tensor(e).resize_((1,1,*e.shape))
    e = F.avg_pool2d(e, ks)
    return torch.max(e)

# Cell
@patch
def predict_tiles(self:Learner, ds_idx=1, dl=None, mc_dropout=False, n_times=1, use_tta=False,
                  tta_merge='mean', energy_T=1, energy_ks=20, padding=(0,0,0,0)): #(-52,-52,-52,-52)
    "Make predictions and reconstruct tiles, optional with dropout and/or tta applied."

    if dl is None: dl = self.dls[ds_idx].new(shuffled=False, drop_last=False)
    if use_tta: tfms=[tta.HorizontalFlip(), tta.Rotate90(angles=[90,180,270])]
    else: tfms=[]

    self.model.eval()
    if mc_dropout: self.apply_dropout()

    smx_means, energy_means, stds = [], [], []
    for data in progress_bar(dl, leave=False):
        if isinstance(data, TensorImage): images = data
        else: images, _, _ = data
        m_smx = tta.Merger()
        m_energy = tta.Merger()
        out_list_smx = []
        for t in tta.Compose(tfms):
            for _ in range(n_times):
                aug_images = t.augment_image(images)
                with torch.no_grad():
                    out = self.model(aug_images)
                out = t.deaugment_mask(out)
                if sum(padding)>0: out = F.pad(out, padding)
                m_smx.append(F.softmax(out, dim=1))
                e = (energy_T*torch.logsumexp(out/energy_T, dim=1)) #negative energy score
                m_energy.append(e)

        stds.append(m_smx.result('std'))
        smx_means.append(m_smx.result())
        energy_means.append(m_energy.result())

    softmax_pred = torch.cat(smx_means).permute(0,2,3,1)
    smx_tiles = [x for x in softmax_pred.cpu().numpy()]

    std_pred = torch.cat(stds).permute(0,2,3,1)
    std_tiles = [x[...,0] for x in std_pred.cpu().numpy()]

    energy_pred = torch.cat(energy_means)#.permute(0,2,3,1)
    energy_tiles = [x for x in energy_pred.cpu().numpy()]

    smxcores = dl.reconstruct_from_tiles(smx_tiles)
    segmentations = [np.argmax(x, axis=-1) for x in smxcores]
    std_deviations = dl.reconstruct_from_tiles(std_tiles)
    energy_scores = dl.reconstruct_from_tiles(energy_tiles)

    if energy_ks is not None:
        energy_scores = [energy_max(e, energy_ks) for e in energy_scores]

    return smxcores, segmentations, std_deviations, energy_scores

# Cell
def save_tmp(pth_tmp, files, results):
    smxs, segs, stds, enrgys = results
    pth_tmp.mkdir(exist_ok=True, parents=True)
    for i, f in progress_bar(enumerate(files), total=len(files), leave=False):
        np.savez(pth_tmp/f'{f.stem}.npz', smx=smxs[i], seg=segs[i], std=stds[i], enrgy=enrgys[i])

# Cell
class EnsembleLearner(GetAttr):
    _default = 'config'
    def __init__(self, image_dir='images', mask_dir=None, config=None, path=None, ensemble_dir=None,
                 label_fn=None, metrics=None, loss_fn=None, cbs=None, ds_kwargs={}):

        self.config = config or Config()
        self.path = Path(path) if path is not None else Path('.')
        self.metrics = metrics or [Iou()] #Dice_f1()
        self.loss_fn = loss_fn or WeightedSoftmaxCrossEntropy(axis=1)
        self.cbs = cbs or [SaveModelCallback(monitor='iou'), ElasticDeformCallback] #ShowGraphCallback
        self.ensemble_dir = ensemble_dir or self.path/'ensemble'

        self.files = get_image_files(self.path/image_dir, recurse=False)
        assert len(self.files)>0, f'Found {len(self.files)} images in "{image_dir}". Please check your images and image folder'
        if any([mask_dir, label_fn]):
            if label_fn: self.label_fn = label_fn
            else: self.label_fn = get_label_fn(self.files[0], self.path/mask_dir)
            #Check if corresponding masks exist
            mask_check = [self.label_fn(x).is_file() for x in self.files]
            chk_str = f'Found {len(self.files)} images in "{image_dir}" and {sum(mask_check)} masks in "{mask_dir}".'
            assert len(self.files)==sum(mask_check) and len(self.files)>0, f'Please check your images and masks (and folders). {chk_str}'
            print(chk_str)

        else:
            self.label_fn = label_fn

        self.n_splits=min(len(self.files), self.max_splits)
        for key, value in get_default_shapes(self.arch).items():
            ds_kwargs.setdefault(key, value)
        self.ds_kwargs = ds_kwargs
        self.item_tfms=[Brightness(max_lighting=self.light)]
        self.models = {}
        self.recorder = {}
        self._set_splits()
        self.ds = RandomTileDataset(self.files, label_fn=self.label_fn, create_weights=False, **self.mw_kwargs, **self.ds_kwargs)
        self.in_channels = self.ds.get_data(max_n=1)[0].shape[-1]
        print('Computing Stats...')
        self.stats = self.stats or self.ds.compute_stats()
        self.df_val, self.df_ens, self.df_model, self.ood = None,None,None,None


    def _set_splits(self):
        kf = KFold(self.n_splits, shuffle=True, random_state=self.random_state)
        self.splits = {key:(self.files[idx[0]], self.files[idx[1]]) for key, idx in zip(range(1,self.n_splits+1), kf.split(self.files))}

    def save_model(self, file, model, pickle_protocol=2):
        state = model.state_dict()
        state = {'model': state, 'arch':self.arch, 'repo':self.repo, 'stats':self.stats, 'c':self.c}
        torch.save(state, file, pickle_protocol=pickle_protocol, _use_new_zipfile_serialization=False)

    def load_model(self, file, with_meta=True, device=None, strict=True):
        if isinstance(device, int): device = torch.device('cuda', device)
        elif device is None: device = 'cpu'
        state = torch.load(file, map_location=device)
        hasopt = set(state)=={'model', 'arch', 'repo', 'stats', 'c'}
        if hasopt:
            model_state = state['model']
            if with_meta:
                self.config.arch = state['arch']
                self.config.repo = state['repo']
                self.config.stats = state['stats']
                self.config.c  = state['c']
        else:
            model_state = state
        model = torch.hub.load(self.repo, self.arch, pretrained=None, n_classes=self.c, in_channels=self.in_channels, pre_ssl=False)
        model.load_state_dict(model_state, strict=strict)
        return model

    def fit(self, i, n_iter=None, lr_max=None, bs=None , n_jobs=-1, verbose=1, **kwargs):
        n_iter = n_iter or self.n_iter
        lr_max = lr_max or self.lr
        bs = bs or self.bs
        name = self.ensemble_dir/f'{self.arch}_model-{i}.pth'
        files_train, files_val = self.splits[i]
        train_ds = RandomTileDataset(files_train, label_fn=self.label_fn, n_jobs=n_jobs, verbose=verbose, **self.mw_kwargs, **self.ds_kwargs)
        valid_ds = TileDataset(files_val, label_fn=self.label_fn, n_jobs=n_jobs, verbose=verbose, **self.mw_kwargs,**self.ds_kwargs)
        batch_tfms = Normalize.from_stats(*self.stats)
        dls = DataLoaders.from_dsets(train_ds, valid_ds, bs=bs, after_item=self.item_tfms, after_batch=batch_tfms)
        pre = None if self.pretrained=='new' else self.pretrained
        model = torch.hub.load(self.repo, self.arch, pretrained=pre, n_classes=dls.c, in_channels=self.in_channels, **kwargs)
        if torch.cuda.is_available(): dls.cuda(), model.cuda()
        learn = Learner(dls, model, metrics=self.metrics, wd=self.wd, loss_func=self.loss_fn, opt_func=_optim_dict[self.optim], cbs=self.cbs)
        learn.model_dir = self.ensemble_dir.parent/'.tmp'
        if self.mpt: learn.to_fp16()
        print(f'Starting training for {name.name}')
        epochs = calc_iterations(n_iter=n_iter,ds_length=len(train_ds), bs=bs)
        learn.fit_one_cycle(epochs, lr_max)

        print(f'Saving model at {name}')
        name.parent.mkdir(exist_ok=True, parents=True)
        self.save_model(name, learn.model)
        self.models[i]=name
        self.recorder[i]=learn.recorder
        del model
        gc.collect()
        torch.cuda.empty_cache()


    def fit_ensemble(self, n_iter, skip=False, **kwargs):
        for i in range(1, self.n+1):
            if skip and (i in self.models): continue
            self.fit(i, n_iter,  **kwargs)

    def set_n(self, n):
        for i in range(n, len(self.models)):
            self.models.pop(i+1, None)
        self.n = n

    def predict(self, files, model_no, bs=None, **kwargs):
        bs = bs or self.bs
        model_path = self.models[model_no]
        batch_tfms = Normalize.from_stats(*self.stats)
        ds = TileDataset(files, **self.ds_kwargs)
        dls = DataLoaders.from_dsets(ds, batch_size=bs, after_batch=batch_tfms, shuffle=False, drop_last=False)
        model = self.load_model(model_path)
        if torch.cuda.is_available(): dls.cuda(), model.cuda()
        learn = Learner(dls, model, loss_func=self.loss_fn)
        if self.mpt: learn.to_fp16()
        results = learn.predict_tiles(dl=dls.train, **kwargs)
        pth_tmp = self.path/'.tmp'/model_path.name
        save_tmp(pth_tmp, files, results)
        return results

    def get_valid_results(self, model_no=None, **kwargs):
        res_list = []
        model_list = self.models if not model_no else [model_no]
        for i in model_list:
            _, files_val = self.splits[i]
            res = self.predict(files_val, i, **kwargs)
            for j, f in enumerate(files_val):
                msk = self.ds.get_data(f, mask=True)[0]
                m_path = self.models[i].name
                df_tmp = pd.Series({'file' : f.name,
                        'model' :  m_path,
                        'model_no' : i,
                        'img_path': f,
                        'msk_path': self.label_fn(f),
                        'res_path': self.path/'.tmp'/m_path/f'{f.stem}.npz',
                        'iou': iou(msk, res[1][j]),
                        'energy_max': res[3][j].numpy()})
                res_list.append(df_tmp)
        self.df_val = pd.DataFrame(res_list)

    def show_valid_results(self, model_no=None, file=None, save_dir=None, filetype='.png', **kwargs):
        if self.df_val is None: self.get_validresults(**kwargs)
        df = self.df_val
        if file is not None: df = df[df.file==file]
        if model_no is not None: df = df[df.model_no==model_no]
        if save_dir:
            save_dir = self.path/save_dir
            pred_path = save_dir/'masks'
            pred_path.mkdir(parents=True, exist_ok=True)
            df.to_csv(save_dir/f'val_results.csv', index=False)
            if self.tta:
                unc_path = save_dir/'uncertainties'
                unc_path.mkdir(parents=True, exist_ok=True)
        for _, r in df.iterrows():
            img = self.ds.get_data(r.img_path)[0]
            msk = self.ds.get_data(r.img_path, mask=True)[0]
            with open(r.res_path, 'rb') as res:
                tmp = np.load(res)
                pred = tmp['seg']
                std = tmp['std']
            print(f'Validation image {r.file} for {r.model}:')
            if self.tta: plot_results(img, msk, pred, std, df=r)
            else: plot_results(img, msk, pred, np.zeros_like(pred), df=r)
            if save_dir:
                save_mask(pred, pred_path/f'{r.file}_{r.model}_mask', filetype)
                if self.tta:
                    save_unc(std, unc_path/f'{r.file}_{r.model}_unc', filetype)

    def get_models(self, path=None):
        path = path or self.ensemble_dir
        models = get_files(path, extensions='.pth', recurse=False)
        assert len(models)>0, f'No models found in {path}'
        self.models = {}
        for m in models:
            model_id = int(m.stem[-1])
            self.models[model_id] = m
        print(f'Found {len(self.models)} models in folder {path}')
        print(self.models)

    def ensemble_results(self, files, save_dir=None, filetype='.png', use_tta=None, **kwargs):
        use_tta = use_tta or self.pred_tta
        pth_out = self.path/'.tmp'/f'{self.arch}_ensemble'
        pth_out.mkdir(exist_ok=True, parents=True)
        if save_dir:
            save_dir = self.path/save_dir
            pred_path = save_dir/'masks'
            pred_path.mkdir(parents=True, exist_ok=True)
            if use_tta:
                unc_path = save_dir/'uncertainties'
                unc_path.mkdir(parents=True, exist_ok=True)
        res_list = []
        for f in files:
            for m in self.models.values():
                pth_tmp = self.path/'.tmp'/m.name/f'{f.stem}.npz'
                m_smx, m_std, m_enrgy = tta.Merger(), tta.Merger(), tta.Merger()
                with open(pth_tmp, 'rb') as file:
                    tmp = np.load(file)
                    m_smx.append(tmp['smx'])
                    m_std.append(tmp['std'])
                    m_enrgy.append(tmp['enrgy'])
            smx = m_smx.result()
            seg = np.argmax(smx, axis=-1)
            std = m_std.result()
            enrgy = m_enrgy.result()
            np.savez(pth_out/f'{f.stem}.npz', smx=smx, seg=seg, std=std, enrgy=enrgy)
            df_tmp = pd.Series({'file' : f.name,
                                'model' :  pth_out.name,
                                'img_path': f,
                                'res_path': pth_out/f'{f.stem}.npz',
                                'energy_max': enrgy.numpy()})
            res_list.append(df_tmp)
            if save_dir:
                save_mask(seg.numpy(), pred_path/f'{df_tmp.file}_{df_tmp.model}_mask', filetype)
                if use_tta:
                    save_unc(std.numpy(), unc_path/f'{df_tmp.file}_{df_tmp.model}_unc', filetype)
        return pd.DataFrame(res_list)

    def get_ensemble_results(self, new_files, save_dir=None, filetype='.png', **kwargs):
        res_list = []
        for i in self.models:
            res = self.predict(new_files, i, **kwargs)
            for j, f in enumerate(new_files):
                m_path = self.models[i].name
                df_tmp = pd.Series({'file' : f.name,
                                    'model_no': i,
                                    'model' :  m_path,
                                    'img_path': f,
                                    'res_path': self.path/'.tmp'/m_path/f'{f.stem}.npz',
                                    'energy_max': res[3][j].numpy()})
                res_list.append(df_tmp)
        self.df_models = pd.DataFrame(res_list)
        self.df_ens  = self.ensemble_results(new_files, save_dir=save_dir, filetype=filetype, **kwargs)

    def show_ensemble_results(self, files=None, model_no=None, unc=True, unc_metric=None):
        if self.df_ens is None: assert print("Please run `get_ensemble_results` first.")
        if model_no is None: df = self.df_ens
        else: df = self.df_models[df_models.model_no==model_no]
        if files is not None: df = df.set_index('file', drop=False).loc[files]
        for _, r in df.iterrows():
            img = _read_img(r.img_path)
            with open(r.res_path, 'rb') as res:
                tmp = np.load(res)
                pred = tmp['seg']
                std = tmp['std']
            if unc: plot_results(img, pred, std, df=r, unc_metric=unc_metric)
            else: plot_results(img, pred, df=r)

    def lr_find(self, files=None, bs=None, n_jobs=-1, verbose=1, **kwargs):
        bs = bs or self.bs
        files = files or self.files
        train_ds = RandomTileDataset(files, label_fn=self.label_fn, n_jobs=n_jobs, verbose=verbose, **self.mw_kwargs, **self.ds_kwargs)
        dls = DataLoaders.from_dsets(train_ds,train_ds, bs=bs)
        pre = None if self.pretrained=='new' else self.pretrained
        model = torch.hub.load(self.repo, self.arch, pretrained=pre, n_classes=dls.c, in_channels=self.in_channels)
        if torch.cuda.is_available(): dls.cuda(), model.cuda()
        learn = Learner(dls, model, metrics=self.metrics, wd=self.wd, loss_func=self.loss_fn, opt_func=_optim_dict[self.optim])
        if self.mpt: learn.to_fp16()
        sug_lrs = learn.lr_find(**kwargs)
        return sug_lrs, learn.recorder

    def show_mask_weights(self, files, figsize=(12,12), **kwargs):
        masks = [self.label_fn(Path(f)) for f in files]
        for m in masks:
            print(self.mw_kwargs)
            print(f'Calculating weights. Please wait...')
            msk = _read_msk(m)
            _, w, _ = calculate_weights(msk, n_dims=self.c, **self.mw_kwargs)
            fig, axes = plt.subplots(nrows=1, ncols=2, figsize=figsize, **kwargs)
            axes[0].imshow(msk)
            axes[0].set_axis_off()
            axes[0].set_title(f'Mask {m.name}')
            axes[1].imshow(w)
            axes[1].set_axis_off()
            axes[1].set_title('Weights')
            plt.show()

    def ood_train(self, features=['energy_max'], **kwargs):
        self.ood = Pipeline([('scaler', StandardScaler()), ('svm',svm.OneClassSVM(**kwargs))])
        self.ood.fit(self.df_ens[features])

    def ood_score(self, features=['energy_max']):
        self.df_ens['ood_score'] = self.ood.score_samples(self.df_ens[features])

    def ood_save(self, path):
        path = Path(path)
        joblib.dump(self.ood, path.with_suffix('.pkl'))
        print(f'Saved OOD model to {path}.pkl')

    def ood_load(self, path):
        path = Path(path)
        try:
            self.ood = joblib.load(path)
            print(f'Successsfully loaded OOD Model from {path}')
        except:
            print('Error! Select valid joblib file (.pkl)')

    def clear_tmp(self):
        try: shutil.rmtree(self.path/'.tmp')
        except: print("No temporary files to delete")

# Cell
add_docs(EnsembleLearner, "Meta class to train and predict model ensembles with `n` models",
         save_model= "Save `model` to `file` along with `arch`, `stats`, and `c` classes",
         load_model="Load `model` from `file` along with `arch`, `stats`, and `c` classes",
         fit="Fit model number `i`",
         fit_ensemble="Fit `i` models and `skip` existing",
         predict="Predict `files` with model at `model_path`",
         get_valid_results="Validate models on validation data and save results",
         show_valid_results="Plot results of all or `file` validation images",
         ensemble_results="Merge single model results",
         get_ensemble_results="Get models and ensemble results",
         show_ensemble_results="Show result of ensemble or `model_no`",
         get_models="Get models saved at `path`",
         set_n="Change to `n` models per ensemble",
         lr_find="Wrapper for learning rate finder",
         show_mask_weights='Plot fn for masks and weights',
         ood_train="Train SVM for OOD Detection",
         ood_score="Get OOD score",
         ood_save='Save OOD model to path',
         ood_load='Load OOD model from path',
         clear_tmp="Clear directory with temporary files"
)