import difflib
import io

import pytest

from TotalDepth.BIT import ReadBIT
from TotalDepth.BIT import ToLAS
from TotalDepth.common import Slice
from . import test_ReadBit


def test_example_file():
    with io.BytesIO(test_ReadBit.EXAMPLE_BIT_FILE) as file:
        assert ReadBIT.is_bit_file(file)


def print_differences(a: str, b: str) -> None:
    differ = difflib.SequenceMatcher()
    differ.set_seqs(a, b)
    for tag, i1, i2, j1, j2 in differ.get_opcodes():
        print('{:7}   a[{}:{}] --> b[{}:{}] {!r:>8} --> {!r}'.format(tag, i1, i2, j1, j2, a[i1:i2], b[j1:j2]))


TYPICAL_LAS_HEADER  = """~Version Information Section
VERS.           2.0                             : CWLS Log ASCII Standard - VERSION 2.0
WRAP.           NO                              : One Line per depth step
PROD.           TotalDepth                      : LAS Producer
PROG.           TotalDepth.BIT.ToLAS 0.1.1      : LAS Program name and version
CREA.           2021-02-05 11:07:32.289769 UTC  : LAS Creation date [YYYY-mm-dd HH MM SS.us UTC]
SOURCE.         bit_path                        : Source File Name
LOGICAL-FILE.   0                               : Logical File number in the Source file
SOURCE_FORMAT.  WESTERN ATLAS BIT FORMAT        : File format of Source file.
"""
TYPICAL_LAS_HEADER_TIMESTAMP = '2021-02-05 11:07:32.289769'

# Slice with these to eliminate variable text
LAS_TIMESTAMP_START = TYPICAL_LAS_HEADER.find(TYPICAL_LAS_HEADER_TIMESTAMP)
LAS_TIMESTAMP_END = LAS_TIMESTAMP_START + len(TYPICAL_LAS_HEADER_TIMESTAMP)


def las_is_effectively_the_same(a, b):
    a_pre = a[:LAS_TIMESTAMP_START]
    b_pre = b[:LAS_TIMESTAMP_START]
    a_post = a[LAS_TIMESTAMP_END:]
    b_post = b[LAS_TIMESTAMP_END:]
    equal_pre = a_pre == b_pre
    equal_post = a_post == b_post

    ret =  equal_pre and equal_post
    if not ret:
        print('Differences:')
        # print_differences(a, b)
        if not equal_pre:
            print('Before timestamp:')
            print_differences(a[:LAS_TIMESTAMP_START], b[:LAS_TIMESTAMP_START])
        if not equal_post:
            print('After timestamp:')
            print_differences(a[LAS_TIMESTAMP_END:], b[LAS_TIMESTAMP_END:])
    return ret


EXPECTED_LAS_ALL_FRAMES_ALL_CHANNELS_16_3F = """~Version Information Section
VERS.           2.0                             : CWLS Log ASCII Standard - VERSION 2.0
WRAP.           NO                              : One Line per depth step
PROD.           TotalDepth                      : LAS Producer
PROG.           TotalDepth.BIT.ToLAS 0.1.1      : LAS Program name and version
CREA.           2021-02-09 11:27:39.216238 UTC  : LAS Creation date [YYYY-mm-dd HH MM SS.us UTC]
SOURCE.         bit_path                        : Source File Name
LOGICAL-FILE.   0                               : Logical File number in the Source file
SOURCE_FORMAT.  WESTERN ATLAS BIT FORMAT        : File format of Source file.
#
# Binary block A: b'OCCIDENTAL PETROLEUM                                                    '
# Binary block B: b'U  15/17-12                                                                '
# BIT Log Pass (claimed): LogPassRange(depth_from=11916.0, depth_to=0.0, spacing=0.25, unknown_a=0.0, unknown_b=32.0)
#
~Well Information Section
#MNEM.UNIT  DATA      DESCRIPTION
#----.----  ----      -----------
STRT.       11916.0   : START
STOP.       11884.25  : STOP
STRP.       -0.25     : STEP
~Curve Information Section
#MNEM.UNIT  Curve Description                
#---------  -----------------                
X   .       : Computed X-axis Dimensions (1,)
SSN .       : SSN  Dimensions (1,)           
LSN .       : LSN  Dimensions (1,)           
SSD .       : SSD  Dimensions (1,)           
LSD .       : LSD  Dimensions (1,)           
CAL .       : CAL  Dimensions (1,)           
TEN .       : TEN  Dimensions (1,)           
SPD .       : SPD  Dimensions (1,)           
THZ .       : THZ  Dimensions (1,)           
UZ  .       : UZ   Dimensions (1,)           
KZ  .       : KZ   Dimensions (1,)           
GR  .       : GR   Dimensions (1,)           
DEN .       : DEN  Dimensions (1,)           
CORR.       : CORR Dimensions (1,)           
CN  .       : CN   Dimensions (1,)           
TH  .       : TH   Dimensions (1,)           
U   .       : U    Dimensions (1,)           
K   .       : K    Dimensions (1,)           
# Array processing information:
# Frame Array: ID: 0 description: b'OCCIDENTAL PETROLEUM                                                    '
# All [18] original channels reproduced here.
# Where a channel has multiple values the reduction method is by "first" value.
# Maximum number of original frames: 128
# Requested frame slicing: <Slice on length=128 start=0 stop=128 step=1>, total number of frames presented here: 128
~A          X                SSN              LSN              SSD              LSD              CAL              TEN              SPD              THZ              UZ               KZ               GR               DEN              CORR             CN               TH               U                K   
       11916.000          278.972          149.985          249.975          108.989            3.047         -109.789           16.998            6.999           32.997           34.996          228.278            2.760            0.066           22.986           21.962           25.558           -0.042
       11915.750          278.984          149.985          249.975          108.989            3.047         -109.627           16.998            6.999           32.997           34.996          228.278            2.760            0.066           22.988           21.962           25.558           -0.042
       11915.500          279.050          149.985          249.975          108.989            3.047         -108.773           16.998            6.999           32.997           34.996          228.278            2.760            0.066           22.998           21.962           25.558           -0.042
       11915.250          279.197          149.985          249.975          108.989            3.047         -106.574           16.998            6.999           32.997           34.996          228.278            2.760            0.066           23.021           21.962           25.558           -0.042
       11915.000          279.410          149.967          249.975          108.989            3.047         -102.650           16.998            6.999           32.997           34.996          228.278            2.760            0.066           23.060           21.962           25.558           -0.042
       11914.750          279.671          149.807          249.975          108.989            3.047          -97.110           16.992            6.999           32.997           34.996          228.278            2.760            0.066           23.148           21.962           25.558           -0.042
       11914.500          279.935          149.247          249.975          108.989            3.047          -90.164           16.969            6.999           32.997           34.996          228.278            2.760            0.066           23.354           21.962           25.558           -0.042
       11914.250          280.133          148.186          249.975          108.989            3.047          -81.425           16.913            6.999           32.997           34.996          228.278            2.760            0.066           23.700           21.962           25.558           -0.042
       11914.000          280.190          146.890          249.975          108.989            3.047          -69.734           16.806            6.999           32.997           34.996          228.278            2.760            0.066           24.102           21.962           25.558           -0.042
       11913.750          280.100          145.785          249.975          108.989            3.047          -53.717           16.651            6.999           32.997           34.996          228.278            2.760            0.066           24.427           21.962           25.558           -0.042
       11913.500          279.901          145.225          249.975          108.989            3.047          -32.697           16.466            6.999           32.997           34.996          228.278            2.760            0.066           24.568           21.962           25.558           -0.042
       11913.250          279.646          145.351          249.975          108.989            3.047           -7.446           16.265            6.999           32.997           34.996          228.278            2.760            0.066           24.488           21.962           25.558           -0.042
       11913.000          279.427          146.130          249.975          108.989            3.047           19.767           16.076            6.999           32.997           34.996          228.278            2.760            0.066           24.212           21.962           25.558           -0.042
       11912.750          279.353          147.269          249.975          108.989            3.047           46.043           15.943            6.999           32.997           34.996          228.278            2.760            0.066           23.853           21.962           25.558           -0.042
       11912.500          279.491          148.334          249.975          108.989            3.047           69.373           15.895            6.999           32.997           34.996          228.278            2.760            0.066           23.555           21.962           25.558           -0.042
       11912.250          279.818          149.007          249.975          108.989            3.047           89.651           15.920            6.999           32.997           34.996          228.278            2.760            0.066           23.406           21.962           25.558           -0.042
       11912.000          280.272          149.216          249.975          108.989            3.047          108.356           15.978            6.999           32.997           34.996          228.278            2.760            0.066           23.416           21.962           25.558           -0.042
       11911.750          280.792          149.195          249.975          108.989            3.047          127.389           16.008            6.999           32.997           34.996          228.278            2.760            0.066           23.504           21.962           25.558           -0.042
       11911.500          281.284          149.036          249.975          108.989            3.047          147.898           15.965            6.999           32.997           34.996          228.278            2.760            0.066           23.629           21.962           25.558           -0.042
       11911.250          281.694          148.725          249.975          108.989            3.047          170.062           15.848            6.999           32.997           34.996          228.278            2.760            0.066           23.786           21.962           25.558           -0.042
       11911.000          282.025          148.427          249.975          108.989            3.047          193.495           15.678            6.999           32.997           34.996          228.278            2.760            0.066           23.928           21.962           25.558           -0.042
       11910.750          282.292          148.293          249.975          108.989            3.047          217.670           15.478            6.999           32.997           34.996          228.278            2.760            0.066           24.011           21.962           25.558           -0.042
       11910.500          282.521          148.332          249.975          108.989            3.047          242.059           15.299            6.999           32.997           34.996          228.278            2.760            0.066           24.035           21.962           25.558           -0.042
       11910.250          282.743          148.376          249.975          108.989            3.047          265.935           15.175            6.999           32.997           34.996          228.278            2.760            0.066           24.057           21.962           25.558           -0.042
       11910.000          282.937          148.298          249.975          108.989            3.047          288.219           15.108            6.999           32.997           34.996          228.278            2.760            0.066           24.111           21.962           25.558           -0.042
       11909.750          283.014          148.027          249.975          108.989            3.047          307.746           15.100            6.999           32.997           34.996          228.278            2.760            0.066           24.206           21.962           25.558           -0.042
       11909.500          282.907          147.470          249.975          108.989            3.047          324.015           15.146            6.999           32.997           34.996          228.278            2.760            0.066           24.358           21.962           25.558           -0.042
       11909.250          282.555          146.624          249.975          108.989            3.047          337.993           15.216            6.999           32.997           34.996          228.278            2.760            0.066           24.562           21.962           25.558           -0.042
       11909.000          281.980          145.604          249.975          108.989            3.047          352.014           15.266            6.999           32.997           34.996          228.278            2.760            0.066           24.786           21.962           25.558           -0.042
       11908.750          281.316          144.667          249.975          108.989            3.047          368.763           15.247            6.999           32.997           34.996          228.278            2.760            0.066           24.973           21.962           25.558           -0.042
       11908.500          280.729          144.094          249.975          108.989            3.047          389.814           15.127            6.999           32.997           34.996          228.278            2.760            0.066           25.069           21.962           25.558           -0.042
       11908.250          280.364          143.962          249.975          108.989            3.047          414.754           14.925            6.999           32.997           34.996          228.278            2.760            0.066           25.048           21.962           25.558           -0.042
       11908.000          280.322          144.454          249.975          108.989            3.047          441.318           14.707            6.999           32.997           34.996          228.278            2.760            0.066           24.879           21.962           25.558           -0.042
       11907.750          280.601          145.577          249.975          109.048            3.047          466.448           14.532            6.999           32.997           34.996          228.278            2.760            0.065           24.572           21.962           25.558           -0.042
       11907.500          281.097          147.126          250.005          109.334            3.047          487.819           14.411            6.999           32.997           34.996          228.278            2.756            0.063           24.175           21.962           25.558           -0.042
       11907.250          281.629          148.717          250.183          109.968            3.047          504.898           14.337            6.999           32.997           34.996          228.278            2.749            0.060           23.778           21.962           25.558           -0.042
       11907.000          282.100          149.959          250.700          110.902            3.046          518.919           14.290            6.999           32.997           34.996          228.278            2.742            0.057           23.484           21.962           25.558           -0.042
       11906.750          282.447          150.537          251.676          111.946            3.045          531.979           14.230            6.999           32.997           34.996          228.278            2.737            0.058           23.368           21.962           25.558           -0.042
       11906.500          282.570          150.375          253.036          112.846            3.042          545.933           14.137            6.999           32.997           34.996          228.278            2.738            0.063           23.435           21.962           25.558           -0.042
       11906.250          282.347          149.626          254.443          113.467            3.039          561.817           14.034            6.999           32.997           34.996          228.278            2.742            0.070           23.621           21.962           25.558           -0.042
       11906.000          281.791          148.717          255.390          113.901            3.036          579.655           13.963            6.999           32.997           34.996          228.278            2.745            0.075           23.804           21.962           25.558           -0.042
       11905.750          281.110          148.012          255.534          114.396            3.033          598.700           13.938            6.999           32.997           34.996          228.278            2.740            0.073           23.908           21.962           25.558           -0.042
       11905.500          280.550          147.664          254.885          115.171            3.032          617.908           13.931            6.999           32.997           34.996          228.278            2.725            0.061           23.924           21.962           25.558           -0.042
       11905.250          280.287          147.422          253.792          116.228            3.032          636.432           13.897            6.999           32.997           34.996          228.278            2.702            0.044           23.955           21.962           25.558           -0.042
       11905.000          280.354          147.045          252.826          117.294            3.032          653.990           13.804            6.999           32.997           34.996          228.278            2.681            0.028           24.080           21.962           25.558           -0.042
       11904.750          280.573          146.516          252.447          117.946            3.034          670.835           13.648            6.999           32.997           34.996          228.278            2.670            0.020           24.277           21.962           25.558           -0.042
       11904.500          280.650          146.041          252.714          117.925            3.034          687.322           13.446            6.999           32.997           34.996          228.278            2.672            0.022           24.436           21.962           25.558           -0.042
       11904.250          280.357          145.990          253.368          117.251            3.035          703.485           13.219            6.999           32.997           34.996          228.278            2.686            0.033           24.405           21.962           25.558           -0.042
       11904.000          279.657          146.591          253.993          116.260            3.035          718.929           13.020            6.999           32.997           34.996          228.278            2.704            0.046           24.108           21.962           25.558           -0.042
       11903.750          278.782          147.677          254.338          115.465            3.036          733.056           12.903            6.999           32.997           34.996          228.278            2.716            0.054           23.639           21.962           25.558           -0.042
       11903.500          278.103          148.772          254.459          115.254            3.036          745.416           12.882            6.999           32.997           34.996          228.278            2.720            0.057           23.205           21.962           25.558           -0.042
       11903.250          277.881          149.416          254.631          115.675            3.037          756.138           12.928            6.999           32.997           34.996          228.278            2.716            0.055           22.981           21.962           25.558           -0.042
       11903.000          278.087          149.489          254.991          116.498            3.037          765.981           13.008            6.999           32.997           34.996          228.278            2.709            0.052           22.992           21.962           25.558           -0.042
       11902.750          278.536          149.208          255.496          117.381            3.038          775.855           13.074            6.999           32.997           34.996          228.278            2.703            0.050           23.145           21.962           25.558           -0.042
       11902.500          279.053          149.038          256.051          118.096            3.040          785.970           13.077            6.999           32.997           34.996          228.278            2.699            0.050           23.276           21.962           25.558           -0.042
       11902.250          279.528          149.331          256.487          118.567            3.042          795.408           13.005            6.999           32.997           34.996          228.278            2.697            0.050           23.265           21.962           25.558           -0.042
       11902.000          279.875          149.970          256.583          118.768            3.044          802.503           12.896            6.999           32.997           34.996          228.278            2.696            0.050           23.132           21.962           25.558           -0.042
       11901.750          280.064          150.628          256.278          118.780            3.046          805.931           12.799            6.999           32.997           34.996          228.278            2.693            0.047           22.970           21.962           25.558           -0.042
       11901.500          280.095          151.036          255.551          118.790            3.047          805.760           12.744            6.999           32.997           34.996          228.278            2.686            0.041           22.857           21.962           25.558           -0.042
       11901.250          280.042          151.037          254.447          118.862            3.047          803.830           12.737            6.999           32.997           34.996          228.278            2.676            0.031           22.848           21.962           25.558           -0.042
       11901.000          280.017          150.624          253.093          118.853            3.046          803.050           12.764            6.999           32.997           34.996          228.278            2.665            0.019           22.964           21.962           25.558           -0.042
       11900.750          280.070          149.878          251.730          118.682            3.045          805.877           12.799            6.999           32.997           34.996          228.278            2.655            0.009           23.189           21.962           25.558           -0.042
       11900.500          280.190          148.810          250.650          118.420            3.044          812.853           12.840            6.999           32.997           34.996          228.278            2.649            0.001           23.523           21.962           25.558           -0.042
       11900.250          280.371          147.429          250.103          118.195            3.043          822.361           12.884            6.999           32.997           34.996          228.278            2.647           -0.002           23.967           21.962           25.558           -0.042
       11900.000          280.565          145.967          250.155          118.153            3.042          831.881           12.909            6.999           32.997           34.996          228.278            2.648           -0.001           24.445           21.962           25.558           -0.042
       11899.750          280.648          144.763          250.665          118.362            3.041          839.742           12.918            6.999           32.997           34.996          228.278            2.649            0.002           24.834           21.962           25.558           -0.042
       11899.500          280.521          144.129          251.437          118.649            3.041          845.813           12.916            6.999           32.997           34.996          228.278            2.653            0.006           25.016           21.962           25.558           -0.042
       11899.250          280.185          144.220          252.278          118.713            3.042          850.899           12.907            6.999           32.997           34.996          228.278            2.659            0.013           24.930           21.962           25.558           -0.042
       11899.000          279.713          144.906          252.919          118.304            3.043          855.855           12.891            6.999           32.997           34.996          228.278            2.670            0.022           24.638           21.962           25.558           -0.042
       11898.750          279.198          145.681          253.078          117.444            3.045          861.171           12.869            6.999           32.997           34.996          228.278            2.681            0.029           24.313           21.962           25.558           -0.042
       11898.500          278.709          146.170          252.721          116.340            3.048          867.049           12.827            6.999           32.997           34.996          228.278            2.692            0.034           24.084           21.962           25.558           -0.042
       11898.250          278.290          146.334          252.164          115.285            3.052          873.175           12.758            6.982           33.014           35.061          228.330            2.700            0.037           23.967           21.906           25.594           -0.041
       11898.000          278.016          146.299          251.844          114.480            3.056          878.400           12.669            6.886           33.122           35.265          228.542            2.707            0.040           23.934           21.607           25.798           -0.038
       11897.750          277.896          146.332          252.093          113.998            3.058          881.320           12.567            6.677           33.396           35.466          228.983            2.715            0.046           23.904           20.951           26.287           -0.044
       11897.500          277.866          146.546          252.974          113.924            3.057          881.619           12.455            6.433           33.705           35.487          229.672            2.724            0.054           23.834           20.184           26.848           -0.058
       11897.250          277.872          146.726          254.150          114.352            3.054          880.413           12.349            6.321           33.690           35.234          230.291            2.728            0.061           23.780           19.834           26.947           -0.068
       11897.000          277.965          146.670          255.144          115.233            3.049          879.900           12.249            6.510           33.114           34.644          230.304            2.726            0.063           23.812           20.428           26.179           -0.063
       11896.750          278.208          146.349          255.671          116.494            3.044          882.458           12.161            7.050           31.980           33.857          229.429            2.715            0.058           23.949           22.121           24.494           -0.039
       11896.500          278.568          145.971          255.838          117.978            3.039          889.538           12.108            7.837           30.517           33.252          227.749            2.699            0.049           24.123           24.591           22.227            0.007
       11896.250          278.917          145.736          255.912          119.330            3.034          900.673           12.102            8.619           29.121           33.175          225.701            2.683            0.040           24.251           27.043           20.032            0.071
       11896.000          279.158          145.774          256.119          120.038            3.030          913.403           12.131            9.121           28.287           33.650          223.863            2.677            0.037           24.278           28.619           18.685            0.211
       11895.750          279.218          146.068          256.420          119.883            3.026          924.361           12.169            9.213           28.439           34.450          222.728            2.681            0.040           24.197           28.908           18.740            0.304
       11895.500          279.087          146.498          256.529          119.084            3.024          931.545           12.186            8.976           29.647           35.223          222.463            2.691            0.047           24.044           28.165           20.185            0.201
       11895.250          278.821          146.983          256.267          118.014            3.023          935.807           12.169            8.608           31.497           35.580          222.796            2.702            0.052           23.854           27.008           22.405            0.055
       11895.000          278.519          147.484          255.768          117.021            3.024          939.868           12.120            8.333           33.222           35.388          223.283            2.710            0.055           23.655           26.147           24.403           -0.030
       11894.750          278.262          147.920          255.278          116.380            3.027          945.659           12.037            8.226           34.117           34.682          223.728            2.713            0.056           23.483           25.809           25.404           -0.096
       11894.500          278.131          148.213          254.955          116.194            3.030          952.962           11.919            8.224           33.900           33.508          224.190            2.713            0.054           23.375           25.804           25.189           -0.130
       11894.250          278.155          148.282          254.774          116.360            3.034          959.929           11.781            8.209           32.665           32.222          224.715            2.709            0.052           23.359           25.757           23.978           -0.123
       11894.000          278.309          148.135          254.577          116.650            3.039          964.472           11.652            8.076           30.914           31.145          225.099            2.704            0.048           23.427           25.341           22.375           -0.087
       11893.750          278.508          147.844          254.213          116.794            3.044          965.556           11.552            7.779           29.402           30.357          225.238            2.699            0.044           23.545           24.409           21.180           -0.051
       11893.500          278.625          147.594          253.722          116.717            3.050          963.773           11.482            7.390           28.764           29.855          225.308            2.696            0.040           23.639           23.187           20.948           -0.044
       11893.250          278.551          147.585          253.223          116.599            3.054          961.045           11.419            7.058           29.093           29.747          225.585            2.693            0.036           23.630           22.146           21.617           -0.064
       11893.000          278.314          147.825          252.801          116.759            3.058          959.470           11.336            6.887           29.914           30.075          226.187            2.687            0.032           23.520           21.610           22.610           -0.089
       11892.750          278.055          148.168          252.504          117.334            3.059          959.990           11.219            6.886           30.707           30.857          227.127            2.678            0.025           23.377           21.607           23.399           -0.095
       11892.500          277.942          148.491          252.308          118.247            3.058          961.953           11.076            7.035           31.205           32.062          228.221            2.665            0.017           23.263           22.074           23.740           -0.071
       11892.250          278.055          148.860          252.199          119.198            3.055          964.092           10.929            7.221           31.360           33.518          228.966            2.653            0.009           23.172           22.656           23.702           -0.023
       11892.000          278.365          149.435          252.202          119.800            3.050          965.760           10.815            7.318           31.222           34.818          228.812            2.646            0.005           23.052           22.962           23.465            0.031
       11891.750          278.701          150.255          252.355          119.796            3.046          967.257           10.758            7.304           30.856           35.641          227.605            2.647            0.006           22.865           22.916           23.116            0.078
       11891.500          278.881          151.163          252.703          119.294            3.042          969.294           10.761            7.212           30.382           35.800          225.773            2.656            0.013           22.632           22.629           22.739            0.116
       11891.250          278.842          151.887          253.202          118.578            3.039          972.185           10.811            7.098           29.937           35.415          223.865            2.669            0.022           22.419           22.270           22.416            0.134
       11891.000          278.622          152.283          253.838          117.875            3.038          975.534           10.890            6.968           29.596           34.804          222.249            2.683            0.033           22.273           21.863           22.211            0.101
       11890.750          278.268          152.333          254.661          117.252            3.039          978.587           10.969            6.788           29.333           34.235          221.257            2.697            0.044           22.204           21.298           22.135            0.090
       11890.500          277.784          152.081          255.640          116.610            3.042          980.840           11.022            6.558           29.159           33.772          221.198            2.713            0.057           22.200           20.577           22.200            0.079
       11890.250          277.189          151.659          256.580          115.778            3.045          982.463           11.030            6.351           29.089           33.468          221.999            2.732            0.071           22.228           19.928           22.344            0.070
       11890.000          276.581          151.157          257.235          114.815            3.045          984.037           10.987            6.220           29.101           33.380          223.187            2.749            0.084           22.276           19.516           22.492            0.065
       11889.750          276.104          150.555          257.366          114.065            3.044          985.774           10.914            6.161           29.139           33.640          224.177            2.759            0.091           22.373           19.332           22.590            0.073
       11889.500          275.909          149.821          256.886          113.914            3.041          987.232           10.864            6.139           29.153           34.270          224.624            2.757            0.088           22.554           19.263           22.627            0.095
       11889.250          276.070          149.091          255.963          114.526            3.038          987.849           10.879            6.114           29.034           35.071          224.413            2.742            0.075           22.791           19.185           22.534            0.219
       11889.000          276.517          148.808          254.966          115.734            3.038          987.518           10.949            6.047           28.651           35.639          223.572            2.718            0.058           22.944           18.973           22.223            0.369
       11888.750          277.051          149.251          254.113          117.125            3.040          986.632           11.019            5.951           27.936           35.678          222.322            2.694            0.040           22.899           18.673           21.611            0.499
       11888.500          277.393          150.330          253.514          118.233            3.043          985.641           11.034            5.869           27.022           35.238          221.025            2.676            0.027           22.639           18.415           20.789            0.592
       11888.250          277.369          151.597          253.217          118.727            3.047          984.878           10.971            5.903           26.225           34.701          219.963            2.667            0.021           22.273           18.522           19.961            0.654
       11888.000          277.018          152.511          253.166          118.496            3.049          984.537           10.846            6.137           25.887           34.419          219.213            2.670            0.022           21.962           19.257           19.383            0.677
       11887.750          276.563          152.678          253.395          117.656            3.051          984.745           10.691            6.565           26.076           34.455          218.670            2.682            0.030           21.845           20.599           19.129            0.659
       11887.500          276.219          152.021          254.038          116.536            3.052          985.473           10.550            7.083           26.633           34.566          218.269            2.701            0.044           21.976           22.225           19.147            0.590
       11887.250          276.141          150.839          254.995          115.482            3.051          986.528           10.449            7.611           27.224           34.519          217.896            2.722            0.060           22.298           23.881           19.189            0.491
       11887.000          276.356          149.540          255.971          114.771            3.048          987.709           10.403            8.010           27.652           34.274          217.405            2.739            0.073           22.705           25.132           19.202            0.389
       11886.750          276.736          148.418          256.565          114.628            3.045          988.947           10.411            8.105           27.919           34.105          216.797            2.746            0.080           23.094           25.430           19.369            0.319
       11886.500          277.077          147.635          256.409          115.231            3.041          989.861           10.473            7.815           28.021           34.288          216.317            2.737            0.074           23.380           24.520           19.770            0.322
       11886.250          277.221          147.192          255.423          116.410            3.038          989.922           10.586            7.218           27.873           34.823          216.142            2.714            0.057           23.536           22.648           20.240            0.413
       11886.000          277.068          147.030          253.925          117.725            3.038          989.112           10.716            6.533           27.391           35.392          216.210            2.685            0.034           23.560           20.498           20.469            0.566
       11885.750          276.639          147.089          252.430          118.663            3.040          988.125           10.798            6.055           26.691           35.621          216.464            2.661            0.015           23.474           19.000           20.267            0.712
       11885.500          276.639          147.089          252.430          118.663            3.040          988.125           10.798            6.055           26.691           35.621          216.464            2.661            0.015           23.474           19.000           20.267            0.712
       11885.250          276.639          147.089          252.430          118.663            3.040          988.125           10.798            6.055           26.691           35.621          216.464            2.661            0.015           23.474           19.000           20.267            0.712
       11885.000          276.639          147.089          252.430          118.663            3.040          988.125           10.798            6.055           26.691           35.621          216.464            2.661            0.015           23.474           19.000           20.267            0.712
       11884.750          276.639          147.089          252.430          118.663            3.040          988.125           10.798            6.055           26.691           35.621          216.464            2.661            0.015           23.474           19.000           20.267            0.712
       11884.500          276.639          147.089          252.430          118.663            3.040          988.125           10.798            6.055           26.691           35.621          216.464            2.661            0.015           23.474           19.000           20.267            0.712
       11884.250          276.639          147.089          252.430          118.663            3.040          988.125           10.798            6.055           26.691           35.621          216.464            2.661            0.015           23.474           19.000           20.267            0.712
"""


EXPECTED_LAS_SLICE_0_8_4_TWO_CHANNELS_16_3F = """~Version Information Section
VERS.           2.0                             : CWLS Log ASCII Standard - VERSION 2.0
WRAP.           NO                              : One Line per depth step
PROD.           TotalDepth                      : LAS Producer
PROG.           TotalDepth.BIT.ToLAS 0.1.1      : LAS Program name and version
CREA.           2021-02-09 11:27:41.154802 UTC  : LAS Creation date [YYYY-mm-dd HH MM SS.us UTC]
SOURCE.         bit_path                        : Source File Name
LOGICAL-FILE.   0                               : Logical File number in the Source file
SOURCE_FORMAT.  WESTERN ATLAS BIT FORMAT        : File format of Source file.
#
# Binary block A: b'OCCIDENTAL PETROLEUM                                                    '
# Binary block B: b'U  15/17-12                                                                '
# BIT Log Pass (claimed): LogPassRange(depth_from=11916.0, depth_to=0.0, spacing=0.25, unknown_a=0.0, unknown_b=32.0)
#
~Well Information Section
#MNEM.UNIT  DATA     DESCRIPTION
#----.----  ----     -----------
STRT.       11916.0  : START
STOP.       11915.0  : STOP
STRP.       -1.0     : STEP
~Curve Information Section
#MNEM.UNIT  Curve Description                
#---------  -----------------                
X   .       : Computed X-axis Dimensions (1,)
CAL .       : CAL  Dimensions (1,)           
# Array processing information:
# Frame Array: ID: 0 description: b'OCCIDENTAL PETROLEUM                                                    '
# Original channels in Frame Array [  18]: X   ,SSN ,LSN ,SSD ,LSD ,CAL ,TEN ,SPD ,THZ ,UZ  ,KZ  ,GR  ,DEN ,CORR,CN  ,TH  ,U   ,K   
# Requested Channels in this LAS file [   2]: X   ,CAL 
# Where a channel has multiple values the reduction method is by "first" value.
# Maximum number of original frames: 128
# Requested frame slicing: <Slice on length=128 start=0 stop=8 step=4>, total number of frames presented here: 2
~A          X                CAL 
       11916.000            3.047
       11915.000            3.047
"""


EXPECTED_LAS_SLICE_0_8_4_TWO_CHANNELS_8_1F = """~Version Information Section
VERS.           2.0                             : CWLS Log ASCII Standard - VERSION 2.0
WRAP.           NO                              : One Line per depth step
PROD.           TotalDepth                      : LAS Producer
PROG.           TotalDepth.BIT.ToLAS 0.1.1      : LAS Program name and version
CREA.           2021-02-09 11:27:41.186710 UTC  : LAS Creation date [YYYY-mm-dd HH MM SS.us UTC]
SOURCE.         bit_path                        : Source File Name
LOGICAL-FILE.   0                               : Logical File number in the Source file
SOURCE_FORMAT.  WESTERN ATLAS BIT FORMAT        : File format of Source file.
#
# Binary block A: b'OCCIDENTAL PETROLEUM                                                    '
# Binary block B: b'U  15/17-12                                                                '
# BIT Log Pass (claimed): LogPassRange(depth_from=11916.0, depth_to=0.0, spacing=0.25, unknown_a=0.0, unknown_b=32.0)
#
~Well Information Section
#MNEM.UNIT  DATA     DESCRIPTION
#----.----  ----     -----------
STRT.       11916.0  : START
STOP.       11915.0  : STOP
STRP.       -1.0     : STEP
~Curve Information Section
#MNEM.UNIT  Curve Description                
#---------  -----------------                
X   .       : Computed X-axis Dimensions (1,)
CAL .       : CAL  Dimensions (1,)           
# Array processing information:
# Frame Array: ID: 0 description: b'OCCIDENTAL PETROLEUM                                                    '
# Original channels in Frame Array [  18]: X   ,SSN ,LSN ,SSD ,LSD ,CAL ,TEN ,SPD ,THZ ,UZ  ,KZ  ,GR  ,DEN ,CORR,CN  ,TH  ,U   ,K   
# Requested Channels in this LAS file [   2]: X   ,CAL 
# Where a channel has multiple values the reduction method is by "first" value.
# Maximum number of original frames: 128
# Requested frame slicing: <Slice on length=128 start=0 stop=8 step=4>, total number of frames presented here: 2
~A  X        CAL 
 11916.0      3.0
 11915.0      3.0
"""


@pytest.mark.parametrize(
    'frame_slice, channels, field_width, float_format, expected',
    (
        (None, set(), 16, '.3f', EXPECTED_LAS_ALL_FRAMES_ALL_CHANNELS_16_3F),
        (Slice.Slice(), set(), 16, '.3f', EXPECTED_LAS_ALL_FRAMES_ALL_CHANNELS_16_3F),
        (Slice.Slice(0, 8, 4), {'X   ', 'CAL ',}, 16, '.3f', EXPECTED_LAS_SLICE_0_8_4_TWO_CHANNELS_16_3F),
        (Slice.Slice(0, 8, 4), {'X   ', 'CAL ',}, 8, '.1f', EXPECTED_LAS_SLICE_0_8_4_TWO_CHANNELS_8_1F),
    )
)
def test_bit_file_to_las(frame_slice, channels, field_width, float_format, expected):
    with io.BytesIO(test_ReadBit.EXAMPLE_BIT_FILE) as bit_file:
        frame_arrays = ReadBIT.create_bit_frame_array_from_file(bit_file)
        frame_array = frame_arrays[0]
        with io.StringIO() as las_file:
            ToLAS.bit_frame_array_to_las_file(
                frame_array, 'bit_path', 0, frame_slice, channels, field_width, float_format, las_file,
            )
            result = las_file.getvalue()
            # print()
            # print(frame_array.long_str())
            print()
            print(result)
            # print_differences(expected, result)
            assert las_is_effectively_the_same(result, expected)

