import io
import math

import pytest

from TotalDepth.BIT import ReadBIT


ISINGL_EXAMPLES = (
    # From RP666V2
    (153.0, b'\x42\x99\x00\x00'),
    (-153.0, b'\xc2\x99\x00\x00'),
    (0.0, b'\x00\x00\x00\x00'),
    # https://en.wikipedia.org/wiki/IBM_hexadecimal_floating-point
    # Example: -118.625 -> b'\xc2\x76\xa0\x00'.
    (-118.625, b'\xc2\x76\xa0\x00'),
    # Positive version of same value, just the sign bit changes.
    (118.625, b'\x42\x76\xa0\x00'),
    # +7.2370051 × 10**75
    (7.2370051e75, b'\x7f\xff\xff\xfe'),
    # +5.397605 × 10−79
    (5.397605668656396e-79, b'\x00\x10\x00\x00'),
    (9.999999615829415e-05, b'\x3d\x68\xdb\x8b'),
    # 443a 6600 4438 fe00          D:f.D8..
    # 0000010c: 4040 0000 0000 0000 4210 0000 4d4e 3233  @@......B...MN23
    # 0000011c: 394a 2031
    (14950.0, b'\x44\x3a\x66\x00'),
    (14590.0, b'\x44\x38\xfe\x00'),
    (0.25, b'\x40\x40\x00\x00'),
    (16.0, b'\x42\x10\x00\x00'),
    (1375640257504053.0, b'\x4d\x4e\x32\x33'),
    (1.0786716607821755e-09, b'\x39\x4a\x20\x31'),
)


@pytest.mark.parametrize(
    'float_value, bytes_value',
    ISINGL_EXAMPLES
)
def test_isingl_bytes_to_float(float_value, bytes_value):
    result = ReadBIT.bytes_to_float(bytes_value)
    assert math.isclose(result, float_value, rel_tol=1e-7)


@pytest.mark.parametrize(
    'bytes_value, expected',
    (
        (b'', "Need at least 4 bytes not b''."),
    )
)
def test_bytes_to_float_raises(bytes_value, expected):
    with pytest.raises(ValueError) as err:
        ReadBIT.bytes_to_float(bytes_value)
    assert err.value.args[0] == expected


@pytest.mark.parametrize(
    'float_value, bytes_value',
    ISINGL_EXAMPLES
)
def test_isingl_float_to_bytes(float_value, bytes_value):
    result = ReadBIT.float_to_bytes(float_value)
    assert result == bytes_value


def test_binary_data_end():
    tell = 0x128
    tell += 4
    frames = 1472
    bytes_per_frame = 40
    tell += frames * bytes_per_frame
    padding = 12 * ((frames // 16) - 1)
    tell += padding
    assert tell == 60272
    assert f'0x{tell:x}' == '0xeb70'


# 15_17-_12/DWL_FILE/15_17-_12_dwl__1646505.bit
EXAMPLE_BIT_FILE = (
    b'\x00\x00\x00\x00\x00\x00\x00\x00 \x01\x00\x00\x00\x00\x00\x00OCCIDENTAL P'
    b'ETROLEUM                                                    \x00\x02\x00\x06'
    b'\x00U  15/17-12                                                            '
    b'    \x00\x07\x00\x17\x008  \x00\x11\x00\x00SSN LSN SSD LSD CAL TEN SPD THZ '
    b'UZ  KZ  GR  DEN CORRCN  TH  U   K               D.\x8c\x00\x00\x00\x00\x00'
    b'@@\x00\x00\x00\x00\x00\x00B \x00\x00X 227D 8\x00\x00\x00\x00\x00\x00\x00\x00'
    b'\xac\t\x00\x00C\x11o\x8eC\x11o\xbfC\x11p\xcbC\x11s&C\x11v\x8fC\x11z\xbc'
    b'C\x11~\xf5C\x11\x82\x1fC\x11\x83\tC\x11\x81\x99C\x11~iC\x11zVC\x11v\xd3'
    b'C\x11u\xa6C\x11w\xdaC\x11}\x16C\x11\x84\\C\x11\x8c\xaaC\x11\x94\x8d'
    b'C\x11\x9b\x19C\x11\xa0fC\x11\xa4\xabC\x11\xa8VC\x11\xab\xe4C\x11\xae\xfe'
    b'C\x11\xb0;C\x11\xae\x84C\x11\xa8\xe1C\x11\x9f\xacC\x11\x95\rC\x11\x8b\xa8'
    b'C\x11\x85\xd2B\x95\xfc)B\x95\xfc)B\x95\xfc)B\x95\xfc)B\x95\xf7\xa1'
    b'B\x95\xce\x9eB\x95?(B\x94/\xb8B\x92\xe3\xbdB\x91\xc8\xd5B\x919\x9aB\x91Y\xdc'
    b'B\x92!\\B\x93D\xdeB\x94U\x8fB\x95\x01\xbdB\x957SB\x951\xccB\x95\t8'
    b'B\x94\xb9\x8dB\x94mWB\x94K\x08B\x94T\xffB\x94`IB\x94L^B\x94\x06\xf6B\x93xK'
    b'B\x92\x9f\xc4B\x91\x9a\xa7B\x90\xaa\xb4B\x90\x17\xfcB\x8f\xf6YB\xf9\xf9\x9c'
    b'B\xf9\xf9\x9cB\xf9\xf9\x9cB\xf9\xf9\x9cB\xf9\xf9\x9cB\xf9\xf9\x9c'
    b'B\xf9\xf9\x9cB\xf9\xf9\x9cB\xf9\xf9\x9cB\xf9\xf9\x9cB\xf9\xf9\x9c'
    b'B\xf9\xf9\x9cB\xf9\xf9\x9cB\xf9\xf9\x9cB\xf9\xf9\x9cB\xf9\xf9\x9c'
    b'B\xf9\xf9\x9cB\xf9\xf9\x9cB\xf9\xf9\x9cB\xf9\xf9\x9cB\xf9\xf9\x9c'
    b'B\xf9\xf9\x9cB\xf9\xf9\x9cB\xf9\xf9\x9cB\xf9\xf9\x9cB\xf9\xf9\x9c'
    b'B\xf9\xf9\x9cB\xf9\xf9\x9cB\xf9\xf9\x9cB\xf9\xf9\x9cB\xf9\xf9\x9c'
    b'B\xf9\xf9\x9cBl\xfd7Bl\xfd7Bl\xfd7Bl\xfd7Bl\xfd7Bl\xfd7Bl\xfd7Bl\xfd7Bl\xfd7'
    b'Bl\xfd7Bl\xfd7Bl\xfd7Bl\xfd7Bl\xfd7Bl\xfd7Bl\xfd7Bl\xfd7Bl\xfd7Bl\xfd7'
    b'Bl\xfd7Bl\xfd7Bl\xfd7Bl\xfd7Bl\xfd7Bl\xfd7Bl\xfd7Bl\xfd7Bl\xfd7Bl\xfd7'
    b'Bl\xfd7Bl\xfd7Bl\xfd7A0\xc1\x11A0\xc1\x11A0\xc1\x11A0\xc1\x11A0\xc1\x11'
    b'A0\xc1\x11A0\xc1\x11A0\xc1\x11A0\xc1\x11A0\xc1\x11A0\xc1\x11A0\xc1\x11'
    b'A0\xc1\x11A0\xc1\x11A0\xc1\x11A0\xc1\x11A0\xc1\x11A0\xc1\x11A0\xc1\x11'
    b'A0\xc1\x11A0\xc1\x11A0\xc1\x11A0\xc1\x11A0\xc1\x11A0\xc1\x11A0\xc1\x11'
    b'A0\xc1\x11A0\xc1\x11A0\xc1\x11A0\xc1\x11A0\xc1\x11A0\xc1\x11\xc2m\xc9\xfc'
    b'\xc2m\xa0\x86\xc2l\xc5\xe6\xc2j\x92\xdd\xc2f\xa6q\xc2a\x1cF\xc2Z)\xed'
    b'\xc2Ql\xb4\xc2E\xbb\xe6\xc25\xb7\x86\xc2 \xb2}\xc1w#\xe3B\x13\xc4IB.\x0b\x1d'
    b'BE_\x9aBY\xa6\xadBl[\x02B\x7fc\x83B\x93\xe5\xd8B\xaa\x0f\xe9B\xc1~\xcc'
    b'B\xd9\xablB\xf2\x0f\x17C\x10\x9e\xf6C\x12\x03\x80C\x13;\xf1C\x14@='
    b'C\x15\x1f\xe3C\x16\x00;C\x17\x0c4C\x18]\x07C\x19\xec\x10B\x10\xff\x91'
    b'B\x10\xff\x91B\x10\xff\x91B\x10\xff\x91B\x10\xff\x91B\x10\xfe\x0f'
    b'B\x10\xf7\xf0B\x10\xe9\xbbB\x10\xcedB\x10\xa6\xafB\x10wfB\x10C\xe5B\x10\x13u'
    b"A\xff\x17[A\xfeS&A\xfe\xb9\xf7A\xff\xa5|B\x10\x02'A\xffrIA\xfd\x92;"
    b'A\xfa\xd7\xddA\xf7\xa6\xb6A\xf4\xc9\xf1A\xf2\xca\xc3A\xf1\xb9%A\xf1\x993'
    b'A\xf2S\xf8A\xf3s\xeeA\xf4@\xb8A\xf3\xf4!A\xf2\tlA\xee\xcc\xcfAo\xfd#Ao\xfd#'
    b'Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#'
    b'Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#'
    b'Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#'
    b'B \xff(B \xff(B \xff(B \xff(B \xff(B \xff(B \xff(B \xff(B \xff(B \xff('
    b'B \xff(B \xff(B \xff(B \xff(B \xff(B \xff(B \xff(B \xff(B \xff(B \xff('
    b'B \xff(B \xff(B \xff(B \xff(B \xff(B \xff(B \xff(B \xff(B \xff(B \xff('
    b'B \xff(B \xff(B"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1a'
    b'B"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1a'
    b'B"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1a'
    b'B"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1a'
    b'B"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1aB\xe4G\x0bB\xe4G\x0b'
    b'B\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4G\x0b'
    b'B\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4G\x0b'
    b'B\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4G\x0b'
    b'B\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4G\x0b'
    b'B\xe4G\x0bB\xe4G\x0bA,*\x13A,*\x13A,*\x13A,*\x13A,*\x13A,*\x13A,*\x13A,*\x13'
    b'A,*\x13A,*\x13A,*\x13A,*\x13A,*\x13A,*\x13A,*\x13A,*\x13A,*\x13A,*\x13'
    b'A,*\x13A,*\x13A,*\x13A,*\x13A,*\x13A,*\x13A,*\x13A,*\x13A,*\x13A,*\x13'
    b'A,*\x13A,*\x13A,*\x13A,*\x13@\x10\xd4\x90@\x10\xd4\x90@\x10\xd4\x90'
    b'@\x10\xd4\x90@\x10\xd4\x90@\x10\xd4\x90@\x10\xd4\x90@\x10\xd4\x90'
    b'@\x10\xd4\x90@\x10\xd4\x90@\x10\xd4\x90@\x10\xd4\x90@\x10\xd4\x90'
    b'@\x10\xd4\x90@\x10\xd4\x90@\x10\xd4\x90@\x10\xd4\x90@\x10\xd4\x90'
    b'@\x10\xd4\x90@\x10\xd4\x90@\x10\xd4\x90@\x10\xd4\x90@\x10\xd4\x90'
    b'@\x10\xd4\x90@\x10\xd4\x90@\x10\xd4\x90@\x10\xd4\x90@\x10\xd4\x90'
    b'@\x10\xd4\x90@\x10\xd4\x90@\x10\xd4\x90@\x10\xd4\x90B\x16\xfc{B\x16\xfc\xf5'
    b'B\x16\xff\x96B\x17\x05|B\x17\x0f[B\x17%\xd0B\x17Z\x98B\x17\xb3RB\x18\x1a\x04'
    b'B\x18mBB\x18\x91\x84B\x18|\xf0B\x1865B\x17\xdaMB\x17\x8d\xfaB\x17h\t'
    b'B\x17j\x83B\x17\x81\x11B\x17\xa0\xfeB\x17\xc9HB\x17\xed\x8cB\x18\x02\xb1'
    b'B\x18\x08\xfbB\x18\x0e\x96B\x18\x1cuB\x184\x9cB\x18[\xb0B\x18\x8f\xc9'
    b'B\x18\xc92B\x18\xf9\rB\x19\x11\xb7B\x19\x0c^B\x15\xf6-B\x15\xf6-B\x15\xf6-'
    b'B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-'
    b'B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-'
    b'B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-'
    b'B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-'
    b'B\x15\xf6-B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4'
    b'B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4'
    b'B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4'
    b'B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4'
    b'B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4'
    b'B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4'
    b'B\x19\x8e\xc4B\x19\x8e\xc4\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`'
    b'\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`'
    b'\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`'
    b'\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`'
    b'\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`\x00\x00\x00\x00'
    b' \x01\x00\x008\x12\x00\x00C\x11\x85%C\x11\x89\x9dC\x11\x91\x8fC\x11\x9a\x12'
    b'C\x11\xa1\x99C\x11\xa7&C\x11\xa9\x1dC\x11\xa5\x8cC\x11\x9c\xa9C\x11\x91\xc2'
    b'C\x11\x88\xceC\x11\x84\x96C\x11\x85\xa9C\x11\x89,C\x11\x8ahC\x11\x85\xb6'
    b'C\x11z\x81C\x11l\x85C\x11a\xa6C\x11^\x18C\x11acC\x11h\x92C\x11p\xd9C\x11xq'
    b'C\x11~\x01C\x11\x81\x06C\x11\x81\x86C\x11\x80\xaeC\x11\x80EC\x11\x81\x1e'
    b'C\x11\x83\tC\x11\x85\xf1B\x90t&B\x91\x93\xd1B\x93 (B\x94\xb7\x8e'
    b'B\x95\xf5\x81B\x96\x89\x80B\x96_\xdfB\x95\xa0GB\x94\xb7\x88B\x94\x03.'
    b'B\x93\xa9\xe9B\x93l\x01B\x93\x0b\x7fB\x92\x84&B\x92\nlB\x91\xfd^B\x92\x972'
    b'B\x93\xad0B\x94\xc5\xa9B\x95jeB\x95}=B\x955+B\x95\t\xd5B\x95T\xbeB\x95\xf8g'
    b'B\x96\xa0\xcaB\x97\t8B\x97\t\x8dB\x96\x9f\xa5B\x95\xe0\xccB\x94\xcfT'
    b'B\x93m\xe1B\xf9\xf9\x9cB\xf9\xf9\x9cB\xfa\x01)B\xfa.\xdbB\xfa\xb3\x14'
    b'B\xfb\xad+B\xfd\tMB\xfeqiB\xffc\xd8B\xff\x88\xc2B\xfe\xe2\x8aB\xfd\xca\xcf'
    b'B\xfc\xd3\x8aB\xfcr\x83B\xfc\xb6\xb6B\xfd^\x1dB\xfd\xfeIB\xfeVuB\xfeuh'
    b'B\xfe\xa1\x97B\xfe\xfd\xb9B\xff\x7f\x16C\x10\x00\xd0C\x10\x07\xcaC\x10\tT'
    b'C\x10\x04sB\xff\x8d!B\xfer{B\xfd\x17\xbcB\xfb\xba\xe8B\xfa\xa6FB\xfa\x1af'
    b'Bl\xfd7Bm\x0cRBmU\x96Bm\xf7\xeaBn\xe7\x05Bo\xf24Bp\xd8\xa1BqwsBq\xe6\xaeBren'
    b'Bs+\xcdBt:sBuK1Bu\xf2/Bu\xec\xc0Bu@(BtB|Bsw BsA\x01Bs\xac\xceBt\x7f~Bua|'
    b'Bv\x18\x81Bv\x91BBv\xc4\x90Bv\xc7\x93Bv\xca*Bv\xdc\x8cBv\xda?Bv\xae\x9eBvke'
    b'Bv2\x00A0\xc1\x11A0\xc1\x11A0\xc0\xaaA0\xbf>A0\xbcHA0\xb6KA0\xac\x88'
    b'A0\x9f\xf2A0\x92\xccA0\x87\xfdA0\x82\x04A0\x81FA0\x84\xafA0\x89IA0\x8cA'
    b'A0\x8d}A0\x8fUA0\x926A0\x94\xccA0\x96\xc7A0\x98\xc0A0\x9b\xf6A0\xa1\xdf'
    b'A0\xaa\xacA0\xb4\x1bA0\xbc"A0\xc0IA0\xc0LA0\xbd[A0\xb9\x0eA0\xb4\x1fA0\xaf\\'
    b'C\x1b\x95\x17C\x1d\'*C\x1e}\x1cC\x1f\x8e_C n\xb3C!?\xaaC"\x1e\xefC#\x1d\x13'
    b"C$:{C%k3C&\x9e\x87C'\xc6\xe8C(\xdf\xd8C)\xed[C*\xf5'C+\xf7\xc1C,\xee\xdb"
    b'C-\xd0\xe4C.\x96\xa8C/B6C/\xdf\xb1C0}\xadC1\x1f\x84C1\xb6\x86C2(\x0bC2^\xe5'
    b'C2\\(C2=HC20\xcdC2^\x06C2\xcd\xa4C3e\xc8A\xebQ\x84A\xe8\x83\x15A\xe6\x91\xd1'
    b'A\xe5e-A\xe4\xa1\xccA\xe3\xadEA\xe22aA\xe0\x8c\x80A\xdfiGA\xdf\x00\xd4'
    b'A\xde\xe62A\xdeZ\xf0A\xdc\xdc\xc8A\xda_\x08A\xd7#;A\xd3\x82BA\xd0PJ'
    b'A\xcep\xa4A\xce\x1cEA\xce\xda\xbbA\xd0!/A\xd1-wA\xd1;fA\xd0\x14\x14'
    b'A\xceU\x9dA\xcc\xc9\x1eA\xcb\xe5\xcbA\xcb\xc9\xf0A\xcc7MA\xcc\xc6\xaaA\xcdrI'
    b'A\xce%\xb1Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#'
    b'Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#'
    b'Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#'
    b'Ao\xfd#Ao\xfd#Ao\xfd#B \xff(B \xff(B \xff(B \xff(B \xff(B \xff(B \xff('
    b'B \xff(B \xff(B \xff(B \xff(B \xff(B \xff(B \xff(B \xff(B \xff(B \xff('
    b'B \xff(B \xff(B \xff(B \xff(B \xff(B \xff(B \xff(B \xff(B \xff(B \xff('
    b'B \xff(B \xff(B \xff(B \xff(B \xff(B"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1a'
    b'B"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1a'
    b'B"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1a'
    b'B"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1a'
    b'B"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1a'
    b'B\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4G\x0b'
    b'B\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4G\x0b'
    b'B\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4G\x0b'
    b'B\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4G\x0b'
    b'B\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4G\x0bA,*\x13A,&\xeeA,\x18\xc8A+\xfd\x9a'
    b'A+\xde\xf7A+\xcb`A+\xcd\x10A+\xde\xc6A+\xe9\xc9A+\xd5\xa9A+\x97\xcbA+<k'
    b'A*\xe5rA*\xb7\xe0A*\xc2UA*\xfa\xa8A+A\xedA+u\xcaA+\x84\xabA+u\x8eA+Y\x0e'
    b"A+>\xf7A+/:A+'NA+ \xdcA+\x15\xa9A*\xfb\xacA*\xd1~A*\xa2EA*z_A*`\xd0A*XM"
    b'@\x10\xd4\x90@\x10\xb6`@\x105 ?\xf5\x89\x00?\xea~\x00?\xec\x9a\x00@\x10\rp'
    b'@\x11\xf4\x00@\x131\xd0@\x12\x90\xa0?\xfa\xce\x00?\xb4\x9a\x00?r\xc3\x00'
    b'?R\n\x00?\\\x15\x00?\x87 \x00?\xba\xbb\x00?\xde\xb6\x00?\xe9Z\x00'
    b'?\xe2\xaf\x00?\xd6\x9b\x00?\xce\x00\x00?\xccT\x00?\xcd\xa4\x00?\xcb\x1c\x00'
    b'?\xc0#\x00?\xa6X\x00?}\x91\x00?N+\x00?"\xf2\x00>D \x00\xbe\x8aP\x00'
    b'B\x18\xe0\xebB\x18\x92bB\x18,\xb7B\x17\xc7IB\x17{\xd6B\x17^@B\x17o]'
    b'B\x17\x9e\xedB\x17\xcd\xd6B\x17\xe8cB\x17\xec\x91B\x17\xf4\x8dB\x18\x14\x9b'
    b'B\x18F\xfcB\x18o\xa8B\x18g\x9bB\x18\x1b\x89B\x17\xa3\x98B\x174\x8aB\x16\xfb0'
    b'B\x16\xfd\xf5B\x17%(B\x17F\xc8B\x17C\xd4B\x17!\xc9B\x16\xf8RB\x16\xdb]'
    b'B\x16\xd9+B\x16\xf6\xc4B\x170\x7fB\x17\x86\x01B\x17\xf7qB\x15\xf6-B\x15\xf6-'
    b'B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-'
    b'B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-'
    b'B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-'
    b'B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-'
    b'B\x15\xf6-B\x15\xf6-B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4'
    b'B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4'
    b'B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4'
    b'B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4'
    b'B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4'
    b'B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4'
    b'B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4\xbf\xac)`\xbf\xac)`\xbf\xac)`'
    b'\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`'
    b'\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`'
    b'\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`'
    b'\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`'
    b'\xbf\xac)`\x00\x00\x00\x00\xac\t\x00\x00\xc4\x1a\x00\x00C\x11\x89\x0c'
    b'C\x11\x8a_C\x11\x88VC\x11\x82\xf5C\x11{hC\x11s+C\x11kXC\x11d\xa5C\x11`@'
    b'C\x11^UC\x11]\xdcC\x11]\xf4C\x11_rC\x11cUC\x11i\x17C\x11n\xaeC\x11r\x89'
    b'C\x11s}C\x11qeC\x11m"C\x11hNC\x11d1C\x11b\x1aC\x11byC\x11d\xf3C\x11h"'
    b'C\x11i\xfeC\x11h\xd0C\x11e\x08C\x11`\xe2C\x11_\x14C\x11`\xe2B\x91\xf7z'
    b'B\x90\xc3[B\x90! B\x9088B\x90\xe7\xe3B\x91\xaeOB\x92+\x8bB\x92UnB\x92Ly'
    b'B\x92U\x05B\x92\x8b\xd3B\x92\xb9\xf5B\x92\xab\x80B\x92YDB\x91\xf8\xa0'
    b'B\x91\xbc\x85B\x91\xc6,B\x92\x11\\B\x92\x7f\x93B\x92\xfb\x9eB\x93|\x02'
    b'B\x93\xebxB\x946nB\x94H0B\x94"uB\x93\xd8\x03B\x93\x98\x1fB\x93\x95\xa2'
    b"B\x93\xd3.B\x94*\xefB\x94}\xb8B\x94\xdc\x10B\xfa'\xc2B\xfa\xaaDB\xfbo\xe6"
    b'B\xfcG\x18B\xfc\xeb^B\xfd\x14\x0cB\xfc\xb8\xa8B\xfc)\xfdB\xfb\xd7\xf6'
    b'B\xfc\x17\xbcB\xfc\xf9sB\xfe&gB\xff$\xe4B\xff\xab\xaeB\xff\xd6{B\xff\xe9^'
    b'C\x10\x01\xe8C\x10\x06\xb7C\x10\x08wC\x10\x04FB\xff\xc4\xb8B\xffGGB\xfe\xf4o'
    b'B\xfe\xc6\x0cB\xfe\x93\xa4B\xfe6\x80B\xfd\xb8\xb6B\xfd9\x11B\xfc\xcd\x15'
    b"B\xfc\x81\x00B\xfcN\xb8B\xfc3\x00Bv'?Bv\\\x9fBv\xa6%Bv\xb6\xa7BvM\xc6Buq\xb9"
    b'BtW\x1fBsH\xd6Brz\xd6Bq\xffqBq\xec\xa6BrZ\nBs;\xc1Bt~\x7fBu\xfa_BwThBx\t\xa7'
    b'Bw\xe1\xffBw\x15hBv\x03\xb0Bu\x05tBta;Bt1\xb2Bt\\5Bt\xa6hBt\xcbKBt\xb7\x95'
    b'Bt\x99KBt\xc2HBuU\x89Bv?(Bw2\xa5A0\xabWA0\xa9\x96A0\xa9\xbfA0\xab\xde'
    b'A0\xb0\x85A0\xb8\xd5A0\xc5\xa8A0\xd6+A0\xe5\x9dA0\xed\xa7A0\xea\x8b'
    b'A0\xdd\xf6A0\xca\xb7A0\xb4\xf6A0\x9f\xf3A0\x8c\x9aA0{\x1eA0l\x00A0`\xf8'
    b'A0]\xf0A0c\x15A0m\xedA0|\x19A0\x8c\xecA0\xa0}A0\xb5\xdaA0\xcbKA0\xde+'
    b'A0\xeb\xb8A0\xf1jA0\xec\xf0A0\xdf\xaeC3\xfe\x17C4{\xdfC4\xdd\x00C5.aC5}\xaf'
    b"C5\xd2\xbcC60\xc9C6\x92\xccC6\xe6fC7\x15\x1fC7\x19\xe7C7\x06\x9cC6\xfehC7'T"
    b'C7\x98\x9cC8J\xc4C9\x16qC9\xc5\xc7C:8\xbaC:|\xe9C:\xbd\xe3C;\x1a\x8cC;\x8fe'
    b'C;\xfe\xdeC<G\x8dC<X\xe5C<<_C<\x10\xb8C;\xf7\x85C;\xff\xd5C<\x1f@C<Ay'
    b'A\xce\x8bDA\xce\xae\x16A\xce\xa7\x1fA\xce\x83|A\xceA\x8aA\xcd\xe63A\xcd;\xce'
    b'A\xcc"\x01A\xca\xb3\x00A\xc9\x117A\xc7FuA\xc5\x93\xe0A\xc3\xfaFA\xc2\x93\xe0'
    b'A\xc1\xba`A\xc1\xa25A\xc2\x19\xd0A\xc2\xb5\rA\xc2\xfb\x18A\xc2\xb5\r'
    b'A\xc1\xe9|A\xc0\x95\xecA\xbe\xb5\xdeA\xbc}\xc0A\xbaojA\xb8\xd5hA\xb7\xb6F'
    b'A\xb6\xb4\xa3A\xb5`\xaaA\xb3\x81\x07A\xb16zA\xae\xdb\x8cAo\xfd#Ao\xfd#'
    b'Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xfd#Ao\xb4\xa3An.JAj\xd6:Af\xeb\xf0Ae#;'
    b'Ah*\x9bAp\xcc\xd0A}e\x95A\x89\xe63A\x91\xf0pA\x93i\xaeA\x8f\x9e\x85'
    b'A\x89\xb9$A\x85TbA\x83\x9c\x10A\x83\x95\x83A\x83WAA\x818\x1fA|w\x9bAv=\t'
    b'Ap\xed\xfbAn2bAn-\xe2Ap\x8f\xc6As\x87\xfdB \xff(B \xff(B \xff(B \xff(B \xff('
    b'B \xff(B \xff(B!\x03\xb0B!\x1f\x1bB!e`B!\xb4\x95B!\xb0\x85B!\x1d\x1c'
    b'B\x1f\xfa\xc7B\x1e\x84bB\x1d\x1f\x00B\x1cI\x99B\x1cpjB\x1d\xa5\x96B\x1f\x7f)'
    b'B!8\xd5B"\x1e\x00B!\xe6SB \xaaDB\x1e\xe9\xe8B\x1dg\x04B\x1c\xc3\xb0'
    b'B\x1d\x17\xb5B\x1d\xe9\xeeB\x1e\xb4\xe4B\x1f4oB\x1f\\\x1cB"\xff\x1a'
    b'B"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1aB"\xff\x1aB#\x0f\xb7'
    b'B#C\xebB#w8B#|\x9fB#;\xeeB"\xa4\xc9B!\xdbJB!@wB!,\xd3B!\xa6LB"s&B#9\x10'
    b'B#\x94nB#caB"\xae\xb8B!\x82\x13B 8\xcfB\x1f%\x0bB\x1e[RB\x1d\xda\xf5'
    b"B\x1d\xbfNB\x1e\x13'B\x1e\xdbRB \x0f\xedB!\x84\xb0B\xe4G\x0bB\xe4G\x0b"
    b'B\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4G\x0bB\xe4T\x91B\xe4\x8a\xab'
    b'B\xe4\xfb\xa3B\xe5\xab\xf2B\xe6J\x8dB\xe6M\xbcB\xe5m\xdaB\xe3\xbf\xca'
    b'B\xe1\xb3\x8fB\xdf\xdc\xf6B\xde\xbakB\xdev\x99B\xde\xcb\xd9B\xdfHhB\xdf\xbak'
    b'B\xe00\x85B\xe0\xb7\x15B\xe1\x19tB\xe1<\xf2B\xe1N\xd0B\xe1\x95\xb5B\xe2/\xca'
    b'B\xe3 \x88B\xe48wB\xe4\xf7?A*\\;A*d1A*q\x97A*\x8cCA*\xb6\xfcA*\xe6\xe4'
    b'A+\x11\x14A+29A+O~A+p\xceA+\x93\xb5A+\xa7kA+\x9d\xb9A+q\x06A+-uA*\xed\xf4'
    b'A*\xd3\x04A*\xe4\xfbA+\x0f\x94A+:\xc4A+Z@A+i\x01A+f\xf2A+XOA+B\xf9A+/\x0c'
    b'A+!\x96A+\x15\xe4A*\xff\x0cA*\xd8\x0cA*\xa4&A*q\xdd\xbeX\xa0\x00>h\x10\x00'
    b'?\x19\x85\x00?5s\x00?X*\x00?w.\x00?\x8b}\x00?\x97\x84\x00?\xa4\x8f\x00'
    b'?\xbc\x1a\x00?\xdd\x83\x00?\xf9\xe4\x00@\x10\x1f\xd0?\xee\x7f\x00?\xc8@\x00'
    b'?\xa3"\x00?\x95\xe6\x00?\xa4\xdf\x00?\xbf\xfa\x00?\xd6F\x00?\xe2.\x00'
    b'?\xe44\x00?\xder\x00?\xd3\x1e\x00?\xc3\x8b\x00?\xb2z\x00?\xa3}\x00?\x95r\x00'
    b'?\x81\xc8\x00?f*\x00?D6\x00?${\x00B\x18r\x03B\x18\xd5\x8cB\x19\x04\x1e'
    b'B\x18\xee)B\x18\xa3AB\x18P?B\x18\x15\x8fB\x17\xf7\x88B\x17\xee\xfbB\x17\xe7r'
    b'B\x17\xd5\x8eB\x17\xc7\xcbB\x17\xcf\xffB\x17\xf2\xffB\x18\x1fbB\x18@HB\x18G>'
    b'B\x182\x8cB\x18\x0bdB\x17\xda\xbcB\x17\xa7\xa3B\x17{\xc3B\x17`\x1eB\x17[\xc8'
    b'B\x17mHB\x17\x8b\x9aB\x17\xa3|B\x17\xa1;B\x17\x85)B\x17`sB\x17CNB\x17+\xfe'
    b'B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-B\x15\xf6-'
    b'B\x15\xe7\xf5B\x15\x9biB\x14\xf3\x88B\x14.\xfaB\x13\xd5jB\x14mxB\x16\x1e\xe7'
    b'B\x18\x97LB\x1b\n\xf5B\x1c\x9e\x97B\x1c\xe8\x91B\x1c*\x1eB\x1b\x02\x1f'
    b'B\x1a%\x8dB\x19\xcf3B\x19\xcd\xeaB\x19\xc1\xb5B\x19W2B\x18h\xa1B\x17/\xed'
    b'B\x16%hB\x15\x9c6B\x15\x9bTB\x16\x12\xefB\x16\xa8\x04B\x19\x8e\xc4'
    b'B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4B\x19\x8e\xc4'
    b'B\x19\x8e\xc4B\x19\x97\xf3B\x19\xcciB\x1aI\x8bB\x1a\xd9\x01B\x1a\xf2x'
    b'B\x1a-\xe3B\x18~\x97B\x16:6B\x14\x08LB\x12\xafEB\x12\xbdzB\x14/vB\x16g\x98'
    b'B\x18g-B\x19g]B\x190uB\x17\xfabB\x16_\xe5B\x15-\xfeB\x14\xf2\xad'
    b'B\x15\x9d\xf7B\x16\x9c\x13B\x17f\x10B\x17\xbdlB\x17\xb3\xbe\xbf\xac)`'
    b'\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xac)`\xbf\xa5\xf2\x10'
    b'\xbf\x9c\xa4\xb0\xbf\xb4\xbd\xb0\xbf\xef\x1f`\xc0\x11R\xde\xc0\x10$\xfc'
    b'\xbf\xa0H\x10?\x1eh\xe0@\x12\x0e\xe3@5\xee[@M\xef\xa5@3c\xb9?\xe0\xbd\xd4'
    b'\xbfy\x9f\xe0\xc0\x18\x84\x9f\xc0!.\xf4\xc0\x1fy\x8a\xc0\x16%\x1f'
    b'\xbf\xd0p\x80\xbf\xb2\xb2\xf0\xc0\x10as\xc0\x16\xc8B\xc0\x18O\xfd\xc0\x12KN'
    b'\xbf_\xc4P\x00\x00\x00\x008\x12\x00\x00P#\x00\x00C\x11e\xd9C\x11k6C\x11n\x1a'
    b'C\x11mwC\x11i\xf2C\x11dJC\x11\\\x8bC\x11S\x06C\x11ILC\x11A\xabC\x11>\x8c'
    b'C\x11A\x1dC\x11HEC\x11P\xd2C\x11VJC\x11U\xe9C\x11PKC\x11I\x00C\x11C\x81'
    b'C\x11BAC\x11E\xb1C\x11K\xc5C\x11Q;C\x11S\x8aC\x11Q\x17C\x11J8C\x11J8C\x11J8'
    b'C\x11J8C\x11J8C\x11J8C\x11J8B\x95oqB\x96ATB\x97)\xbbB\x97\xe2\xfeB\x98HX'
    b'B\x98U\x1fB\x98\x14\xd6B\x97\xa8\xbaB\x97(+B\x96\x8e)B\x95\xd2\x1aB\x95\x17?'
    b'B\x94\xce\xc0B\x95@4B\x96TaB\x97\x98\xc7B\x98\x82\xb1B\x98\xad~B\x98\x05?'
    b'B\x96\xd6\xdcB\x95\x8aJB\x94j\xe8B\x93\xa2oB\x931!B\x93\x07\xa0B\x93\x16\xbc'
    b'B\x93\x16\xbcB\x93\x16\xbcB\x93\x16\xbcB\x93\x16\xbcB\x93\x16\xbc'
    b'B\x93\x16\xbcB\xfc3\xc6B\xfcZ\xfbB\xfc\xb4\x13B\xfd3\xc5B\xfd\xd6fB\xfe\xa9R'
    b'B\xff\xa3\xecC\x10\tGC\x10\x13\xc3C\x10\x15\xdaC\x10\x0e.B\xff\xf6\xa2'
    b'B\xfe\xf7RB\xfe\x1c\xe9B\xfd\x83vB\xfd7\xa2B\xfd*yB\xfde\x12B\xfe\t\x9a'
    b'B\xfe\xfe\xa0B\xff\xf8\x9bC\x10\t\x0cC\x10\x06\x8aB\xfflDB\xfd\xec\xc8'
    b'B\xfcn\tB\xfcn\tB\xfcn\tB\xfcn\tB\xfcn\tB\xfcn\tB\xfcn\tBw\xcc\xaeBw\xcb\xbb'
    b'BwKJBv\x93\xe5Bu\xe0\x01Bu@cBt\x9c\x16Bs\xc7\x11Br\xd0\xb1Br\x10\xa6'
    b'Bq\xea\x03Br\x86\xa2Bs\xbb\xceBu \x1cBv;\xafBv\xba\x1dBv~\xf4Bu\xa8\nBt\x898'
    b'Bs{lBr\xc5FBr\xa0\xd9Bs;6Bth\xd6Bu\xb9\xa7Bv\xa9\xcfBv\xa9\xcfBv\xa9\xcf'
    b'Bv\xa9\xcfBv\xa9\xcfBv\xa9\xcfBv\xa9\xcfA0\xcd\x9fA0\xba\xd1A0\xaa\x0f'
    b'A0\x9e\xa4A0\x9a\xfbA0\xa0MA0\xab\x91A0\xb6SA0\xb9\x83A0\xb2\xdfA0\xa6O'
    b'A0\x9c\x19A0\x9aJA0\xa1\xfaA0\xaf\xecA0\xbe\xbeA0\xcatA0\xd1\x8fA0\xd3\xa0'
    b'A0\xcf\xfaA0\xc6\x1cA0\xb7\x0fA0\xa7iA0\x9dnA0\x9d\xa1A0\xa5kA0\xa5kA0\xa5k'
    b"A0\xa5kA0\xa5kA0\xa5kA0\xa5kC<\\'C<t\x1eC<\x94\xb3C<\xc2\xf6C<\xf8\x8aC=)d"
    b'C=MoC=giC=\x80\x97C=\x9caC=\xb3\xb7C=\xbd\x96C=\xb8KC=\xaa\x1cC=\x9aA'
    b'C=\x8e\rC=\x88\x96C=\x8b\xebC=\x97\x90C=\xa8sC=\xbbWC=\xcf(C=\xdd\xc5'
    b'C=\xde\xc1C=\xd1\xcaC=\xc1\xffC=\xc1\xffC=\xc1\xffC=\xc1\xffC=\xc1\xff'
    b'C=\xc1\xffC=\xc1\xffA\xad\x08\x9aA\xac\x1e\xb9A\xac,\xa5A\xac\xfb\x7fA\xae>C'
    b"A\xaf\x81pA\xb0[XA\xb0z\xe2A\xaf\xcb*A\xae\xa0'A\xad\xd3\xc4A\xae\x0f\x93"
    b'A\xaf/\xedA\xb0M\xd4A\xb0\x899A\xaf\x89\xa1A\xad\x88\xcfA\xab\r\xee'
    b'A\xa8\xca\xc1A\xa7-\xe1A\xa6p\xa4A\xa6\x93\xdfA\xa7\x8f^A\xa9^6A\xabu&'
    b'A\xac\xc2\xfaA\xac\xc2\xfaA\xac\xc2\xfaA\xac\xc2\xfaA\xac\xc2\xfa'
    b'A\xac\xc2\xfaA\xac\xc2\xfaAu\x16\xf1At\xdb%Asd\xc3Aq\x8f\xc6Ao|\x85Al\x9al'
    b'Ah\xed\x92Ae\x9e\x1dAc\x84NAb\x94\xb0Ab:+Aa\xd4-A`\xc0\x1bA_7\xb6A]\xe7m'
    b"A^s\x82Ab1\xf9Ai\n>AqTaAy\xc7\x12A\x80'SA\x81\xac\xdbA}\t\x05As|\x86Ah\x86["
    b'A`\xe2\x85A`\xe2\x85A`\xe2\x85A`\xe2\x85A`\xe2\x85A`\xe2\x85A`\xe2\x85'
    b'B\x1f8\xdcB\x1e\xdb/B\x1ea\xd1B\x1d\xef\xedB\x1d\x98\xa8B\x1dU3B\x1d(\xcf'
    b"B\x1d\x16\xd5B\x1d\x19\xd4B\x1d#\xb0B\x1d'9B\x1d\x08\xb9B\x1c\xa6\xa1"
    b'B\x1b\xef\x8bB\x1b\x05\xb5B\x1a9\xb4B\x19\xe2\xf8B\x1a\x13\x8fB\x1a\xa2\x0c'
    b'B\x1b9kB\x1b\xa6\xeaB\x1b\xebQB\x1c\x05MB\x1b\xdfiB\x1bd\x19B\x1a\xb0\xdf'
    b'B\x1a\xb0\xdfB\x1a\xb0\xdfB\x1a\xb0\xdfB\x1a\xb0\xdfB\x1a\xb0\xdf'
    b'B\x1a\xb0\xdfB"\xd1UB#\xa4\x18B#\xcc\xc6B#j)B"\xcd\xe7B"<7B!\xc5\xbcB!w\xdd'
    b'B!aNB!\xa3\xecB"E\x18B#\x12,B#\xa3\x81B#\xad\x7fB#=\x0eB"\xb3\x8fB"k#B"t\x8e'
    b'B"\x90\xffB"\x84\xf1B"F\x1eB"\x1a\xc8B"I\xdbB"\xd2\xaaB#dZB#\x9e\xe7'
    b'B#\x9e\xe7B#\x9e\xe7B#\x9e\xe7B#\x9e\xe7B#\x9e\xe7B#\x9e\xe7B\xe4\xcf\xfa'
    b'B\xe3\x9a\xf3B\xe1\xc5\xc2B\xdf\xddhB\xde?\xb6B\xddA\xdeB\xdd2\xc2'
    b'B\xdd\xff\xb1B\xdf/\xeeB\xe0-QB\xe0\x9f\xd5B\xe0i\xbfB\xdf\x92\x86B\xdeR~'
    b'B\xdd\x06KB\xdb\xf6\x93B\xdb6\x8dB\xda\xab\x97B\xdaD\xf1B\xd9\xe5}B\xd9g\xa2'
    b'B\xd8\xcb\xf0B\xd8Q\x1aB\xd8$>B\xd85\xb6B\xd8v\xcfB\xd8v\xcfB\xd8v\xcf'
    b'B\xd8v\xcfB\xd8v\xcfB\xd8v\xcfB\xd8v\xcfA*T\xc2A*Z^A*\x7f\x02A*\xb3\x8f'
    b'A*\xecjA+(\x1cA+j5A+\xb4\xa6A+\xfb\xd6A,&eA,\x1dhA+\xde\x13A+~`A+\x1b\x1b'
    b'A*\xcfkA*\xac\xcaA*\xb6HA*\xe7\x95A+5\xb9A+\x8c\x00A+\xd1\xf9A+\xee\x00'
    b'A+\xca\x1fA+l\x96A*\xf6\xdfA*\x94\x15A*\x94\x15A*\x94\x15A*\x94\x15'
    b'A*\x94\x15A*\x94\x15A*\x94\x15?\x13\n\x00?\x18\x93\x00?3\x7f\x00?Z\x18\x00'
    b'?\x85.\x00?\xb4\x98\x00?\xe9\xfb\x00@\x12=@@\x15z\xf0@\x170\xe0@\x16p\x10'
    b'@\x13A@?\xec\xad\x00?\xa5\x19\x00?o7\x00?V=\x00?[8\x00?|\r\x00?\xb4\x06\x00'
    b'?\xf5B\x00@\x12\xce\xb0@\x14a0@\x12\xe5\x80?\xe8`\x00?\x8c\xb1\x00?<K\x00'
    b'?<K\x00?<K\x00?<K\x00?<K\x00?<K\x00?<K\x00B\x17\r6B\x16\xdd\x81B\x16\xa1\xb6'
    b'B\x16kGB\x16E\xe3B\x164TB\x163SB\x16:RB\x16F\xa6B\x16_\x92B\x16\x8d\xcf'
    b'B\x16\xca~B\x16\xf1\xc2B\x16\xe6\x1dB\x16\xa3\xadB\x16E\xfdB\x15\xf6<'
    b'B\x15\xd8RB\x15\xf9\xd3B\x16LEB\x16\xb4\x8bB\x17\x17\xfcB\x17a[B\x17\x890'
    b'B\x17\x8fjB\x17yOB\x17yOB\x17yOB\x17yOB\x17yOB\x17yOB\x17yOB\x16\xf6A'
    b'B\x16\xea\x87B\x16\xa1\x1bB\x16E#B\x15\xdc\xf4B\x15L5B\x14\x93\xb4'
    b'B\x13\xed\x83B\x13\x84\x0bB\x13U\x0eB\x13CNB\x13/MB\x12\xf9*B\x12\xac6'
    b'B\x12jDB\x12\x85\xbcB\x13A\xb2B\x14\x99SB\x169}B\x17\xe1\x9bB\x19!\xb3'
    b'B\x19n\x16B\x18\x85%B\x16\xa5\xc4B\x14\x7fvB\x12\xff\xe9B\x12\xff\xe9'
    b'B\x12\xff\xe9B\x12\xff\xe9B\x12\xff\xe9B\x12\xff\xe9B\x12\xff\xe9B\x17v\xf0'
    b'B\x17\x1d\xb5B\x16\xbdHB\x16jdB\x165\xfbB\x16"\x9fB\x163HB\x16X(B\x16}\xe0'
    b'B\x16\x97(B\x16\xa0\x84B\x16\x88\xcbB\x169$B\x15\x9c\x8aB\x14\xc9\xe1'
    b'B\x13\xf6\x18B\x13a\xfcB\x13!\x14B\x13%\x9aB\x130UB\x133\xb6B\x13^\x86'
    b'B\x13\xc5\x14B\x14=TB\x14x\rB\x14DGB\x14DGB\x14DGB\x14DGB\x14DGB\x14DG'
    b'B\x14DG?\x80\x12\x80@\x13\xe4k@\x1d\x97\x06@"]W@\x19\xd7\xdb@\x16\xf9m'
    b'@\x144\x8c@\x11\xcf5@\x10\xa5\xf2@\x12\x96\xdd@\x18[\x92@8*\xa1@^\x81\xae'
    b'@\x7f\x9fe@\x97\xa5\xb6@\xa7i\xf5@\xad\\\xae@\xa8\xc2 @\x96\xfc%@}\xb8\xdd'
    b'@c\x8cr@Q\xac\xbe@RUF@i\xaa0@\x90\xcfx@\xb6&+@\xb6&+@\xb6&+@\xb6&+@\xb6&+'
    b'@\xb6&+@\xb6&+\x01\x00\x00\x00\xc4\x1a\x00\x00\\#\x00\x00\x01\x00\x00\x00'
    b'P#\x00\x00h#\x00\x00\x01\x00\x00\x00\\#\x00\x00t#\x00\x00'
)


def test_is_bit_file():
    with io.BytesIO(EXAMPLE_BIT_FILE) as file:
        assert ReadBIT.is_bit_file(file)


def test_create_bit_frame_array_from_file_has_one_log_pass():
    with io.BytesIO(EXAMPLE_BIT_FILE) as file:
        frame_array = ReadBIT.create_bit_frame_array_from_file(file)
        assert len(frame_array) == 1


def test_create_bit_frame_array_from_file_temporary_frames_empty():
    with io.BytesIO(EXAMPLE_BIT_FILE) as file:
        frame_array = ReadBIT.create_bit_frame_array_from_file(file)
        assert frame_array[0]._temporary_frames == []


@pytest.mark.parametrize(
    'attribute, expected',
    (
        ('description', b'OCCIDENTAL PETROLEUM                                                    '),
        ('channel_names',
         ['SSN ', 'LSN ', 'SSD ', 'LSD ', 'CAL ', 'TEN ', 'SPD ', 'THZ ', 'UZ  ', 'KZ  ', 'GR  ', 'DEN ', 'CORR',
          'CN  ', 'TH  ', 'U   ', 'K   ']),
        ('frame_count', 128),
        ('ident', '0'),
        ('len_channels', 17),
        ('unknown_b', b'U  15/17-12                                                                '),
    )
)
def test_create_bit_frame_array_from_file_attributes(attribute, expected):
    with io.BytesIO(EXAMPLE_BIT_FILE) as file:
        frame_array = ReadBIT.create_bit_frame_array_from_file(file)
        # print(frame_array)
        assert hasattr(frame_array[0], attribute)
        assert getattr(frame_array[0], attribute) == expected


@pytest.mark.parametrize(
    'attribute, expected',
    (
        ('depth_from', 11916.0),
        ('depth_to', 0.0),
        ('spacing', 0.25),
        ('unknown_a', 0.0),
        ('unknown_b', 32.0),
        ('frames', 47665),
        ('is_increasing', False),
    )
)
def test_create_bit_frame_array_from_file_bit_log_pass_range(attribute, expected):
    with io.BytesIO(EXAMPLE_BIT_FILE) as file:
        frame_array = ReadBIT.create_bit_frame_array_from_file(file)
        # print(frame_array)
        assert hasattr(frame_array[0].bit_log_pass_range, attribute)
        assert math.isclose(getattr(frame_array[0].bit_log_pass_range, attribute), expected, rel_tol=1e-7)


def test_create_bit_frame_array_from_file_long_str():
    with io.BytesIO(EXAMPLE_BIT_FILE) as file:
        frame_array = ReadBIT.create_bit_frame_array_from_file(file)
        # print()
        # print(frame_array[0].long_str())
        assert frame_array[0].long_str().startswith('BITFrameArray: ident="0"')


@pytest.mark.parametrize(
    'offset, expected_value',
    (
        (0x104, 11916.0,),  # Start
        (0x108, 0.0,),  # Stop
        (0x10c, 0.25,),  # Spacing
        (0x110, 0.0,),  # ???
        (0x114, 32.0,),  # ???
    )
)
def test_floats_at_offset(offset, expected_value):
    with io.BytesIO(EXAMPLE_BIT_FILE) as file:
        file.seek(offset)
        result = ReadBIT.read_float(file)
        assert math.isclose(result, expected_value, rel_tol=1e-7)
