# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pydantic_sqlite']

package_data = \
{'': ['*']}

install_requires = \
['pydantic>=1.8.2,<2.0.0',
 'sqlite-utils>=3.19,<4.0',
 'typing-inspect>=0.7.1,<0.8.0']

setup_kwargs = {
    'name': 'pydantic-sqlite',
    'version': '0.2.1',
    'description': 'Simple package for storing pydantic BaseModels in an in-memory SQLite database.',
    'long_description': '# pydantic_sqlite\nSimple package for storing pydantic BaseModels in an in-memory SQLite database.\n\n## Installation\n\n    pip install pydantic-sqlite\n\n## Basic Example\nCreate two objects of the type TestCase and add them to the database in the table \'Test\'. Later, all values in the table are printed while iteration over the Table \'Test\'.\n\n``` python\nfrom pydantic_sqlite import DataBase\nfrom pydantic import BaseModel\nfrom uuid import uuid4\n\nclass TestCase(BaseModel):\n    uuid: str\n    name: str \n    age: int\n        \ntest1 = TestCase(uuid=str(uuid4()), name="Bob", age=12)\ntest2 = TestCase(uuid=str(uuid4()), name="Alice", age=28)\n\ndb = DataBase()\ndb.add("Test", test1)\ndb.add("Test", test2)\n\nfor x in db("Test"):\n    assert issubclass(x.__class__, BaseModel)\n    assert isinstance(x, TestCase)\n    print(x)\n\n#>>> uuid=\'10d002bc-9941-4943-a46b-82b8214bf618\' name=\'Bob\' age=12\n#>>> uuid=\'595fd605-4684-4f78-96a5-8420bdb3fc0f\' name=\'Alice\' age=28\n\n```\n\n## Nested Example\nCreate one object of the type address and two objects of the type person. Each person has an attribute of the type address. \nWhen adding the person to the database, the database needs the foreign_table \'Adresses\' to create the foreign key. This means that when iterating over the table \'Persons\', a complete object "Person" can be created again, which has an attribute of the type \'Address\'.\n\n\n```python\nfrom pydantic_sqlite import DataBase\nfrom pydantic import BaseModel\nfrom uuid import uuid4\n\nclass Address(BaseModel):\n    uuid: str\n    town: str\n    street: str\n    number: int\n        \nclass Person(BaseModel):\n    uuid: str\n    name: str \n    address: Address\n\naddress = Address(uuid=str(uuid4()), town="Berlin", street="Bahnhofstraße", number=67)\nperson1 = Person(uuid=str(uuid4()), name="Bob", address=address)\nperson2 = Person(uuid=str(uuid4()), name="Alice", address=address)\n\ndb = DataBase()\ndb.add("Adresses", address)\ndb.add("Persons", person1, foreign_tables={\'address\': \'Adresses\'})\ndb.add("Persons", person2, foreign_tables={\'address\': \'Adresses\'})\n\nfor x in db("Adresses"):\n    assert issubclass(x.__class__, BaseModel)\n    assert isinstance(x, Address)\n    print(x)\n\nfor y in db("Persons"):\n    assert issubclass(y.__class__, BaseModel)\n    assert isinstance(y, Person)\n    print(y)\n\n#>>> uuid=\'7cd5410e-cfaa-481e-a201-ad04cd959719\' town=\'Berlin\' street=\'Bahnhofstraße\' number=67\n#>>> uuid=\'cc1cedaf-dac5-4fc2-a11a-41c6631271a5\' name=\'Bob\' address=Address(uuid=\'7cd5410e-cfaa-481e-a201-ad04cd959719\', town=\'Berlin\', street=\'Bahnhofstraße\', number=67)\n#>>> uuid=\'b144ed22-d8a4-46da-8a18-e34c260d7c45\' name=\'Alice\' address=Address(uuid=\'7cd5410e-cfaa-481e-a201-ad04cd959719\', town=\'Berlin\', street=\'Bahnhofstraße\', number=67)\n\n```\n\n# Nested Example without foreign Table\nIf you do not want to have an additional table, you can save an object of the BaseModel type differently.\n\nIn this example, the address object is not saved as an additional table. It is stored as a string in a column of the table \'Persons\'. To realise this, the class `SQConfig` is added to the Address class. This class must contain the method `convert`, which determines how the object is to be stored in SQLite. During the subsequent loading, an object of the type Address is created again from the string with the function pydantic.validator.\n\n```python\nfrom pydantic_sqlite import DataBase\nfrom pydantic import BaseModel, validator\nfrom uuid import uuid4\n\nclass Address(BaseModel):\n    town: str\n    street: str\n        \n    class SQConfig:\n        special_insert: bool = True\n\n        def convert(obj):\n            return f"{obj.town},{obj.street}"\n\nclass Person(BaseModel):\n    uuid: str\n    name: str \n    address: Address\n        \n    @validator(\'address\', pre=True)\n    def validate(cls, v):\n        if isinstance(v, Address):\n            return v\n        town, street = v.split(\',\')\n        return Address(town=town, street=street)\n\naddress = Address(town="Berlin", street="Bahnhofstraße 67")\nperson1 = Person(uuid=str(uuid4()), name="Bob", address=address)\nperson2 = Person(uuid=str(uuid4()), name="Alice", address=address)\n\ndb = DataBase()\ndb.add("Persons", person1)\ndb.add("Persons", person2)\n\nfor y in db("Persons"):\n    assert issubclass(y.__class__, BaseModel)\n    assert isinstance(y, Person)\n    print(y)\n\n#>>> uuid=\'802f50d6-b6a2-47f4-bb96-4375790daed9\' name=\'Bob\' address=Address(town=\'Berlin\', street=\'Bahnhofstraße 67\')\n#>>> uuid=\'79488c0d-44c8-4a6a-afa3-1ed0b88af4a2\' name=\'Alice\' address=Address(town=\'Berlin\', street=\'Bahnhofstraße 67\')\n```\n\n# DB_Handler\nThe DB_handler provides a wrapper for the DataBase. The database returned by the context manager can be used in the same way as in the previous examples. \n\nHowever, the handler has the advantage that if an exception occurs, e.g. a \'ZeroDevisionError\', a database with the last values is saved as \'<<dbname_crash>>.db\'. If this file already exists, the file name is incremented.\n\nThis example creates two files hello.db and hello_crash.db If you run this script twice, three files are created: hello.db, hello_crash.db and hello_crash_(1).db\n```python\nfrom pydantic_sqlite import DataBase, DB_Handler\nfrom pydantic import BaseModel, validator\nfrom uuid import uuid4\n\nclass TestCase(BaseModel):\n    uuid: str\n    name: str \n    age: int\n\nwith DB_Handler("hello") as db:\n    test1 = TestCase(uuid=str(uuid4()), name="Bob", age=12)\n    db.add("Test", test1)\n    for x in db("Test"):\n        assert issubclass(x.__class__, BaseModel)\n        assert isinstance(x, TestCase)\n        print(x)\n    db.save("hello_world.db")\n    \n    1/0\n\n#>>> uuid=\'04d6dfad-0ce5-4222-8686-22348e1f0c0b\' name=\'Bob\' age=12\n#>>> ---------------------------------------------------------------------------\n#>>> ZeroDivisionError    Traceback (most recent call last)\n#>>> ~\\AppData\\Local\\Temp/ipykernel_20124/1430346317.py in <module>\n#>>>      17     db.save("hello_world.db")\n#>>>      18 \n#>>> ---> 19     1/0\n#>>> \n#>>> ZeroDivisionError: division by zero\n```',
    'author': 'Your Name',
    'author_email': 'you@example.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/Phil997/pydantic-sqlite',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
