import { Callback, Context } from 'aws-lambda';
import AWS = require('aws-sdk');
/**
 * The event passed to the Lambda handler
 */
export interface LambdaEvent {
    [key: string]: any;
}
/**
 * The event passed through the promises.
 */
export interface Event extends LambdaEvent {
    /**
     * Adds values to the response returned to CloudFormation
     */
    addResponseValue: (key: string, value: any) => void;
    /**
     * Set the physical ID of the resource
     */
    setPhysicalResourceId: (value: any) => void;
}
/**
 * Function signature
 */
export declare type func = (event: Event) => Promise<Event | AWS.AWSError>;
/**
 * Custom CloudFormation resource helper
 */
export declare class CustomResource {
    /**
     * Stores functions executed when resource creation is requested
     */
    createFunctions: func[];
    /**
     * Stores functions executed when resource update is requested
     */
    updateFunctions: func[];
    /**
     * Stores functions executed when resource deletion is requested
     */
    deleteFunctions: func[];
    /**
     * The context passed to the Lambda handler
     */
    context: Context;
    /**
     * The callback function passed to the Lambda handler
     */
    callback: Callback;
    /**
     * Stores values returned to CloudFormation
     */
    ResponseData: {
        [key: string]: any;
    };
    /**
     * Stores values physical ID of the resource
     */
    PhysicalResourceId?: string;
    /**
     * Logger class
     */
    logger: Logger;
    constructor(context: Context, callback: Callback, logger?: Logger);
    /**
     * Adds a function to the CREATE queue
     */
    onCreate(func: func): this;
    /**
     * Adds a function to the UPDATE queue
     */
    onUpdate(func: func): this;
    /**
     * Adds a function to the DELETE queue
     */
    onDelete(func: func): this;
    /**
     * Handles the Lambda event
     */
    handle(event: LambdaEvent): this;
    /**
     * Sends CloudFormation response just before the Lambda times out
     */
    timeout(event: LambdaEvent): void;
    /**
     * Sends CloudFormation response
     */
    sendResponse(event: LambdaEvent, responseStatus: string, responseData: string): void;
}
/**
 * Logger class
 */
export interface Logger {
    log(message: any, ...optionalParams: any[]): void;
    info(message: any, ...optionalParams: any[]): void;
    debug(message: any, ...optionalParams: any[]): void;
    warn(message: any, ...optionalParams: any[]): void;
    error(message: any, ...optionalParams: any[]): void;
}
/**
 * LogLevels supported by the logger
 */
export declare const enum LogLevel {
    ERROR = 0,
    WARN = 1,
    INFO = 2,
    DEBUG = 3
}
/**
 * Standard logger class
 */
export declare class StandardLogger {
    /**
     * The log level
     *
     * @default LogLevel.WARN
     */
    level: LogLevel;
    constructor(level?: LogLevel);
    /**
     * Logs message with level ERROR
     */
    error(message: any, ...optionalParams: any[]): void;
    /**
     * Logs message with level WARN
     */
    warn(message: any, ...optionalParams: any[]): void;
    /**
     * Logs message with level INFO
     */
    info(message: any, ...optionalParams: any[]): void;
    /**
     * Logs message with level DEBUG
     */
    debug(message: any, ...optionalParams: any[]): void;
    /**
     * Alias for info
     */
    log(message: any, ...optionalParams: any[]): void;
}
