from selenium.webdriver.support import expected_conditions as EC
from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.common.by import By
from selenium.common import exceptions as SeleniumExceptions
import undetected_chromedriver as uc

from requests.adapters import HTTPAdapter
from datetime import datetime, timedelta
from urllib.parse import unquote
import requests
import base64
import uuid
import json
import re


class ChatGPT:
    '''
    An unofficial Python wrapper for OpenAI's ChatGPT API
    '''

    def __init__(
        self,
        session_token: str = None,
        email: str = None,
        password: str = None,
        conversation_id: str = None,
        parent_id: str = None,
        proxy: str = None,
        cf_refresh_interval: int = 30,
    ) -> None:
        '''
        Initialize the ChatGPT class\n
        Either provide a session token or email and password\n
        Parameters:
        - session_token: (optional) Your session token in cookies named as `__Secure-next-auth.session-token` from https://chat.openai.com/chat
        - email: (optional) Your OpenAI email
        - password: (optional) Your OpenAI password
        - conversation_id: (optional) The conversation ID if you want to continue a conversation
        - parent_id: (optional) The parent ID if you want to continue a conversation
        - proxy: (optional) The proxy to use, in URL format (i.e. `https://ip:port`)
        - cf_refresh_interval: (optional) The interval in minutes to refresh the Cloudflare cookies
        '''
        self.conversation_id = conversation_id
        if parent_id:
            self.parent_id = parent_id
        else:
            self.parent_id = str(uuid.uuid4())

        self.proxy = proxy
        self.cookies = []
        self.headers = {
            'accept': '*/*',
            'accept-encoding': 'gzip, deflate',
            'origin': 'https://chat.openai.com',
            'referer': 'https://chat.openai.com/chat',
            'x-openai-assistant-app-id': '',
        }
        self.session = requests.Session()
        self.session.headers = self.headers
        self.session.proxies = {'http': proxy, 'https': proxy} if self.proxy else {}
        self.session.mount('https://chat.openai.com', HTTPAdapter(max_retries=5))

        self.last_cf = None
        self.cf_refresh_interval = cf_refresh_interval
        self.session_token = session_token
        if not self.session_token:
            if not email or not password:
                raise ValueError(
                    'Either session_token or email and password must be provided'
                )
            self.session_token = self.__login(email, password)

        self.session.cookies.set('__Secure-next-auth.session-token', self.session_token)
        self.__refresh_auth()

    def __get_cf_cookies(self) -> None:
        '''
        Get the Cloudflare cookies & user-agent
        '''
        # Don't refresh the cf cookies if they are less than 30 minutes old
        if self.last_cf and datetime.now() - self.last_cf < timedelta(
            minutes=self.cf_refresh_interval
        ):
            return

        # Start the browser
        options = uc.ChromeOptions()
        options.add_argument('--window-size=1,1')
        if self.proxy:
            options.add_argument(f'--proxy-server={self.proxy}')
        try:
            self.driver = uc.Chrome(options=options)
        except TypeError as e:
            if str(e) == 'expected str, bytes or os.PathLike object, not NoneType':
                raise ValueError('Chrome installation not found')
            raise e

        # Set the user-agent to the one from the browser
        self.headers['user-agent'] = self.driver.execute_script(
            'return navigator.userAgent'
        )
        # Restore the cf cookies if they exist
        for cookie in self.cookies:
            self.driver.execute_cdp_cmd(
                'Network.setCookie',
                {
                    'domain': cookie['domain'],
                    'path': cookie['path'],
                    'name': cookie['name'],
                    'value': cookie['value'],
                    'httpOnly': cookie['httpOnly'],
                    'secure': cookie['secure'],
                },
            )

        # Get the Cloudflare challenge
        self.driver.get('https://chat.openai.com/api/auth/session')
        try:
            WebDriverWait(self.driver, 15).until(
                EC.presence_of_element_located((By.TAG_NAME, 'pre'))
            )
        except SeleniumExceptions.TimeoutException:
            raise ValueError(f'Cloudflare challenge failed: {self.driver.page_source}')

        # We only need the cf cookies
        self.cookies = [
            i
            for i in self.driver.get_cookies()
            if i['name'] in ['__cf_bm', 'cf_clearance']
        ]
        for cookie in self.cookies:
            self.session.cookies.set(cookie['name'], cookie['value'])
        self.last_cf = datetime.now()

        # Close the browser
        self.driver.quit()

    def __login(self, email: str, password: str) -> str:
        '''
        Login to OpenAI\n
        Parameters:
        - email: Your OpenAI email
        - password: Your OpenAI password\n
        Returns the session token
        '''
        self.__get_cf_cookies()

        # Get the CSRF token
        resp = self.session.get('https://chat.openai.com/api/auth/csrf')
        if resp.status_code != 200:
            raise ValueError(f'Status code {resp.status_code}: {resp.text}')
        csrf_token = resp.json()['csrfToken']

        # Get state
        resp = self.session.post(
            'https://chat.openai.com/api/auth/signin/auth0?prompt=login',
            data={'callbackUrl': '/', 'csrfToken': csrf_token, 'json': 'true'},
        )
        if resp.status_code != 200:
            raise ValueError(f'Status code {resp.status_code}: {resp.text}')
        redirect_url = resp.json()['url']

        # Redirect to auth0 /login/identifier
        resp = self.session.get(redirect_url)
        if resp.status_code != 200:
            raise ValueError(f'Status code {resp.status_code}: {resp.text}')

        captcha = None
        if '<img alt="captcha"' in resp.text:
            try:
                import svg_captcha_solver
            except ModuleNotFoundError:
                raise ModuleNotFoundError(
                    'Please install svg_captcha_solver to solve captchas: pip install https://github.com/dinhitcom/svg-captcha-solver/archive/master.zip'
                )
            captcha_src = re.findall(r'<img alt="captcha" src="([^"]+)"', resp.text)[0]
            svg = base64.b64decode(captcha_src.split(',')[1])
            captcha = svg_captcha_solver.solve_captcha(svg)

        pattern = r'<input type="hidden" name="state" value="(.*)" \/>'
        results = re.findall(pattern, resp.text)
        if not results:
            raise ValueError(f'Could not get state: {resp.text}')
        state = results[0]

        # Post email
        payload = {
            'state': state,
            'username': email,
            'js-available': 'true',
            'webauthn-available': 'true',
            'is-brave': 'false',
            'webauthn-platform-available': 'false',
            'action': 'default',
        }
        if captcha:
            payload['captcha'] = captcha
        resp = self.session.post(
            f'https://auth0.openai.com/u/login/identifier?state={state}',
            data=payload,
        )
        if resp.status_code != 200:
            raise ValueError(f'Status code {resp.status_code}: {resp.text}')

        # Post password
        resp = self.session.post(
            f'https://auth0.openai.com/u/login/password?state={state}',
            data={
                'state': state,
                'username': email,
                'password': password,
                'action': 'default',
            },
        )
        if resp.status_code != 200:
            raise ValueError(f'Status code {resp.status_code}: {resp.text}')

        # Get session token in CookieJar
        cookies = self.session.cookies.get_dict()
        if '__Secure-next-auth.session-token' not in cookies:
            if resp.url.startswith('https://chat.openai.com/auth/error?error='):
                raise ValueError(unquote(resp.url.split('error=')[1]))
            raise ValueError(f'Could not get session token: {resp.text}')
        return cookies['__Secure-next-auth.session-token']

    def __refresh_auth(self) -> None:
        '''
        Refresh the session's authorization
        '''
        self.__get_cf_cookies()

        resp = self.session.get('https://chat.openai.com/api/auth/session')
        if resp.status_code != 200:
            raise ValueError(f'Invalid session token: {resp.text}')

        data = resp.json()
        if not data:
            raise ValueError('Invalid session token')
        access_token = data['accessToken']
        self.headers['authorization'] = f'Bearer {access_token}'

    def __moderation(self) -> None:
        '''
        Send a fake moderation request
        '''
        self.__get_cf_cookies()

        resp = self.session.post(
            'https://chat.openai.com/backend-api/moderations',
            json={'input': 'Hello', 'model': 'text-moderation-playground'},
        )
        if resp.status_code != 200:
            raise ValueError(f'Status code {resp.status_code}: {resp.text}')
        data = resp.json()
        if data['blocked'] or data['flagged']:
            raise ValueError(f'Message is blocked or flagged: {resp.text}')

    def send_message(self, message: str) -> dict:
        '''
        Send a message to the chatbot\n
        Parameters:
        - message: The message you want to send\n
        Returns a `dict` with the following keys:
        - message: The message the chatbot sent
        - conversation_id: The conversation ID
        - parent_id: The parent message ID
        '''
        self.__refresh_auth()
        self.__moderation()
        resp = self.session.post(
            'https://chat.openai.com/backend-api/conversation',
            json={
                'action': 'next',
                'messages': [
                    {
                        'id': str(uuid.uuid4()),
                        'role': 'user',
                        'content': {'content_type': 'text', 'parts': [message]},
                    }
                ],
                'conversation_id': self.conversation_id,
                'parent_message_id': self.parent_id,
                'model': 'text-davinci-002-render',
            },
            stream=True,
        )
        if resp.status_code != 200:
            raise ValueError(f'Status code {resp.status_code}: {resp.text}')

        data = list(resp.iter_lines())[-4].decode('utf-8').lstrip('data: ')
        data = json.loads(data)
        self.conversation_id = data['conversation_id']
        self.parent_id = data['message']['id']
        return {
            'message': data['message']['content']['parts'][0],
            'conversation_id': self.conversation_id,
            'parent_id': self.parent_id,
        }

    def reset_conversation(self) -> None:
        '''
        Reset the conversation
        '''
        self.conversation_id = None
        self.parent_id = str(uuid.uuid4())
