import smtplib
import os
import io
import sys

from datetime import datetime
from email.mime.multipart import MIMEMultipart
from email.mime.text import MIMEText

ENVIRON_VAR = ['LOG_SENDER_EMAIL_ADDRESS',
               'LOG_SENDER_EMAIL_PASSWORD',
               'LOG_RECEIVER_EMAIL_ADDRESS']


def log_function(function):
    email, password, recipient = get_email_info()
    session = authenticate_email(email, password)
    message = create_message(email, recipient)

    def log_function_wrapper(*args, **kwargs):
        arguments = get_function_arguments(args, kwargs)

        # Add subject to email
        subject = "Function '{}' execution log".format(function.__name__)
        message['Subject'] = subject

        # Start email body text
        text = "Function {}({}) finished its execution.\n\n".format(
            function.__name__, arguments)

        start_time = datetime.now()
        text += "Start time: {0:%b %d %H:%M:%S}\n".format(start_time)

        with CapturingOutput() as text_output:
            return_value = function(*args, **kwargs)

        if text_output:
            text += 'Function text output:\n'
            for op in text_output:
                text += op + '\n'
        else:
            text += 'No text output\n'

        text += 'Function returned: {}\n'.format(
            return_value) if return_value else 'No returned value\n'

        end_time = datetime.now()
        text += "End time: {0:%b %d %H:%M:%S}\n".format(end_time)

        total = (end_time - start_time).seconds
        hours, remainder = divmod(total, 3600)
        minutes, seconds = divmod(remainder, 60)
        text += '\nTotal execution time: {0:02d}:{1:02d}:{2:02d}\n'.format(
            hours, minutes, seconds)

        # Add body to email
        message.attach(MIMEText(text, 'plain'))
        body = message.as_string()

        session.sendmail(email, recipient, body)
        session.quit()

    return log_function_wrapper


def get_email_info():
    for var in ENVIRON_VAR:
        yield os.environ.get(var) if os.environ.get(var) else input('{}: '.format(var))


def authenticate_email(email, password):
    session = smtplib.SMTP('smtp.gmail.com', 587)
    session.starttls()
    session.login(email, password)

    return session


def create_message(email, recipient):
    message = MIMEMultipart()
    message['From'] = email
    message['To'] = recipient

    return message


def get_function_arguments(args, kwargs):
    args_repr = [repr(a) for a in args]
    kwargs_repr = ["{0}={1!r}".format(k, v) for k, v in kwargs.items()]
    arguments = ", ".join(args_repr + kwargs_repr)

    return arguments


class CapturingOutput(list):
    def __enter__(self):
        self._stdout = sys.stdout
        sys.stdout = self._stringio = io.StringIO()
        return self
    def __exit__(self, *args):
        self.extend(self._stringio.getvalue().splitlines())
        del self._stringio
        sys.stdout = self._stdout