# wom.py - An asynchronous wrapper for the Wise Old Man API.
# Copyright (c) 2023-present Jonxslays
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

from __future__ import annotations

import typing as t

from wom import models
from wom import result
from wom import routes

from . import BaseService

__all__ = ("NameChangeService",)

ValueT = t.TypeVar("ValueT")
ResultT = result.Result[ValueT, models.HttpErrorResponse]


class NameChangeService(BaseService):
    """Handles endpoints related to name changes."""

    __slots__ = ()

    async def search_name_changes(
        self,
        username: str | None = None,
        *,
        status: models.NameChangeStatus | None = None,
        limit: int | None = None,
        offset: int | None = None,
    ) -> ResultT[list[models.NameChange]]:
        """Searches for name changes.

        Args:
            username: The optional username to search for.

        Keyword Args:
            status: The optional name change status to filter on.
                Defaults to `None`.

            limit: The optional maximum items to return on this page
                from the API. Defaults to `None`.

            offset: The optional page offset. Defaults to
                `None`.

        Returns:
            A [`Result`][wom.Result] containing a list of name changes.

        ??? example

            ```py
            import wom

            client = wom.Client(...)

            result = await client.names.search_name_changes(
                "Jonxslays", limit=1
            )
            ```
        """
        params = self._generate_map(username=username, status=status, limit=limit, offset=offset)
        route = routes.SEARCH_NAME_CHANGES.compile().with_params(params)
        data = await self._http.fetch(route, self._list)

        if isinstance(data, models.HttpErrorResponse):
            return result.Err(data)

        return result.Ok([self._serializer.deserialize_name_change(c) for c in data])

    async def submit_name_change(self, old_name: str, new_name: str) -> ResultT[models.NameChange]:
        """Submits a new name change.

        Args:
            old_name: The old name for the player.

            new_name: The new name for the player.

        Returns:
            A [`Result`][wom.Result] containing the name change.

        ??? example

            ```py
            import wom

            client = wom.Client(...)

            result = await client.names.submit_name_change(
                "Jonxslays", "I Mahatma I"
            )
            ```
        """
        payload = self._generate_map(oldName=old_name, newName=new_name)
        route = routes.SUBMIT_NAME_CHANGE.compile()
        data = await self._http.fetch(route, self._dict, payload=payload)

        if isinstance(data, models.HttpErrorResponse):
            return result.Err(data)

        return result.Ok(self._serializer.deserialize_name_change(data))

    async def get_name_change_details(self, id: int) -> ResultT[models.NameChangeDetail]:
        """Gets the details for an existing name change.

        Args:
            id: The ID of the name change.

        Returns:
            A [`Result`][wom.Result] containing the name change details.

        ??? example

            ```py
            import wom

            client = wom.Client(...)

            result = await client.names.get_name_change_details(123)
            ```
        """
        route = routes.NAME_CHANGE_DETAILS.compile(id)
        data = await self._http.fetch(route, self._dict)

        if isinstance(data, models.HttpErrorResponse):
            return result.Err(data)

        return result.Ok(self._serializer.deserialize_name_change_detail(data))
