from __future__ import annotations

from typing import Dict, Tuple

SSL_TEST_PRIVATE_CA_CRT = b"""-----BEGIN CERTIFICATE-----
MIIDKTCCAhGgAwIBAgIUD5VUdvJQlRhGJg2WJ+/K8I2sYZcwDQYJKoZIhvcNAQEL
BQAwRDENMAsGA1UECgwEQ2hpYTEQMA4GA1UEAwwHQ2hpYSBDQTEhMB8GA1UECwwY
T3JnYW5pYyBGYXJtaW5nIERpdmlzaW9uMB4XDTIyMDMyMzE3MjkyN1oXDTMyMDMy
MDE3MjkyN1owRDENMAsGA1UECgwEQ2hpYTEQMA4GA1UEAwwHQ2hpYSBDQTEhMB8G
A1UECwwYT3JnYW5pYyBGYXJtaW5nIERpdmlzaW9uMIIBIjANBgkqhkiG9w0BAQEF
AAOCAQ8AMIIBCgKCAQEA47sS3nsWACyhnX2pvqdSMSfDWyeF1t+kmVvap2hk22yA
oAM56XCtHifhP8Zu8nI8uumP7Bnh8sW4yqE1Tl/Y5gkYD57r4ZH9o5lj1FvnZnoJ
lERpCpgeR9Lpw6xqfoPt+9a/KR6mQ93AwqH6tyqSicl+7TTNp8UAdAN/oFnymQ3T
VcBXV30YgiX3GIYNvcV0f+hglgA3BsprZLJRuaCv9fLT8PlhCqrFyJ67XflT98nE
0pyaGIAEcvPCPQstAkcc9nnhmARkoAq7/3dsOWRr31paoWAnrSzXJtb+NWomlNEQ
ZADbNd68TIV40H6bcN45ff1nOfC78NJkXIjwdnFmuwIDAQABoxMwETAPBgNVHRMB
Af8EBTADAQH/MA0GCSqGSIb3DQEBCwUAA4IBAQAnl8bxBpQL41lIPB/o1Fe629IK
pFpJDjLC8xyUZrcoZigEYZ+O06Kbc8W52n3awRlLgSMjPUjwb3JExhPW1JcL2DAJ
ARoiu5XDCoTVAADeUAyf60YfARlu9y+hRDUSgYrUknuvJ07fTobJ4JbpMvP/mI9T
ILNZNJZVcaSD4ectURKYkveGtm5R9F3LIO0gscEFgWkhrPeLjiU6Hm1uuZgjRW8C
o57vHwW4yfmiWaXb3dwIjh8SBMgv/g9iDZskIkTCfRBEp8jM4ZlCAII0sNejSLkG
NjUDOtD3AYD46mLU9uOUkv/ecDn13usCaJSnQ7RfO7S/yGZQEcNw+DHHQXbS
-----END CERTIFICATE-----
"""

SSL_TEST_PRIVATE_CA_KEY = b"""-----BEGIN RSA PRIVATE KEY-----
MIIEpgIBAAKCAQEA47sS3nsWACyhnX2pvqdSMSfDWyeF1t+kmVvap2hk22yAoAM5
6XCtHifhP8Zu8nI8uumP7Bnh8sW4yqE1Tl/Y5gkYD57r4ZH9o5lj1FvnZnoJlERp
CpgeR9Lpw6xqfoPt+9a/KR6mQ93AwqH6tyqSicl+7TTNp8UAdAN/oFnymQ3TVcBX
V30YgiX3GIYNvcV0f+hglgA3BsprZLJRuaCv9fLT8PlhCqrFyJ67XflT98nE0pya
GIAEcvPCPQstAkcc9nnhmARkoAq7/3dsOWRr31paoWAnrSzXJtb+NWomlNEQZADb
Nd68TIV40H6bcN45ff1nOfC78NJkXIjwdnFmuwIDAQABAoIBAQDbXQOTDaumpVdU
loyhGxngQbY7ozmYPruWAlYANl4Yw06vVAw/4orP5ohiX2nV3tbDEWLSgigVLAKS
JXFsuoT7KT1lJqd/FJrnMUVpdNHU868whClOzDM26mLFWtsGXV0r3rMrD9wGuaAJ
m/Ae6kJXuaGrKtcDSY/jMM+KjbpTVJ0iWVS2yQtOQffgOc2l9SXi7EwxIccBGrbF
H+a+JOZIbLsPMXwCzo1i0PMeAkG/FneqkCG7BEc2RWeQl6nzkoxANydlLLwnJmzG
Xd+UPAJo2TTRlkXpK06Jb/+wsPSpEq3Sim9IQQMHp8kxtFSjgr2sDG28g0SRfV/K
3RMX8A3xAoGBAP8qXGX8QNl4IbGJl786lk5cCzVlEzmCSgPcKVv7V6hPWknRUInO
rB4UbI66hFAK3SGJrs6+gyj/1GB4mM88TSdGJaifZnt4dekA3NZdZbrMINoIcS64
KFGpFyFlj81zxwXyEDitchwYrkMEQ8Y+wO20lepCPoEuMnhbJ+kH22hTAoGBAOR5
viyu7m1DU9qv5ZWhFoS1+tDT3g3zHsKSrrKz5sE07tPz5t9FQztKfflmv9jCurDq
C+Dro00v+WnxUQSuDYXNk9JXU0Ly+Z/Tt7S3AEZQkarjya/jkDECsEQJuuHtOo/C
Lc/req4PeG3IHqoAIwJR2lLoSmA1/bFkBqzwvZr5AoGBAJ75Rm3n1oNm/B7/aYKj
vsd9QyJ2IZ8f1HtJLJ95HajoH4mEFlh1Yfivot7dx9eSnSjq/fUi7taZZTjhidr7
Z9pDMu02uoPMjjGn7u7Da8EQspsEzXddFwmhfp65Bvnq2yGSKZcWQNTOb6ujOZKG
wG6Ypxo3QXf8T0d7C4d89K6vAoGBAKbwxaJEl3ZCsayixbAVKi0CAyg/BotOz3UA
VKHCjaMymCdKmbEeHMff4DgxeZBaiKrn+JH52zJ1EVr1tmsi+kKAObiBkhbjDr5X
pm1nIc4+5JDnK+FW9264Ni8gpOqbuiR8j5jSyKsSJr6gcLdqrintwQa/MEkoahwJ
qE8JAKGRAoGBALTTp3skePMl/wCpdzvOqNMvJzoDV9xX1ttPCKF0pNOKMmFdSRib
8JN6dpRuQ5CugLYqNQgMExxgpytduGVRsALp/hKNo7MYCmLRsRAcSIe6YpgvnVYw
y+ufj7QhG3hCPBMCAC2YHLTAAAkzdiookyhTmtfWtSZJuZlUYgLsMQ9g
-----END RSA PRIVATE KEY-----
"""

SSL_TEST_FULLNODE_PRIVATE_CRT = b"""-----BEGIN CERTIFICATE-----
MIIDLDCCAhSgAwIBAgIUI/qIR41vNfZrImHDASkbd5CdhdkwDQYJKoZIhvcNAQEL
BQAwRDENMAsGA1UECgwEQ2hpYTEQMA4GA1UEAwwHQ2hpYSBDQTEhMB8GA1UECwwY
T3JnYW5pYyBGYXJtaW5nIERpdmlzaW9uMCAXDTIyMDMyMjEwMjkyN1oYDzIxMDAw
ODAyMDAwMDAwWjBBMQ0wCwYDVQQDDARDaGlhMQ0wCwYDVQQKDARDaGlhMSEwHwYD
VQQLDBhPcmdhbmljIEZhcm1pbmcgRGl2aXNpb24wggEiMA0GCSqGSIb3DQEBAQUA
A4IBDwAwggEKAoIBAQClBLYsg7P6WBNexxAiF04sCZN6saN+dI0Q/KAN1Rp2jfTX
nApvuNImoc2fozCViausJmEWKtx7RHQheDJmx+0LSpQ0fgdJJSNK5UVRme/pgVyE
/FACdD+fBuiasX1LoaBUqk7mpfeyQpfwKW5D1q3HgDK5ROmiaMe5VnMD0BvW/QYc
1T3TDuBLXYZpAg7c+fdqmBKx4wnE1v9PqChdUd2AQF2INdQuz3QOHGAQ+cJsXqvy
cwep0LvlK9/j9dy08qKFvc40EZwsYqOh0HT6KzUupVh80qhZxOz3QSok6gNhGpnI
JHOFvJ+i2P48Wb05dCiqBqYkEnNQ252b1l7o+grjAgMBAAGjFzAVMBMGA1UdEQQM
MAqCCGNoaWEubmV0MA0GCSqGSIb3DQEBCwUAA4IBAQBYYc7lM9pM40R/SUMzhI4C
D59d0VMwKjTqXRy2ifrcZYVxmc9CAldj+nk04piXC+YxUihsIRSOxppd2I72lbo7
bCRPxj0rYK/iNBlkzFwvVCQ2e/chcKcFOEcXzEpKGzVpL2GyLv/NpxH87hLLJwSG
FPGsMm5MznnMDuCX1ALedLFKyvMoy8SSg/+z7TNiblJxEabxgEADpC/pbkLnJvlv
qQBTPZk9LD02ZXivI624ltlQt92tqIUFZBfQXKe6WDjz2IzeeaYSlhTBKvqzXXXg
aUXnGR1lA3Q7dBaCPSNkUimOm0qT6t81DLkvoSyQu3P0OvKFCmZNLxDbxEO5LBAK
-----END CERTIFICATE-----
"""

SSL_TEST_FULLNODE_PRIVATE_KEY = b"""-----BEGIN RSA PRIVATE KEY-----
MIIEpAIBAAKCAQEApQS2LIOz+lgTXscQIhdOLAmTerGjfnSNEPygDdUado3015wK
b7jSJqHNn6MwlYmrrCZhFirce0R0IXgyZsftC0qUNH4HSSUjSuVFUZnv6YFchPxQ
AnQ/nwbomrF9S6GgVKpO5qX3skKX8CluQ9atx4AyuUTpomjHuVZzA9Ab1v0GHNU9
0w7gS12GaQIO3Pn3apgSseMJxNb/T6goXVHdgEBdiDXULs90DhxgEPnCbF6r8nMH
qdC75Svf4/XctPKihb3ONBGcLGKjodB0+is1LqVYfNKoWcTs90EqJOoDYRqZyCRz
hbyfotj+PFm9OXQoqgamJBJzUNudm9Ze6PoK4wIDAQABAoIBAGKIOa7CxUiKJqbE
/eEdQVQSM7ljMFhlh4XJEliwEikQAk5rod+r3++pVXoomu3vUk5sbDQsS5JfAWiq
uI2eeu/vDCd7ySXnwvcJhyZ+YD5xO6k8bxnIB+UQ3vrfWA086NslBg+6sfgCw25n
jqt7GDCzX5Y4i7iqD/eeovZ34OexH+J6dE0rtlO6hblbnrk8jgJJU/CsK+fPybky
U/Dpls6AMEI/s/6sC7Ajieq4HN8lJpnOS03LD7Xo5XQRzHDHzKfryFP6y/Ng0lGn
H9WD1kRWtvQ2JAct9V2VTgNUFcdVGICqn01uiob+3jI/LyfWylMqhoPOJv/K7i3B
pmi+FIECgYEA0WIeWEAREuLvs2sFFdxkUbfzJpEVrPiVy57csc7+q/5db6fFd4nR
Arcr8VoFQcVbEzMLXcJQzV8lXJUacPetjwNh+91NUdM6dFRotG+AYc9Gb0wd/YMg
5PCyqwWQfRXHj7OLB2iQ7jI2oZSi+MSNLBuNZiPoP5D1qSvyJlY43JMCgYEAycH/
/dWP8/ukyxcZuwf+qeseZ7StCprTXY1+eEm6ov8okegvUfPWXUSFp3vu7nUf+jQb
j3KyemkUcDmxNXNKsWGyxpdUG/2kmg0Mktky+ndVhXIKJ/bkiBFfZagxnIO5MyBU
BoD9Kr55dCw4BwIDQoQ8h5SjBuKcQNe6Atx3WnECgYEAprJHFltrPG4QB/ecBNuB
ws2Ad6Pqbds65vGDIsx5VNUd7lbTj2fZTyAli/DvXn2RFKf+1JZvXhHx3NOYSXfm
ZYV77NkzRYG1eAf7lCtxOm2a1eDzv3E2Lw6yMBYuAmfkiTOX7D4DpJ+1fs6nMQbK
neMRn8/Wh0URT639Cyh5/KECgYEAl1DorXd+f348GhU5NwzH8qhkJWNcZJe57nAw
agbh811rEAMTemNmYo0BsJtCl5VAZmauROfA9A3bRk/+mBff7SlY65iTfGq/CMrA
o2JC2ZamjW76Lv78ZxCge52pl7kzLSUR387eETDJ5ldDrf3UdA65VvYjTwamKfuh
hoPTAAECgYBp0yhXkzMrGtsTfiiKSWAGe0HSakvz8H3bQDuQnUw67iEH6FbMhRtZ
Pu/QTafLpAMKrtbDMuls+2pxf2BN294naZyB+fI0zAmefXtTjx23hxGoFnh3FEVx
eMX8OdBvkvk/Ifbfa8PqD9/bTQTcez+RkCnzANn04+h/tUCzJExfhA==
-----END RSA PRIVATE KEY-----
"""

SSL_TEST_FULLNODE_PUBLIC_CRT = b"""-----BEGIN CERTIFICATE-----
MIIDLDCCAhSgAwIBAgIUBJH2JfwoCndkFqwVE6FeUMjDVx4wDQYJKoZIhvcNAQEL
BQAwRDENMAsGA1UECgwEQ2hpYTEQMA4GA1UEAwwHQ2hpYSBDQTEhMB8GA1UECwwY
T3JnYW5pYyBGYXJtaW5nIERpdmlzaW9uMCAXDTIyMDMyMjEwMjkyOFoYDzIxMDAw
ODAyMDAwMDAwWjBBMQ0wCwYDVQQDDARDaGlhMQ0wCwYDVQQKDARDaGlhMSEwHwYD
VQQLDBhPcmdhbmljIEZhcm1pbmcgRGl2aXNpb24wggEiMA0GCSqGSIb3DQEBAQUA
A4IBDwAwggEKAoIBAQClPnyntGjxicj7xn5m1Ujkgid5yz+/d/g6kdRTJt0Faeiz
oqm+9o0YUsLeKWSjOhjgcKGPh542JPzYqTaV531R8gIYJEDs9hGR3dbjmeu5RzVd
Ycv3Ru9BInMudjH5GeelUos0nT3OUJlb75QcNB2Q3miP2SfPcHNtGHa72VXCP7mN
zmiadi6gEJyPz32eBAh64CPoV177/sJqqlrFERmo+LyUgNNiTpFqwbKYIW2dCozG
3V/8dz25xQVvOMbVG7GnmRHXr6M6wTOefqsciOiGpPyfsTGfx++W+poGHrT1xqCL
IQNUVJZ+jip0BkWheR//bwe3RqCR/fCpbsd8CwTFAgMBAAGjFzAVMBMGA1UdEQQM
MAqCCGNoaWEubmV0MA0GCSqGSIb3DQEBCwUAA4IBAQAP3lZ7n12+Qq96lT5Q28/m
rldsTbJguDNPxOQYx8rvZO21Bf3YjJZmqAFM0H6wBgQf7d0CBru+3NxlwZs9RiLf
bpThwzl+oWWVw9dxyYmcBl6RXf9DvisbIxKvQloUWjPAGfRS64qyOqjjFwBc5dlc
r9AOzgsN1pM7f52o5Ks5frq0vfO20lVbkvFb9TfzcuzRh0xNROlrAVK2q3XEaEab
nBTNTYt5kq8KYU+s0jstlPFF9gGx+NOXKfZmdKtOfribuPJIQ5ADvxGDaWEmQHhF
PlgF7Y0kmP/vUknuY8rwMrGi+cncaafO+GIiaBXDGxa92SNWe2J7RtRWDC5NF1en
-----END CERTIFICATE-----
"""

SSL_TEST_FULLNODE_PUBLIC_KEY = b"""-----BEGIN RSA PRIVATE KEY-----
MIIEowIBAAKCAQEApT58p7Ro8YnI+8Z+ZtVI5IInecs/v3f4OpHUUybdBWnos6Kp
vvaNGFLC3ilkozoY4HChj4eeNiT82Kk2led9UfICGCRA7PYRkd3W45nruUc1XWHL
90bvQSJzLnYx+RnnpVKLNJ09zlCZW++UHDQdkN5oj9knz3BzbRh2u9lVwj+5jc5o
mnYuoBCcj899ngQIeuAj6Fde+/7CaqpaxREZqPi8lIDTYk6RasGymCFtnQqMxt1f
/Hc9ucUFbzjG1Ruxp5kR16+jOsEznn6rHIjohqT8n7Exn8fvlvqaBh609cagiyED
VFSWfo4qdAZFoXkf/28Ht0agkf3wqW7HfAsExQIDAQABAoIBAQCEiL3KdLTHJdAB
wI6ZQ/AW9x1aXl5e34ZE07oMh1s2gF+X1Jt5Ap7gJ2EMdv60vGA/g5bRW+DVZI03
6WBSkzy8gPKVEl7Qr/wflmJIYqfiDu8KWuoRBt/Wyh6/4STruo9E1hO4u3VbCOI4
DswoYtRj4T7UQUPg6Txq//gbapGSHefwrZ5hH9A/YB01ouVafnSxxh33JqodIjqe
1uL1LInxF+ghmH8hlifP2xmD6O0Y9Y7kq+6715gLmz+UToGvLwrEiJoN5JLfbCnI
lw5z+1LDzlBfS2jlyZTg7nUQrJtTnlcg5oQdaaM5nDo+R2Cl/kU7/2Rir0sKIefF
Y32lvkfxAoGBANlyjRYTLuvsi5yQo+yrMspDiWHrIkJZgl8Zp7DAWPz6BwsOqLFw
EuBBE+OtTYokLDKsprrWPLNKafXAQrRdV3WfQuEvM5WMUK3qMGb3UX1L6S7iMa3u
46DPbk+TYkljLH82mvTmg595g9UyTJIEodfWzkNT4R69TtgtHb5kzR+bAoGBAMKK
iughFvrb9dnEOFnFgWTxfFt+SRzZmJQDhu50dvjKmSB1uvxHfswdX0sJINVqylOB
x74bCQVERGOrEwfcp7lqGHEvxHXsoKxo0A2pYvIPz8pSvkMHFU8ZMjhpukb8eSeq
SmGOB4I3FvoXxl6XQVIfNzKaCtWIpRyyaz3+xiMfAoGASdPNuR/8P1fJsvec4F7j
2K6Dd35o3FQdooJIYAd+by3ItGVeuxfNINyXjyao++z6IJ5j4W2ZoFn0jd6gBzLl
3eabw27OOckxH0sy8dKolTzvx94+dcp0+IVU86mbpkUIt4xnohydtciBDSKNwsnV
1T12PH+/IDuwCE+RGLKKvl8CgYAn7mAMvZ4F59hS1p/qbUvd+GNaNFW1gkR7PS4Z
C/bPomZD72ZuGg7AvtashDclF0JoWZ8yOkJjvlt7ScYiOvyCZnCV6wQcYHGqceki
pL4WiYBROyKqh22sSrOwOKNvpqsHpUCXPZv08nHY68Bf8hQKUzX1QQEwS1f4x/aT
RxRMaQKBgHfwK+DSGjtojrwo10CKYQ94ZDwPnviuIPcZfNQ+X/KxRY+wCl2ft3iC
H23dOdNZ52NDlRVXafCCI7FFhy50/jojJ4MupVX3pTQ3oDiHkii0eHt4NwgAzqm4
Um7BAuOB20ogIzQd5Hozd2eQTI0AFs4Zd3yBxTgG2B6BHPSwvxjd
-----END RSA PRIVATE KEY-----
"""

SSL_TEST_WALLET_PRIVATE_CRT = b"""-----BEGIN CERTIFICATE-----
MIIDLDCCAhSgAwIBAgIUT34idpoOlGXOYcUny2DMZVL5T4owDQYJKoZIhvcNAQEL
BQAwRDENMAsGA1UECgwEQ2hpYTEQMA4GA1UEAwwHQ2hpYSBDQTEhMB8GA1UECwwY
T3JnYW5pYyBGYXJtaW5nIERpdmlzaW9uMCAXDTIyMDMyMjEwMjkyOFoYDzIxMDAw
ODAyMDAwMDAwWjBBMQ0wCwYDVQQDDARDaGlhMQ0wCwYDVQQKDARDaGlhMSEwHwYD
VQQLDBhPcmdhbmljIEZhcm1pbmcgRGl2aXNpb24wggEiMA0GCSqGSIb3DQEBAQUA
A4IBDwAwggEKAoIBAQDCzGdbpwHSb2Cr1yhXKPZ2aCWtHEksIl7ao2UCDO+ljCb0
n6PxHAbgVHKJ0vy7mC6Y1VXdstr4FS1Ak22V+4ceo1UW6Tsls4I9RunAjkfAl9J6
CzKCVr31L6GvWhIjgQBkxswmYK9AFx3WbOQfz7kN46usBX3YgAWiUwBseOS8KBuh
EWfguDqxL8yyPHf6v5kWy2QyfwUBITuLQE6AanbLrkoT5bRrE+06X7+PeKidHzXx
b72fSvuJ7ERbNPiHmmVoila6kb3PegcKnKsm7d6Blg9MwCeOcvTXRrXsCFHlJnlZ
YWczAKAvtd/OJeTEC4de4UTuuyuDm2+00FUjVKITAgMBAAGjFzAVMBMGA1UdEQQM
MAqCCGNoaWEubmV0MA0GCSqGSIb3DQEBCwUAA4IBAQAgc1DcbsLsxrKHZEoIENxp
2pB5LM2Kq284Pll7Ix5x7OH8JRenLLZCGMLK+AeW25UchHZ8iagHYR6tQrc+7CLI
piE6YKchJu340c3RONXcxDZUKqXBxalUyVcb3z8I/syiqsvDrE94hnGfp4X9Yg/M
7EIdv+I4y41qTxMD4HvVTcXVU5gapEC2mpOQ7mfN/rnIU1Gv6kpwG1FrnUyH+aP/
2lpVSke8eSY59W4hThqZNf5cSGw/t95CBHigePoTqoD5MfmGhhrwGrjr9JHXVMpX
IUNgsbuPb6N5J7a63Ilgr+26X2ZC2zco7Ff5zKz1xKPMvLQ4tGBjm4rwQ1r7F4jH
-----END CERTIFICATE-----
"""

SSL_TEST_WALLET_PRIVATE_KEY = b"""-----BEGIN RSA PRIVATE KEY-----
MIIEpAIBAAKCAQEAwsxnW6cB0m9gq9coVyj2dmglrRxJLCJe2qNlAgzvpYwm9J+j
8RwG4FRyidL8u5gumNVV3bLa+BUtQJNtlfuHHqNVFuk7JbOCPUbpwI5HwJfSegsy
gla99S+hr1oSI4EAZMbMJmCvQBcd1mzkH8+5DeOrrAV92IAFolMAbHjkvCgboRFn
4Lg6sS/Msjx3+r+ZFstkMn8FASE7i0BOgGp2y65KE+W0axPtOl+/j3ionR818W+9
n0r7iexEWzT4h5plaIpWupG9z3oHCpyrJu3egZYPTMAnjnL010a17AhR5SZ5WWFn
MwCgL7XfziXkxAuHXuFE7rsrg5tvtNBVI1SiEwIDAQABAoIBAQCnQns85xlZZujK
o+OvcyysXqB7E16A1sI7WTDRoenja4PHZ4uM61O8KsZuMQtwyCq6b/NaeMgrW4OP
fwcJUP+j/vqAwaJXrNqXYtwyyfgyFipTQGoOIAzbChr6RYxtj8aWwVtpWHshvLeR
9c2qwuFSW7p76fs0ejhcOIiUmvlL1GTF8fWvZ8PibxcMEHbNaclx58rpWdBKOZpI
WXQ593AOGwufWX8EJFSdv+zjJs+JIIrVVOqLUMhww3B9SKRkYvMGABTRFY53U6Ru
ixR5jBRCrSdot5S7YIktZ+BoyZYDl4LNzcEsWv8KDqD35RbjRUXH6pPRD5xaTucZ
nKCGCGUBAoGBAOUyXq7KRqkrDlXpgvbkfI99ESS9rHCBa88SLjIKIdGszE1AIUVy
xHjXH/eYWQQ1u9ghqEgU0uyPcDbGwR2DiKnV6M9Y1RqOxgMVe7b02SjeIzqp+6zw
ZeqJsaaUPaRQFOG0gKSjpxA3FrQZO2HRnnUIZT7jmXXsCSMxcsYmuIVpAoGBANmU
Og7mag5iTgpsP38O+vHkCi9dXOif+20xI04Wuyt26EapPC3oNW3TwsDkCTX7ljpo
yI0yRQQgNUvPu/trsBaqDhTOXTaFCvH3+EA9SbXKKoFD7XiQ2iPJ5s/wv9aAsu/p
zpE8UI619RWx1NNp9qzPPYRaT5QQWbw0vG4ASBAbAoGAG/XQHSyqABszjYGGK45m
L68HN184OSVgvIswPYSE6apV5hqrtGgy1bcQvEjGmz/e4ZgBioGX1aoN+3C1wF76
fhWNaDYjETFXsKqfRAuMrnKjA5YXENrm4/zWQkcVWgU29u3pP6yDRfQifegSGXXU
CdzzVP+5Cey4lKWEpe21VvkCgYBQscq+j/Ixl3/+GZuLEhE7+SlV27iDhiIHozvG
0GNmgVkwK7/n/sEiFHGcfHZ2EPLsgsQuqcAzAYcCsO61FoFeRn+mc8raV4lxqFfn
61MVGv+cpYbDOmXM9CqoYE+lzGGQ3RQBhW8THqThAO0/44LbIJleJuhwr4e4Z46i
9XPElQKBgQDZDFu6mSo2MdYpEQucOgJAaLrwyUPCD9Ja/Gd+Mdb+2AC1kovp6v6u
xg+COGdQaLddU1yWbxB39lKC42yi7AKDb24gCBRmyj5lsgaipnlQkGtNNK10MWg2
ff90qGZpBO6tE4L/GMNsEcvQxxFGkbrTnU08Cp9mz8zJr47rlRoRig==
-----END RSA PRIVATE KEY-----
"""

SSL_TEST_WALLET_PUBLIC_CRT = b"""-----BEGIN CERTIFICATE-----
MIIDLDCCAhSgAwIBAgIUUFx52dcZRD27Ek1S8a4FJOg5WTwwDQYJKoZIhvcNAQEL
BQAwRDENMAsGA1UECgwEQ2hpYTEQMA4GA1UEAwwHQ2hpYSBDQTEhMB8GA1UECwwY
T3JnYW5pYyBGYXJtaW5nIERpdmlzaW9uMCAXDTIyMDMyMjEwMjkyOFoYDzIxMDAw
ODAyMDAwMDAwWjBBMQ0wCwYDVQQDDARDaGlhMQ0wCwYDVQQKDARDaGlhMSEwHwYD
VQQLDBhPcmdhbmljIEZhcm1pbmcgRGl2aXNpb24wggEiMA0GCSqGSIb3DQEBAQUA
A4IBDwAwggEKAoIBAQCvCRPBz1Wjq7GEdOtWDZUp0BTB4pTIuXP2F7bdAwPTJ7Pd
T6mjIPxM85E5sJL0pHvF49BZSFPm3OcrVv/4CHSZ9OV9aDGeioHZj5DdFZJDFv21
0E02uZSDZBJ6b22zbA0dqq/wWm/s6J6dbypcW4895QeH/OqMU//A4dFPeDJzBf0b
1BVjt/1QsFoEymb2nPRd0Pi05ROhuEaM4QuNSmncaFucLnDnn2UE3OB/O5K8zfyG
hsXxJx3RwS+0SLAIbzde9GxQUbU3SS0CFHN9J+UbeOX+fIBOvPrQsWZwRdOveTFS
tcqiEe0InWYXIIsfBxfdI0JNrsSrKlwMAhITg7yfAgMBAAGjFzAVMBMGA1UdEQQM
MAqCCGNoaWEubmV0MA0GCSqGSIb3DQEBCwUAA4IBAQAuBrB0tzL1s5Eaxv04bVeY
s2bktuk6zFw87kN9mJBEAQSkVAzx/5DfL/XT4uFhls3VyLOU2yMu7Vhxm2ZuMVXk
QdnhZ43T87lBnDBbZHd1NkrCY8UBqePB6cgA4JpmR1R8ypZmT6/V+POjqKGGwxJu
sxjF0f5VB78yJlPEoAnzYiqZUitZ1Lptq8K4ICORtN2yWG+dsq8Cv/dIvMOvEMpy
O6ozbk1qY0EaRuZAl7S/xghUq3QQOFZLQIkNGSJ5Q5xN1lqPJiDBinAjSb/r/zYn
edxjLvMjHDSLEfzyD9g1nmNq9z/8QsiYdc0mVMkkuvhXGDdCMOFmnZ0Zc3nPdvQ7
-----END CERTIFICATE-----
"""

SSL_TEST_WALLET_PUBLIC_KEY = b"""-----BEGIN RSA PRIVATE KEY-----
MIIEowIBAAKCAQEArwkTwc9Vo6uxhHTrVg2VKdAUweKUyLlz9he23QMD0yez3U+p
oyD8TPORObCS9KR7xePQWUhT5tznK1b/+Ah0mfTlfWgxnoqB2Y+Q3RWSQxb9tdBN
NrmUg2QSem9ts2wNHaqv8Fpv7OienW8qXFuPPeUHh/zqjFP/wOHRT3gycwX9G9QV
Y7f9ULBaBMpm9pz0XdD4tOUTobhGjOELjUpp3GhbnC5w559lBNzgfzuSvM38hobF
8Scd0cEvtEiwCG83XvRsUFG1N0ktAhRzfSflG3jl/nyATrz60LFmcEXTr3kxUrXK
ohHtCJ1mFyCLHwcX3SNCTa7EqypcDAISE4O8nwIDAQABAoIBABWYkx14jWJyTqa/
HkIxztOn3sfFQwI5D3uvlVpTgudMoL6Pk6iS6047RxHNHXSdzr+tygkaCXAUlVkd
lNtvf6N3OdFtR6H+CONk+USt4qvwbEFAgCZeY1qKwFTm5qwaisu8QZTwM5sRYHCg
RKO8kEHcvj8jD3Vc79NGiWK6u87BHAUp84GLLgyuju1aNaZ8n0NM+P6wEWxlduFU
l/yNoSJkQXAs8okIIA1yR1BGoCcgzXJnGY+WGglQpmSxWigay7mgg6odLoWGf2jh
eL7AsB389cgGm7xfxDj9cfh3lSnZJCbSDFcD2nl3DlgktrDq2JpndI3IL46hBp4i
rJnVPQECgYEA5EVznienuuGegGAod0Faaj9W+eEngZmikB0JaykRJpR9D9NsXXxM
P5iMAhbTl/HWh1ISXD5M8DOBdNYHHJS1sbxflWrHxR4yHqPboZw3sFUOEp6yyZ1K
5eGfXPEWm4s0w9hytC5YoLp+8DP8wqBdEtYLL/CNKRhukzwYmkYEtkECgYEAxEwm
0u/eYHOcBnnbJTDZwQTzjySAlyVPa2DaLlEtPfYhP93Uy74P7hh4e4yABwVxhO7p
rnES/NFDOMSOe2hsd8E4amknI76C2/xNaRP92we6jU/Rsw2HFbuqapFvPzApY+rm
mtGu/kEG2TKCOqvTUar2azmV8jtFZrGavR8Oet8CgYBQ8Yuw7jDc5Lo2nWJb/1GX
UUR/MOa4Xwk0/wbi99n8CzVZkkff0n6bInWrG/sF6xx+OY0HJpRTrH/gNTCqYtUW
5EwkiIu0CyQMl1//K4zXuyFVWu7c7Sis4VeINnkkyLT6KVtrJvAIdnkipdygCwtV
7Y5j/nGfu+khznKO6fsrwQKBgDcdbpUxVk3ciAVld3t0wN8qJbyeMiC+EBOFKR1y
G3LXRbSp81KVdeGTK1j9NQFDrQD3F9Or2fn0Q5yPy7WotzPUo6N1DNFDb3iW2aFU
RKWYAPAmZimQ9cnLXGj5lrO2MRjl3oO/thObbHHVda1Fi1rV1sR6cbiGndDcsybA
ro8fAoGBAMreL4+KxIzdIBQ36bN70pxqUYREoeT+d3AUFhfFyYaH0U0PFXOf/XtO
kC7k75CYwsLiarYne531tuo6AvY39aDePK6D0uHF4BqI/PZ/rsIKOVAkUhWSNDU2
b4NkEZRl1ejdxk/PjGR8oH+9S1X4Y1+kzJKpqt/k96Uct2eSxiQM
-----END RSA PRIVATE KEY-----
"""

SSL_TEST_FARMER_PRIVATE_CRT = b"""-----BEGIN CERTIFICATE-----
MIIDLDCCAhSgAwIBAgIUR15D8Dk0zvppH7mRgS1FK1xRwf8wDQYJKoZIhvcNAQEL
BQAwRDENMAsGA1UECgwEQ2hpYTEQMA4GA1UEAwwHQ2hpYSBDQTEhMB8GA1UECwwY
T3JnYW5pYyBGYXJtaW5nIERpdmlzaW9uMCAXDTIyMDMyMjEwMjkyOFoYDzIxMDAw
ODAyMDAwMDAwWjBBMQ0wCwYDVQQDDARDaGlhMQ0wCwYDVQQKDARDaGlhMSEwHwYD
VQQLDBhPcmdhbmljIEZhcm1pbmcgRGl2aXNpb24wggEiMA0GCSqGSIb3DQEBAQUA
A4IBDwAwggEKAoIBAQC+v0wEABCQFqugGlVovy1kKU9YYWmPmK431Z2Gm8CECi6Q
GujhMJ4WYp9g/NhGcP7U6uQnuI1Ab0paDkPW4mMhqpfrrFE+dy0q+G/vlJZnD32U
BvAPHHd6z8QBJ3bM9t1mMZMPwi7iCadwQMDHaz+DOLC9EJO1FhsATAw8FLri8lJz
Acvs7LUwbRqoDn+7gR5lFg+kxSkEr/E0rVQZW27n+5nDE7lwU15O4czbczN+z1jA
ELSn/wJOFNyVsrDANALQ7ibHafWzk7T5wWDUyMsOSOd+E3uYz+myOJej9XpfwL2q
ANPkt8sa9XmJwiG2O/O45Lt/6nsjJzQbVuOW6jcHAgMBAAGjFzAVMBMGA1UdEQQM
MAqCCGNoaWEubmV0MA0GCSqGSIb3DQEBCwUAA4IBAQAqb3wJxEyRg3Z4UFzufrmK
WeWqEOCndJj5l3ppvf3yVW2VMoiqVssnfhtQ6LXJUZrl2qrkAYE0LFlWLpr9DQPG
i0Atr3dFH6+6f41pMe4GX2uusR/MdmRoU1mF2TpjxgkS8B5Q2aB23h/q6SR4n4bR
SWO7i3ARFljj9iAMVTOqrpyWbb+wuGWav709BqAXM7ZUYga6ifKaxL7ahr7K7HTY
yXbmisTBUY4yT7RYfvau1GyT93fU7K8cdtXBDHUBpVOWUJFQBHPKVKRivO7DkDTR
sPcphGcNjvGSNqjwGmN4Hzp+WC5/uJahRlhL7Y3+v3pfIeGUnOSatau6UuJpXNl1
-----END CERTIFICATE-----
"""

SSL_TEST_FARMER_PRIVATE_KEY = b"""-----BEGIN RSA PRIVATE KEY-----
MIIEpQIBAAKCAQEAvr9MBAAQkBaroBpVaL8tZClPWGFpj5iuN9WdhpvAhAoukBro
4TCeFmKfYPzYRnD+1OrkJ7iNQG9KWg5D1uJjIaqX66xRPnctKvhv75SWZw99lAbw
Dxx3es/EASd2zPbdZjGTD8Iu4gmncEDAx2s/gziwvRCTtRYbAEwMPBS64vJScwHL
7Oy1MG0aqA5/u4EeZRYPpMUpBK/xNK1UGVtu5/uZwxO5cFNeTuHM23Mzfs9YwBC0
p/8CThTclbKwwDQC0O4mx2n1s5O0+cFg1MjLDkjnfhN7mM/psjiXo/V6X8C9qgDT
5LfLGvV5icIhtjvzuOS7f+p7Iyc0G1bjluo3BwIDAQABAoIBAQCK62N/XlSxu/EO
z174xJNYaUWiRn/M1xO7ElaBnJmfEJNM3GxS5UedYgJVbpBM7SqLAu9bhmtJTERI
Ri9eJs6vzECMoZkh90XsD7fmMr8/G+cHke3v83mI9hv4AzfgmsIwFVbXmULv2Lwb
Yu2DzzYYaYc+iv4oWosbskcO0sIxWax7UffFAM6fw1SYboL4ubxQrCYJKMvSCPE7
tDtuvndO1tr3EFrAo84E5KuPFRR1ncaI07FPC2H7tK8qMf8N/ks9CNxA7jmnPS5L
yaIoNtwk4i1rFTVzZY4hPQD5cN/b947XEMiWSZtcKF8W9La2vbjcJk2U6dTQUre2
4ej1dx1BAoGBAOKrQ8jFjB/eeQ7OnrKsyh+Fk52F0qEOWdnh/OmZ0BVPrt1R4Rtc
hEbYk4Q614AlHApiSP/SnQ0vtBx/C0If46ATRmh76hsmqeH0dK2FHNgih2IYsXdM
T+AiZDzhTrNdSiZu6ZiegeQeNUhx2dbJfLIaEbP0l/A9xPmrrjxaZCRxAoGBANdu
EqG/VTxxnqLNEsIWfPT4SVTXAfOHOMyxFvMJe8Nrqe7yAuIfybw/JNJHE8J7grvB
0aZLYwDTo6kmQW7cviNIkcc7TJCUczJjqlu0XxEIcu70J+1cel2Ekh/2l6qyUG4y
WaputvX2pgWcFiyBhIG0SGfjPn7Izzmk/th6Cu73AoGAWDkYtPhOxcitjZnWIu0l
7obrIZCInq8bQba33aREXUcaZIIN/7xOr9Rz9i+A2RuYgmImz4uGHS/IBp9cHmyT
CCb4bIpuYAr4bD3de/wncehekUvTJv/IxxAbu6ZmH2HcDoF04jYvkD5BtuS5SOQ7
aIbFBSnaSWOBWzv7IPp9p8ECgYEAmu/BWq8fAE+/BdtzqD9AM3b2Vpwj/R+9jAm5
NaYWat/d4coQpMyz7D2Fn+4amf72cU7eFzmUxJ1x0U+AM9lyrjGdFqrhUAJUL2Na
vIWm8bsf3hefN6kRIH0eEvro7Y09J2Py9Q6Xa6KumM4/bFbiE96zTHQ1GD4YO58c
4lgz0NUCgYEA3vVVE9JMVlN/KXW0rrw5WHCX3PpYhs3ohvXinur1vHSNdYi+thHp
QoyG/jA2t1jVQlrsbe245QNXzoh+2mZKtzEukY2B7ivoszSF0k5P8CK52efof6Ss
nejt8xMdtQlTxcFx2EOnsvPjn3nmMMuye4bUsZPprPqNrDtzkzJVccQ=
-----END RSA PRIVATE KEY-----
"""

SSL_TEST_FARMER_PUBLIC_CRT = b"""-----BEGIN CERTIFICATE-----
MIIDLDCCAhSgAwIBAgIUbNfOeBL5Eg+Tt37NytSIPVDVDKowDQYJKoZIhvcNAQEL
BQAwRDENMAsGA1UECgwEQ2hpYTEQMA4GA1UEAwwHQ2hpYSBDQTEhMB8GA1UECwwY
T3JnYW5pYyBGYXJtaW5nIERpdmlzaW9uMCAXDTIyMDMyMjEwMjkyOFoYDzIxMDAw
ODAyMDAwMDAwWjBBMQ0wCwYDVQQDDARDaGlhMQ0wCwYDVQQKDARDaGlhMSEwHwYD
VQQLDBhPcmdhbmljIEZhcm1pbmcgRGl2aXNpb24wggEiMA0GCSqGSIb3DQEBAQUA
A4IBDwAwggEKAoIBAQDdwdMrIMRVxTYSO0s1krlXy/sAk/BYSKWEUIe6Htzc2GGn
gAVNdvOiVsXc+T213Euw7yvBb0pfRUmS5f1O+uWZespgDTsZUy0SJYOuHhGFmI9I
UelOX6Oy8fRDWVz8lovFKRcr2ytywXTAb1mTGad7VHwPW3DlnE/xeAfmLPjENQxy
qiUjxqgqK1U+mdhxIxMg5hScsFjWdZDvngKzH0bojgicnKwWAymG7eJDd31TFQ3d
Cs0SNPzXmNAOKCRkiAMgaln3hG3FXASQbl09JvEdYPpxcJXUePxBIXh6D8KhkAGS
LoFq4uXSw4cm1fMO/TGHtZgVOFS9+EprA8L2KSjpAgMBAAGjFzAVMBMGA1UdEQQM
MAqCCGNoaWEubmV0MA0GCSqGSIb3DQEBCwUAA4IBAQC3ZKaUzSnUlaeNCdUvyRL4
RYVbzr5RuUPL085/JEtSpXUF7cZPMbpvpw/EWIUp+EPVG8h4/89F4L67MmHwCxgU
VwL2N87lk0b+TpOkLO93mO8gIGy7jJuIvDGqhWS13ocKI0Mm5Gl7n3zPS7pbqj2Z
F82nU6532WK16UBiYBYa4k+ZG4puYXpJc33/M8eiUQK5eY3sUL//OTJVz6WO8vCd
QVUn3+FwKZSJIaMaosKgb1SdHSqeti9RDNdQ44k6bfklZoz/NLOXi3bGUOt7RUp+
PEWLB7UnsI5qUY8aUU6g7PE3qt96dTy/9c+NTcpEnHj20wykofuPkaZOgDNgx51X
-----END CERTIFICATE-----
"""

SSL_TEST_FARMER_PUBLIC_KEY = b"""-----BEGIN RSA PRIVATE KEY-----
MIIEowIBAAKCAQEA3cHTKyDEVcU2EjtLNZK5V8v7AJPwWEilhFCHuh7c3Nhhp4AF
TXbzolbF3Pk9tdxLsO8rwW9KX0VJkuX9TvrlmXrKYA07GVMtEiWDrh4RhZiPSFHp
Tl+jsvH0Q1lc/JaLxSkXK9srcsF0wG9Zkxmne1R8D1tw5ZxP8XgH5iz4xDUMcqol
I8aoKitVPpnYcSMTIOYUnLBY1nWQ754Csx9G6I4InJysFgMphu3iQ3d9UxUN3QrN
EjT815jQDigkZIgDIGpZ94RtxVwEkG5dPSbxHWD6cXCV1Hj8QSF4eg/CoZABki6B
auLl0sOHJtXzDv0xh7WYFThUvfhKawPC9iko6QIDAQABAoIBAFMzntRGpgN8S85J
Wu2N0GaFjPZpizrEfv2G7XXCkKF3uiQLX3HyGHUnU3OWPDYYonmPMv6Pj/rw1yr7
ia/xaOXN8VJahHr6/yUY2DAi7fYPCGtc+ElEjvnb2AbQ55eJsIVX5m64+7NBrFlr
LdOIQ5N9XlKwv0oW/NOfcm9FHk0W23qsFG6+lsPnGuCCvzJrzz2OxFsGhRD66e0L
3Zv4xMjxIKV2nHyI8Ca/rWgQnYJ465TWMbz8qfIcH7FKK2T7/poajaOBqiVQ0FvQ
JxU5LPoCDfqN1IQxt354KhOEy33a8vOlt6NkbN+0BUdB6Mw5JPTg1I6EVMKwZMlr
lN98wbUCgYEA7u4p8+WM16VoUWmwu+4C/GNziQdg7sDFD5sALARgaAle8ql+/pU2
v4PbgY0H7AQ4YdLQkMG9k/JyitL1uIcdt9I3NErWahFZsxu6uMa6loqERW1/1JZa
+yJbDQk0s3P4XxJ16XEB9RHVpjM4K+70I2Hj6wBZ/KCABPJ9+lMkGGsCgYEA7ZmT
2Pf7RPZZUBDx/BcRXvgFQp0UBzcUCW+gEfcmuXUFsfxpEML3p6jkYTG5exmiu74V
JSMfJQbxgjx/8jV6qjEWrOI19ONmrLwAn6YSi4kTSFOzy/nunbO90yZS1EPWHTgt
1KwIpVV3YQwx8unAj3w+ioNKW2pNdKqgruD9qPsCgYEAoRnrJa2oupqn7QUDiqOg
VcdE1qcn/7Pw62XKxuMokd2HrL0GiOfZY2an8gFCzyvFTIhV4tUPW5tFCj/WONa5
/gD7AgJX8mrCfZb73qz4kJrTNvDvGUtMCzsOA4sqgrl2RbxvzrhI5pJg6VBt1omn
CdhuUj0Jb+JUj4Fl2gyHMQMCgYBbKzHS8BxnMsvEcCwaGdPdh/ekvtcfiZ/TmxRl
sFsYkHGWAQ9jkhwrJiAztzu/o2+Q8Hf2nwplT8u9uG8zxheCVIGp5sbjZ5pPL3hh
yfyVtAxFFXEJFKbdtklhSvQ/ZKQ05AiZ8UUEA85h26cS2WS86eSpi4UtnJWJ7aro
T4alnwKBgGQfkEQvkthUo1xwkYQEgbJzk5aunQ3aZ0467L/kkEJZ2SknAXKLURin
uErwhm3wgKHMCy3p5SLU7uv1BihG6nNvLuskrD6mAkUPBMJmXCX1NEIBLwcMM4YH
H7IipX9XJOjCnauv9AsChOkz9pgG4/0BaaxkQ/Cm/zKm+wUXIfUx
-----END RSA PRIVATE KEY-----
"""

SSL_TEST_HARVESTER_PRIVATE_CRT = b"""-----BEGIN CERTIFICATE-----
MIIDLDCCAhSgAwIBAgIUZGh0kY4W3yL0nzJVIYYmCTQM0LwwDQYJKoZIhvcNAQEL
BQAwRDENMAsGA1UECgwEQ2hpYTEQMA4GA1UEAwwHQ2hpYSBDQTEhMB8GA1UECwwY
T3JnYW5pYyBGYXJtaW5nIERpdmlzaW9uMCAXDTIyMDMyMjEwMjkyOFoYDzIxMDAw
ODAyMDAwMDAwWjBBMQ0wCwYDVQQDDARDaGlhMQ0wCwYDVQQKDARDaGlhMSEwHwYD
VQQLDBhPcmdhbmljIEZhcm1pbmcgRGl2aXNpb24wggEiMA0GCSqGSIb3DQEBAQUA
A4IBDwAwggEKAoIBAQDVy8kOTXSTWhLK3BILK95b5/tY1dq7ARqrv4tVDWcXuyGV
vPPNj2BibF3Iv+4Gr/6XeZEWmH2ybbEKrAws9aBIw7ETQhTNjoBlEvkoKvISBkkR
KaQ3lUQ0Km0BONsdSwxkgcgzn61TeSOlWgpOuW8EOXZNSEUJTxp82GbbdVH1TAXH
28oxUvrV/hPltxuQJ+3tcqN78VD2cGPwp4+RNYwklG04Jcj97wjv6v41jVrQ8IR2
k5GYuI8f9V6y1zw45A4XasZDb1/UmKk5cROSI/nJgOqZLmAhjK6eQMx1XqPYm2B/
e5J7V2nrOVhsasddwNH8fx+QhBz8cZwd7uK17oWdAgMBAAGjFzAVMBMGA1UdEQQM
MAqCCGNoaWEubmV0MA0GCSqGSIb3DQEBCwUAA4IBAQCpby2yro0LmC4xuhhuqs8q
pn4r6E5i5QTDHa5uVZ/GtNgYDP7x5MsSaWcVFHsq5TONiQY8rFg3iTMfoQTqvVqg
+4Eyopx/b7a8SryImxJXNVCNnY2GZ9DQIojYxDqiqsqDpIlgeR7FRCrjdy0W+5h3
Icbugg7sDeLYnerD9RAQciuClMC8jgAUc8LWXvI715kxTbrr2//8wu0YpfBLHln7
zL3zdyKcaAnqvbBx1FHgj4YI6ozfdpqYX88Uj4aeirCzHzT3QQGoAHXoDLFdg2yX
s44rft3s9/+MKiFdcvaxVkh5UIajc2q2ZZJ/mqdRtW/paO9cv+aNSdgPoRcn9hzb
-----END CERTIFICATE-----
"""

SSL_TEST_HARVESTER_PRIVATE_KEY = b"""-----BEGIN RSA PRIVATE KEY-----
MIIEpAIBAAKCAQEA1cvJDk10k1oSytwSCyveW+f7WNXauwEaq7+LVQ1nF7shlbzz
zY9gYmxdyL/uBq/+l3mRFph9sm2xCqwMLPWgSMOxE0IUzY6AZRL5KCryEgZJESmk
N5VENCptATjbHUsMZIHIM5+tU3kjpVoKTrlvBDl2TUhFCU8afNhm23VR9UwFx9vK
MVL61f4T5bcbkCft7XKje/FQ9nBj8KePkTWMJJRtOCXI/e8I7+r+NY1a0PCEdpOR
mLiPH/Vestc8OOQOF2rGQ29f1JipOXETkiP5yYDqmS5gIYyunkDMdV6j2Jtgf3uS
e1dp6zlYbGrHXcDR/H8fkIQc/HGcHe7ite6FnQIDAQABAoIBAGzlaJotX5RyS6Ii
J9qi1JaReZfNDaU0scYLQUfoFLukqD2hybjJL7nSIo7PG8kyOYYj+H++2j9Y2dVq
vqTj02lcq8P4Y6ozzDgvStNn1HbT1sG0KR3anKvJdE4sDIdGI6DJ7hRFX75ltH9T
dm6UyvQm5JgGmIteTEqM2w8rGbzLtsg6SakuNoyGSL17wlc6GmYiO1N383KZaaqR
yfKz0yQDRIzHiGv9bQP53uPNZv0YL0+LJQg8lNGyq8+grwslT89zMw81pnCcWEjJ
Bm3g+97tyIzxBW3CBh3KCfAtggrYNZ0BAvC+Hh0+mExQSEJOK5Xh06r/OnUYLcKK
JTeK3CECgYEA7W/aWfOXwLz18qt4AqYSBWV8/MzAj2UGPvbHbyM/y5FOpBNHWRge
vwrCgLXWAFLYnSrbLs+7CcLdlURGo5aAT0Xfxefa2MKUK9HVBLGjzQAhL+g1Z7KU
RibPgxcxf9rEE0bfS59AamD9y3psRm6yzqfatYvvAQeMy8aKx30hGHUCgYEA5oLG
eNuHxRICGZNbV/4Hgj1dn9Sc+28wVU50Km0aT82rTddH2K7xX3EqN3+bS0npdOwB
2c47MeRJWwa0/QII+5MOPNcbhz6pHy6WhPkfbphjBfURtTlInJ4QBbe5Zdm7uGTs
zVSXyQ5vA2bhkGHlLR5pTf8hbQ/UipAglRXQ04kCgYAbMZ5egwm0Z2/71Suvkfmq
aI9CHOlJAxcOCxGYZv8df4z0OckIKT0MQF06hJj3/IUWqxvjZqNTYOAkAmfcwJX9
flE8x2tHPDHgOu4c1YOfLQ6gAAZt1AQrbLKDnxpqPFIMqydT7+9ev7ERpvnTr0YH
aS4hcN90zLqRdDHc2zX5KQKBgQCU9aeqTDtlp6iBvI8Hwto5WMyaz7uOZCD4z5UU
IaMiNQdhayA7r3m9spSUfrrtIVtjOmxLwxDPwfWBhEkLQvhpEnPd7ygT8vM5elK8
pXyhbIp5PBBwaw84XpB+EWcWoWjE1TuNyP+5ZsFokmBrfEesK33jcF66asNvs4nK
relf+QKBgQDREMabtiYvEmJhG+oVnqcWCpQnYetgrUYeXKj0sOeRvC15XRSt0Jh5
TjxpY3OV37lvXDi9QCst2akbKxahzP4CDX1//RQsT2fgzLK5yc6xmPm5hdLb0EtR
K3WDy96qj7+5xWNpqSGollLAFxXmdZ4NOrnU3mMHxsLDYe0+9diGQQ==
-----END RSA PRIVATE KEY-----
"""

SSL_TEST_TIMELORD_PRIVATE_CRT = b"""-----BEGIN CERTIFICATE-----
MIIDLDCCAhSgAwIBAgIUBingUaQUNXQpmRUh3SWSFMUaSqkwDQYJKoZIhvcNAQEL
BQAwRDENMAsGA1UECgwEQ2hpYTEQMA4GA1UEAwwHQ2hpYSBDQTEhMB8GA1UECwwY
T3JnYW5pYyBGYXJtaW5nIERpdmlzaW9uMCAXDTIyMDMyMjEwMjkyOFoYDzIxMDAw
ODAyMDAwMDAwWjBBMQ0wCwYDVQQDDARDaGlhMQ0wCwYDVQQKDARDaGlhMSEwHwYD
VQQLDBhPcmdhbmljIEZhcm1pbmcgRGl2aXNpb24wggEiMA0GCSqGSIb3DQEBAQUA
A4IBDwAwggEKAoIBAQDTqgBgTsNxYUYJrT8MLFe8Myww/wTuLUhZWTkCqGzQ7grt
BPV1d5BTN2gthV1sltE3pjVowQrakbhnYyKQtxkuuaL2j+99cd1ZAC7BopxnDgnw
wNp7lkwsuV9de5GcykCz5wLzkfj5webvaOWsYptahtFkAv2Q2EgeXp8ZK7xoiW9p
uTZ/74+7i+FqY4M3DcM7/RATcQTtKW8qvqF3oFUXlFxZb7j0UMjqPGfzjP0ccze2
zKxes1lgcic8TzCHWztOC4sYpSEfK/djDv2vSVjBBsGOWK+hi8rDCmoOKJonO9Fc
iw431LSHlvVyhg73Xp3HmBz+92IY1/EtJDrMGooHAgMBAAGjFzAVMBMGA1UdEQQM
MAqCCGNoaWEubmV0MA0GCSqGSIb3DQEBCwUAA4IBAQCpHVgrVBHGBx+s+CllkVJS
re1AACf94K1PUrnCe3nSPIjsjlauMO1vJEg39XnOdxz1GIj0ZxPVoanRWjJWRC6E
TGivunOZzfREEFrMmvxO/azlgLn2VjhPxXxgyvyhjVE1ZUTabuNCMQtvwvon79Ko
TX0Na8Rkkdx23TPR33YIkQXeGnFHY3Cikc8UiXPus7LMMP89PmVh7ACi4t5v+DRu
CImEvP+jr18Sjhul9aA+CtL5PFPzn8bWlROCAgGI7GZPMhJvugCusHHWSzG8UblK
hWV8Aa5CTndzTOya2regyeZyy/ftXEmjqKjKSH5kfL83MI5MWr7AJMJ2oeEoC1W0
-----END CERTIFICATE-----
"""

SSL_TEST_TIMELORD_PRIVATE_KEY = b"""-----BEGIN RSA PRIVATE KEY-----
MIIEpAIBAAKCAQEA06oAYE7DcWFGCa0/DCxXvDMsMP8E7i1IWVk5Aqhs0O4K7QT1
dXeQUzdoLYVdbJbRN6Y1aMEK2pG4Z2MikLcZLrmi9o/vfXHdWQAuwaKcZw4J8MDa
e5ZMLLlfXXuRnMpAs+cC85H4+cHm72jlrGKbWobRZAL9kNhIHl6fGSu8aIlvabk2
f++Pu4vhamODNw3DO/0QE3EE7SlvKr6hd6BVF5RcWW+49FDI6jxn84z9HHM3tsys
XrNZYHInPE8wh1s7TguLGKUhHyv3Yw79r0lYwQbBjlivoYvKwwpqDiiaJzvRXIsO
N9S0h5b1coYO916dx5gc/vdiGNfxLSQ6zBqKBwIDAQABAoIBAER9H74J/meziAwd
iOcReholnMkF27yN3nk6y522je9U9ygWkK4/z+19YjJIkyBN9MjC+Uk6HZ73w+mA
RHD7LiTWYIrTrDS1IWBpFLgHPACEpGbIoGmZCnD2UotXLpjDQQh1WZkixKNcQLN7
B8Yt6gLy2NI5vX6e3UcmauikJETWjMm2CBa4O5s0sKl/ZtzCUrzy4osyDdaOwpHW
2vPTHNgTVkETlI9UZehqC6iXOsOGzZR2lpcY9KGoS9rXAnc31tfRCfolyHvKHm/I
YCEFLzLypHp48hI5qgikkBhJFC9GnvOWJtvz9XQlkUahBIGx+xRSvNXHZS/OVPqG
dYZJp2ECgYEA+AttR6dXwChtGNjEwOI/F2X3RzBKMxS7tLaqEGD1SYTM3y/Fz7FY
aqiE5052Q5CXJj7juVB1YMfPXhq6ae6wCFqjZC9mzc6DhDs9NZ8Vx4sCsIZQI6uO
DAR9P4FyXW8mMAijN8oRQb4dkyk9NgWJ+1lA83cYAXWn6hGDHvT8cH8CgYEA2nPf
Hi8AOu0aDYWrBRyobttWvvy8t8+JT7qvRcyLgUL05mblKkt7QzrJuv5WvRYdMhiQ
Fh3eaXZpmBrb0O1jAeTlzOlkJtKYJlq0l6My9gLVEJuDN2TjZpcqSYQswUQWImx+
wc71MUI03rLQnsXbuTgpzjxSp2nZ5XDRCMHnonkCgYBUz1NmFImkYi4rOLLNYI5B
6dNKu/Ai1wv3Fh2/y+cSZB3IkgUIppKcpVIL5H/7SgzkPsyOKDx6AtecTXc/kSUP
Ni53qt1zsRC2vaqIEwTXXWhmAwaTq6y93ysT86au27oq4kksYiLUX7StBISxW+N5
1WKn3CiRFzF6jtJOW2WU8QKBgQCPew2wAITxLywud34VlHYYod3/AXdeb2Srxl06
xhdW5kh+UFTyi7NBPAql1EnaVf0FG1b//8xGjqzaZ8G7H0ApTVOWXDEYwreGcYsu
EdX1+XRCPTJZHt2xU1ZGBRd0NSkooQQ8fZdULD0XjHEz55bit73VpI9RQFgKuCfD
zcA7WQKBgQCTBD1+CnwFHqrgp4tSjz+4kgclI89Hf+K0wSzi9hL5I4uvaJRvfVVD
mNs+eID0NbRALpNc9M+P6+EBdYFsSMdcOEeV672DFEukdy7e2ldbiQ221yTU3mSs
SHfBOBR7KJ6cmreUG9Cj9QLdsBuF2FEPfrzpP/YX/d7z/OdmStbX8w==
-----END RSA PRIVATE KEY-----
"""

SSL_TEST_TIMELORD_PUBLIC_CRT = b"""-----BEGIN CERTIFICATE-----
MIIDLDCCAhSgAwIBAgIUcWOy/+2e2VlxzEedhpSZS3UdmKUwDQYJKoZIhvcNAQEL
BQAwRDENMAsGA1UECgwEQ2hpYTEQMA4GA1UEAwwHQ2hpYSBDQTEhMB8GA1UECwwY
T3JnYW5pYyBGYXJtaW5nIERpdmlzaW9uMCAXDTIyMDMyMjEwMjkyOFoYDzIxMDAw
ODAyMDAwMDAwWjBBMQ0wCwYDVQQDDARDaGlhMQ0wCwYDVQQKDARDaGlhMSEwHwYD
VQQLDBhPcmdhbmljIEZhcm1pbmcgRGl2aXNpb24wggEiMA0GCSqGSIb3DQEBAQUA
A4IBDwAwggEKAoIBAQDHS48UM5B+l3SpuKg1ZncimLWwOSyHyCASgN1dTFhFMW7/
WN1hGGrB/2lw8CAdEBb1y+i2+PHo/HQq+2tn6GmT3744GnwS3vZOqvhN0LBA4Eg8
DtE6utFBb4TZKcU2GQqGOHS3BdewXcVdoMfzM4IslqmHaQOXiqVwMrCGjiVKzF2l
SFzZdIsUBFiwSv02ZvWKhb0IbTsm7YK4NTGYKZxP4fsTHHZeBICjL2GEKlkXx47Z
JwthnpxB1AgeIdD+6tp8oCGPZxzd4lJxxTV385WH9zFg7jSp6poAOETra8ZnnceT
YgfHVvY5PEiKB+uZzMiPf9VIoVE3KwhW7JLWktnZAgMBAAGjFzAVMBMGA1UdEQQM
MAqCCGNoaWEubmV0MA0GCSqGSIb3DQEBCwUAA4IBAQDGEr3k2P9Z4lOIwghhuVRf
T0YYzq39Vd/cDpQ/v1Ohs5kMdAEP/sDt5JWamtNLkbUDhbdgrE4Z5uKd2zTmL8Nb
aiT4vJ8E9hYYEheatE0qNJl0v3qxhz2mDGDfuYxGPPnssuxr/6cGWA2u1FasInYo
obLx7tBB3HNgKAtqSJZSwAMAhebJataR3REtkmBkn8s9zaTczwGsavH4AFit9ipB
oXkddyfvr5Ogpo1BEydLIOCv1c94zh9yI4gn9/Gz1uEgQZq6xg+A1VV4p3Oy0LWf
4OjEohsV2dX8KMVbVJwht4HiTv58M+iL9kxvcYe2Ao/z6CnvObxiFgD0eVZjNsVq
-----END CERTIFICATE-----
"""

SSL_TEST_TIMELORD_PUBLIC_KEY = b"""-----BEGIN RSA PRIVATE KEY-----
MIIEogIBAAKCAQEAx0uPFDOQfpd0qbioNWZ3Ipi1sDksh8ggEoDdXUxYRTFu/1jd
YRhqwf9pcPAgHRAW9cvotvjx6Px0KvtrZ+hpk9++OBp8Et72Tqr4TdCwQOBIPA7R
OrrRQW+E2SnFNhkKhjh0twXXsF3FXaDH8zOCLJaph2kDl4qlcDKwho4lSsxdpUhc
2XSLFARYsEr9Nmb1ioW9CG07Ju2CuDUxmCmcT+H7Exx2XgSAoy9hhCpZF8eO2ScL
YZ6cQdQIHiHQ/urafKAhj2cc3eJSccU1d/OVh/cxYO40qeqaADhE62vGZ53Hk2IH
x1b2OTxIigfrmczIj3/VSKFRNysIVuyS1pLZ2QIDAQABAoIBAHfJS4tp3tyn7gAh
NEE8gDj0LqhoBQLr35Mfj24HeDrGlp+16wH0Kh7GhLrUKYkuZY6BbDOl98kBDFXV
Xl5LDEJMRkn0aUTybHsmOi6TU4z7AY/PnCS/qwy4mdHpUFbOwEuiXUywn06rXfiD
rgrYl9JzEByKmxUXyY5heUa242VL1YxVgKhtU41cTxo8TaaZ9mhQdjnMY09piuKe
Nb1pK5TzjIl/ci9VXp/JWtOcES+EGNLlboQydZAM6RXC9/GQKcdA39bibq/nLg1b
u2ha4NQPkxYUI3Qlzjz0X+uPvIr6vUHsvzUo5e6GFhVM3tm26iGtojogrP83/c6u
p12TDgECgYEA++fbGx/5q1Qhf5bbEZykseTmbOk+Qm969IEYpcDS2ENGVntc/zpf
xbwonZJB3gk1P9YWfOF0wR3IueG13D4cifRbSCfW8gkT9DzeNJO2QXsOFM96lb7r
+oUXIYqecno5T0V0Rhs4miIoj+M5owJiJcQHEbolKblFEWoKLb7AAxkCgYEAyojM
SM1mO8bT8WU4AQ3kGq0Bhy3fpQtAmkoOKWkcYsWwzOLMQDl9RM11r3k3ha92zzQn
8z5Dq+CMreO/7k2AO+iPQUxbVxSWcDfeBgFzUgRDrhGfVfInn1BFeyqTu8achZV9
FVwg/b+qhNbnoOOL9qjsw6LYyG6w2uDEbzeJJMECgYASEb6QwolpNb61ZT/Pkpnp
vM6ej1iM8Wwzb7Mx6JP3OVQpTe4MzMYj6+xJ6TzaTXTJYHT6fj2ewKEuXDdsVfQ8
HRATK/BqKiaJRXh28wWGyunmqj9T0H8i2Dshh8z3zzafLg86HCfCPqMENztiKwCZ
Tudm6SBTTtvadO3H/D/PuQKBgHM4yB9S2yp+vuzCDM5svKyAaDIjYuxeLm6YWt4s
Mj3vAdpN3K1dB2sxCBa/+1pOrGTM4z162rzZjq8Oto44ARAUCgu4CO7Ng5hr8B+y
i4zJaEeLNv6KOQhlfmHHoyxcnYdRuEuCkjJJ7BSOvk/FpEa7a5cQ+gttDag33wj2
q0sBAoGAchejiA7NRNoudW5Nj5AZz5Lao2mY5VABIQEe3fbZKmNN8whsDul0bpEi
AboLaJVCj71HEwBKEPGcqgK9eAWRJqgmkZXQm1nXjmJeFo4PF98L98TBsM7JxnBr
IDrNl1cpFWosD7XWnabJnVLZ3ZfRbXHE1egjnTjtJ9fQhlI9Du8=
-----END RSA PRIVATE KEY-----
"""

SSL_TEST_CRAWLER_PRIVATE_CRT = b"""-----BEGIN CERTIFICATE-----
MIIDLDCCAhSgAwIBAgIUDNXuwE8yqviMvf47HApsgcMIG6kwDQYJKoZIhvcNAQEL
BQAwRDENMAsGA1UECgwEQ2hpYTEQMA4GA1UEAwwHQ2hpYSBDQTEhMB8GA1UECwwY
T3JnYW5pYyBGYXJtaW5nIERpdmlzaW9uMCAXDTIyMDMyMjEwMjkyOFoYDzIxMDAw
ODAyMDAwMDAwWjBBMQ0wCwYDVQQDDARDaGlhMQ0wCwYDVQQKDARDaGlhMSEwHwYD
VQQLDBhPcmdhbmljIEZhcm1pbmcgRGl2aXNpb24wggEiMA0GCSqGSIb3DQEBAQUA
A4IBDwAwggEKAoIBAQC4r+RhnufHjAar0Rj+PEnVQFi2DtJGOVh4u96xaEbiY2EJ
3Z+jZG+pV2ivKfKZJiVCWsMa/XZ9GoDckAM/7/BACMozRYdBXGXJQ8luOaBLh6oy
w+nCMQ7zs6J3meJKm8AXiEpq6y7AGJ2P0EcG5qY7jiKqkNckFvJayuJ7mt9DGxS8
OCNu0ja4waEtcVsMerM11zl4hQAxpm/+zRbTeRdHe6jY7Oxg3/V5e7W9kj6klzAt
LEFq3GNqztXlxEvnNCf2ItJ4RZ6P6dzho7hKQjIICkmzBrApcEZAj2Rhr47q2Prx
Ztw9E3iBTc3qpRkadbhQAtXUHDxTX/wUUzXgkBfXAgMBAAGjFzAVMBMGA1UdEQQM
MAqCCGNoaWEubmV0MA0GCSqGSIb3DQEBCwUAA4IBAQDH2bDcb+NYjee2vV48E2Bc
16MvrBDfgQLA9Nv6oczcE2yfD1qOQTnoLs40QtKhDq2PMivTLSS+x6pstg0R63bH
7cCbWtGOYRDsjbjqTHwAFczQCgxZA3uKUeTaOv5M1jhQ4oP7Oz/oQv4yH7Fv9T0j
QOyCHe+CnLDP5VY0+3UWDSudCDIBWD/BxvsCBva+dstPfz3heIDciGiCRqrSXCh8
ROiBKUh1TaASOtcIKYIJfrLFs5zpHT0zZSppKQcxPeKimCR5q4ZC+CuPXMJzvzrX
xPlO7OvgQ6dJE1/nOnXdXvl085ycWrrefNb7IvJ2Oc2/yB0957jbm0+13tuikXgG
-----END CERTIFICATE-----
"""

SSL_TEST_CRAWLER_PRIVATE_KEY = b"""-----BEGIN RSA PRIVATE KEY-----
MIIEowIBAAKCAQEAuK/kYZ7nx4wGq9EY/jxJ1UBYtg7SRjlYeLvesWhG4mNhCd2f
o2RvqVdorynymSYlQlrDGv12fRqA3JADP+/wQAjKM0WHQVxlyUPJbjmgS4eqMsPp
wjEO87Oid5niSpvAF4hKausuwBidj9BHBuamO44iqpDXJBbyWsrie5rfQxsUvDgj
btI2uMGhLXFbDHqzNdc5eIUAMaZv/s0W03kXR3uo2OzsYN/1eXu1vZI+pJcwLSxB
atxjas7V5cRL5zQn9iLSeEWej+nc4aO4SkIyCApJswawKXBGQI9kYa+O6tj68Wbc
PRN4gU3N6qUZGnW4UALV1Bw8U1/8FFM14JAX1wIDAQABAoIBAQC1MsVETe6jpitp
aXY3AZmC42XmATam9V5q3hAISQG63bv4PPuCCGlRYNPxf5toTLejmX7wvRsCdP1W
6Oh3U4aInNcdZKrmIqwuwhYf0eQGFq2Srwpqn01HJshQxabygkr+C5bPvbyZdc0o
qzT0fVk89x/l7sNVwjm3gHSixvkjUQKBxDwVDVLbNJLdeN6LIkep2PJoZCi8R441
ySR1GkMlJjpCCkrv35/syAQpevbdM+7wBnuZkgfnzE7ECNKPvSMtTcgSf7Gs+943
ydMau+e7sL+uDME2EIq0cveNceMQaHn6mYr2PVULsD6Sp42YlJWWZCgO053jPQxd
+qKREFZBAoGBAPGt0E5nN+FCsRXgYI3dAD9HOqjp2NjVF+qUtnHIw/rRnfsxPhSg
dQxjDMrBc0bYk5vim7q3Kd5W59ElSLFBjeUIbE+41dlJot99D25eD54ipWZqWJ89
410y9O8/BOlmAalrlWmFUPvrIo/afVt4KbfZShvtCYkVHU5Uu7+LKmYhAoGBAMOh
iAGl6eP5KJBLJMe1ANNt7imVxOCzsEDryP2SPUk2tBpapxRCoYtHv0NFORKXUSX3
g86eJL2EzFIvTHta4ksWs5xb6RdTET+CZlcA2FZik0kb50URt5R24HhwNuvSP9ET
E31E9n2V+gX5OtaqHJUqRpf2FY9Y9LyF9PebWc73AoGAHoP24F9yE9RK7ds19dOo
21SQGJHmWoKVkX4Th0x1Sm87RmNSVmJoRRcgn11Lw+9GvBEh3rKP684HWUYe5prD
Yt7sZiiNf/EnSpbM4v+ncy0mu0bER98VnUf421iWsy25Gf5GkFtP/W8UHxvUZj9G
7TrXe60zXga3e5OBXU8iHGECgYANnibrhnMbjXKHaNOsmyrtjM9xUZ2czqVMP52E
GTPu7sHmu7y5qlG4ckwcEKuCYcoiTHJ+ZX/FOYNHjUdTpuXtuCzUi4lUOFIRC7C4
CbUFfmMmSaz1n+AZk5TYjWclT0Nl+F/47l4CK/h1hf3Uh47py0GSvmyNx/FxVhnR
Sp0bzQKBgHPyCjMEW/f17QG+NOTth87uBn0qxQ/3N5u3TK8j697GYz6GrZk8FsxY
2dO/O5Bp87LxlrDmI17SpbhP4OIwT53eQoMESa1MrUuOSl5TNQDLfP/ZRS2A5FZR
azPRUP1FibFtIF9+Oil/V6rWU+oOnvSSYt+P0Npy+WXnyDdRi+qm
-----END RSA PRIVATE KEY-----
"""

SSL_TEST_DAEMON_PRIVATE_CRT = b"""-----BEGIN CERTIFICATE-----
MIIDLDCCAhSgAwIBAgIUaVjP5toSZjz+kCPgpM9eyglT71IwDQYJKoZIhvcNAQEL
BQAwRDENMAsGA1UECgwEQ2hpYTEQMA4GA1UEAwwHQ2hpYSBDQTEhMB8GA1UECwwY
T3JnYW5pYyBGYXJtaW5nIERpdmlzaW9uMCAXDTIyMDMyMjEwMjkyOVoYDzIxMDAw
ODAyMDAwMDAwWjBBMQ0wCwYDVQQDDARDaGlhMQ0wCwYDVQQKDARDaGlhMSEwHwYD
VQQLDBhPcmdhbmljIEZhcm1pbmcgRGl2aXNpb24wggEiMA0GCSqGSIb3DQEBAQUA
A4IBDwAwggEKAoIBAQDKT+/vp2gbgVymuRMyDDgpefM0auPvmIFcNGIS7MiCIc+N
qpYCfjaCdZc5gCo0pgS4Z1rqbl88puYH47Awav+b6YXz5jvU66ZssVlRHXdrnD1r
kXTdRGQxEMBy/iUfEdQgOgsk779QoEpAWAyxyHp2r3LSdphnRpHua+04cMdpWiRN
Jv90l/qHL5DEubwd9GIrnOa4BmfCfpyn3waJITN49ngSNumyQiCOoMKIcCRDdt1K
yxTw00cLLUVhhaaOlMGZ2Xvmlpl1UhIpXS6BPnztV7lbAQzgfAKFL7f+ae3hAsrq
jTKKU4BpzwNLsggE0/UaPqrvDP22rcQ0GBtYLUTbAgMBAAGjFzAVMBMGA1UdEQQM
MAqCCGNoaWEubmV0MA0GCSqGSIb3DQEBCwUAA4IBAQACY+ZfFoP7D93ZJmwjFm36
AYwDLDCWbrwk1+MuLGVa3ppnbhb1ZnMpG+SUUoGUZ6BB0/Hq/945Y1igMPsElSRX
qrwq+Rtbr3f3ZqZe/JIVSlpFpPYYdZkXFJYrMkLSGJP8DkJVpmlskbXGqtaqcIQR
CurKKhkBsYsY3Z/YTEf82EHwXifukX2OlCmNWIEJe87/Dx/01GC9jD6iaHgJA76a
RIBwzf9YZqpgBEgOoTH5Pvz88tRf63njMdupgJVsxC03AqVBLFacDQ6aRjYgLedS
8gJnwq0CY6nfO63UkPH1wHBLd8UFSw75FVIUxyeh9JNjiaPj/vlr7sOVseqTaKZK
-----END CERTIFICATE-----
"""

SSL_TEST_DAEMON_PRIVATE_KEY = b"""-----BEGIN RSA PRIVATE KEY-----
MIIEpAIBAAKCAQEAyk/v76doG4FcprkTMgw4KXnzNGrj75iBXDRiEuzIgiHPjaqW
An42gnWXOYAqNKYEuGda6m5fPKbmB+OwMGr/m+mF8+Y71OumbLFZUR13a5w9a5F0
3URkMRDAcv4lHxHUIDoLJO+/UKBKQFgMsch6dq9y0naYZ0aR7mvtOHDHaVokTSb/
dJf6hy+QxLm8HfRiK5zmuAZnwn6cp98GiSEzePZ4EjbpskIgjqDCiHAkQ3bdSssU
8NNHCy1FYYWmjpTBmdl75paZdVISKV0ugT587Ve5WwEM4HwChS+3/mnt4QLK6o0y
ilOAac8DS7IIBNP1Gj6q7wz9tq3ENBgbWC1E2wIDAQABAoIBABWBz/kDK+8Ynnfe
3uqIIARQwYkF//s9lCrwKqjmYR5sZ7sgslfLgRH3XD8xn8r/BytvWVvopdk1yOh3
zQgop8m0VpgqFpw4/PU7GUqlPkfDUUg5K1pGZfxNUpgm6l1WMN3ILd0cW5M/pwUW
FytjK5moZyV2lBcOin71HD/OxXQuf0wKp4xKswag0DzUAB0AVLHmaIIMf9Sp0hXc
NqsdL5hnxHaMJ78lA9a14dVQ3Kx2Hovo3EFShqcEOqQH/Ekqj1QNlivn5TiarkhS
FFcIwuJisFg3RKZiSETEL3LHYr/tx9Q8g44GzgQb3lwBznG/0WjtatEqXUNqJ/nz
s5fp2+kCgYEA/7yAkYMKyIEYO6c+31NYrONkOFvD1EkQe9boSb+CUZbjGNHw1NDK
ChUYTbKOSV9PLV2SQKghQ4KVN+ZReVeEfdGptax03tWfrz6woRMzLTyo9hxlZLU5
8xsNBt58f9XqjPsPU25FWSY2Q5C8AEykwNRiCBgsuW1wOmv+5IP39I8CgYEAyoVV
pJ8fy36CX+T+oYWyGMc1B0QCIWBPRZwA+EPgHWNBrNgd30JfndC9BJwq519eEaqW
x313/rM43B+mXgsMkndszIdwpt+i2Poqvv6FO6iwIdQsSe6FiUIRMYxwxccrzIp3
0PrTLThIl3Z2gnaDFh1pj6yKHxvO/vTUHTtoSPUCgYEAvb4tokkZChUGVtRLCMW9
KGF6rXogfPRM/6U/KeCmn0SVcIfr6OqUzRBDvPo7KeNGSsF/Tuuv6ngLoLMJRj+R
9QxHn5y6Bc94MD0SLiChuvGj5x7AeA0iVuZ+LteKhgrAgSORhnRtjayVXGIZqxsy
rkOmDJ1rKZUyTJZuQzxLD68CgYBzpMWt1kuCmP/7oDiipUR4j7BBMN2Cj4tUtWt8
WM3uGhKTj2knZrBQ3rRAoKiDLFr7/YxR59yQbWjnkY8fnnGUZWuUMzbpo0RLYDbu
j0onMgE5n/2WTgSkxzwoLgjrdvsUtnEKvV+/L6eMDdJIV0Ita6guqZjJgMinn8hL
1POMzQKBgQD5HFgGlb0YEpEu9aS2cQiBGW5s5Ww64Sjpqui7PRm8J25V7mGIcpAk
N2LxBYnCSt5vOcpx7MiWqsR50ExROkS97383VD7tb+gSLc3NiGHnL2Q0A4DXbNc9
VB9enMPAbSaXQ8dSj3l+GkC45xm5mRcespWoD1Q8N+FHENlkXYRj4Q==
-----END RSA PRIVATE KEY-----
"""

SSL_TEST_INTRODUCER_PUBLIC_CRT = b"""-----BEGIN CERTIFICATE-----
MIIDLDCCAhSgAwIBAgIUOwoUduReWasvp/QDg0CoLyhjMnAwDQYJKoZIhvcNAQEL
BQAwRDENMAsGA1UECgwEQ2hpYTEQMA4GA1UEAwwHQ2hpYSBDQTEhMB8GA1UECwwY
T3JnYW5pYyBGYXJtaW5nIERpdmlzaW9uMCAXDTIyMDMyMjEwMjkyOVoYDzIxMDAw
ODAyMDAwMDAwWjBBMQ0wCwYDVQQDDARDaGlhMQ0wCwYDVQQKDARDaGlhMSEwHwYD
VQQLDBhPcmdhbmljIEZhcm1pbmcgRGl2aXNpb24wggEiMA0GCSqGSIb3DQEBAQUA
A4IBDwAwggEKAoIBAQC6ghTGOkfeWwqPLfK1hMgIbY5HaowuUHRXkCyxCF0E994w
CVGnU+dCSJIy+xAqRDvmiQc4DyG4SSskTSCq1tv2xCDmdCb8BFjFVoKwtTpDjIkF
mZBRAWJWnuDxE1zUH1kwR9p/d44Ln+SJQV9/tHdLrn1exmAl7TZ3fMHFKyyXsded
qJ7K/9mlNAHs4Mz2hMZ6h93363bShVRP0clOzdJ0g0/OpXgRkRDSntDwATlBpsYU
p0mWrpyAdPIvADHBWlFZik2Ts/DLpdzPaHG0jUmQ3OFAOUNWBGvGTbkPeWsv3grk
4+m+MYbmoCg30l0ReeWy66J6OdjJMAiKuxoLfuRBAgMBAAGjFzAVMBMGA1UdEQQM
MAqCCGNoaWEubmV0MA0GCSqGSIb3DQEBCwUAA4IBAQCYEncQIRyINhibeDZaPwhF
exLV0IfnbLFMpICG36F/sK7b9RFAg1x3SrPk6SXe7odKfmEvjKivptKXnUJ1mojA
cufGQiMY9k+xiLzL/x9nTKMuT6hcGAwdpSZl9T1xQOalI67PVl9OH02Qdab1h1bR
Un5OISmD4D0vtFH4CYXjG//r/SpfbC7d5h7v9/h8rRhZPN/uGgezHE1s6IJdeRas
zRhieSXDW0aW3Y7agkgwdfrPbz4wpEYUpWHDM0dyRMpqww/aXgw32L7Iv2wYJroW
R8FESaGFlY+7aBORNlOqULtg6SEzCy+Bb2Z90iUA1A8UuuGQpIPvZPKfv68Hr/mI
-----END CERTIFICATE-----
"""

SSL_TEST_INTRODUCER_PUBLIC_KEY = b"""-----BEGIN RSA PRIVATE KEY-----
MIIEpgIBAAKCAQEAuoIUxjpH3lsKjy3ytYTICG2OR2qMLlB0V5AssQhdBPfeMAlR
p1PnQkiSMvsQKkQ75okHOA8huEkrJE0gqtbb9sQg5nQm/ARYxVaCsLU6Q4yJBZmQ
UQFiVp7g8RNc1B9ZMEfaf3eOC5/kiUFff7R3S659XsZgJe02d3zBxSssl7HXnaie
yv/ZpTQB7ODM9oTGeofd9+t20oVUT9HJTs3SdINPzqV4EZEQ0p7Q8AE5QabGFKdJ
lq6cgHTyLwAxwVpRWYpNk7Pwy6Xcz2hxtI1JkNzhQDlDVgRrxk25D3lrL94K5OPp
vjGG5qAoN9JdEXnlsuuiejnYyTAIirsaC37kQQIDAQABAoIBAQCHRA6jEePLzYWk
ADQWGB4nlqVpAEp40JsAYq03HBSypL59YlTZIY4I0a9O0C2SKfizo1AcBUV0bYSf
iFyX6sqPdwFRgo4/mztyq+KPHHEMSqGZ8Rzs0y8qGYKU9XEO20mJLaO3bNMVdTp/
nW0QpqcauYttsDxNZXRqRo4WXNCzluf5HEargy2A3/44MGc34vmwvbUYPk+2DpFL
glsVXrRyiB17pwUMw17/h40t0sSTsm2GlubfGzz6PWTtJ7tqSXLMeFDn3zLOd89N
GGWyi/d682xzXmBJ1KA+rEKFCoC3S+enIJEMedJU2LQ9TtyPVy//8ocPksijlcah
Rg/bInIdAoGBAN/dH/3bVaLg0C0lrEdHVa38QnjbBce6nUkzjPqTE5H6AlUDReEM
kKBWbWXT03MYrsoR5+SP6YAmzrdFnVQIVgeEOCtTn1MTP0O8vgSaS+koA1K3E12V
Hnn/Ca2Auqhmq9Lu+PMgTlrfHmHlvfljHTeM8XCsk1Z1RnfCTB0VAP47AoGBANVI
J/v0Y6LhFAsGSDuM6Bxrz64rFOXDpwGMyIi9Joxe7Lhlx957So/vLMb7UyJhnN/U
0Fo/xNtuP6X0YtuOtfW/hfM4F7Q4IX9RWWEf3o4M9ePgmy221RNBghiCW3o409y9
C9q9Ic0LWjFccRJYviitnU2I+vRTMv29mUGmIFOzAoGBAMVumJNjiGAw04itrAXl
FNxwGRPLXoqqu6Lekxw+8EiS7NGbJFr5oOe3sZ3XtqnQf/740cJTZcS+9RrPi5Sc
EXtpkbwscNxsoiQUZqVai05jGqMtkSe8pvdDBX7+hCCXBDzww6BwwooEMyB15Rj3
38GpTRppKTxcC7z9H/GaH5kXAoGBALyT8haKPVaUoD4nLdLJgakvgg9DNFT7Kzzp
l52fcOgIPlHEXHZQmNqu5O6C5TyvBh+6cQZ6/7nsvwYjEQ2EH0rMLsmgPQUx5BxI
dzizDvqWZws3Wr2OvNMrxrWY/P0SJfE0jbi4JZzAftzKGYfZQzmAWGdnyEwjDCDi
dRFgq5CVAoGBAKaxSX+aKLw+rGS2c+gJj3WpmAjiyrpd/T/rWr/Vb7WODIENTZNx
cV6qrEx/K3KLNWWt/RSGF1WcwMogfIQeFSCuYUFC1OTaFIIzHXHoJ5kfNDuc58dY
5KBvJhlub9s4tAwlsEFHtHiKoOwpqDQqrMVk5T74cZnSY/01aMJzya8w
-----END RSA PRIVATE KEY-----
"""

SSL_TEST_PRIVATE_CA_CERT_AND_KEY_6: Tuple[bytes, bytes] = (SSL_TEST_PRIVATE_CA_CRT, SSL_TEST_PRIVATE_CA_KEY)

SSL_TEST_NODE_CERTS_AND_KEYS_6: Dict[str, Dict[str, Dict[str, bytes]]] = {
    "full_node": {
        "private": {"crt": SSL_TEST_FULLNODE_PRIVATE_CRT, "key": SSL_TEST_FULLNODE_PRIVATE_KEY},
        "public": {"crt": SSL_TEST_FULLNODE_PUBLIC_CRT, "key": SSL_TEST_FULLNODE_PUBLIC_KEY},
    },
    "wallet": {
        "private": {"crt": SSL_TEST_WALLET_PRIVATE_CRT, "key": SSL_TEST_WALLET_PRIVATE_KEY},
        "public": {"crt": SSL_TEST_WALLET_PUBLIC_CRT, "key": SSL_TEST_WALLET_PUBLIC_KEY},
    },
    "farmer": {
        "private": {"crt": SSL_TEST_FARMER_PRIVATE_CRT, "key": SSL_TEST_FARMER_PRIVATE_KEY},
        "public": {"crt": SSL_TEST_FARMER_PUBLIC_CRT, "key": SSL_TEST_FARMER_PUBLIC_KEY},
    },
    "harvester": {
        "private": {"crt": SSL_TEST_HARVESTER_PRIVATE_CRT, "key": SSL_TEST_HARVESTER_PRIVATE_KEY},
    },
    "timelord": {
        "private": {"crt": SSL_TEST_TIMELORD_PRIVATE_CRT, "key": SSL_TEST_TIMELORD_PRIVATE_KEY},
        "public": {"crt": SSL_TEST_TIMELORD_PUBLIC_CRT, "key": SSL_TEST_TIMELORD_PUBLIC_KEY},
    },
    "crawler": {
        "private": {"crt": SSL_TEST_CRAWLER_PRIVATE_CRT, "key": SSL_TEST_CRAWLER_PRIVATE_KEY},
    },
    "daemon": {
        "private": {"crt": SSL_TEST_DAEMON_PRIVATE_CRT, "key": SSL_TEST_DAEMON_PRIVATE_KEY},
    },
    "introducer": {
        "public": {"crt": SSL_TEST_INTRODUCER_PUBLIC_CRT, "key": SSL_TEST_INTRODUCER_PUBLIC_KEY},
    },
}
