"""Tests for the NetworkHealthCollector using test helpers."""

from __future__ import annotations

from unittest.mock import MagicMock

from meraki_dashboard_exporter.collectors.network_health import NetworkHealthCollector
from meraki_dashboard_exporter.core.constants import UpdateTier
from tests.helpers.base import BaseCollectorTest
from tests.helpers.factories import DeviceFactory, NetworkFactory, OrganizationFactory


class TestNetworkHealthCollector(BaseCollectorTest):
    """Test NetworkHealthCollector functionality."""

    collector_class = NetworkHealthCollector
    update_tier = UpdateTier.MEDIUM

    async def test_collect_with_no_networks(self, collector, mock_api_builder, metrics):
        """Test collection when no networks exist."""
        # Set up test data
        org = OrganizationFactory.create(org_id="123", name="Test Org")

        # Configure mock API
        api = mock_api_builder.with_organizations([org]).with_networks([], org_id=org["id"]).build()
        collector.api = api

        # Run collection
        await self.run_collector(collector)

        # Verify success
        self.assert_collector_success(collector, metrics)
        self.assert_api_call_tracked(collector, metrics, "getOrganizations")

    async def test_collect_channel_utilization(self, collector, mock_api_builder, metrics):
        """Test collection of channel utilization metrics."""
        # Set up test data
        org = OrganizationFactory.create(org_id="123", name="Test Org")
        network = NetworkFactory.create(
            network_id="N_123",
            name="Test Network",
            product_types=["wireless"],
            org_id=org["id"],
        )
        devices = [
            DeviceFactory.create_mr(
                serial="Q2KD-XXXX",
                name="AP1",
                model="MR36",
                network_id=network["id"],
            ),
            DeviceFactory.create_mr(
                serial="Q2KD-YYYY",
                name="AP2",
                model="MR46",
                network_id=network["id"],
            ),
        ]

        channel_util_data = [
            {
                "serial": "Q2KD-XXXX",
                "model": "MR36",
                "wifi0": [  # 2.4GHz
                    {"utilization": 45, "wifi": 30, "nonWifi": 15}
                ],
                "wifi1": [  # 5GHz
                    {"utilization": 25, "wifi": 20, "nonWifi": 5}
                ],
            },
            {
                "serial": "Q2KD-YYYY",
                "model": "MR46",
                "wifi0": [{"utilization": 55, "wifi": 40, "nonWifi": 15}],
                "wifi1": [{"utilization": 35, "wifi": 30, "nonWifi": 5}],
            },
        ]

        # Configure mock API
        api = (
            mock_api_builder.with_organizations([org])
            .with_networks([network], org_id=org["id"])
            .with_devices(devices, org_id=org["id"])
            .with_custom_response("getNetworkNetworkHealthChannelUtilization", channel_util_data)
            .build()
        )

        # The RF health collector calls getOrganizationDevices with specific params
        # We need to ensure it returns the devices when called with networkIds filter
        api.organizations.getOrganizationDevices = MagicMock(return_value=devices)

        collector.api = api
        # Update sub-collectors' API references
        collector.rf_health_collector.api = api
        collector.connection_stats_collector.api = api
        collector.data_rates_collector.api = api
        collector.bluetooth_collector.api = api

        # Run collection
        await self.run_collector(collector)

        # Verify success
        self.assert_collector_success(collector, metrics)

        # Verify API calls were tracked
        self.assert_api_call_tracked(collector, metrics, "getOrganizationDevices")
        self.assert_api_call_tracked(
            collector, metrics, "getNetworkNetworkHealthChannelUtilization"
        )

        # Verify metrics were set
        metrics.assert_gauge_value(
            "meraki_ap_channel_utilization_2_4ghz_percent",
            45,
            serial="Q2KD-XXXX",
            name="AP1",
            model="MR36",
            network_id=network["id"],
            network_name=network["name"],
            type="total",  # Channel utilization metrics include a type label
        )

    async def test_collect_connection_stats(self, collector, mock_api_builder, metrics):
        """Test collection of wireless connection statistics."""
        # Set up test data
        org = OrganizationFactory.create(org_id="123", name="Test Org")
        network = NetworkFactory.create(
            network_id="N_123",
            name="Test Network",
            product_types=["wireless"],
            org_id=org["id"],
        )

        connection_stats_data = {
            "assoc": 95,
            "auth": 98,
            "dhcp": 92,
            "dns": 99,
            "success": 90,
        }

        # Configure mock API
        api = (
            mock_api_builder.with_organizations([org])
            .with_networks([network], org_id=org["id"])
            .with_devices([], org_id=org["id"])
            .with_custom_response("getNetworkNetworkHealthChannelUtilization", [])
            .with_custom_response("getNetworkWirelessConnectionStats", connection_stats_data)
            .build()
        )
        collector.api = api
        # Update sub-collectors' API references
        collector.connection_stats_collector.api = api

        # Run collection
        await self.run_collector(collector)

        # Verify success
        self.assert_collector_success(collector, metrics)
        self.assert_api_call_tracked(collector, metrics, "getNetworkWirelessConnectionStats")

        # Verify metrics were set
        metrics.assert_gauge_value(
            "meraki_network_wireless_connection_stats_total",
            95,
            stat_type="assoc",
            network_id=network["id"],
            network_name=network["name"],
        )

    async def test_collect_data_rates(self, collector, mock_api_builder, metrics):
        """Test collection of wireless data rate metrics."""
        # Set up test data
        org = OrganizationFactory.create(org_id="123", name="Test Org")
        network = NetworkFactory.create(
            network_id="N_123",
            name="Test Network",
            product_types=["wireless"],
            org_id=org["id"],
        )

        data_rate_history = [
            {
                "endTs": "2024-01-01T12:00:00Z",
                "downloadKbps": 25000,
                "uploadKbps": 10000,
            },
            {
                "endTs": "2024-01-01T11:55:00Z",
                "downloadKbps": 20000,
                "uploadKbps": 8000,
            },
        ]

        # Configure mock API
        api = (
            mock_api_builder.with_organizations([org])
            .with_networks([network], org_id=org["id"])
            .with_devices([], org_id=org["id"])
            .with_custom_response("getNetworkNetworkHealthChannelUtilization", [])
            .with_custom_response("getNetworkWirelessConnectionStats", {})
            .with_custom_response("getNetworkWirelessDataRateHistory", data_rate_history)
            .build()
        )
        collector.api = api
        # Update sub-collectors' API references
        collector.data_rates_collector.api = api

        # Run collection
        await self.run_collector(collector)

        # Verify success
        self.assert_collector_success(collector, metrics)
        self.assert_api_call_tracked(collector, metrics, "getNetworkWirelessDataRateHistory")

        # Verify metrics were set (should use most recent data point)
        metrics.assert_gauge_value(
            "meraki_network_wireless_download_kbps",
            25000,
            network_id=network["id"],
            network_name=network["name"],
        )

    async def test_collect_handles_empty_channel_util_data(
        self, collector, mock_api_builder, metrics
    ):
        """Test handling of empty channel utilization data."""
        # Set up test data
        org = OrganizationFactory.create(org_id="123", name="Test Org")
        network = NetworkFactory.create(
            network_id="N_123",
            name="Test Network",
            product_types=["wireless"],
            org_id=org["id"],
        )

        # Configure mock API with empty responses
        api = (
            mock_api_builder.with_organizations([org])
            .with_networks([network], org_id=org["id"])
            .with_devices([], org_id=org["id"])
            .with_custom_response("getNetworkNetworkHealthChannelUtilization", [])
            .with_custom_response("getNetworkWirelessConnectionStats", {})
            .with_custom_response("getNetworkWirelessDataRateHistory", [])
            .build()
        )
        collector.api = api

        # Run collection - should handle gracefully
        await self.run_collector(collector)

        # Verify success
        self.assert_collector_success(collector, metrics)

    async def test_collect_handles_api_errors(self, collector, mock_api_builder, metrics):
        """Test handling of API errors."""
        # Set up test data
        org = OrganizationFactory.create(org_id="123", name="Test Org")
        network = NetworkFactory.create(
            network_id="N_123",
            name="Test Network",
            product_types=["wireless"],
            org_id=org["id"],
        )

        # Configure mock API with errors
        api = (
            mock_api_builder.with_organizations([org])
            .with_networks([network], org_id=org["id"])
            .with_devices([], org_id=org["id"])
            .with_error("getNetworkNetworkHealthChannelUtilization", 400)
            .with_error("getNetworkWirelessConnectionStats", 404)
            .with_error("getNetworkWirelessDataRateHistory", 500)
            .build()
        )
        collector.api = api

        # Run collection - should handle errors gracefully
        await self.run_collector(collector)

        # Verify collector still marks as successful (error handling decorators)
        self.assert_collector_success(collector, metrics)

    async def test_collect_non_wireless_networks_skipped(
        self, collector, mock_api_builder, metrics
    ):
        """Test that non-wireless networks are skipped."""
        # Set up test data
        org = OrganizationFactory.create(org_id="123", name="Test Org")
        networks = [
            NetworkFactory.create(
                network_id="N_123",
                name="Switch Network",
                product_types=["switch"],  # Not wireless
                org_id=org["id"],
            ),
            NetworkFactory.create(
                network_id="N_456",
                name="Camera Network",
                product_types=["camera"],  # Not wireless
                org_id=org["id"],
            ),
        ]

        # Configure mock API
        api = (
            mock_api_builder.with_organizations([org])
            .with_networks(networks, org_id=org["id"])
            .build()
        )
        collector.api = api

        # Run collection
        await self.run_collector(collector)

        # Verify success
        self.assert_collector_success(collector, metrics)

        # Should not call wireless-specific APIs
        # We can't directly check that methods weren't called with the mock builder,
        # but we can verify that no wireless-specific API calls were tracked
        try:
            self.assert_api_call_tracked(
                collector, metrics, "getNetworkWirelessConnectionStats", count=0
            )
        except AssertionError:
            # Expected - the API call wasn't tracked
            pass

    def test_set_metric_value_handles_none(self, collector):
        """Test that None values are handled properly."""
        # This tests the _set_metric_value method directly
        labels = {"network_id": "N_123", "network_name": "Test"}

        # Should skip None values without error
        collector._set_metric_value("_network_utilization_2_4ghz", labels, None)

    def test_update_tier(self, collector):
        """Test that network health collector has correct update tier."""
        assert collector.update_tier == UpdateTier.MEDIUM
        assert self.update_tier == UpdateTier.MEDIUM

    async def test_collect_bluetooth_clients(self, collector, mock_api_builder, metrics):
        """Test collection of Bluetooth client metrics."""
        # Set up test data
        org = OrganizationFactory.create(org_id="123", name="Test Org")
        network = NetworkFactory.create(
            network_id="N_123",
            name="Test Network",
            product_types=["wireless"],
            org_id=org["id"],
        )

        bluetooth_data = [
            {
                "startTs": "2024-01-01T12:00:00Z",
                "endTs": "2024-01-01T12:05:00Z",
                "bluetoothDeviceCount": 42,
            }
        ]

        # Configure mock API
        api = (
            mock_api_builder.with_organizations([org])
            .with_networks([network], org_id=org["id"])
            .with_devices([], org_id=org["id"])
            .with_custom_response("getNetworkNetworkHealthChannelUtilization", [])
            .with_custom_response("getNetworkWirelessConnectionStats", {})
            .with_custom_response("getNetworkWirelessDataRateHistory", [])
            .with_custom_response("getNetworkWirelessBluetoothClients", bluetooth_data)
            .build()
        )

        # Manually configure the bluetooth clients API (if needed)
        api.wireless.getNetworkWirelessBluetoothClients = MagicMock(return_value=bluetooth_data)

        collector.api = api
        # Update sub-collectors' API references
        if hasattr(collector, "bluetooth_collector"):
            collector.bluetooth_collector.api = api

        # Run collection
        await self.run_collector(collector)

        # Verify success
        self.assert_collector_success(collector, metrics)
