"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const ssm = require("./ssm.generated");
const util_1 = require("./util");
/**
 * Basic features shared across all types of SSM Parameters.
 */
class ParameterBase extends core_1.Resource {
    grantRead(grantee) {
        if (this.encryptionKey) {
            this.encryptionKey.grantDecrypt(grantee);
        }
        return iam.Grant.addToPrincipal({
            grantee,
            actions: [
                'ssm:DescribeParameters',
                'ssm:GetParameters',
                'ssm:GetParameter',
                'ssm:GetParameterHistory',
            ],
            resourceArns: [this.parameterArn],
        });
    }
    grantWrite(grantee) {
        if (this.encryptionKey) {
            this.encryptionKey.grantEncrypt(grantee);
        }
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['ssm:PutParameter'],
            resourceArns: [this.parameterArn],
        });
    }
}
/**
 * SSM parameter type
 */
var ParameterType;
(function (ParameterType) {
    /**
     * String
     */
    ParameterType["STRING"] = "String";
    /**
     * Secure String
     * Parameter Store uses an AWS Key Management Service (KMS) customer master key (CMK) to encrypt the parameter value.
     */
    ParameterType["SECURE_STRING"] = "SecureString";
    /**
     * String List
     */
    ParameterType["STRING_LIST"] = "StringList";
    /**
     * An Amazon EC2 image ID, such as ami-0ff8a91507f77f867
     */
    ParameterType["AWS_EC2_IMAGE_ID"] = "AWS::EC2::Image::Id";
})(ParameterType = exports.ParameterType || (exports.ParameterType = {}));
/**
 * SSM parameter tier
 */
var ParameterTier;
(function (ParameterTier) {
    /**
     * String
     */
    ParameterTier["ADVANCED"] = "Advanced";
    /**
     * String
     */
    ParameterTier["INTELLIGENT_TIERING"] = "Intelligent-Tiering";
    /**
     * String
     */
    ParameterTier["STANDARD"] = "Standard";
})(ParameterTier = exports.ParameterTier || (exports.ParameterTier = {}));
/**
 * Creates a new String SSM Parameter.
 * @resource AWS::SSM::Parameter
 */
class StringParameter extends ParameterBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, {
            physicalName: props.parameterName,
        });
        if (props.allowedPattern) {
            _assertValidValue(props.stringValue, props.allowedPattern);
        }
        if (this.physicalName.length > 2048) {
            throw new Error('Name cannot be longer than 2048 characters.');
        }
        if (props.description && ((_a = props.description) === null || _a === void 0 ? void 0 : _a.length) > 1024) {
            throw new Error('Description cannot be longer than 1024 characters.');
        }
        const resource = new ssm.CfnParameter(this, 'Resource', {
            allowedPattern: props.allowedPattern,
            description: props.description,
            name: this.physicalName,
            tier: props.tier,
            type: props.type || ParameterType.STRING,
            value: props.stringValue,
        });
        this.parameterName = this.getResourceNameAttribute(resource.ref);
        this.parameterArn = util_1.arnForParameterName(this, this.parameterName, {
            physicalName: props.parameterName || util_1.AUTOGEN_MARKER,
            simpleName: props.simpleName,
        });
        this.parameterType = resource.attrType;
        this.stringValue = resource.attrValue;
    }
    /**
     * Imports an external string parameter by name.
     */
    static fromStringParameterName(scope, id, stringParameterName) {
        return this.fromStringParameterAttributes(scope, id, { parameterName: stringParameterName });
    }
    /**
     * Imports an external string parameter with name and optional version.
     */
    static fromStringParameterAttributes(scope, id, attrs) {
        if (!attrs.parameterName) {
            throw new Error('parameterName cannot be an empty string');
        }
        const type = attrs.type || ParameterType.STRING;
        const stringValue = attrs.version
            ? new core_1.CfnDynamicReference(core_1.CfnDynamicReferenceService.SSM, `${attrs.parameterName}:${attrs.version}`).toString()
            : new core_1.CfnParameter(scope, `${id}.Parameter`, { type: `AWS::SSM::Parameter::Value<${type}>`, default: attrs.parameterName }).valueAsString;
        class Import extends ParameterBase {
            constructor() {
                super(...arguments);
                this.parameterName = attrs.parameterName;
                this.parameterArn = util_1.arnForParameterName(this, attrs.parameterName, { simpleName: attrs.simpleName });
                this.parameterType = type;
                this.stringValue = stringValue;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Imports a secure string parameter from the SSM parameter store.
     */
    static fromSecureStringParameterAttributes(scope, id, attrs) {
        const stringValue = new core_1.CfnDynamicReference(core_1.CfnDynamicReferenceService.SSM_SECURE, `${attrs.parameterName}:${attrs.version}`).toString();
        class Import extends ParameterBase {
            constructor() {
                super(...arguments);
                this.parameterName = attrs.parameterName;
                this.parameterArn = util_1.arnForParameterName(this, attrs.parameterName, { simpleName: attrs.simpleName });
                this.parameterType = ParameterType.SECURE_STRING;
                this.stringValue = stringValue;
                this.encryptionKey = attrs.encryptionKey;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Reads the value of an SSM parameter during synthesis through an
     * environmental context provider.
     *
     * Requires that the stack this scope is defined in will have explicit
     * account/region information. Otherwise, it will fail during synthesis.
     */
    static valueFromLookup(scope, parameterName) {
        const value = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.SSM_PARAMETER_PROVIDER,
            props: { parameterName },
            dummyValue: `dummy-value-for-${parameterName}`,
        }).value;
        return value;
    }
    /**
     * Returns a token that will resolve (during deployment) to the string value of an SSM string parameter.
     * @param scope Some scope within a stack
     * @param parameterName The name of the SSM parameter.
     * @param version The parameter version (recommended in order to ensure that the value won't change during deployment)
     */
    static valueForStringParameter(scope, parameterName, version) {
        return StringParameter.valueForTypedStringParameter(scope, parameterName, ParameterType.STRING, version);
    }
    /**
     * Returns a token that will resolve (during deployment) to the string value of an SSM string parameter.
     * @param scope Some scope within a stack
     * @param parameterName The name of the SSM parameter.
     * @param type The type of the SSM parameter.
     * @param version The parameter version (recommended in order to ensure that the value won't change during deployment)
     */
    static valueForTypedStringParameter(scope, parameterName, type = ParameterType.STRING, version) {
        const stack = core_1.Stack.of(scope);
        const id = makeIdentityForImportedValue(parameterName);
        const exists = stack.node.tryFindChild(id);
        if (exists) {
            return exists.stringValue;
        }
        return this.fromStringParameterAttributes(stack, id, { parameterName, version, type }).stringValue;
    }
    /**
     * Returns a token that will resolve (during deployment)
     * @param scope Some scope within a stack
     * @param parameterName The name of the SSM parameter
     * @param version The parameter version (required for secure strings)
     */
    static valueForSecureStringParameter(scope, parameterName, version) {
        const stack = core_1.Stack.of(scope);
        const id = makeIdentityForImportedValue(parameterName);
        const exists = stack.node.tryFindChild(id);
        if (exists) {
            return exists.stringValue;
        }
        return this.fromSecureStringParameterAttributes(stack, id, { parameterName, version }).stringValue;
    }
}
exports.StringParameter = StringParameter;
/**
 * Creates a new StringList SSM Parameter.
 * @resource AWS::SSM::Parameter
 */
class StringListParameter extends ParameterBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, {
            physicalName: props.parameterName,
        });
        if (props.stringListValue.find(str => !core_1.Token.isUnresolved(str) && str.indexOf(',') !== -1)) {
            throw new Error('Values of a StringList SSM Parameter cannot contain the \',\' character. Use a string parameter instead.');
        }
        if (props.allowedPattern && !core_1.Token.isUnresolved(props.stringListValue)) {
            props.stringListValue.forEach(str => _assertValidValue(str, props.allowedPattern));
        }
        if (this.physicalName.length > 2048) {
            throw new Error('Name cannot be longer than 2048 characters.');
        }
        if (props.description && ((_a = props.description) === null || _a === void 0 ? void 0 : _a.length) > 1024) {
            throw new Error('Description cannot be longer than 1024 characters.');
        }
        const resource = new ssm.CfnParameter(this, 'Resource', {
            allowedPattern: props.allowedPattern,
            description: props.description,
            name: this.physicalName,
            tier: props.tier,
            type: ParameterType.STRING_LIST,
            value: props.stringListValue.join(','),
        });
        this.parameterName = this.getResourceNameAttribute(resource.ref);
        this.parameterArn = util_1.arnForParameterName(this, this.parameterName, {
            physicalName: props.parameterName || util_1.AUTOGEN_MARKER,
            simpleName: props.simpleName,
        });
        this.parameterType = resource.attrType;
        this.stringListValue = core_1.Fn.split(',', resource.attrValue);
    }
    /**
     * Imports an external parameter of type string list.
     * Returns a token and should not be parsed.
     */
    static fromStringListParameterName(scope, id, stringListParameterName) {
        class Import extends ParameterBase {
            constructor() {
                super(...arguments);
                this.parameterName = stringListParameterName;
                this.parameterArn = util_1.arnForParameterName(this, this.parameterName);
                this.parameterType = ParameterType.STRING_LIST;
                this.stringListValue = core_1.Fn.split(',', new core_1.CfnDynamicReference(core_1.CfnDynamicReferenceService.SSM, stringListParameterName).toString());
            }
        }
        return new Import(scope, id);
    }
}
exports.StringListParameter = StringListParameter;
/**
 * Validates whether a supplied value conforms to the allowedPattern, granted neither is an unresolved token.
 *
 * @param value          the value to be validated.
 * @param allowedPattern the regular expression to use for validation.
 *
 * @throws if the ``value`` does not conform to the ``allowedPattern`` and neither is an unresolved token (per
 *         ``cdk.unresolved``).
 */
function _assertValidValue(value, allowedPattern) {
    if (core_1.Token.isUnresolved(value) || core_1.Token.isUnresolved(allowedPattern)) {
        // Unable to perform validations against unresolved tokens
        return;
    }
    if (!new RegExp(allowedPattern).test(value)) {
        throw new Error(`The supplied value (${value}) does not match the specified allowedPattern (${allowedPattern})`);
    }
}
function makeIdentityForImportedValue(parameterName) {
    return `SsmParameterValue:${parameterName}:C96584B6-F00A-464E-AD19-53AFF4B05118`;
}
//# sourceMappingURL=data:application/json;base64,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