#  -----------------------------------------------------------------------------------------
#  (C) Copyright IBM Corp. 2025.
#  https://opensource.org/licenses/BSD-3-Clause
#  -----------------------------------------------------------------------------------------
from __future__ import annotations

from datetime import datetime, timedelta
from typing import Callable, TYPE_CHECKING

from ibm_watsonx_ai.utils.auth.base_auth import RefreshableTokenAuth, TokenInfo
from ibm_watsonx_ai.wml_client_error import (
    CannotAutogenerateBedrockUrl,
    InvalidCredentialsError,
    WMLClientError,
)

if TYPE_CHECKING:
    from ibm_watsonx_ai import APIClient


class ICPAuth(RefreshableTokenAuth):
    """ICP token authentication method class.

    :param api_client: initialized APIClient object with set project or space ID
    :type api_client: APIClient

    :param on_token_refresh: callback which allows to notify about token refresh
    :type on_token_refresh: function which takes no params and returns nothing
    """

    def __init__(
        self, api_client: APIClient, on_token_refresh: Callable[[], None]
    ) -> None:
        RefreshableTokenAuth.__init__(
            self, api_client, on_token_refresh, timedelta(minutes=50)
        )
        self._has_attr_is_bedrock_url_autogenerated = hasattr(
            api_client, "_is_bedrock_url_autogenerated"
        )
        self._is_bedrock_url_autogenerated = (
            getattr(api_client, "_is_bedrock_url_autogenerated")
            if self._has_attr_is_bedrock_url_autogenerated
            else None
        )

        if api_client.CLOUD_PLATFORM_SPACES:
            raise WMLClientError("Authentication method supported only on ICP.")

        if self._credentials.username is None:
            raise WMLClientError("`username` missing in credentials.")

        if self._credentials.api_key is None and self._credentials.password is None:
            raise WMLClientError(
                "Api key or password needs to be provided in the credentials."
            )

    def _generate_token(self) -> TokenInfo:
        """Generate token for ICP.

        :returns: token info to be used by auth method
        :rtype: TokenInfo
        """
        if (
            self._credentials.bedrock_url is not None
            and self._credentials.password is not None
        ):
            try:
                return self._get_cpd_token_from_request_new_auth_flow()
            except Exception as e1:
                if not self._has_attr_is_bedrock_url_autogenerated:
                    raise e1

                try:
                    token_info = self._get_cpd_token_from_request_old_auth_flow()
                    # if it worked when iamintegration=False, then removing bedrock_url will shorten the path
                    self._credentials.bedrock_url = None
                    return token_info
                except Exception as e2:
                    if (
                        self._has_attr_is_bedrock_url_autogenerated
                        and self._is_bedrock_url_autogenerated
                    ):
                        raise CannotAutogenerateBedrockUrl(e1, e2)
                    else:
                        raise e2
        else:
            return self._get_cpd_token_from_request_old_auth_flow()

    def _get_cpd_token_from_request_old_auth_flow(self) -> TokenInfo:
        token_url = self._href_definitions.get_cpd_token_endpoint_href()
        response = self._session.post(
            token_url,
            headers={"Content-Type": "application/json"},
            data=self._get_cpd_auth_pair(),
        )

        if response.status_code == 200:
            return TokenInfo(response.json().get("token"))
        else:
            raise InvalidCredentialsError(reason=response.text)

    def _get_cpd_token_from_request_new_auth_flow(self) -> TokenInfo:
        bedrock_url = self._href_definitions.get_cpd_bedrock_token_endpoint_href()
        response = self._session.post(
            bedrock_url,
            headers={"Content-Type": "application/x-www-form-urlencoded;charset=UTF-8"},
            data=self._get_cpd_bedrock_auth_data(),
        )

        if response.status_code != 200:
            raise InvalidCredentialsError(reason=response.text, logg_messages=False)

        iam_token = response.json()["access_token"]
        expiration_datetime = datetime.now() + timedelta(
            seconds=response.json()["expires_in"]
        )
        # refresh_token = response.json()['refresh_token']

        token_url = self._href_definitions.get_cpd_validation_token_endpoint_href()
        response = self._session.get(
            token_url,
            headers={
                "username": self._credentials.username,
                "iam-token": iam_token,
            },
        )

        if response.status_code == 200:
            return TokenInfo(response.json()["accessToken"], expiration_datetime)
        else:
            raise InvalidCredentialsError(reason=response.text)

    def _get_cpd_auth_pair(self) -> str:
        """Get a pair of credentials required for the token generation.

        :return: string representing a dictionary of authentication credentials
                         (username & password) or (username & api_key).
        :rtype: str
        """
        if self._credentials.api_key is not None:
            return f'{{"username": "{self._credentials.username}", "api_key": "{self._credentials.api_key}"}}'
        else:
            return f'{{"username": "{self._credentials.username}", "password": "{self._credentials.password}"}}'

    def _get_cpd_bedrock_auth_data(self) -> str:
        """Get the data required for the token generation.

        :return: string representing a dictionary of authentication credentials
        :rtype: str
        """
        return f"grant_type=password&username={self._credentials.username}&password={self._credentials.password}&scope=openid"
