import logging

from django.contrib.postgres.fields import ArrayField
from django.db.models import DecimalField, BigIntegerField, TextField, BooleanField, DateTimeField, PositiveIntegerField

from isc_common.auth.models.user import User
from isc_common.common import blinkString
from isc_common.datetime import DateTimeToStr
from isc_common.fields.code_field import CodeField
from isc_common.fields.related import ForeignKeyProtect, ForeignKeyCascade
from isc_common.managers.common_manager import CommonManager
from isc_common.models.base_ref import Hierarcy
from isc_common.number import ToDecimal, DecimalToStr, Set
from kaf_pas.ckk.models.item import Item
from kaf_pas.planing.models.operation_types import Operation_types
from kaf_pas.planing.models.production_order import Production_orderQuerySet
from kaf_pas.production.models.launches import Launches

logger = logging.getLogger(__name__)


class Production_order_per_launchManager(CommonManager):
    @staticmethod
    def getRecord(record):
        value_sum = ToDecimal(record.value_sum)
        if value_sum != 0:
            percents = round(ToDecimal(record.value_made) * 100 / ToDecimal(record.value_sum), 2)
        else:
            percents = 0
        percents_str = "%.2f" % percents

        res = {
            'creator__short_name': record.creator.get_short_name,
            # 'date': record.date,
            'description': record.description,
            'edizm__name': ' / '.join(record.edizm_arr) if record.edizm_arr != None else None,
            'id': record.id,
            'item_id': record.item.id,
            'parent_item_id': record.parent_item.id if record.parent_item else None,
            'item__STMP_1__value_str': record.item.STMP_1.value_str if record.item.STMP_1 else None,
            'item__STMP_2__value_str': record.item.STMP_2.value_str if record.item.STMP_2 else None,
            'launch_id': record.launch.id,
            'launch__code': record.launch.code,
            'launch__date': record.launch.date,
            'location_sector_ids': Set(record.location_sector_ids).get_set_sorted_as_original,
            'location_sectors_full_name': '<br>'.join(Set(record.location_sectors_full_name).get_set_sorted_as_original),
            'num': record.num,
            'isFolder': record.isFolder,
            'cnt_opers': record.cnt_opers,
            'value_sum': DecimalToStr(record.value_sum),
            'value1_sum': DecimalToStr(record.value1_sum),
            # 'value1_sum_len': len(record.value1_sum) if record.value1_sum != None else None,
            'value_made': DecimalToStr(record.value_made),
            'value_made_str': f'''{blinkString(DecimalToStr(record.value_made), blink=True if percents >= 100 else False, color="blue", bold=True)}({percents_str}%)''',
            'value_start': DecimalToStr(record.value_start),
            'value_odd': DecimalToStr(record.value_odd),
            'opertype__full_name': record.opertype.full_name,
            'opertype_id': record.opertype.id,
            'parent_id': record.parent.id if record.parent else None,
            'status__code': record.status.code,
            'status__name': blinkString(text=record.status.name, blink=False, color=record.status.color, bold=False),
            'status_id': record.status.id,
            'isDeleted': record.isDeleted,
        }
        return res

    def get_queryset(self):
        return Production_orderQuerySet(self.model, using=self._db)


class Production_order_per_launch(Hierarcy):
    from kaf_pas.planing.models.status_operation_types import Status_operation_types
    from kaf_pas.planing.models.operation_refs import Operation_refsManager

    date = DateTimeField(default=None)
    isFolder = BooleanField(default=None)
    num = CodeField()

    description = TextField(null=True, blank=True)

    opertype = ForeignKeyProtect(Operation_types, related_name='Production_order_per_launch_opertype')
    creator = ForeignKeyProtect(User, related_name='Production_order_per_launch_creator')
    exucutors = ArrayField(BigIntegerField(), default=list)
    location_sector_ids = ArrayField(BigIntegerField(), default=list)
    location_sectors_full_name = ArrayField(TextField(), default=list)
    arranges_exucutors = ArrayField(BigIntegerField(), default=list)
    status = ForeignKeyProtect(Status_operation_types)
    launch = ForeignKeyCascade(Launches)
    location_ids = ArrayField(BigIntegerField(), default=list)
    edizm_arr = ArrayField(CodeField(null=True, blank=True))
    cnt_opers = PositiveIntegerField()
    isDeleted = BooleanField()

    item = ForeignKeyProtect(Item, related_name='Production_order_per_launch_item')
    parent_item = ForeignKeyProtect(Item, null=True, blank=True, related_name='Production_order_per_launch_parent_item')

    value_sum = DecimalField(decimal_places=4, max_digits=19)
    value1_sum = DecimalField(decimal_places=4, max_digits=19)
    value_start = DecimalField(decimal_places=4, max_digits=19, null=True, blank=True)
    value_made = DecimalField(decimal_places=4, max_digits=19, null=True, blank=True)
    value_odd = DecimalField(decimal_places=4, max_digits=19)

    props = Operation_refsManager.props()

    objects = Production_order_per_launchManager()

    started_partly = Status_operation_types.objects.get(code='started_partly')
    started = Status_operation_types.objects.get(code='started')

    def __str__(self):
        return f'id: {self.id}, ' \
               f'date: {DateTimeToStr(self.date)}, ' \
               f'num: {self.num}, ' \
               f'description: {self.description}, ' \
               f'opertype: [{self.opertype}], ' \
               f'creator: [{self.creator}], ' \
               f'exucutors: [{self.exucutors}], ' \
               f'status: [{self.status}], ' \
               f'launch: [{self.launch}], ' \
               f'edizm: [{self.edizm_arr}], ' \
               f'item: [{self.item}], ' \
               f'parent_item: [{self.parent_item}], ' \
               f'cnt_opers: {self.cnt_opers}, ' \
               f'value_sum: {self.value_sum},' \
               f'value1_sum: {self.value1_sum},' \
               f'value_start: {self.value_start},' \
               f'value_made: {self.value_made},' \
               f'value_odd: {self.value_odd}'

    def __repr__(self):
        return self.__str__()

    class Meta:
        verbose_name = 'Заказы на производство'
        managed = False
        # db_table = 'planing_production_order_per_launch_view'
        db_table = 'planing_production_order_per_launch_tbl'
