"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Schedule = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Schedule for scheduled scaling actions.
 *
 * @stability stable
 */
class Schedule {
    /**
     * @stability stable
     */
    constructor() {
    }
    /**
     * Construct a schedule from a literal schedule expression.
     *
     * @param expression The expression to use.
     * @stability stable
     */
    static expression(expression) {
        return new LiteralSchedule(expression);
    }
    /**
     * Construct a schedule from an interval and a time unit.
     *
     * @stability stable
     */
    static rate(duration) {
        if (duration.isUnresolved()) {
            const validDurationUnit = ['minute', 'minutes', 'hour', 'hours', 'day', 'days'];
            if (!validDurationUnit.includes(duration.unitLabel())) {
                throw new Error("Allowed units for scheduling are: 'minute', 'minutes', 'hour', 'hours', 'day' or 'days'");
            }
            return new LiteralSchedule(`rate(${duration.formatTokenToNumber()})`);
        }
        if (duration.toSeconds() === 0) {
            throw new Error('Duration cannot be 0');
        }
        let rate = maybeRate(duration.toDays({ integral: false }), 'day');
        if (rate === undefined) {
            rate = maybeRate(duration.toHours({ integral: false }), 'hour');
        }
        if (rate === undefined) {
            rate = makeRate(duration.toMinutes({ integral: true }), 'minute');
        }
        return new LiteralSchedule(rate);
    }
    /**
     * Construct a Schedule from a moment in time.
     *
     * @stability stable
     */
    static at(moment) {
        return new LiteralSchedule(`at(${formatISO(moment)})`);
    }
    /**
     * Create a schedule from a set of cron fields.
     *
     * @stability stable
     */
    static cron(options) {
        if (options.weekDay !== undefined && options.day !== undefined) {
            throw new Error('Cannot supply both \'day\' and \'weekDay\', use at most one');
        }
        const minute = fallback(options.minute, '*');
        const hour = fallback(options.hour, '*');
        const month = fallback(options.month, '*');
        const year = fallback(options.year, '*');
        // Weekday defaults to '?' if not supplied. If it is supplied, day must become '?'
        const day = fallback(options.day, options.weekDay !== undefined ? '?' : '*');
        const weekDay = fallback(options.weekDay, '?');
        return new LiteralSchedule(`cron(${minute} ${hour} ${day} ${month} ${weekDay} ${year})`);
    }
}
exports.Schedule = Schedule;
_a = JSII_RTTI_SYMBOL_1;
Schedule[_a] = { fqn: "@aws-cdk/aws-applicationautoscaling.Schedule", version: "1.104.0" };
class LiteralSchedule extends Schedule {
    constructor(expressionString) {
        super();
        this.expressionString = expressionString;
    }
}
function fallback(x, def) {
    return x === undefined ? def : x;
}
function formatISO(date) {
    if (!date) {
        return undefined;
    }
    return date.getUTCFullYear() +
        '-' + pad(date.getUTCMonth() + 1) +
        '-' + pad(date.getUTCDate()) +
        'T' + pad(date.getUTCHours()) +
        ':' + pad(date.getUTCMinutes()) +
        ':' + pad(date.getUTCSeconds());
    function pad(num) {
        if (num < 10) {
            return '0' + num;
        }
        return num;
    }
}
/**
 * Return the rate if the rate is whole number
 */
function maybeRate(interval, singular) {
    if (interval === 0 || !Number.isInteger(interval)) {
        return undefined;
    }
    return makeRate(interval, singular);
}
/**
 * Return 'rate(${interval} ${singular}(s))` for the interval
 */
function makeRate(interval, singular) {
    return interval === 1 ? `rate(1 ${singular})` : `rate(${interval} ${singular}s)`;
}
//# sourceMappingURL=data:application/json;base64,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