"""
    Node basis class

    Usage:

    >>> from poly_escape_graph import Node
    >>> class MyNode(Node):
            def listen(self):
                return "It's my node"
    >>> MyNode().run(port=5000)
"""

import sys
from typing import List

import requests
from flask import Flask, request, jsonify
from flask_sqlalchemy import SQLAlchemy


__all__ = ['Node']

OK = 'OK'
POST = 'POST'
TAGS = 'tags'
SUBSCRIBER = 'subscriber'
SQLALCHEMY_DATABASE_URI = 'SQLALCHEMY_DATABASE_URI'
SQLALCHEMY_TRACK_MODIFICATIONS = 'SQLALCHEMY_TRACK_MODIFICATIONS'


class Node(Flask):
    """
        Base class for Plugins.
        Define the listen, subscribe and unsubscribe routes.
        The listen route may be overwritten by the subclasses.
        The method publish is here to send a message to the instance subscribers, it must not be overwritten.
        A subclass can add another route by using the method
            self.add_url_rule('/listen', 'listen', self.listen, methods=[POST])
        This method will, however, not be public and therefore, is only for internal uses
    """

    def __init__(self, name=None, **kwargs):
        super().__init__(name or __name__)
        self._config(**kwargs)
        self._create_db()
        self.add_url_rule('/listen', 'listen', self.listen, methods=[POST])
        self.add_url_rule('/subscribe', 'subscribe', self.subscribe, methods=[POST])
        self.add_url_rule('/unsubscribe', 'unsubscribe', self.unsubscribe, methods=[POST])

    def _config(self, **kwargs):
        database_uri = f'sqlite:///{sys.argv[0][:-3]}.node.db'
        kwargs.setdefault(SQLALCHEMY_DATABASE_URI, database_uri)
        kwargs.setdefault(SQLALCHEMY_TRACK_MODIFICATIONS, False)
        self.config.update(**kwargs)

    def _create_db(self):
        self.db = SQLAlchemy(self)
        self.plugin = self._create_model('Plugin', self.db.Model,
                                         id=self.db.Column(self.db.Integer, primary_key=True),
                                         url=self.db.Column(self.db.String(200)),
                                         tag=self.db.Column(self.db.String(50)))
        self.db.create_all()

    def _create_model(self, name, *bases, **attributes):
        return type(name, bases, attributes)

    def listen(self):
        """
            Overwrite this
            The data of the POST http request can be found in the property request.json
        :return: {"status": "OK"}
        """
        return jsonify(status=OK)

    def publish(self, tags: List[str], files=None, **data):
        """
            This method send a message to all the subscribers
            This method may not be overwritten.
        :param files: files: (optional) Dictionary of couple {'name': file-tuple}
        :param tags: the list of tags used in the message
        :param data: a dict with the data for the subscribers
        :return: None
        """
        subscribers = set(plugin for tag in tags for plugin in self.db.session.query(self.plugin).filter_by(tag=tag))
        for subscriber in subscribers:
            requests.post(f'{subscriber.url}/listen', json={"tags": tags, "data": data}, files=files)

    def subscribe(self):
        """
            This method subscribe the applicant to this service
            This method may not be overwritten.
        :return: {"status": "OK"}
        """
        data = request.json
        for tag in data[TAGS]:
            plugin = self.plugin(url=data[SUBSCRIBER], tag=tag)
            self.db.session.add(plugin)
        self.db.session.commit()
        return jsonify(status=OK)

    def unsubscribe(self):
        """
            This method unsubscribe the applicant from this service
            This method may not be overwritten.
        :return: {"status": "OK"}
        """
        data = request.json
        for tag in data[TAGS]:
            plugin = self.db.session.query(self.plugin).filter_by(url=data[SUBSCRIBER], tag=tag).first()
            if plugin is not None:
                self.db.session.delete(plugin)
        self.db.session.commit()
        return jsonify(status=OK)


if __name__ == '__main__':
    app = Node()
    app.run(port=5000)
