"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const have_resource_1 = require("@aws-cdk/assert/lib/assertions/have-resource");
const ec2 = require("@aws-cdk/aws-ec2");
const ecr = require("@aws-cdk/aws-ecr");
const ecs = require("@aws-cdk/aws-ecs");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const batch = require("../lib");
describe('Batch Job Definition', () => {
    let stack;
    let jobDefProps;
    beforeEach(() => {
        stack = new cdk.Stack();
        const role = new iam.Role(stack, 'job-role', {
            assumedBy: new iam.ServicePrincipal('batch.amazonaws.com'),
        });
        const linuxParams = new ecs.LinuxParameters(stack, 'job-linux-params', {
            initProcessEnabled: true,
            sharedMemorySize: 1,
        });
        jobDefProps = {
            jobDefinitionName: 'test-job',
            container: {
                command: ['echo "Hello World"'],
                environment: {
                    foo: 'bar',
                },
                jobRole: role,
                gpuCount: 1,
                image: ecs.EcrImage.fromRegistry('docker/whalesay'),
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.T2, ec2.InstanceSize.MICRO),
                linuxParams,
                memoryLimitMiB: 1,
                mountPoints: new Array(),
                privileged: true,
                readOnly: true,
                ulimits: new Array(),
                user: 'root',
                vcpus: 2,
                volumes: new Array(),
            },
            nodeProps: {
                count: 2,
                mainNode: 1,
                rangeProps: new Array(),
            },
            parameters: {
                foo: 'bar',
            },
            retryAttempts: 2,
            timeout: cdk.Duration.seconds(30),
        };
    });
    test('renders the correct cloudformation properties', () => {
        // WHEN
        new batch.JobDefinition(stack, 'job-def', jobDefProps);
        // THEN
        expect(stack).toHaveResourceLike('AWS::Batch::JobDefinition', {
            JobDefinitionName: jobDefProps.jobDefinitionName,
            ContainerProperties: jobDefProps.container ? {
                Command: jobDefProps.container.command,
                Environment: [
                    {
                        Name: 'foo',
                        Value: 'bar',
                    },
                ],
                InstanceType: jobDefProps.container.instanceType ? jobDefProps.container.instanceType.toString() : '',
                LinuxParameters: {},
                Memory: jobDefProps.container.memoryLimitMiB,
                MountPoints: [],
                Privileged: jobDefProps.container.privileged,
                ReadonlyRootFilesystem: jobDefProps.container.readOnly,
                ResourceRequirements: [{ Type: 'GPU', Value: String(jobDefProps.container.gpuCount) }],
                Ulimits: [],
                User: jobDefProps.container.user,
                Vcpus: jobDefProps.container.vcpus,
                Volumes: [],
            } : undefined,
            NodeProperties: jobDefProps.nodeProps ? {
                MainNode: jobDefProps.nodeProps.mainNode,
                NodeRangeProperties: [],
                NumNodes: jobDefProps.nodeProps.count,
            } : undefined,
            Parameters: {
                foo: 'bar',
            },
            RetryStrategy: {
                Attempts: jobDefProps.retryAttempts,
            },
            Timeout: {
                AttemptDurationSeconds: jobDefProps.timeout ? jobDefProps.timeout.toSeconds() : -1,
            },
            Type: 'container',
        }, have_resource_1.ResourcePart.Properties);
    });
    test('can use an ecr image', () => {
        // WHEN
        const repo = new ecr.Repository(stack, 'image-repo');
        new batch.JobDefinition(stack, 'job-def', {
            container: {
                image: ecs.ContainerImage.fromEcrRepository(repo),
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Batch::JobDefinition', {
            ContainerProperties: {
                Image: {
                    'Fn::Join': [
                        '',
                        [
                            {
                                'Fn::Select': [
                                    4,
                                    {
                                        'Fn::Split': [
                                            ':',
                                            {
                                                'Fn::GetAtt': [
                                                    'imagerepoD116FAF0',
                                                    'Arn',
                                                ],
                                            },
                                        ],
                                    },
                                ],
                            },
                            '.dkr.ecr.',
                            {
                                'Fn::Select': [
                                    3,
                                    {
                                        'Fn::Split': [
                                            ':',
                                            {
                                                'Fn::GetAtt': [
                                                    'imagerepoD116FAF0',
                                                    'Arn',
                                                ],
                                            },
                                        ],
                                    },
                                ],
                            },
                            '.',
                            {
                                Ref: 'AWS::URLSuffix',
                            },
                            '/',
                            {
                                Ref: 'imagerepoD116FAF0',
                            },
                            ':latest',
                        ],
                    ],
                },
                Memory: 4,
                Privileged: false,
                ReadonlyRootFilesystem: false,
                Vcpus: 1,
            },
        }, have_resource_1.ResourcePart.Properties);
    });
    test('can use a registry image', () => {
        // WHEN
        new batch.JobDefinition(stack, 'job-def', {
            container: {
                image: ecs.ContainerImage.fromRegistry('docker/whalesay'),
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Batch::JobDefinition', {
            ContainerProperties: {
                Image: 'docker/whalesay',
                Memory: 4,
                Privileged: false,
                ReadonlyRootFilesystem: false,
                Vcpus: 1,
            },
        }, have_resource_1.ResourcePart.Properties);
    });
    test('can be imported from an ARN', () => {
        // WHEN
        const importedJob = batch.JobDefinition.fromJobDefinitionArn(stack, 'job-def-clone', 'arn:aws:batch:us-east-1:123456789012:job-definition/job-def-name:1');
        // THEN
        expect(importedJob.jobDefinitionName).toEqual('job-def-name:1');
        expect(importedJob.jobDefinitionArn).toEqual('arn:aws:batch:us-east-1:123456789012:job-definition/job-def-name:1');
    });
    test('can be imported from a name', () => {
        // WHEN
        const importedJob = batch.JobDefinition.fromJobDefinitionName(stack, 'job-def-clone', 'job-def-name');
        // THEN
        expect(importedJob.jobDefinitionName).toEqual('job-def-name');
        expect(importedJob.jobDefinitionArn)
            .toEqual('arn:${Token[AWS.Partition.3]}:batch:${Token[AWS.Region.4]}:${Token[AWS.AccountId.0]}:job-definition/job-def-name');
    });
});
//# sourceMappingURL=data:application/json;base64,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