"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const assert_2 = require("assert");
const batch = require("../lib");
describe('Batch Compute Evironment', () => {
    let expectedManagedDefaultComputeProps;
    let defaultServiceRole;
    let stack;
    let vpc;
    beforeEach(() => {
        // GIVEN
        stack = new cdk.Stack();
        vpc = new ec2.Vpc(stack, 'test-vpc');
        defaultServiceRole = {
            ServiceRole: {
                'Fn::GetAtt': [
                    'testcomputeenvResourceServiceInstanceRole105069A5',
                    'Arn',
                ],
            },
        };
        expectedManagedDefaultComputeProps = (overrides) => {
            return {
                ComputeResources: {
                    AllocationStrategy: batch.AllocationStrategy.BEST_FIT,
                    InstanceRole: {
                        'Fn::GetAtt': [
                            'testcomputeenvInstanceProfileCBD87EAB',
                            'Arn',
                        ],
                    },
                    InstanceTypes: [
                        'optimal',
                    ],
                    MaxvCpus: 256,
                    MinvCpus: 0,
                    Subnets: [
                        {
                            Ref: 'testvpcPrivateSubnet1Subnet865FB50A',
                        },
                        {
                            Ref: 'testvpcPrivateSubnet2Subnet23D3396F',
                        },
                    ],
                    Type: batch.ComputeResourceType.ON_DEMAND,
                    ...overrides,
                },
            };
        };
    });
    describe('when validating props', () => {
        test('should deny setting compute resources when using type unmanaged', () => {
            // THEN
            assert_2.throws(() => {
                // WHEN
                new batch.ComputeEnvironment(stack, 'test-compute-env', {
                    managed: false,
                    computeResources: {
                        vpc,
                    },
                });
            });
        });
        test('should deny if creating a managed environment with no provided compute resource props', () => {
            // THEN
            assert_2.throws(() => {
                // WHEN
                new batch.ComputeEnvironment(stack, 'test-compute-env', {
                    managed: true,
                });
            });
        });
    });
    describe('using spot resources', () => {
        test('should provide a spot fleet role if one is not given and allocationStrategy is BEST_FIT', () => {
            // WHEN
            new batch.ComputeEnvironment(stack, 'test-compute-env', {
                managed: true,
                computeResources: {
                    type: batch.ComputeResourceType.SPOT,
                    allocationStrategy: batch.AllocationStrategy.BEST_FIT,
                    vpc,
                },
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Batch::ComputeEnvironment', {
                Type: 'MANAGED',
                ...expectedManagedDefaultComputeProps({
                    Type: batch.ComputeResourceType.SPOT,
                    SpotIamFleetRole: {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition',
                                },
                                ':iam::',
                                {
                                    Ref: 'AWS::AccountId',
                                },
                                ':role/aws-service-role/spotfleet.amazonaws.com/AWSServiceRoleForEC2SpotFleet',
                            ],
                        ],
                    },
                }),
            }, assert_1.ResourcePart.Properties));
        });
        describe('with a bid percentage', () => {
            test('should deny my bid if set below 0', () => {
                // THEN
                assert_2.throws(() => {
                    // WHEN
                    new batch.ComputeEnvironment(stack, 'test-compute-env', {
                        managed: true,
                        computeResources: {
                            vpc,
                            type: batch.ComputeResourceType.SPOT,
                            bidPercentage: -1,
                        },
                    });
                });
            });
            test('should deny my bid if above 100', () => {
                // THEN
                assert_2.throws(() => {
                    // WHEN
                    new batch.ComputeEnvironment(stack, 'test-compute-env', {
                        managed: true,
                        computeResources: {
                            vpc,
                            type: batch.ComputeResourceType.SPOT,
                            bidPercentage: 101,
                        },
                    });
                });
            });
        });
    });
    describe('with properties specified', () => {
        test('renders the correct cloudformation properties', () => {
            // WHEN
            const props = {
                computeEnvironmentName: 'my-test-compute-env',
                computeResources: {
                    allocationStrategy: batch.AllocationStrategy.BEST_FIT,
                    vpc,
                    computeResourcesTags: {
                        'Name': 'AWS Batch Instance - C4OnDemand',
                        'Tag Other': 'Has other value',
                    },
                    desiredvCpus: 1,
                    ec2KeyPair: 'my-key-pair',
                    image: new ecs.EcsOptimizedAmi({
                        generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
                        hardwareType: ecs.AmiHardwareType.STANDARD,
                    }),
                    instanceRole: new iam.CfnInstanceProfile(stack, 'Instance-Profile', {
                        roles: [new iam.Role(stack, 'Ecs-Instance-Role', {
                                assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
                                managedPolicies: [
                                    iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2ContainerServiceforEC2Role'),
                                ],
                            }).roleName],
                    }).attrArn,
                    instanceTypes: [
                        ec2.InstanceType.of(ec2.InstanceClass.T2, ec2.InstanceSize.MICRO),
                    ],
                    maxvCpus: 4,
                    minvCpus: 1,
                    securityGroups: [
                        new ec2.SecurityGroup(stack, 'test-sg', {
                            vpc,
                            allowAllOutbound: true,
                        }),
                    ],
                    type: batch.ComputeResourceType.ON_DEMAND,
                    vpcSubnets: {
                        subnetType: ec2.SubnetType.PRIVATE,
                    },
                },
                enabled: false,
                managed: true,
            };
            new batch.ComputeEnvironment(stack, 'test-compute-env', props);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Batch::ComputeEnvironment', {
                ComputeEnvironmentName: 'my-test-compute-env',
                Type: 'MANAGED',
                State: 'DISABLED',
                ServiceRole: {
                    'Fn::GetAtt': [
                        'testcomputeenvResourceServiceInstanceRole105069A5',
                        'Arn',
                    ],
                },
                ComputeResources: {
                    AllocationStrategy: batch.AllocationStrategy.BEST_FIT,
                    DesiredvCpus: props.computeResources.desiredvCpus,
                    Ec2KeyPair: props.computeResources.ec2KeyPair,
                    ImageId: {
                        Ref: 'SsmParameterValueawsserviceecsoptimizedamiamazonlinux2recommendedimageidC96584B6F00A464EAD1953AFF4B05118Parameter',
                    },
                    InstanceRole: {
                        'Fn::GetAtt': [
                            props.computeResources.instanceRole ? 'InstanceProfile' : '',
                            'Arn',
                        ],
                    },
                    InstanceTypes: [
                        props.computeResources.instanceTypes ? props.computeResources.instanceTypes[0].toString() : '',
                    ],
                    MaxvCpus: props.computeResources.maxvCpus,
                    MinvCpus: props.computeResources.minvCpus,
                    SecurityGroupIds: [
                        {
                            'Fn::GetAtt': [
                                'testsg872EB48A',
                                'GroupId',
                            ],
                        },
                    ],
                    Subnets: [
                        {
                            Ref: `${vpc.node.uniqueId}PrivateSubnet1Subnet865FB50A`,
                        },
                        {
                            Ref: `${vpc.node.uniqueId}PrivateSubnet2Subnet23D3396F`,
                        },
                    ],
                    Tags: {
                        'Name': 'AWS Batch Instance - C4OnDemand',
                        'Tag Other': 'Has other value',
                    },
                    Type: 'EC2',
                },
            }, assert_1.ResourcePart.Properties));
        });
        describe('with no allocation strategy specified', () => {
            test('should default to a best_fit strategy', () => {
                // WHEN
                new batch.ComputeEnvironment(stack, 'test-compute-env', {
                    managed: true,
                    computeResources: {
                        vpc,
                    },
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Batch::ComputeEnvironment', {
                    Type: 'MANAGED',
                    ServiceRole: {
                        'Fn::GetAtt': [
                            'testcomputeenvResourceServiceInstanceRole105069A5',
                            'Arn',
                        ],
                    },
                }, assert_1.ResourcePart.Properties));
            });
        });
        describe('with a min vcpu value', () => {
            test('should deny less than 0', () => {
                // THEN
                assert_2.throws(() => {
                    // WHEN
                    new batch.ComputeEnvironment(stack, 'test-compute-env', {
                        computeResources: {
                            vpc,
                            minvCpus: -1,
                        },
                    });
                });
            });
            test('cannot be greater than the max vcpu value', () => {
                // THEN
                assert_2.throws(() => {
                    // WHEN
                    new batch.ComputeEnvironment(stack, 'test-compute-env', {
                        computeResources: {
                            vpc,
                            minvCpus: 2,
                            maxvCpus: 1,
                        },
                    });
                });
            });
        });
        describe('with no min vcpu value provided', () => {
            test('should default to 0', () => {
                // WHEN
                new batch.ComputeEnvironment(stack, 'test-compute-env', {
                    managed: true,
                    computeResources: {
                        vpc,
                    },
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Batch::ComputeEnvironment', {
                    ...defaultServiceRole,
                    ...expectedManagedDefaultComputeProps({
                        MinvCpus: 0,
                    }),
                }, assert_1.ResourcePart.Properties));
            });
        });
        describe('with no max vcpu value provided', () => {
            test('should default to 256', () => {
                // WHEN
                new batch.ComputeEnvironment(stack, 'test-compute-env', {
                    managed: true,
                    computeResources: {
                        vpc,
                    },
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Batch::ComputeEnvironment', {
                    ...expectedManagedDefaultComputeProps({
                        MaxvCpus: 256,
                    }),
                }, assert_1.ResourcePart.Properties));
            });
        });
        describe('with no instance role specified', () => {
            test('should generate a role for me', () => {
                // WHEN
                new batch.ComputeEnvironment(stack, 'test-compute-env', {
                    managed: true,
                    computeResources: {
                        vpc,
                    },
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResource('AWS::Batch::ComputeEnvironment'));
                assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role'));
            });
        });
        describe('with no instance type defined', () => {
            test('should default to optimal matching', () => {
                // WHEN
                new batch.ComputeEnvironment(stack, 'test-compute-env', {
                    managed: true,
                    computeResources: {
                        vpc,
                    },
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Batch::ComputeEnvironment', {
                    ...expectedManagedDefaultComputeProps({
                        InstanceTypes: ['optimal'],
                    }),
                }, assert_1.ResourcePart.Properties));
            });
        });
        describe('with no type specified', () => {
            test('should default to EC2', () => {
                // WHEN
                new batch.ComputeEnvironment(stack, 'test-compute-env', {
                    managed: true,
                    computeResources: {
                        vpc,
                    },
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Batch::ComputeEnvironment', {
                    ...expectedManagedDefaultComputeProps({
                        Type: batch.ComputeResourceType.ON_DEMAND,
                    }),
                }, assert_1.ResourcePart.Properties));
            });
        });
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29tcHV0ZS1lbnZpcm9ubWVudC50ZXN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiY29tcHV0ZS1lbnZpcm9ubWVudC50ZXN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEsNENBQXVGO0FBQ3ZGLGdDQUE4QjtBQUM5Qix3Q0FBd0M7QUFDeEMsd0NBQXdDO0FBQ3hDLHdDQUF3QztBQUN4QyxxQ0FBcUM7QUFDckMsbUNBQWdDO0FBQ2hDLGdDQUFnQztBQUVoQyxRQUFRLENBQUMsMEJBQTBCLEVBQUUsR0FBRyxFQUFFO0lBQ3hDLElBQUksa0NBQXVDLENBQUM7SUFDNUMsSUFBSSxrQkFBdUIsQ0FBQztJQUU1QixJQUFJLEtBQWdCLENBQUM7SUFDckIsSUFBSSxHQUFZLENBQUM7SUFFakIsVUFBVSxDQUFDLEdBQUcsRUFBRTtRQUNkLFFBQVE7UUFDUixLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDeEIsR0FBRyxHQUFHLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxLQUFLLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFFckMsa0JBQWtCLEdBQUc7WUFDbkIsV0FBVyxFQUFFO2dCQUNYLFlBQVksRUFBRTtvQkFDWixtREFBbUQ7b0JBQ25ELEtBQUs7aUJBQ047YUFDRjtTQUNGLENBQUM7UUFFRixrQ0FBa0MsR0FBRyxDQUFDLFNBQWMsRUFBRSxFQUFFO1lBQ3RELE9BQU87Z0JBQ0wsZ0JBQWdCLEVBQUU7b0JBQ2hCLGtCQUFrQixFQUFFLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxRQUFRO29CQUNyRCxZQUFZLEVBQUU7d0JBQ1osWUFBWSxFQUFFOzRCQUNaLHVDQUF1Qzs0QkFDdkMsS0FBSzt5QkFDTjtxQkFDRjtvQkFDRCxhQUFhLEVBQUU7d0JBQ2IsU0FBUztxQkFDVjtvQkFDRCxRQUFRLEVBQUUsR0FBRztvQkFDYixRQUFRLEVBQUUsQ0FBQztvQkFDWCxPQUFPLEVBQUU7d0JBQ1A7NEJBQ0UsR0FBRyxFQUFFLHFDQUFxQzt5QkFDM0M7d0JBQ0Q7NEJBQ0UsR0FBRyxFQUFFLHFDQUFxQzt5QkFDM0M7cUJBQ0Y7b0JBQ0QsSUFBSSxFQUFFLEtBQUssQ0FBQyxtQkFBbUIsQ0FBQyxTQUFTO29CQUN6QyxHQUFHLFNBQVM7aUJBQ2I7YUFDRixDQUFDO1FBQ0osQ0FBQyxDQUFDO0lBQ0osQ0FBQyxDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMsdUJBQXVCLEVBQUUsR0FBRyxFQUFFO1FBQ3JDLElBQUksQ0FBQyxpRUFBaUUsRUFBRSxHQUFHLEVBQUU7WUFDM0UsT0FBTztZQUNQLGVBQU0sQ0FBQyxHQUFHLEVBQUU7Z0JBQ1YsT0FBTztnQkFDUCxJQUFJLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLEVBQUUsa0JBQWtCLEVBQUU7b0JBQ3RELE9BQU8sRUFBRSxLQUFLO29CQUNkLGdCQUFnQixFQUFFO3dCQUNoQixHQUFHO3FCQUNKO2lCQUNGLENBQUMsQ0FBQztZQUNMLENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsdUZBQXVGLEVBQUUsR0FBRyxFQUFFO1lBQ2pHLE9BQU87WUFDUCxlQUFNLENBQUMsR0FBRyxFQUFFO2dCQUNWLE9BQU87Z0JBQ1AsSUFBSSxLQUFLLENBQUMsa0JBQWtCLENBQUMsS0FBSyxFQUFFLGtCQUFrQixFQUFFO29CQUN0RCxPQUFPLEVBQUUsSUFBSTtpQkFDZCxDQUFDLENBQUM7WUFDTCxDQUFDLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMsc0JBQXNCLEVBQUUsR0FBRyxFQUFFO1FBQ3BDLElBQUksQ0FBQyx5RkFBeUYsRUFBRSxHQUFHLEVBQUU7WUFDbkcsT0FBTztZQUNQLElBQUksS0FBSyxDQUFDLGtCQUFrQixDQUFDLEtBQUssRUFBRSxrQkFBa0IsRUFBRTtnQkFDdEQsT0FBTyxFQUFFLElBQUk7Z0JBQ2IsZ0JBQWdCLEVBQUU7b0JBQ2hCLElBQUksRUFBRSxLQUFLLENBQUMsbUJBQW1CLENBQUMsSUFBSTtvQkFDcEMsa0JBQWtCLEVBQUUsS0FBSyxDQUFDLGtCQUFrQixDQUFDLFFBQVE7b0JBQ3JELEdBQUc7aUJBQ0o7YUFDRixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxnQ0FBZ0MsRUFBRTtnQkFDbEUsSUFBSSxFQUFFLFNBQVM7Z0JBQ2YsR0FBRyxrQ0FBa0MsQ0FBQztvQkFDcEMsSUFBSSxFQUFFLEtBQUssQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJO29CQUNwQyxnQkFBZ0IsRUFBRTt3QkFDaEIsVUFBVSxFQUFFOzRCQUNWLEVBQUU7NEJBQ0Y7Z0NBQ0UsTUFBTTtnQ0FDTjtvQ0FDRSxHQUFHLEVBQUUsZ0JBQWdCO2lDQUN0QjtnQ0FDRCxRQUFRO2dDQUNSO29DQUNFLEdBQUcsRUFBRSxnQkFBZ0I7aUNBQ3RCO2dDQUNELDhFQUE4RTs2QkFDL0U7eUJBQ0Y7cUJBQ0Y7aUJBQ0YsQ0FBQzthQUNILEVBQUUscUJBQVksQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDO1FBQy9CLENBQUMsQ0FBQyxDQUFDO1FBRUgsUUFBUSxDQUFDLHVCQUF1QixFQUFFLEdBQUcsRUFBRTtZQUNyQyxJQUFJLENBQUMsbUNBQW1DLEVBQUUsR0FBRyxFQUFFO2dCQUM3QyxPQUFPO2dCQUNQLGVBQU0sQ0FBQyxHQUFHLEVBQUU7b0JBQ1YsT0FBTztvQkFDUCxJQUFJLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLEVBQUUsa0JBQWtCLEVBQUU7d0JBQ3RELE9BQU8sRUFBRSxJQUFJO3dCQUNiLGdCQUFnQixFQUFFOzRCQUNoQixHQUFHOzRCQUNILElBQUksRUFBRSxLQUFLLENBQUMsbUJBQW1CLENBQUMsSUFBSTs0QkFDcEMsYUFBYSxFQUFFLENBQUMsQ0FBQzt5QkFDbEI7cUJBQ0YsQ0FBQyxDQUFDO2dCQUNMLENBQUMsQ0FBQyxDQUFDO1lBQ0wsQ0FBQyxDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsaUNBQWlDLEVBQUUsR0FBRyxFQUFFO2dCQUMzQyxPQUFPO2dCQUNQLGVBQU0sQ0FBQyxHQUFHLEVBQUU7b0JBQ1YsT0FBTztvQkFDUCxJQUFJLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLEVBQUUsa0JBQWtCLEVBQUU7d0JBQ3RELE9BQU8sRUFBRSxJQUFJO3dCQUNiLGdCQUFnQixFQUFFOzRCQUNoQixHQUFHOzRCQUNILElBQUksRUFBRSxLQUFLLENBQUMsbUJBQW1CLENBQUMsSUFBSTs0QkFDcEMsYUFBYSxFQUFFLEdBQUc7eUJBQ25CO3FCQUNGLENBQUMsQ0FBQztnQkFDTCxDQUFDLENBQUMsQ0FBQztZQUNMLENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQywyQkFBMkIsRUFBRSxHQUFHLEVBQUU7UUFDekMsSUFBSSxDQUFDLCtDQUErQyxFQUFFLEdBQUcsRUFBRTtZQUN6RCxPQUFPO1lBQ1AsTUFBTSxLQUFLLEdBQUc7Z0JBQ1osc0JBQXNCLEVBQUUscUJBQXFCO2dCQUM3QyxnQkFBZ0IsRUFBRTtvQkFDaEIsa0JBQWtCLEVBQUUsS0FBSyxDQUFDLGtCQUFrQixDQUFDLFFBQVE7b0JBQ3JELEdBQUc7b0JBQ0gsb0JBQW9CLEVBQUU7d0JBQ3BCLE1BQU0sRUFBRSxpQ0FBaUM7d0JBQ3pDLFdBQVcsRUFBRSxpQkFBaUI7cUJBQy9CO29CQUNELFlBQVksRUFBRSxDQUFDO29CQUNmLFVBQVUsRUFBRSxhQUFhO29CQUN6QixLQUFLLEVBQUUsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO3dCQUM3QixVQUFVLEVBQUUsR0FBRyxDQUFDLHFCQUFxQixDQUFDLGNBQWM7d0JBQ3BELFlBQVksRUFBRSxHQUFHLENBQUMsZUFBZSxDQUFDLFFBQVE7cUJBQzNDLENBQUM7b0JBQ0YsWUFBWSxFQUFFLElBQUksR0FBRyxDQUFDLGtCQUFrQixDQUFDLEtBQUssRUFBRSxrQkFBa0IsRUFBRTt3QkFDbEUsS0FBSyxFQUFFLENBQUMsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxtQkFBbUIsRUFBRTtnQ0FDL0MsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLG1CQUFtQixDQUFDO2dDQUN4RCxlQUFlLEVBQUU7b0NBQ2YsR0FBRyxDQUFDLGFBQWEsQ0FBQyx3QkFBd0IsQ0FBQyxrREFBa0QsQ0FBQztpQ0FDL0Y7NkJBQ0YsQ0FBQyxDQUFDLFFBQVEsQ0FBQztxQkFDYixDQUFDLENBQUMsT0FBTztvQkFDVixhQUFhLEVBQUU7d0JBQ2IsR0FBRyxDQUFDLFlBQVksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxFQUFFLEVBQUUsR0FBRyxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUM7cUJBQ2xFO29CQUNELFFBQVEsRUFBRSxDQUFDO29CQUNYLFFBQVEsRUFBRSxDQUFDO29CQUNYLGNBQWMsRUFBRTt3QkFDZCxJQUFJLEdBQUcsQ0FBQyxhQUFhLENBQUMsS0FBSyxFQUFFLFNBQVMsRUFBRTs0QkFDdEMsR0FBRzs0QkFDSCxnQkFBZ0IsRUFBRSxJQUFJO3lCQUN2QixDQUFDO3FCQUNIO29CQUNELElBQUksRUFBRSxLQUFLLENBQUMsbUJBQW1CLENBQUMsU0FBUztvQkFDekMsVUFBVSxFQUFFO3dCQUNWLFVBQVUsRUFBRSxHQUFHLENBQUMsVUFBVSxDQUFDLE9BQU87cUJBQ25DO2lCQUN3QjtnQkFDM0IsT0FBTyxFQUFFLEtBQUs7Z0JBQ2QsT0FBTyxFQUFFLElBQUk7YUFDZCxDQUFDO1lBRUYsSUFBSSxLQUFLLENBQUMsa0JBQWtCLENBQUMsS0FBSyxFQUFFLGtCQUFrQixFQUFFLEtBQUssQ0FBQyxDQUFDO1lBRS9ELE9BQU87WUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLGdDQUFnQyxFQUFFO2dCQUNsRSxzQkFBc0IsRUFBRSxxQkFBcUI7Z0JBQzdDLElBQUksRUFBRSxTQUFTO2dCQUNmLEtBQUssRUFBRSxVQUFVO2dCQUNqQixXQUFXLEVBQUU7b0JBQ1gsWUFBWSxFQUFFO3dCQUNaLG1EQUFtRDt3QkFDbkQsS0FBSztxQkFDTjtpQkFDRjtnQkFDRCxnQkFBZ0IsRUFBRTtvQkFDaEIsa0JBQWtCLEVBQUUsS0FBSyxDQUFDLGtCQUFrQixDQUFDLFFBQVE7b0JBQ3JELFlBQVksRUFBRSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsWUFBWTtvQkFDakQsVUFBVSxFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxVQUFVO29CQUM3QyxPQUFPLEVBQUU7d0JBQ1AsR0FBRyxFQUFFLG1IQUFtSDtxQkFDekg7b0JBQ0QsWUFBWSxFQUFFO3dCQUNaLFlBQVksRUFBRTs0QkFDWixLQUFLLENBQUMsZ0JBQWdCLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLENBQUMsRUFBRTs0QkFDNUQsS0FBSzt5QkFDTjtxQkFDRjtvQkFDRCxhQUFhLEVBQUU7d0JBQ2IsS0FBSyxDQUFDLGdCQUFnQixDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDLENBQUMsRUFBRTtxQkFDL0Y7b0JBQ0QsUUFBUSxFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxRQUFRO29CQUN6QyxRQUFRLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLFFBQVE7b0JBQ3pDLGdCQUFnQixFQUFFO3dCQUNoQjs0QkFDRSxZQUFZLEVBQUU7Z0NBQ1osZ0JBQWdCO2dDQUNoQixTQUFTOzZCQUNWO3lCQUNGO3FCQUNGO29CQUNELE9BQU8sRUFBRTt3QkFDUDs0QkFDRSxHQUFHLEVBQUUsR0FBRyxHQUFHLENBQUMsSUFBSSxDQUFDLFFBQVEsOEJBQThCO3lCQUN4RDt3QkFDRDs0QkFDRSxHQUFHLEVBQUUsR0FBRyxHQUFHLENBQUMsSUFBSSxDQUFDLFFBQVEsOEJBQThCO3lCQUN4RDtxQkFDRjtvQkFDRCxJQUFJLEVBQUU7d0JBQ0osTUFBTSxFQUFFLGlDQUFpQzt3QkFDekMsV0FBVyxFQUFFLGlCQUFpQjtxQkFDL0I7b0JBQ0QsSUFBSSxFQUFFLEtBQUs7aUJBQ1o7YUFDRixFQUFFLHFCQUFZLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQztRQUMvQixDQUFDLENBQUMsQ0FBQztRQUVILFFBQVEsQ0FBQyx1Q0FBdUMsRUFBRSxHQUFHLEVBQUU7WUFDckQsSUFBSSxDQUFDLHVDQUF1QyxFQUFFLEdBQUcsRUFBRTtnQkFDakQsT0FBTztnQkFDUCxJQUFJLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLEVBQUUsa0JBQWtCLEVBQUU7b0JBQ3RELE9BQU8sRUFBRSxJQUFJO29CQUNiLGdCQUFnQixFQUFFO3dCQUNoQixHQUFHO3FCQUNKO2lCQUNGLENBQUMsQ0FBQztnQkFFSCxPQUFPO2dCQUNQLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsZ0NBQWdDLEVBQUU7b0JBQ2xFLElBQUksRUFBRSxTQUFTO29CQUNmLFdBQVcsRUFBRTt3QkFDWCxZQUFZLEVBQUU7NEJBQ1osbURBQW1EOzRCQUNuRCxLQUFLO3lCQUNOO3FCQUNGO2lCQUNGLEVBQUUscUJBQVksQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDO1lBQy9CLENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCxRQUFRLENBQUMsdUJBQXVCLEVBQUUsR0FBRyxFQUFFO1lBQ3JDLElBQUksQ0FBQyx5QkFBeUIsRUFBRSxHQUFHLEVBQUU7Z0JBQ25DLE9BQU87Z0JBQ1AsZUFBTSxDQUFDLEdBQUcsRUFBRTtvQkFDVixPQUFPO29CQUNQLElBQUksS0FBSyxDQUFDLGtCQUFrQixDQUFDLEtBQUssRUFBRSxrQkFBa0IsRUFBRTt3QkFDdEQsZ0JBQWdCLEVBQUU7NEJBQ2hCLEdBQUc7NEJBQ0gsUUFBUSxFQUFFLENBQUMsQ0FBQzt5QkFDYjtxQkFDRixDQUFDLENBQUM7Z0JBQ0wsQ0FBQyxDQUFDLENBQUM7WUFDTCxDQUFDLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQywyQ0FBMkMsRUFBRSxHQUFHLEVBQUU7Z0JBQ3JELE9BQU87Z0JBQ1AsZUFBTSxDQUFDLEdBQUcsRUFBRTtvQkFDVixPQUFPO29CQUNQLElBQUksS0FBSyxDQUFDLGtCQUFrQixDQUFDLEtBQUssRUFBRSxrQkFBa0IsRUFBRTt3QkFDdEQsZ0JBQWdCLEVBQUU7NEJBQ2hCLEdBQUc7NEJBQ0gsUUFBUSxFQUFFLENBQUM7NEJBQ1gsUUFBUSxFQUFFLENBQUM7eUJBQ1o7cUJBQ0YsQ0FBQyxDQUFDO2dCQUNMLENBQUMsQ0FBQyxDQUFDO1lBQ0wsQ0FBQyxDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUVILFFBQVEsQ0FBQyxpQ0FBaUMsRUFBRSxHQUFHLEVBQUU7WUFDL0MsSUFBSSxDQUFDLHFCQUFxQixFQUFFLEdBQUcsRUFBRTtnQkFDL0IsT0FBTztnQkFDUCxJQUFJLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLEVBQUUsa0JBQWtCLEVBQUU7b0JBQ3RELE9BQU8sRUFBRSxJQUFJO29CQUNiLGdCQUFnQixFQUFFO3dCQUNoQixHQUFHO3FCQUNKO2lCQUNGLENBQUMsQ0FBQztnQkFFSCxPQUFPO2dCQUNQLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsZ0NBQWdDLEVBQUU7b0JBQ2xFLEdBQUcsa0JBQWtCO29CQUNyQixHQUFHLGtDQUFrQyxDQUFDO3dCQUNwQyxRQUFRLEVBQUUsQ0FBQztxQkFDWixDQUFDO2lCQUNILEVBQUUscUJBQVksQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDO1lBQy9CLENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCxRQUFRLENBQUMsaUNBQWlDLEVBQUUsR0FBRyxFQUFFO1lBQy9DLElBQUksQ0FBQyx1QkFBdUIsRUFBRSxHQUFHLEVBQUU7Z0JBQ2pDLE9BQU87Z0JBQ1AsSUFBSSxLQUFLLENBQUMsa0JBQWtCLENBQUMsS0FBSyxFQUFFLGtCQUFrQixFQUFFO29CQUN0RCxPQUFPLEVBQUUsSUFBSTtvQkFDYixnQkFBZ0IsRUFBRTt3QkFDaEIsR0FBRztxQkFDSjtpQkFDRixDQUFDLENBQUM7Z0JBRUgsT0FBTztnQkFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLGdDQUFnQyxFQUFFO29CQUNsRSxHQUFHLGtDQUFrQyxDQUFDO3dCQUNwQyxRQUFRLEVBQUUsR0FBRztxQkFDZCxDQUFDO2lCQUNILEVBQUUscUJBQVksQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDO1lBQy9CLENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCxRQUFRLENBQUMsaUNBQWlDLEVBQUUsR0FBRyxFQUFFO1lBQy9DLElBQUksQ0FBQywrQkFBK0IsRUFBRSxHQUFHLEVBQUU7Z0JBQ3pDLE9BQU87Z0JBQ1AsSUFBSSxLQUFLLENBQUMsa0JBQWtCLENBQUMsS0FBSyxFQUFFLGtCQUFrQixFQUFFO29CQUN0RCxPQUFPLEVBQUUsSUFBSTtvQkFDYixnQkFBZ0IsRUFBRTt3QkFDaEIsR0FBRztxQkFDSjtpQkFDRixDQUFDLENBQUM7Z0JBRUgsT0FBTztnQkFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsZ0NBQWdDLENBQUMsQ0FBQyxDQUFDO2dCQUNqRSxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDO1lBQ25ELENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCxRQUFRLENBQUMsK0JBQStCLEVBQUUsR0FBRyxFQUFFO1lBQzdDLElBQUksQ0FBQyxvQ0FBb0MsRUFBRSxHQUFHLEVBQUU7Z0JBQzlDLE9BQU87Z0JBQ1AsSUFBSSxLQUFLLENBQUMsa0JBQWtCLENBQUMsS0FBSyxFQUFFLGtCQUFrQixFQUFFO29CQUN0RCxPQUFPLEVBQUUsSUFBSTtvQkFDYixnQkFBZ0IsRUFBRTt3QkFDaEIsR0FBRztxQkFDSjtpQkFDRixDQUFDLENBQUM7Z0JBRUgsT0FBTztnQkFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLGdDQUFnQyxFQUFFO29CQUNsRSxHQUFHLGtDQUFrQyxDQUFDO3dCQUNwQyxhQUFhLEVBQUUsQ0FBQyxTQUFTLENBQUM7cUJBQzNCLENBQUM7aUJBQ0gsRUFBRSxxQkFBWSxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUM7WUFDL0IsQ0FBQyxDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUVILFFBQVEsQ0FBQyx3QkFBd0IsRUFBRSxHQUFHLEVBQUU7WUFDdEMsSUFBSSxDQUFDLHVCQUF1QixFQUFFLEdBQUcsRUFBRTtnQkFDakMsT0FBTztnQkFDUCxJQUFJLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLEVBQUUsa0JBQWtCLEVBQUU7b0JBQ3RELE9BQU8sRUFBRSxJQUFJO29CQUNiLGdCQUFnQixFQUFFO3dCQUNoQixHQUFHO3FCQUNKO2lCQUNGLENBQUMsQ0FBQztnQkFFSCxPQUFPO2dCQUNQLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsZ0NBQWdDLEVBQUU7b0JBQ2xFLEdBQUcsa0NBQWtDLENBQUM7d0JBQ3BDLElBQUksRUFBRSxLQUFLLENBQUMsbUJBQW1CLENBQUMsU0FBUztxQkFDMUMsQ0FBQztpQkFDSCxFQUFFLHFCQUFZLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQztZQUMvQixDQUFDLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IGV4cGVjdCwgaGF2ZVJlc291cmNlLCBoYXZlUmVzb3VyY2VMaWtlLCBSZXNvdXJjZVBhcnQgfSBmcm9tICdAYXdzLWNkay9hc3NlcnQnO1xuaW1wb3J0ICdAYXdzLWNkay9hc3NlcnQvamVzdCc7XG5pbXBvcnQgKiBhcyBlYzIgZnJvbSAnQGF3cy1jZGsvYXdzLWVjMic7XG5pbXBvcnQgKiBhcyBlY3MgZnJvbSAnQGF3cy1jZGsvYXdzLWVjcyc7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyB0aHJvd3MgfSBmcm9tICdhc3NlcnQnO1xuaW1wb3J0ICogYXMgYmF0Y2ggZnJvbSAnLi4vbGliJztcblxuZGVzY3JpYmUoJ0JhdGNoIENvbXB1dGUgRXZpcm9ubWVudCcsICgpID0+IHtcbiAgbGV0IGV4cGVjdGVkTWFuYWdlZERlZmF1bHRDb21wdXRlUHJvcHM6IGFueTtcbiAgbGV0IGRlZmF1bHRTZXJ2aWNlUm9sZTogYW55O1xuXG4gIGxldCBzdGFjazogY2RrLlN0YWNrO1xuICBsZXQgdnBjOiBlYzIuVnBjO1xuXG4gIGJlZm9yZUVhY2goKCkgPT4ge1xuICAgIC8vIEdJVkVOXG4gICAgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgdnBjID0gbmV3IGVjMi5WcGMoc3RhY2ssICd0ZXN0LXZwYycpO1xuXG4gICAgZGVmYXVsdFNlcnZpY2VSb2xlID0ge1xuICAgICAgU2VydmljZVJvbGU6IHtcbiAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgJ3Rlc3Rjb21wdXRlZW52UmVzb3VyY2VTZXJ2aWNlSW5zdGFuY2VSb2xlMTA1MDY5QTUnLFxuICAgICAgICAgICdBcm4nLFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICB9O1xuXG4gICAgZXhwZWN0ZWRNYW5hZ2VkRGVmYXVsdENvbXB1dGVQcm9wcyA9IChvdmVycmlkZXM6IGFueSkgPT4ge1xuICAgICAgcmV0dXJuIHtcbiAgICAgICAgQ29tcHV0ZVJlc291cmNlczoge1xuICAgICAgICAgIEFsbG9jYXRpb25TdHJhdGVneTogYmF0Y2guQWxsb2NhdGlvblN0cmF0ZWd5LkJFU1RfRklULFxuICAgICAgICAgIEluc3RhbmNlUm9sZToge1xuICAgICAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgICAgICd0ZXN0Y29tcHV0ZWVudkluc3RhbmNlUHJvZmlsZUNCRDg3RUFCJyxcbiAgICAgICAgICAgICAgJ0FybicsXG4gICAgICAgICAgICBdLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgSW5zdGFuY2VUeXBlczogW1xuICAgICAgICAgICAgJ29wdGltYWwnLFxuICAgICAgICAgIF0sXG4gICAgICAgICAgTWF4dkNwdXM6IDI1NixcbiAgICAgICAgICBNaW52Q3B1czogMCxcbiAgICAgICAgICBTdWJuZXRzOiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIFJlZjogJ3Rlc3R2cGNQcml2YXRlU3VibmV0MVN1Ym5ldDg2NUZCNTBBJyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIFJlZjogJ3Rlc3R2cGNQcml2YXRlU3VibmV0MlN1Ym5ldDIzRDMzOTZGJyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgXSxcbiAgICAgICAgICBUeXBlOiBiYXRjaC5Db21wdXRlUmVzb3VyY2VUeXBlLk9OX0RFTUFORCxcbiAgICAgICAgICAuLi5vdmVycmlkZXMsXG4gICAgICAgIH0sXG4gICAgICB9O1xuICAgIH07XG4gIH0pO1xuXG4gIGRlc2NyaWJlKCd3aGVuIHZhbGlkYXRpbmcgcHJvcHMnLCAoKSA9PiB7XG4gICAgdGVzdCgnc2hvdWxkIGRlbnkgc2V0dGluZyBjb21wdXRlIHJlc291cmNlcyB3aGVuIHVzaW5nIHR5cGUgdW5tYW5hZ2VkJywgKCkgPT4ge1xuICAgICAgLy8gVEhFTlxuICAgICAgdGhyb3dzKCgpID0+IHtcbiAgICAgICAgLy8gV0hFTlxuICAgICAgICBuZXcgYmF0Y2guQ29tcHV0ZUVudmlyb25tZW50KHN0YWNrLCAndGVzdC1jb21wdXRlLWVudicsIHtcbiAgICAgICAgICBtYW5hZ2VkOiBmYWxzZSxcbiAgICAgICAgICBjb21wdXRlUmVzb3VyY2VzOiB7XG4gICAgICAgICAgICB2cGMsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSk7XG4gICAgICB9KTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ3Nob3VsZCBkZW55IGlmIGNyZWF0aW5nIGEgbWFuYWdlZCBlbnZpcm9ubWVudCB3aXRoIG5vIHByb3ZpZGVkIGNvbXB1dGUgcmVzb3VyY2UgcHJvcHMnLCAoKSA9PiB7XG4gICAgICAvLyBUSEVOXG4gICAgICB0aHJvd3MoKCkgPT4ge1xuICAgICAgICAvLyBXSEVOXG4gICAgICAgIG5ldyBiYXRjaC5Db21wdXRlRW52aXJvbm1lbnQoc3RhY2ssICd0ZXN0LWNvbXB1dGUtZW52Jywge1xuICAgICAgICAgIG1hbmFnZWQ6IHRydWUsXG4gICAgICAgIH0pO1xuICAgICAgfSk7XG4gICAgfSk7XG4gIH0pO1xuXG4gIGRlc2NyaWJlKCd1c2luZyBzcG90IHJlc291cmNlcycsICgpID0+IHtcbiAgICB0ZXN0KCdzaG91bGQgcHJvdmlkZSBhIHNwb3QgZmxlZXQgcm9sZSBpZiBvbmUgaXMgbm90IGdpdmVuIGFuZCBhbGxvY2F0aW9uU3RyYXRlZ3kgaXMgQkVTVF9GSVQnLCAoKSA9PiB7XG4gICAgICAvLyBXSEVOXG4gICAgICBuZXcgYmF0Y2guQ29tcHV0ZUVudmlyb25tZW50KHN0YWNrLCAndGVzdC1jb21wdXRlLWVudicsIHtcbiAgICAgICAgbWFuYWdlZDogdHJ1ZSxcbiAgICAgICAgY29tcHV0ZVJlc291cmNlczoge1xuICAgICAgICAgIHR5cGU6IGJhdGNoLkNvbXB1dGVSZXNvdXJjZVR5cGUuU1BPVCxcbiAgICAgICAgICBhbGxvY2F0aW9uU3RyYXRlZ3k6IGJhdGNoLkFsbG9jYXRpb25TdHJhdGVneS5CRVNUX0ZJVCxcbiAgICAgICAgICB2cGMsXG4gICAgICAgIH0sXG4gICAgICB9KTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkJhdGNoOjpDb21wdXRlRW52aXJvbm1lbnQnLCB7XG4gICAgICAgIFR5cGU6ICdNQU5BR0VEJyxcbiAgICAgICAgLi4uZXhwZWN0ZWRNYW5hZ2VkRGVmYXVsdENvbXB1dGVQcm9wcyh7XG4gICAgICAgICAgVHlwZTogYmF0Y2guQ29tcHV0ZVJlc291cmNlVHlwZS5TUE9ULFxuICAgICAgICAgIFNwb3RJYW1GbGVldFJvbGU6IHtcbiAgICAgICAgICAgICdGbjo6Sm9pbic6IFtcbiAgICAgICAgICAgICAgJycsXG4gICAgICAgICAgICAgIFtcbiAgICAgICAgICAgICAgICAnYXJuOicsXG4gICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgUmVmOiAnQVdTOjpQYXJ0aXRpb24nLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgJzppYW06OicsXG4gICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgUmVmOiAnQVdTOjpBY2NvdW50SWQnLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgJzpyb2xlL2F3cy1zZXJ2aWNlLXJvbGUvc3BvdGZsZWV0LmFtYXpvbmF3cy5jb20vQVdTU2VydmljZVJvbGVGb3JFQzJTcG90RmxlZXQnLFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICB9LFxuICAgICAgICB9KSxcbiAgICAgIH0sIFJlc291cmNlUGFydC5Qcm9wZXJ0aWVzKSk7XG4gICAgfSk7XG5cbiAgICBkZXNjcmliZSgnd2l0aCBhIGJpZCBwZXJjZW50YWdlJywgKCkgPT4ge1xuICAgICAgdGVzdCgnc2hvdWxkIGRlbnkgbXkgYmlkIGlmIHNldCBiZWxvdyAwJywgKCkgPT4ge1xuICAgICAgICAvLyBUSEVOXG4gICAgICAgIHRocm93cygoKSA9PiB7XG4gICAgICAgICAgLy8gV0hFTlxuICAgICAgICAgIG5ldyBiYXRjaC5Db21wdXRlRW52aXJvbm1lbnQoc3RhY2ssICd0ZXN0LWNvbXB1dGUtZW52Jywge1xuICAgICAgICAgICAgbWFuYWdlZDogdHJ1ZSxcbiAgICAgICAgICAgIGNvbXB1dGVSZXNvdXJjZXM6IHtcbiAgICAgICAgICAgICAgdnBjLFxuICAgICAgICAgICAgICB0eXBlOiBiYXRjaC5Db21wdXRlUmVzb3VyY2VUeXBlLlNQT1QsXG4gICAgICAgICAgICAgIGJpZFBlcmNlbnRhZ2U6IC0xLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9KTtcbiAgICAgICAgfSk7XG4gICAgICB9KTtcblxuICAgICAgdGVzdCgnc2hvdWxkIGRlbnkgbXkgYmlkIGlmIGFib3ZlIDEwMCcsICgpID0+IHtcbiAgICAgICAgLy8gVEhFTlxuICAgICAgICB0aHJvd3MoKCkgPT4ge1xuICAgICAgICAgIC8vIFdIRU5cbiAgICAgICAgICBuZXcgYmF0Y2guQ29tcHV0ZUVudmlyb25tZW50KHN0YWNrLCAndGVzdC1jb21wdXRlLWVudicsIHtcbiAgICAgICAgICAgIG1hbmFnZWQ6IHRydWUsXG4gICAgICAgICAgICBjb21wdXRlUmVzb3VyY2VzOiB7XG4gICAgICAgICAgICAgIHZwYyxcbiAgICAgICAgICAgICAgdHlwZTogYmF0Y2guQ29tcHV0ZVJlc291cmNlVHlwZS5TUE9ULFxuICAgICAgICAgICAgICBiaWRQZXJjZW50YWdlOiAxMDEsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0pO1xuICAgICAgICB9KTtcbiAgICAgIH0pO1xuICAgIH0pO1xuICB9KTtcblxuICBkZXNjcmliZSgnd2l0aCBwcm9wZXJ0aWVzIHNwZWNpZmllZCcsICgpID0+IHtcbiAgICB0ZXN0KCdyZW5kZXJzIHRoZSBjb3JyZWN0IGNsb3VkZm9ybWF0aW9uIHByb3BlcnRpZXMnLCAoKSA9PiB7XG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCBwcm9wcyA9IHtcbiAgICAgICAgY29tcHV0ZUVudmlyb25tZW50TmFtZTogJ215LXRlc3QtY29tcHV0ZS1lbnYnLFxuICAgICAgICBjb21wdXRlUmVzb3VyY2VzOiB7XG4gICAgICAgICAgYWxsb2NhdGlvblN0cmF0ZWd5OiBiYXRjaC5BbGxvY2F0aW9uU3RyYXRlZ3kuQkVTVF9GSVQsXG4gICAgICAgICAgdnBjLFxuICAgICAgICAgIGNvbXB1dGVSZXNvdXJjZXNUYWdzOiB7XG4gICAgICAgICAgICAnTmFtZSc6ICdBV1MgQmF0Y2ggSW5zdGFuY2UgLSBDNE9uRGVtYW5kJyxcbiAgICAgICAgICAgICdUYWcgT3RoZXInOiAnSGFzIG90aGVyIHZhbHVlJyxcbiAgICAgICAgICB9LFxuICAgICAgICAgIGRlc2lyZWR2Q3B1czogMSxcbiAgICAgICAgICBlYzJLZXlQYWlyOiAnbXkta2V5LXBhaXInLFxuICAgICAgICAgIGltYWdlOiBuZXcgZWNzLkVjc09wdGltaXplZEFtaSh7XG4gICAgICAgICAgICBnZW5lcmF0aW9uOiBlYzIuQW1hem9uTGludXhHZW5lcmF0aW9uLkFNQVpPTl9MSU5VWF8yLFxuICAgICAgICAgICAgaGFyZHdhcmVUeXBlOiBlY3MuQW1pSGFyZHdhcmVUeXBlLlNUQU5EQVJELFxuICAgICAgICAgIH0pLFxuICAgICAgICAgIGluc3RhbmNlUm9sZTogbmV3IGlhbS5DZm5JbnN0YW5jZVByb2ZpbGUoc3RhY2ssICdJbnN0YW5jZS1Qcm9maWxlJywge1xuICAgICAgICAgICAgcm9sZXM6IFtuZXcgaWFtLlJvbGUoc3RhY2ssICdFY3MtSW5zdGFuY2UtUm9sZScsIHtcbiAgICAgICAgICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2VjMi5hbWF6b25hd3MuY29tJyksXG4gICAgICAgICAgICAgIG1hbmFnZWRQb2xpY2llczogW1xuICAgICAgICAgICAgICAgIGlhbS5NYW5hZ2VkUG9saWN5LmZyb21Bd3NNYW5hZ2VkUG9saWN5TmFtZSgnc2VydmljZS1yb2xlL0FtYXpvbkVDMkNvbnRhaW5lclNlcnZpY2Vmb3JFQzJSb2xlJyksXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICB9KS5yb2xlTmFtZV0sXG4gICAgICAgICAgfSkuYXR0ckFybixcbiAgICAgICAgICBpbnN0YW5jZVR5cGVzOiBbXG4gICAgICAgICAgICBlYzIuSW5zdGFuY2VUeXBlLm9mKGVjMi5JbnN0YW5jZUNsYXNzLlQyLCBlYzIuSW5zdGFuY2VTaXplLk1JQ1JPKSxcbiAgICAgICAgICBdLFxuICAgICAgICAgIG1heHZDcHVzOiA0LFxuICAgICAgICAgIG1pbnZDcHVzOiAxLFxuICAgICAgICAgIHNlY3VyaXR5R3JvdXBzOiBbXG4gICAgICAgICAgICBuZXcgZWMyLlNlY3VyaXR5R3JvdXAoc3RhY2ssICd0ZXN0LXNnJywge1xuICAgICAgICAgICAgICB2cGMsXG4gICAgICAgICAgICAgIGFsbG93QWxsT3V0Ym91bmQ6IHRydWUsXG4gICAgICAgICAgICB9KSxcbiAgICAgICAgICBdLFxuICAgICAgICAgIHR5cGU6IGJhdGNoLkNvbXB1dGVSZXNvdXJjZVR5cGUuT05fREVNQU5ELFxuICAgICAgICAgIHZwY1N1Ym5ldHM6IHtcbiAgICAgICAgICAgIHN1Ym5ldFR5cGU6IGVjMi5TdWJuZXRUeXBlLlBSSVZBVEUsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSBhcyBiYXRjaC5Db21wdXRlUmVzb3VyY2VzLFxuICAgICAgICBlbmFibGVkOiBmYWxzZSxcbiAgICAgICAgbWFuYWdlZDogdHJ1ZSxcbiAgICAgIH07XG5cbiAgICAgIG5ldyBiYXRjaC5Db21wdXRlRW52aXJvbm1lbnQoc3RhY2ssICd0ZXN0LWNvbXB1dGUtZW52JywgcHJvcHMpO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6QmF0Y2g6OkNvbXB1dGVFbnZpcm9ubWVudCcsIHtcbiAgICAgICAgQ29tcHV0ZUVudmlyb25tZW50TmFtZTogJ215LXRlc3QtY29tcHV0ZS1lbnYnLFxuICAgICAgICBUeXBlOiAnTUFOQUdFRCcsXG4gICAgICAgIFN0YXRlOiAnRElTQUJMRUQnLFxuICAgICAgICBTZXJ2aWNlUm9sZToge1xuICAgICAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAgICAgJ3Rlc3Rjb21wdXRlZW52UmVzb3VyY2VTZXJ2aWNlSW5zdGFuY2VSb2xlMTA1MDY5QTUnLFxuICAgICAgICAgICAgJ0FybicsXG4gICAgICAgICAgXSxcbiAgICAgICAgfSxcbiAgICAgICAgQ29tcHV0ZVJlc291cmNlczoge1xuICAgICAgICAgIEFsbG9jYXRpb25TdHJhdGVneTogYmF0Y2guQWxsb2NhdGlvblN0cmF0ZWd5LkJFU1RfRklULFxuICAgICAgICAgIERlc2lyZWR2Q3B1czogcHJvcHMuY29tcHV0ZVJlc291cmNlcy5kZXNpcmVkdkNwdXMsXG4gICAgICAgICAgRWMyS2V5UGFpcjogcHJvcHMuY29tcHV0ZVJlc291cmNlcy5lYzJLZXlQYWlyLFxuICAgICAgICAgIEltYWdlSWQ6IHtcbiAgICAgICAgICAgIFJlZjogJ1NzbVBhcmFtZXRlclZhbHVlYXdzc2VydmljZWVjc29wdGltaXplZGFtaWFtYXpvbmxpbnV4MnJlY29tbWVuZGVkaW1hZ2VpZEM5NjU4NEI2RjAwQTQ2NEVBRDE5NTNBRkY0QjA1MTE4UGFyYW1ldGVyJyxcbiAgICAgICAgICB9LFxuICAgICAgICAgIEluc3RhbmNlUm9sZToge1xuICAgICAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgICAgIHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuaW5zdGFuY2VSb2xlID8gJ0luc3RhbmNlUHJvZmlsZScgOiAnJyxcbiAgICAgICAgICAgICAgJ0FybicsXG4gICAgICAgICAgICBdLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgSW5zdGFuY2VUeXBlczogW1xuICAgICAgICAgICAgcHJvcHMuY29tcHV0ZVJlc291cmNlcy5pbnN0YW5jZVR5cGVzID8gcHJvcHMuY29tcHV0ZVJlc291cmNlcy5pbnN0YW5jZVR5cGVzWzBdLnRvU3RyaW5nKCkgOiAnJyxcbiAgICAgICAgICBdLFxuICAgICAgICAgIE1heHZDcHVzOiBwcm9wcy5jb21wdXRlUmVzb3VyY2VzLm1heHZDcHVzLFxuICAgICAgICAgIE1pbnZDcHVzOiBwcm9wcy5jb21wdXRlUmVzb3VyY2VzLm1pbnZDcHVzLFxuICAgICAgICAgIFNlY3VyaXR5R3JvdXBJZHM6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgICAgICAgJ3Rlc3RzZzg3MkVCNDhBJyxcbiAgICAgICAgICAgICAgICAnR3JvdXBJZCcsXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIF0sXG4gICAgICAgICAgU3VibmV0czogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBSZWY6IGAke3ZwYy5ub2RlLnVuaXF1ZUlkfVByaXZhdGVTdWJuZXQxU3VibmV0ODY1RkI1MEFgLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgUmVmOiBgJHt2cGMubm9kZS51bmlxdWVJZH1Qcml2YXRlU3VibmV0MlN1Ym5ldDIzRDMzOTZGYCxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgXSxcbiAgICAgICAgICBUYWdzOiB7XG4gICAgICAgICAgICAnTmFtZSc6ICdBV1MgQmF0Y2ggSW5zdGFuY2UgLSBDNE9uRGVtYW5kJyxcbiAgICAgICAgICAgICdUYWcgT3RoZXInOiAnSGFzIG90aGVyIHZhbHVlJyxcbiAgICAgICAgICB9LFxuICAgICAgICAgIFR5cGU6ICdFQzInLFxuICAgICAgICB9LFxuICAgICAgfSwgUmVzb3VyY2VQYXJ0LlByb3BlcnRpZXMpKTtcbiAgICB9KTtcblxuICAgIGRlc2NyaWJlKCd3aXRoIG5vIGFsbG9jYXRpb24gc3RyYXRlZ3kgc3BlY2lmaWVkJywgKCkgPT4ge1xuICAgICAgdGVzdCgnc2hvdWxkIGRlZmF1bHQgdG8gYSBiZXN0X2ZpdCBzdHJhdGVneScsICgpID0+IHtcbiAgICAgICAgLy8gV0hFTlxuICAgICAgICBuZXcgYmF0Y2guQ29tcHV0ZUVudmlyb25tZW50KHN0YWNrLCAndGVzdC1jb21wdXRlLWVudicsIHtcbiAgICAgICAgICBtYW5hZ2VkOiB0cnVlLFxuICAgICAgICAgIGNvbXB1dGVSZXNvdXJjZXM6IHtcbiAgICAgICAgICAgIHZwYyxcbiAgICAgICAgICB9LFxuICAgICAgICB9KTtcblxuICAgICAgICAvLyBUSEVOXG4gICAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpCYXRjaDo6Q29tcHV0ZUVudmlyb25tZW50Jywge1xuICAgICAgICAgIFR5cGU6ICdNQU5BR0VEJyxcbiAgICAgICAgICBTZXJ2aWNlUm9sZToge1xuICAgICAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgICAgICd0ZXN0Y29tcHV0ZWVudlJlc291cmNlU2VydmljZUluc3RhbmNlUm9sZTEwNTA2OUE1JyxcbiAgICAgICAgICAgICAgJ0FybicsXG4gICAgICAgICAgICBdLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sIFJlc291cmNlUGFydC5Qcm9wZXJ0aWVzKSk7XG4gICAgICB9KTtcbiAgICB9KTtcblxuICAgIGRlc2NyaWJlKCd3aXRoIGEgbWluIHZjcHUgdmFsdWUnLCAoKSA9PiB7XG4gICAgICB0ZXN0KCdzaG91bGQgZGVueSBsZXNzIHRoYW4gMCcsICgpID0+IHtcbiAgICAgICAgLy8gVEhFTlxuICAgICAgICB0aHJvd3MoKCkgPT4ge1xuICAgICAgICAgIC8vIFdIRU5cbiAgICAgICAgICBuZXcgYmF0Y2guQ29tcHV0ZUVudmlyb25tZW50KHN0YWNrLCAndGVzdC1jb21wdXRlLWVudicsIHtcbiAgICAgICAgICAgIGNvbXB1dGVSZXNvdXJjZXM6IHtcbiAgICAgICAgICAgICAgdnBjLFxuICAgICAgICAgICAgICBtaW52Q3B1czogLTEsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0pO1xuICAgICAgICB9KTtcbiAgICAgIH0pO1xuXG4gICAgICB0ZXN0KCdjYW5ub3QgYmUgZ3JlYXRlciB0aGFuIHRoZSBtYXggdmNwdSB2YWx1ZScsICgpID0+IHtcbiAgICAgICAgLy8gVEhFTlxuICAgICAgICB0aHJvd3MoKCkgPT4ge1xuICAgICAgICAgIC8vIFdIRU5cbiAgICAgICAgICBuZXcgYmF0Y2guQ29tcHV0ZUVudmlyb25tZW50KHN0YWNrLCAndGVzdC1jb21wdXRlLWVudicsIHtcbiAgICAgICAgICAgIGNvbXB1dGVSZXNvdXJjZXM6IHtcbiAgICAgICAgICAgICAgdnBjLFxuICAgICAgICAgICAgICBtaW52Q3B1czogMixcbiAgICAgICAgICAgICAgbWF4dkNwdXM6IDEsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0pO1xuICAgICAgICB9KTtcbiAgICAgIH0pO1xuICAgIH0pO1xuXG4gICAgZGVzY3JpYmUoJ3dpdGggbm8gbWluIHZjcHUgdmFsdWUgcHJvdmlkZWQnLCAoKSA9PiB7XG4gICAgICB0ZXN0KCdzaG91bGQgZGVmYXVsdCB0byAwJywgKCkgPT4ge1xuICAgICAgICAvLyBXSEVOXG4gICAgICAgIG5ldyBiYXRjaC5Db21wdXRlRW52aXJvbm1lbnQoc3RhY2ssICd0ZXN0LWNvbXB1dGUtZW52Jywge1xuICAgICAgICAgIG1hbmFnZWQ6IHRydWUsXG4gICAgICAgICAgY29tcHV0ZVJlc291cmNlczoge1xuICAgICAgICAgICAgdnBjLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pO1xuXG4gICAgICAgIC8vIFRIRU5cbiAgICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkJhdGNoOjpDb21wdXRlRW52aXJvbm1lbnQnLCB7XG4gICAgICAgICAgLi4uZGVmYXVsdFNlcnZpY2VSb2xlLFxuICAgICAgICAgIC4uLmV4cGVjdGVkTWFuYWdlZERlZmF1bHRDb21wdXRlUHJvcHMoe1xuICAgICAgICAgICAgTWludkNwdXM6IDAsXG4gICAgICAgICAgfSksXG4gICAgICAgIH0sIFJlc291cmNlUGFydC5Qcm9wZXJ0aWVzKSk7XG4gICAgICB9KTtcbiAgICB9KTtcblxuICAgIGRlc2NyaWJlKCd3aXRoIG5vIG1heCB2Y3B1IHZhbHVlIHByb3ZpZGVkJywgKCkgPT4ge1xuICAgICAgdGVzdCgnc2hvdWxkIGRlZmF1bHQgdG8gMjU2JywgKCkgPT4ge1xuICAgICAgICAvLyBXSEVOXG4gICAgICAgIG5ldyBiYXRjaC5Db21wdXRlRW52aXJvbm1lbnQoc3RhY2ssICd0ZXN0LWNvbXB1dGUtZW52Jywge1xuICAgICAgICAgIG1hbmFnZWQ6IHRydWUsXG4gICAgICAgICAgY29tcHV0ZVJlc291cmNlczoge1xuICAgICAgICAgICAgdnBjLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pO1xuXG4gICAgICAgIC8vIFRIRU5cbiAgICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkJhdGNoOjpDb21wdXRlRW52aXJvbm1lbnQnLCB7XG4gICAgICAgICAgLi4uZXhwZWN0ZWRNYW5hZ2VkRGVmYXVsdENvbXB1dGVQcm9wcyh7XG4gICAgICAgICAgICBNYXh2Q3B1czogMjU2LFxuICAgICAgICAgIH0pLFxuICAgICAgICB9LCBSZXNvdXJjZVBhcnQuUHJvcGVydGllcykpO1xuICAgICAgfSk7XG4gICAgfSk7XG5cbiAgICBkZXNjcmliZSgnd2l0aCBubyBpbnN0YW5jZSByb2xlIHNwZWNpZmllZCcsICgpID0+IHtcbiAgICAgIHRlc3QoJ3Nob3VsZCBnZW5lcmF0ZSBhIHJvbGUgZm9yIG1lJywgKCkgPT4ge1xuICAgICAgICAvLyBXSEVOXG4gICAgICAgIG5ldyBiYXRjaC5Db21wdXRlRW52aXJvbm1lbnQoc3RhY2ssICd0ZXN0LWNvbXB1dGUtZW52Jywge1xuICAgICAgICAgIG1hbmFnZWQ6IHRydWUsXG4gICAgICAgICAgY29tcHV0ZVJlc291cmNlczoge1xuICAgICAgICAgICAgdnBjLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pO1xuXG4gICAgICAgIC8vIFRIRU5cbiAgICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6QmF0Y2g6OkNvbXB1dGVFbnZpcm9ubWVudCcpKTtcbiAgICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6SUFNOjpSb2xlJykpO1xuICAgICAgfSk7XG4gICAgfSk7XG5cbiAgICBkZXNjcmliZSgnd2l0aCBubyBpbnN0YW5jZSB0eXBlIGRlZmluZWQnLCAoKSA9PiB7XG4gICAgICB0ZXN0KCdzaG91bGQgZGVmYXVsdCB0byBvcHRpbWFsIG1hdGNoaW5nJywgKCkgPT4ge1xuICAgICAgICAvLyBXSEVOXG4gICAgICAgIG5ldyBiYXRjaC5Db21wdXRlRW52aXJvbm1lbnQoc3RhY2ssICd0ZXN0LWNvbXB1dGUtZW52Jywge1xuICAgICAgICAgIG1hbmFnZWQ6IHRydWUsXG4gICAgICAgICAgY29tcHV0ZVJlc291cmNlczoge1xuICAgICAgICAgICAgdnBjLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pO1xuXG4gICAgICAgIC8vIFRIRU5cbiAgICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkJhdGNoOjpDb21wdXRlRW52aXJvbm1lbnQnLCB7XG4gICAgICAgICAgLi4uZXhwZWN0ZWRNYW5hZ2VkRGVmYXVsdENvbXB1dGVQcm9wcyh7XG4gICAgICAgICAgICBJbnN0YW5jZVR5cGVzOiBbJ29wdGltYWwnXSxcbiAgICAgICAgICB9KSxcbiAgICAgICAgfSwgUmVzb3VyY2VQYXJ0LlByb3BlcnRpZXMpKTtcbiAgICAgIH0pO1xuICAgIH0pO1xuXG4gICAgZGVzY3JpYmUoJ3dpdGggbm8gdHlwZSBzcGVjaWZpZWQnLCAoKSA9PiB7XG4gICAgICB0ZXN0KCdzaG91bGQgZGVmYXVsdCB0byBFQzInLCAoKSA9PiB7XG4gICAgICAgIC8vIFdIRU5cbiAgICAgICAgbmV3IGJhdGNoLkNvbXB1dGVFbnZpcm9ubWVudChzdGFjaywgJ3Rlc3QtY29tcHV0ZS1lbnYnLCB7XG4gICAgICAgICAgbWFuYWdlZDogdHJ1ZSxcbiAgICAgICAgICBjb21wdXRlUmVzb3VyY2VzOiB7XG4gICAgICAgICAgICB2cGMsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgLy8gVEhFTlxuICAgICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6QmF0Y2g6OkNvbXB1dGVFbnZpcm9ubWVudCcsIHtcbiAgICAgICAgICAuLi5leHBlY3RlZE1hbmFnZWREZWZhdWx0Q29tcHV0ZVByb3BzKHtcbiAgICAgICAgICAgIFR5cGU6IGJhdGNoLkNvbXB1dGVSZXNvdXJjZVR5cGUuT05fREVNQU5ELFxuICAgICAgICAgIH0pLFxuICAgICAgICB9LCBSZXNvdXJjZVBhcnQuUHJvcGVydGllcykpO1xuICAgICAgfSk7XG4gICAgfSk7XG4gIH0pO1xufSk7XG4iXX0=