"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.JobDefinition = void 0;
const core_1 = require("@aws-cdk/core");
const batch_generated_1 = require("./batch.generated");
const job_definition_image_config_1 = require("./job-definition-image-config");
/**
 * Batch Job Definition.
 *
 * Defines a batch job definition to execute a specific batch job.
 */
class JobDefinition extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.jobDefinitionName,
        });
        this.imageConfig = new job_definition_image_config_1.JobDefinitionImageConfig(this, props.container);
        const jobDef = new batch_generated_1.CfnJobDefinition(this, 'Resource', {
            jobDefinitionName: props.jobDefinitionName,
            containerProperties: this.buildJobContainer(props.container),
            type: 'container',
            nodeProperties: props.nodeProps
                ? {
                    mainNode: props.nodeProps.mainNode,
                    nodeRangeProperties: this.buildNodeRangeProps(props.nodeProps),
                    numNodes: props.nodeProps.count,
                }
                : undefined,
            parameters: props.parameters,
            retryStrategy: {
                attempts: props.retryAttempts || 1,
            },
            timeout: {
                attemptDurationSeconds: props.timeout ? props.timeout.toSeconds() : undefined,
            },
        });
        this.jobDefinitionArn = this.getResourceArnAttribute(jobDef.ref, {
            service: 'batch',
            resource: 'job-definition',
            resourceName: this.physicalName,
        });
        this.jobDefinitionName = this.getResourceNameAttribute(jobDef.ref);
    }
    /**
     * Imports an existing batch job definition by its amazon resource name.
     *
     * @param scope
     * @param id
     * @param jobDefinitionArn
     */
    static fromJobDefinitionArn(scope, id, jobDefinitionArn) {
        const stack = core_1.Stack.of(scope);
        const jobDefName = stack.parseArn(jobDefinitionArn).resourceName;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.jobDefinitionArn = jobDefinitionArn;
                this.jobDefinitionName = jobDefName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Imports an existing batch job definition by its name.
     * If name is specified without a revision then the latest active revision is used.
     *
     * @param scope
     * @param id
     * @param jobDefinitionName
     */
    static fromJobDefinitionName(scope, id, jobDefinitionName) {
        const stack = core_1.Stack.of(scope);
        const jobDefArn = stack.formatArn({
            service: 'batch',
            resource: 'job-definition',
            sep: '/',
            resourceName: jobDefinitionName,
        });
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.jobDefinitionArn = jobDefArn;
                this.jobDefinitionName = jobDefinitionName;
            }
        }
        return new Import(scope, id);
    }
    deserializeEnvVariables(env) {
        const vars = new Array();
        if (env === undefined) {
            return undefined;
        }
        Object.keys(env).map((name) => {
            vars.push({ name, value: env[name] });
        });
        return vars;
    }
    buildJobContainer(container) {
        if (container === undefined) {
            return undefined;
        }
        return {
            command: container.command,
            environment: this.deserializeEnvVariables(container.environment),
            image: this.imageConfig.imageName,
            instanceType: container.instanceType && container.instanceType.toString(),
            jobRoleArn: container.jobRole && container.jobRole.roleArn,
            linuxParameters: container.linuxParams
                ? { devices: container.linuxParams.renderLinuxParameters().devices }
                : undefined,
            memory: container.memoryLimitMiB || 4,
            mountPoints: container.mountPoints,
            privileged: container.privileged || false,
            resourceRequirements: container.gpuCount
                ? [{ type: 'GPU', value: String(container.gpuCount) }]
                : undefined,
            readonlyRootFilesystem: container.readOnly || false,
            ulimits: container.ulimits,
            user: container.user,
            vcpus: container.vcpus || 1,
            volumes: container.volumes,
        };
    }
    buildNodeRangeProps(multiNodeProps) {
        const rangeProps = new Array();
        for (const prop of multiNodeProps.rangeProps) {
            rangeProps.push({
                container: this.buildJobContainer(prop.container),
                targetNodes: `${prop.fromNodeIndex || 0}:${prop.toNodeIndex || multiNodeProps.count}`,
            });
        }
        return rangeProps;
    }
}
exports.JobDefinition = JobDefinition;
//# sourceMappingURL=data:application/json;base64,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