"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ComputeEnvironment = exports.AllocationStrategy = exports.ComputeResourceType = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const batch_generated_1 = require("./batch.generated");
/**
 * Property to specify if the compute environment
 * uses On-Demand or SpotFleet compute resources.
 */
var ComputeResourceType;
(function (ComputeResourceType) {
    /**
     * Resources will be EC2 On-Demand resources.
     */
    ComputeResourceType["ON_DEMAND"] = "EC2";
    /**
     * Resources will be EC2 SpotFleet resources.
     */
    ComputeResourceType["SPOT"] = "SPOT";
})(ComputeResourceType = exports.ComputeResourceType || (exports.ComputeResourceType = {}));
/**
 * Properties for how to prepare compute resources
 * that are provisioned for a compute environment.
 */
var AllocationStrategy;
(function (AllocationStrategy) {
    /**
     * Batch will use the best fitting instance type will be used
     * when assigning a batch job in this compute environment.
     */
    AllocationStrategy["BEST_FIT"] = "BEST_FIT";
    /**
     * Batch will select additional instance types that are large enough to
     * meet the requirements of the jobs in the queue, with a preference for
     * instance types with a lower cost per unit vCPU.
     */
    AllocationStrategy["BEST_FIT_PROGRESSIVE"] = "BEST_FIT_PROGRESSIVE";
    /**
     * This is only available for Spot Instance compute resources and will select
     * additional instance types that are large enough to meet the requirements of
     * the jobs in the queue, with a preference for instance types that are less
     * likely to be interrupted.
     */
    AllocationStrategy["SPOT_CAPACITY_OPTIMIZED"] = "SPOT_CAPACITY_OPTIMIZED";
})(AllocationStrategy = exports.AllocationStrategy || (exports.AllocationStrategy = {}));
/**
 * Batch Compute Environment.
 *
 * Defines a batch compute environment to run batch jobs on.
 */
class ComputeEnvironment extends core_1.Resource {
    constructor(scope, id, props = { enabled: true, managed: true }) {
        super(scope, id, {
            physicalName: props.computeEnvironmentName,
        });
        this.validateProps(props);
        const spotFleetRole = this.getSpotFleetRole(props);
        let computeResources;
        // Only allow compute resources to be set when using MANAGED type
        if (props.computeResources && this.isManaged(props)) {
            computeResources = {
                allocationStrategy: props.computeResources.allocationStrategy
                    || (props.computeResources.type === ComputeResourceType.SPOT
                        ? AllocationStrategy.SPOT_CAPACITY_OPTIMIZED
                        : AllocationStrategy.BEST_FIT),
                bidPercentage: props.computeResources.bidPercentage,
                desiredvCpus: props.computeResources.desiredvCpus,
                ec2KeyPair: props.computeResources.ec2KeyPair,
                imageId: props.computeResources.image && props.computeResources.image.getImage(this).imageId,
                instanceRole: props.computeResources.instanceRole
                    ? props.computeResources.instanceRole
                    : new iam.CfnInstanceProfile(this, 'Instance-Profile', {
                        roles: [new iam.Role(this, 'Ecs-Instance-Role', {
                                assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
                                managedPolicies: [
                                    iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2ContainerServiceforEC2Role'),
                                ],
                            }).roleName],
                    }).attrArn,
                instanceTypes: this.buildInstanceTypes(props.computeResources.instanceTypes),
                launchTemplate: props.computeResources.launchTemplate,
                maxvCpus: props.computeResources.maxvCpus || 256,
                minvCpus: props.computeResources.minvCpus || 0,
                securityGroupIds: this.buildSecurityGroupIds(props.computeResources.vpc, props.computeResources.securityGroups),
                spotIamFleetRole: spotFleetRole ? spotFleetRole.roleArn : undefined,
                subnets: props.computeResources.vpc.selectSubnets(props.computeResources.vpcSubnets).subnetIds,
                tags: props.computeResources.computeResourcesTags,
                type: props.computeResources.type || ComputeResourceType.ON_DEMAND,
            };
        }
        const computeEnvironment = new batch_generated_1.CfnComputeEnvironment(this, 'Resource', {
            computeEnvironmentName: this.physicalName,
            computeResources,
            serviceRole: props.serviceRole
                ? props.serviceRole.roleArn
                : new iam.Role(this, 'Resource-Service-Instance-Role', {
                    managedPolicies: [
                        iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSBatchServiceRole'),
                    ],
                    assumedBy: new iam.ServicePrincipal('batch.amazonaws.com'),
                }).roleArn,
            state: props.enabled === undefined ? 'ENABLED' : (props.enabled ? 'ENABLED' : 'DISABLED'),
            type: this.isManaged(props) ? 'MANAGED' : 'UNMANAGED',
        });
        if (props.computeResources && props.computeResources.vpc) {
            this.node.addDependency(props.computeResources.vpc);
        }
        this.computeEnvironmentArn = this.getResourceArnAttribute(computeEnvironment.ref, {
            service: 'batch',
            resource: 'compute-environment',
            resourceName: this.physicalName,
        });
        this.computeEnvironmentName = this.getResourceNameAttribute(computeEnvironment.ref);
    }
    /**
     * Fetches an existing batch compute environment by its amazon resource name.
     *
     * @param scope
     * @param id
     * @param computeEnvironmentArn
     */
    static fromComputeEnvironmentArn(scope, id, computeEnvironmentArn) {
        const stack = core_1.Stack.of(scope);
        const computeEnvironmentName = stack.parseArn(computeEnvironmentArn).resourceName;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.computeEnvironmentArn = computeEnvironmentArn;
                this.computeEnvironmentName = computeEnvironmentName;
            }
        }
        return new Import(scope, id);
    }
    isManaged(props) {
        return props.managed === undefined ? true : props.managed;
    }
    /**
     * Validates the properties provided for a new batch compute environment.
     */
    validateProps(props) {
        if (props === undefined) {
            return;
        }
        if (!this.isManaged(props) && props.computeResources !== undefined) {
            throw new Error('It is not allowed to set computeResources on an AWS unmanaged compute environment');
        }
        if (this.isManaged(props) && props.computeResources === undefined) {
            throw new Error('computeResources is missing but required on a managed compute environment');
        }
        // Setting a bid percentage is only allowed on SPOT resources +
        // Cannot use SPOT_CAPACITY_OPTIMIZED when using ON_DEMAND
        if (props.computeResources) {
            if (props.computeResources.type === ComputeResourceType.ON_DEMAND) {
                // VALIDATE FOR ON_DEMAND
                // Bid percentage is not allowed
                if (props.computeResources.bidPercentage !== undefined) {
                    throw new Error('Setting the bid percentage is only allowed for SPOT type resources on a batch compute environment');
                }
                // SPOT_CAPACITY_OPTIMIZED allocation is not allowed
                if (props.computeResources.allocationStrategy && props.computeResources.allocationStrategy === AllocationStrategy.SPOT_CAPACITY_OPTIMIZED) {
                    throw new Error('The SPOT_CAPACITY_OPTIMIZED allocation strategy is only allowed if the environment is a SPOT type compute environment');
                }
            }
            else {
                // VALIDATE FOR SPOT
                // Bid percentage must be from 0 - 100
                if (props.computeResources.bidPercentage !== undefined &&
                    (props.computeResources.bidPercentage < 0 || props.computeResources.bidPercentage > 100)) {
                    throw new Error('Bid percentage can only be a value between 0 and 100');
                }
            }
            if (props.computeResources.minvCpus) {
                // minvCpus cannot be less than 0
                if (props.computeResources.minvCpus < 0) {
                    throw new Error('Minimum vCpus for a batch compute environment cannot be less than 0');
                }
                // minvCpus cannot exceed max vCpus
                if (props.computeResources.maxvCpus &&
                    props.computeResources.minvCpus > props.computeResources.maxvCpus) {
                    throw new Error('Minimum vCpus cannot be greater than the maximum vCpus');
                }
            }
        }
    }
    buildInstanceTypes(instanceTypes) {
        if (instanceTypes === undefined) {
            return [
                'optimal',
            ];
        }
        return instanceTypes.map((type) => type.toString());
    }
    buildSecurityGroupIds(vpc, securityGroups) {
        if (securityGroups === undefined) {
            return [
                new ec2.SecurityGroup(this, 'Resource-Security-Group', { vpc }).securityGroupId,
            ];
        }
        return securityGroups.map((group) => group.securityGroupId);
    }
    /**
     * Generates an AWS IAM role for provisioning spotfleet resources
     * if the allocation strategy is set to BEST_FIT or not defined.
     *
     * @param props - the compute environment construct properties
     */
    getSpotFleetRole(props) {
        var _a;
        if (((_a = props.computeResources) === null || _a === void 0 ? void 0 : _a.allocationStrategy) && props.computeResources.allocationStrategy !== AllocationStrategy.BEST_FIT) {
            return undefined;
        }
        if (props.computeResources) {
            if (props.computeResources.spotFleetRole) {
                return props.computeResources.spotFleetRole;
            }
            else if (props.computeResources.type === ComputeResourceType.SPOT) {
                return iam.Role.fromRoleArn(this, 'Resource-SpotFleet-Role', `arn:${this.stack.partition}:iam::${this.stack.account}:role/aws-service-role/spotfleet.amazonaws.com/AWSServiceRoleForEC2SpotFleet`);
            }
        }
        return undefined;
    }
}
exports.ComputeEnvironment = ComputeEnvironment;
//# sourceMappingURL=data:application/json;base64,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