# -*- coding: utf-8 -*-
"""
    pip_services3_commons.validate.ValidationResult
    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    Validation result implementation

    :copyright: Conceptual Vision Consulting LLC 2018-2019, see AUTHORS for more details.
    :license: MIT, see LICENSE for more details.
"""
from typing import Any, Dict

from ..convert.JsonConverter import JsonConverter
from ..validate import ValidationResultType


class ValidationResult:
    """
    Result generated by schema validation
    """

    def __init__(self, path: str = None, type: ValidationResultType = None, code: str = None, message: str = None,
                 expected: Any = None, actual: Any = None):
        """
        Creates a new instance of validation ressult and sets its values.

        :param path: a dot notation path of the validated element.

        :param type: a type of the validation result: Information, Warning, or Error.

        :param code: an error code.

        :param message: a human readable message.

        :param expected: an args expected by schema validation.

        :param actual: an actual args found by schema validation.
        """
        self.__path: str = path
        self.__type: ValidationResultType = type
        self.__code: str = code
        self.__message: str = message
        self.__expected: Any = expected
        self.__actual: Any = actual

    def __str__(self):
        return JsonConverter.to_json(self.to_json())

    def get_path(self) -> str:
        """
        Gets dot notation path of the validated element.

        :return: the path of the validated element.
        """
        return self.__path

    def get_type(self) -> ValidationResultType:
        """
        Gets the type of the validation result: Information, Warning, or Error.
        See :class: `ValidationResultType`

        :return: the type of the validation result.
        """
        return self.__type

    def get_code(self) -> str:
        """
        Gets the error code.

        :return: the error code
        """
        return self.__code

    def get_message(self) -> str:
        """
        Gets the human readable message.

        :return: the result message.
        """
        return self.__message

    def get_expected(self) -> Any:
        """
        Gets the args expected by schema validation.

        :return: the expected args.
        """
        return self.__expected

    def get_actual(self) -> Any:
        """
        Gets the actual args found by schema validation.

        :return: the actual args.
        """
        return self.__actual

    def to_json(self) -> Dict[str, Any]:
        return {
            'path': self.__path,
            'type': self.__type,
            'code': self.__code,
            'message': self.__message,
            'expected': self.__expected,
            'actual': self.__actual
        }
